/*
 * 	  Created on 09-Sep-2003
 * 
 *    ALMA - Atacama Large Millimiter Array
 *    (c) European Southern Observatory, 2002
 *    Copyright by ESO (in the framework of the ALMA collaboration),
 *    All rights reserved
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License as published by the Free Software Foundation; either
 *    version 2.1 of the License, or (at your option) any later version.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
 *    MA 02111-1307  USA
 */
package alma.archive.database.xmldb;


import java.net.URI;
import java.util.Iterator;
import java.util.Vector;
import java.util.logging.Logger;

import org.xmldb.api.base.Resource;
import org.xmldb.api.base.ResourceIterator;
import org.xmldb.api.base.ResourceSet;
import org.xmldb.api.base.XMLDBException;

import alma.archive.database.helpers.DBConfiguration;
import alma.archive.database.interfaces.DBCursor;
import alma.archive.exceptions.ArchiveException;
import alma.archive.exceptions.cursor.CursorClosedException;
import alma.archive.exceptions.general.DatabaseException;
import alma.archive.wrappers.ResultStruct;

/**
 * @author simon
 * 
 * This is the XML:DB implementation of the Database DBCursor class.
 * The main function of this class is to manage and filter results returned
 * by an XPath query. When a query is presented to the database, results for
 * all of the schema are returned, this class reduces the results to just those
 * associated with the appropriate schema.
 */
public class XmldbCursor implements DBCursor
{
	private DBConfiguration dbConfig;
	private XmldbConnector connector;
	
	private XmldbDatabase database;
	
	private String schema = "";
	private String user = "";
	private boolean dirtyRead;
	
	private Vector resourceList = null;
	private Vector uriList = null;
	private Iterator resourceIterator;
	
	/**
	 * Create a cursor, requires a ResourceSet returned by a query.
	 * The ResourceSet is filtered by the constructor and all of
	 * the resources extracted to a local Vector.
	 * @param database requires access to the database to fetch filtering
	 * 		information
	 * @param resourceset the results ResourceSet
	 * @param schema the name od the schema used to filter the results
	 * @param dirtyRead if true allow results flagged as dirty
	 * @param allRead allows all results to be read regardless of flags
	 * @param user used to find user read permissions
	 * @param logger
	 * @throws XMLDBException
	 * @throws DatabaseException
	 */
	public XmldbCursor(
			XmldbDatabase database, 
			ResourceSet resourceset, 
			String schema, 
			boolean dirtyRead, 
			boolean allRead, 
			String user,
			Logger logger) 
				throws 
					XMLDBException,
					DatabaseException
	{
		this.schema = schema;
		this.database = database;
		this.user = user;
		this.dirtyRead = dirtyRead;
		
		if (dbConfig == null) 
		{
			dbConfig=DBConfiguration.instance(logger);
		}
		
		connector = XmldbConnector.instance(logger,dbConfig.testMode);
		
		resourceList = new Vector();
		uriList = new Vector();
		
		ResourceIterator iter = resourceset.getIterator();
		while (iter.hasMoreResources())
		{
			Resource current = iter.nextResource();
			URI currentUri = connector.extractURI(current);
			if (filter(currentUri,allRead))
			{
				resourceList.add(current);
				uriList.add(currentUri);
			}
		}
		this.resourceIterator = resourceList.iterator();
	}
	
	/**
	 * Used to filter the results and allow only those that pass into the
	 * results vector. It is important to note that when the Database is in
	 * test mode only those results that are in the test range are allowed.
	 * @param uri
	 * @param allRead
	 * @return
	 */
	private boolean filter(URI uri, boolean allRead)
		throws
			DatabaseException
	{
		if (database.descriptorExists(uri))
		{
			DocumentDataXml documentData = database.getDescriptor(uri);
			if (!documentData.getSchemaName().equalsIgnoreCase(schema)) return false;
			//When in test mode only the results in the test range are allowed
			if (!allRead)
			{
				if (documentData.isDeleted()) return false;
				if (documentData.isDirty() && !dirtyRead) return false;
				if (!documentData.checkReadPermision(user)) return false;
			}
			return true;
		}
		else return false;
	}

	/**
	 * Returns true if the iteration has more elements.
	 * @throws DatabaseException
	 * @throws ArchiveException
	 */
	public boolean hasNext() 
		throws 
			DatabaseException, 
			ArchiveException
	{
		if (resourceIterator == null) {return false;} 
		else {return resourceIterator.hasNext();} 
	}
	
	/**
	 * Returns the next ResultStruct, will throw and exception if they run out 
	 */
	public ResultStruct next() 
		throws 
			CursorClosedException, 
			DatabaseException, 
			ArchiveException
	{
		if (resourceIterator == null) throw new CursorClosedException(
			"The cursor has been closed, or there are no more results");
		if (resourceIterator.hasNext())
		{
			try
			{
				Resource current = (Resource)resourceIterator.next();
				String content = (String)current.getContent();
				URI currentUri = connector.extractURI(current); 
				content = connector.cleanResult(content);
				ResultStruct struct = new ResultStruct(currentUri,content);
				return struct;
			}
			catch (XMLDBException e)
			{
				throw new DatabaseException(
					"Problems accessing the resource iterator",e);
			}
		}
		else
		{
			resourceIterator = null;
			throw new CursorClosedException(
				"There are no more results in the Cursor");
		}
	}

	/**
	 * Returns an array of the size give in blockSize, if there are
	 * not enough entries the remainder will be null
	 */
	public ResultStruct[] nextBlock(short blockSize) 
		throws 
			CursorClosedException, 
			DatabaseException, 
			ArchiveException
	{
		ResultStruct[] res = new ResultStruct[blockSize];
		for (int x = 0; x < blockSize; x++)
		{
			if (this.hasNext())
			{
				res[x] = this.next();
			}
			else res[x]=null;
		}
		return res;
	}
	
	/**
	 * Returns a list of the URI's from the cursor, this may be a very
	 * large array.
	 * @return
	 */
	public URI[] uriList()
	{
		URI[] list = new URI[uriList.size()];
		list = (URI[])uriList.toArray(list);	
		return list;
	}

	/**
	 * Closes the TestManager and sets the resource iterator to null
	 */
	public void close() 
		throws 
			CursorClosedException, 
			DatabaseException, 
			ArchiveException
	{
		resourceIterator = null;
	}
}
