/*
 * 	  Created on 19-Feb-2004
 * 
 *    ALMA - Atacama Large Millimiter Array
 *    (c) European Southern Observatory, 2002
 *    Copyright by ESO (in the framework of the ALMA collaboration),
 *    All rights reserved
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License as published by the Free Software Foundation; either
 *    version 2.1 of the License, or (at your option) any later version.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
 *    MA 02111-1307  USA
 */
package alma.archive.database.xmldb;

import java.net.URI;
import java.net.URISyntaxException;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.xmldb.api.DatabaseManager;
import org.xmldb.api.base.Collection;
import org.xmldb.api.base.Database;
import org.xmldb.api.base.ErrorCodes;
import org.xmldb.api.base.Resource;
import org.xmldb.api.base.XMLDBException;
import org.xmldb.api.modules.CollectionManagementService;

import alma.archive.database.helpers.DBConfiguration;
import alma.archive.exceptions.general.DatabaseException;

/**
 * @author simon
 * 
 * This class is used to connect to the differnt kinds of XML:DB Database. Although
 * the differnt implementations of XML:DB are similar there are some differnces that
 * need to be taken into account. Hopefully at some point in the future it should be
 * possible just to specify a location and a driver and connect to any database.
 * 
 * This class is implemented as a singleton but is used in very close connection with
 * the XmldbDatabase class, all of the closing of reaources is actually perfomed by
 * the XmldbDatabase class and not in this connector. The connector exists purely to
 * consolidate differnt implementations of the XML:DB standard.
 */
public class XmldbConnector
{
	public static DBConfiguration dbConfig;
	
	private Logger logger;
	
	private final String location;
	private final String name;
	private final String driver;
	
	private Database database = null;
	
	private boolean testMode;
	
	private static XmldbConnector _instance = null;
	private static int access_count = 0;
	
	/**
	 * Returns an instance of the Connector
	 * @param logger
	 * @return
	 * @throws DatabaseException
	 * @throws XMLDBException
	 */
	public static XmldbConnector instance(Logger logger, boolean testMode)
		throws 
			DatabaseException, 
			XMLDBException
	{
		 if (_instance == null)
		 {
		 	_instance = new XmldbConnector(logger,testMode);
		 }
		 access_count++;
		 return _instance;
	}
	
	/**
	 * Constructor for the connector
	 * @param logger a Logger for any messages
	 * @throws XMLDBException thrown when there are problems connecting
	 * 			to the database
	 * @throws DatabaseException thrown when no configuration information
	 * 			is found
	 */
	protected XmldbConnector(Logger logger, boolean testMode) 
		throws 
			XMLDBException, 
			DatabaseException
	{
		this.logger = logger;
		this.testMode = testMode;
		if (testMode){
			logger.info("***Archive running in test mode***");
		}else{
			logger.info("***Archive running in operational mode***");
		}
		dbConfig = DBConfiguration.instance(logger);
		location = dbConfig.get("archive.xmldb.location");
		logger.info("Archive location: " + location);
		if ((location == null) || (location.equalsIgnoreCase(""))){
			throw new DatabaseException(
				"The archive.xmldb.location has not been set");
		}
		name = dbConfig.get("archive.xmldb.name");
		logger.info("Archive name: " + name);
		if ((name == null) || (name.equalsIgnoreCase(""))){
			throw new DatabaseException(
				"The archive.xmldb.name config has not been set");
		}
		driver = dbConfig.get("archive.xmldb.driver");
		logger.info("Archive driver: " + driver);
		if ((driver == null) || (driver.equalsIgnoreCase(""))){
			throw new DatabaseException(
				"The archive.xmldb.driver config has not been set");
		}
		registerDB();
	}
	
	/**
	 * Looks at the configuration information and configures the appropriate
	 * driver
	 * @throws XMLDBException
	 * @throws DatabaseException
	 */
	private void registerDB() 
		throws 
			XMLDBException, 
			DatabaseException
	{
		if (database == null)
		{	
			try
			{
				Class c= Class.forName(driver);
				if (c == null){
					throw new DatabaseException("Could not find the driver: " +
						driver);
				}
				database= (Database) c.newInstance();
				database.setProperty("create-database","true");
			}
			catch (Exception e)
			{
				throw new DatabaseException("Could not get driver "+driver+":\n"+e);
				//MPA: no logging
			}
			DatabaseManager.registerDatabase(database);
		}
	}

	/**
	 * This will return the top level collection for the database.
	 * @return
	 * @throws XMLDBException
	 */
	private Collection getTopCollection() 
		throws 
			XMLDBException
	{
		try
		{	
			Collection topCollection = DatabaseManager.getCollection(
				location + "/" + name);
			
			if (topCollection != null)
			{
				return topCollection;
			}
			else throw new XMLDBException();
		}
		catch (XMLDBException e)
		{
			logger.log(Level.SEVERE, 
				"Could not find top level collection: " 
				+ location + "/" + name + "::" + e.errorCode);
			e.printStackTrace();
			throw new XMLDBException(
				ErrorCodes.INVALID_COLLECTION,
				"Could not find top level collection: " 
				+ location + "/" + name);
			//MPA: is it not better to also pass the exception e ?
			
		}
	}
	
	/**
	 * Opens and returns the requested collection. If the collection
	 * cannot be found it will attempt to create a new collection
	 * @param cn the name of the collection to be opened
	 * @throws XMLDBException
	 */
	private Collection getCollection(String cn) 
		throws 
			XMLDBException
	{	
		Collection topCollection = getTopCollection();
		Collection collection = topCollection.getChildCollection(cn);
		//Try to create the collection
		if (collection == null)
		{
			CollectionManagementService service =
				(org.xmldb.api.modules.CollectionManagementService)
				topCollection.getService(
					"CollectionManagementService", "1.0");
			return service.createCollection(cn);
		}
		else
		{
			return collection;
		}
	}
	
	private final String _descriptor = "descriptor";
	private final String _latest = "latest";
	private final String _documents = "documents";
	
	/**
	 * Returns an instance of the Descriptor collection
	 * @return
	 * @throws XMLDBException
	 */
	public Collection getDescriptor() 
		throws 
			XMLDBException
	{
		if (testMode){
			return getCollection(_descriptor + "test");
		}
		else{
			return getCollection(_descriptor);
		}
	}
	
	/**
	 * Returns and instance of the Latest collection
	 * @return
	 * @throws XMLDBException
	 */
	public Collection getLatest() 
		throws 
			XMLDBException
	{
		if (testMode){
			return getCollection(_latest + "test");
		}
		else{
			return getCollection(_latest);
		}
	}
	
	/**
	 * Returns an instance of the Documents collection
	 * @return
	 * @throws XMLDBException
	 */
	public Collection getDocuments() 
		throws 
			XMLDBException
	{
		if (testMode){
			return getCollection(_documents + "test");
		}
		else{
			return getCollection(_documents);
		}
	}
	
	/**
	 * Returns a specified collection
	 * @param name
	 * @return
	 * @throws XMLDBException
	 */
	public Collection getOther(String name)
		throws
			XMLDBException
	{
		if (testMode){
			return getCollection(name + "test");
		}
		else{
			return getCollection(name);
		}
	}
	
	/**
	 * This will erase the contents of all of the test collections.
	 * @throws DatabaseException
	 */
	public void cleanTestCollections() 
		throws 
			DatabaseException
	{
		try
		{
			Collection testDescriptor = getCollection(_descriptor + "test");
			clean(testDescriptor);
			shutdownCollection(testDescriptor);
			Collection testLatest = getCollection(_latest + "test");
			clean(testLatest);
			shutdownCollection(testLatest);
			Collection testDocuments = getCollection(_documents + "test");
			clean(testDocuments);
			shutdownCollection(testDocuments);
		}
		catch (XMLDBException e){
			throw new DatabaseException(e);
		}
	}
	
	/**
	 * Erases the contents of the operational database
	 * @throws DatabaseException
	 */
	public void cleanCollections() 
		throws 
			DatabaseException
	{
		try
		{
			logger.info("Cleaning collections! Mode="+(testMode ? "test" : "operational"));
			Collection descriptor = getDescriptor();
			clean(descriptor);
			shutdownCollection(descriptor);
			Collection latest = getLatest();
			clean(latest);
			shutdownCollection(latest);
			Collection documents = getDocuments();
			clean(documents);
			shutdownCollection(documents);
			XmldbDatabase.instance(logger).refresh();
			latest=getDescriptor();
		}
		catch (XMLDBException e){
			throw new DatabaseException(e);
		}
	}
	
	/**
	 * Erases the contents of a particular collection
	 * @param col
	 * @throws XMLDBException
	 */
	private void clean(Collection col) 
		throws 
			XMLDBException
	{
		String[] res = col.listResources();
		for (int x = 0; x < res.length; x++){
			col.removeResource(col.getResource(res[x]));
		}
	}
	
	/**
	 * Uses the appropriate DatabaseInstanceManager for the database to close
	 * the collection. 
	 * @param col the collection ot be closed
	 * @throws XMLDBException
	 */
	public void shutdownCollection(Collection col) 
		throws 
			DatabaseException
	{	
		/*
		 * Unnecessary now that a remote server is being used, must be careful
		 * if a local implementation is being used.
		try
		{
			logger.info("Shuting down collection " + 
				col.getName());
			
			String driver = dbConfig.get("archive.xmldb.driver");
			if (driver.equals("org.apache.xindice.client.xmldb.DatabaseImpl")){
				org.apache.xindice.client.xmldb.services.DatabaseInstanceManager m=
					(org.apache.xindice.client.xmldb.services.DatabaseInstanceManager)
					col.getService("DatabaseInstanceManager","1.0");
				m.shutdown();
			}
			else if (dbConfig.dbBackend.equalsIgnoreCase("xindice")){
				org.apache.xindice.client.xmldb.services.DatabaseInstanceManager m=
					(org.apache.xindice.client.xmldb.services.DatabaseInstanceManager)
					col.getService("DatabaseInstanceManager","1.0");
				m.shutdown();
			} 
			else if (driver.equals("org.exist.xmldb.DatabaseImpl")){
				org.exist.xmldb.RemoteDatabaseInstanceManager m=
					(org.exist.xmldb.RemoteDatabaseInstanceManager)
					col.getService("DatabaseInstanceManager","1.0");
				m.shutdown();
			}
			else if (dbConfig.dbBackend.equalsIgnoreCase("exist")){
				org.exist.xmldb.RemoteDatabaseInstanceManager m=
					(org.exist.xmldb.RemoteDatabaseInstanceManager)
					col.getService("DatabaseInstanceManager","1.0");
				m.shutdown();
			}
			else {
				throw new DatabaseException("The XMLDB driver class: " + 
					driver + " is not recognised ");
			}
		}
		catch (XMLDBException e)
		{
			throw new DatabaseException(e);
		}
		*/
	}
	
	/**
	 * Used to extract the URI from a given resource 
	 * @param resource
	 * @return
	 * @throws XMLDBException
	 * @throws DatabaseException
	 */
	public URI extractURI(Resource resource) 
		throws 
			DatabaseException
	{
		try
		{
			String xmldbDriver = dbConfig.get("archive.xmldb.driver");
			if (xmldbDriver.equals("org.exist.xmldb.DatabaseImpl")){
				return extractExist(resource);
			}
			else if (dbConfig.dbBackend.equalsIgnoreCase("exist")){
				return extractExist(resource);
			}
			else {
				throw new DatabaseException("The XMLDB driver class: " + xmldbDriver + " is not recognised ");
			}
		}
		catch (XMLDBException e)
		{
			throw new DatabaseException(e);
		}
	}
	
	
	/**
	 * Extract the URI from an eXist resource
	 * @param resource
	 * @return
	 * @throws XMLDBException
	 * @throws DatabaseException
	 */
	private URI extractExist(Resource resource) 
		throws 
			XMLDBException, 
			DatabaseException
	{
		try
		{
			org.exist.xmldb.RemoteXMLResource res = (org.exist.xmldb.RemoteXMLResource)resource;
			String id = (String)res.getId();
			id = id.replaceAll("schemeSeperator","://");
			id = id.replaceAll("slash","/");
			id = id.replaceAll("hash","#");
			
			int lastus = id.indexOf("_");
			if (lastus != -1){
				id = id.substring(0,lastus);
			}
			return new URI(id);
		}
		catch (URISyntaxException e)
		{
			throw new DatabaseException(e);
		}
	}
	
	/**
	 * Used to remove artifacts added to the XML by the database
	 * @param str
	 * @return
	 * @throws XMLDBException
	 */
	public String cleanResult(String str) throws DatabaseException
	{
		String dbdriver = dbConfig.get("archive.xmldb.driver");
		if (dbdriver.equals("org.exist.xmldb.DatabaseImpl")){
			return cleanExist(str);
		}
		else if (dbConfig.dbBackend.equalsIgnoreCase("exist")){
			return cleanExist(str);
		}
		else {
			throw new DatabaseException("The XMLDB driver class: " + 
					dbdriver + " is not recognised ");
		}
	}
	
	
	/**
	 * Originally meant to remove artifacts added to the XML by xindice (see CVS history).
	 * TODO: figure out if exist needs anything like this.
	 */
	private String cleanExist(String str)
	{
		return str;
	}
}
