/*
 * 	  Created on 24-May-2004
 * 
 *    ALMA - Atacama Large Millimiter Array
 *    (c) European Southern Observatory, 2002
 *    Copyright by ESO (in the framework of the ALMA collaboration),
 *    All rights reserved
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License as published by the Free Software Foundation; either
 *    version 2.1 of the License, or (at your option) any later version.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
 *    MA 02111-1307  USA
 */
package alma.archive.database.vdoc;

import java.io.IOException;
import java.io.StringReader;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Set;
import java.util.logging.Logger;

import org.jdom.Document;
import org.jdom.Element;
import org.jdom.JDOMException;
import org.jdom.Namespace;
import org.jdom.input.SAXBuilder;
import org.jdom.output.Format;
import org.jdom.output.XMLOutputter;
import org.jdom.xpath.XPath;
import org.xmldb.api.base.XMLDBException;

import alma.archive.database.interfaces.IdentifierManager;
import alma.archive.database.interfaces.InternalIF;
import alma.archive.database.interfaces.InternalIFFactory;
import alma.archive.database.interfaces.SchemaManager;
import alma.archive.database.xmldb.XmldbDatabase;
import alma.archive.database.xmldb.XmldbSchemaManager;
import alma.archive.exceptions.ArchiveException;
import alma.archive.exceptions.ArchiveGeneralException;
import alma.archive.exceptions.ModuleCriticalException;
import alma.archive.exceptions.access.EntityDirtyException;
import alma.archive.exceptions.access.PermissionDeniedException;
import alma.archive.exceptions.general.DatabaseException;
import alma.archive.exceptions.general.EntityDoesNotExistException;
import alma.archive.exceptions.general.HistoryInconsistencyException;
import alma.archive.exceptions.general.UnknownSchemaException;
import alma.archive.exceptions.general.VDocException;
import alma.archive.exceptions.identifier.LocalRangeExceededException;
import alma.archive.exceptions.syntax.IllegalHistoryNumberException;
import alma.archive.exceptions.syntax.MalformedURIException;
import alma.archive.exceptions.syntax.MalformedXMLException;
import alma.archive.exceptions.user.UserDoesNotExistException;
import alma.archive.wrappers.ArchiveTimeStamp;
import alma.archive.wrappers.DocumentData;
import alma.archive.wrappers.Permissions;

import alma.acs.container.archive.Range;
import alma.JavaContainerError.wrappers.AcsJContainerServicesEx;
import alma.acs.container.archive.UIDLibrary;
import alma.ArchiveIdentifierError.wrappers.AcsJUidAlreadyExistsEx;
import alma.ArchiveIdentifierError.wrappers.AcsJIdentifierUnavailableEx;
import alma.ArchiveIdentifierError.wrappers.AcsJRangeUnavailableEx;
import alma.ArchiveIdentifierError.wrappers.AcsJIdentifierUnexpectedEx;

/**
 * @author simon
 * 
 * The VDocManager provides a unified interface for dealing with Virtual 
 * Documents and other associated items. It mostly exists to keep track of 
 * Contexts and allow the editing and updating of VDocs.
 */
public class VDocManager
{
	
	private static VDocManager _instance = null;
	private static int access_count = 0;
	
	private static final String _vdocmanager = "archive://vdocmanager";
	private URI vdocmanager = null;
	
	/**
	 * Return a pointer to the VDocManager, the VDocManager is implemented as a
	 * singleton class. All of the subsystem interfaces are passed here to allow
	 * the setup to use proxy versions of the interface. The first one passed 
	 * will be used later calls will not affect the interfaces used. 
	 * @param internal
	 * @param identifier
	 * @param logger
	 * @return
	 * @throws VDocException
	 */
	public static VDocManager instance(
			InternalIF internal, 
			IdentifierManager identifier, 
			Logger logger)
		throws 
			VDocException
	{
		if (_instance == null)
		{
			_instance = new VDocManager(internal,identifier,logger);
		}
		access_count++;
		return _instance;
	}
	
	/**
	 * Close the VDocManager to free resource, it is vital that this is called.
	 * @throws DatabaseException
	 * @throws ModuleCriticalException
	 */
	public void close() 
		throws 
			DatabaseException, 
			ModuleCriticalException
	{
		access_count--;
		if (access_count == 0)
		{
			identifier.close();
			_instance = null;
		}
	}
	
	private InternalIF internal = null;
	private IdentifierManager identifier = null;
	private Logger logger = null;
	
	private Element definitions = null;
	private Element vdocContexts = null;
	private Element definitionContexts = null;
	private Element vdocs = null;
	private Element root = null;
	private Range range;
	
	/**
	 * Used to create an instance of the VDocManager, connects to the subsystem
	 * interfaces. Checks to see if an existing XML document defing it's
	 * structure is in the Database, if not it will create an empty one and
	 * store it in the Database.
	 * @param internal
	 * @param identifier
	 * @param logger
	 * @throws VDocException
	 */
	private VDocManager(
			InternalIF internal, 
			IdentifierManager identifier, 
			Logger logger)
		throws 
			VDocException
	{
		this.internal = internal;
		this.identifier = identifier;
		this.logger = logger;
		
		try {
		range = new Range(identifier.getNewRange());
		} catch (Exception e) {
			throw new VDocException(e);
		}
		
		try{vdocmanager = new URI(_vdocmanager);}
		catch (URISyntaxException e){throw new VDocException(e);}
		//MPA: no logging
		
		try
		{	
			
			String xml = internal.get(vdocmanager,"VDocManager");
			root = createElement(xml);
			definitions = root.getChild("definitions");
			vdocContexts = root.getChild("vdocContexts");
			definitionContexts = root.getChild("definitionContexts");
			vdocs = root.getChild("vdocs");
		}
		catch (EntityDoesNotExistException e)
		{
			root = new Element("vdocmanager");
			definitions = new Element("definitions");
			vdocContexts = new Element("vdocContexts");
			definitionContexts = new Element("definitionContexts");
			vdocs = new Element("vdocs");
			root.addContent(definitions);
			root.addContent(vdocContexts);
			root.addContent(definitionContexts);
			root.addContent(vdocs);
			
			/*
			internal.store(
				vdocmanager,
				this.toString(),
				testns,
				"test",
				"VDocManager",
				new Permissions(),
				"VDocManager");
			*/
			//MPA: logging missing
		}
		catch (ArchiveException e){
			throw new VDocException(e);
		}
		catch (ModuleCriticalException e){
			throw new VDocException(e);
		}
		//MPA: logging missing
	}
	
	/**
	 * Returns a JODM Element from any given valid XML string.
	 * @param xml
	 * @return
	 * @throws VDocException
	 */
	private Element createElement(String xml) 
		throws 
			VDocException
	{
		SAXBuilder builder = new SAXBuilder();
		builder.setIgnoringElementContentWhitespace(true);
		try
		{
			Document vdocdoc = builder.build(
				new StringReader(xml));
			Element root = vdocdoc.getRootElement();
			return root;
		}
		catch (JDOMException e) {throw new VDocException(e);}
		catch (IOException e) {throw new VDocException(e);}
		//MPA: logging missing
	}
	
	/*
	private void save() 
		throws 
			VDocException
	{
		update(vdocmanager,this.toString(),"VDocManager");
	}
	*/
	
	private void save()
	{
		print();
	}
	
	
	private void print()
	{
		System.out.println(this.toString());
	}
	
	public String toString()
	{
		XMLOutputter out = new XMLOutputter(Format.getPrettyFormat());
		return out.outputString(root);
	}
	
	/**
	 * locally caches a UID and increments the local part, this is to conserve
	 * UIDs. Watches to make sure that the local namespace doesn't go out of
	 * range.
	 * @return
	 * @throws VDocException
	 */
	private URI getId() 
		throws 
			VDocException
	{
		try {
		return range.next();
		} catch (Exception e) {
			throw new VDocException(e);
		}
		}

	/**
	 * Registers an Entity as a virtual document. All this method really does
	 * is set the virtual flag associated with the docuemnt to true
	 * @param uri The UID of the entity to be registered as virtual
	 * @throws VDocException
	 */
	public void registerVDoc(URI uri, String user) 
		throws 
			VDocException
	{
		try
		{
			internal.setVirtual(uri,user,true);
		}
		catch (ModuleCriticalException e){
			throw new VDocException(e);
//			MPA: no logging
		}
		catch (ArchiveException e){
			throw new VDocException(e);
//			MPA: no logging
		}
	}
	
	/**
	 * Unregisters an Entity as a virtual document. All this method really does 
	 * is set the virtual flag associated with the docuemnt to false
	 * @param uri The UID of the entity to be unregistered as virtual
	 * @throws VDocException
	 */
	public void unregisterVDoc(URI uri, String user) 
		throws 
			VDocException
	{
		try
		{
			internal.setVirtual(uri,user,false);
		}
		catch (ModuleCriticalException e){
			throw new VDocException(e);
//			MPA: no logging
		}
		catch (ArchiveException e){
			throw new VDocException(e);
//			MPA: no logging
		}
	}
	
	/**
	 * Creates and stores contexts for either VDocs or Definitions.
	 * @param name
	 * @param description
	 * @param xml
	 * @param user
	 * @param owner
	 * @param parent
	 * @param type
	 * @param contextSchema
	 * @return
	 * @throws VDocException
	 */
	private URI addContext(
			String name, 
			String description, 
			String xml, 
			String user,
			String owner,
			Element parent,
			String type,
			URI contextSchema) 
		throws 
			VDocException
	{
		try
		{
			URI uid = getId();
			
			//Store the document
			SchemaManager smanager = internal.getSchemaManager(user);
			String schemaName = smanager.getSchemaName(contextSchema);
			internal.store(
				uid,xml,contextSchema,schemaName,
				owner,new Permissions(),user,true);
			//Create the describing element
			Element context = new Element(type);
			context.setAttribute("name",name);
			context.setAttribute("uri",uid.toASCIIString());
			context.setAttribute("user",user);
			context.setAttribute("owner",owner);
			context.setText(description);
			parent.addContent(context);
			save();
			
			return uid;
		}
		catch (ModuleCriticalException e){
			throw new VDocException(e);
//			MPA: no logging
		}
		catch (ArchiveException e){
			throw new VDocException(e);
//			MPA: no logging
		}
	}
	
	/**
	 * Store a VDoc Context transform in the archive with a name and description
	 * and uri, neither the name or the description need be unique. A URI is
	 * returned for the context.
	 * @param name			A name for the Context
	 * @param description	Description of the context
	 * @param xml			The context (transofrm) itself
	 * @return				Returns a URI pointing to the context
	 * @throws VDocException
	 */
	public URI addVDocContext(
			String name, 
			String description, 
			String xml, 
			String user,
			String owner,
			URI contextSchema) 
		throws 
			VDocException
	{
		try
		{
			URI uri = addContext(name,description,xml,user,
				owner,vdocContexts,"vdocContext",contextSchema);
			logger.finest("Added vdocContext at: " + uri.toASCIIString());
			return uri;
		}
		catch (ArchiveException e){
			throw new VDocException(e);
//			MPA: no logging
		}
	}
	
	/**
	 * Updates a particular VDoc context, must be done with care as many VDocs
	 * may already be using this context.
	 * @param uri			The uri of the context to be updated
	 * @param xml			The  context (transform) itself
	 * @throws VDocException
	 */
	public void updateVDocContext(
			URI uri, 
			String xml, 
			String user) 
		throws 
			VDocException
	{
		try
		{
			DocumentData dd = internal.status(uri,user);
			ArchiveTimeStamp timeStamp = dd.getTimestamp();
			URI contextSchema = dd.getSchema();
			logger.finest("Updated  vdocContext at: " + uri.toASCIIString());
			internal.update(uri,timeStamp,xml,contextSchema,true,user);
		}
		catch (ModuleCriticalException e){
			throw new VDocException(e);
//			MPA: no logging
		}
		catch (ArchiveException e){
			throw new VDocException(e);
//			MPA: no logging
		}
	}
	
	/**
	 * Returns an element describing the VDoc Contexts
	 * @return
	 */
	public Element getVDocContexts()
	{
		return vdocContexts;
	}
	
	/**
	 * Store a Definition Context transform in the archive with a name,
	 * description and uri neither the name or description need to be unique. A
	 * URI is returned for the context.
	 * @param name			The name of the context
	 * @param description	Description of the context
	 * @param xml			The context (transform) itself
	 * @return				Returns a URI pointing to the context
	 * @throws VDocException
	 */
	public URI addDefinitionContext(
			String name, 
			String description, 
			String xml, 
			String user, 
			String owner,
			URI contextSchema) 
		throws 
			VDocException
	{
		try
		{
			URI uri = addContext(name,description,xml,user,
				owner,definitionContexts,"definitionContext",contextSchema);
			logger.finest("Added definitionContext at: " + uri.toASCIIString());
			return uri;
		}
		catch (ArchiveException e){
			throw new VDocException(e);
//			MPA: no logging
		}
	}
	
	/**
	 * Updates a particular Definition context, must be done with care as many 
	 * VDocs may already be using this context.
	 * @param uri			The uri of the context to be updated
	 * @param xml			The  context (transform) itself
	 * @throws VDocException
	 */
	public void updateDefinitionsContext(
			URI uri, 
			String xml, 
			String user) 
		throws 
			VDocException
	{
		try
		{
			DocumentData dd = internal.status(uri,user);
			ArchiveTimeStamp timeStamp = dd.getTimestamp();
			URI contextSchema = dd.getSchema();
			logger.finest("Updated definitionContext at: " + uri.toASCIIString());
			internal.update(uri,timeStamp,xml,contextSchema,true,user);
		}
		catch (ModuleCriticalException e){
			throw new VDocException(e);
//			MPA: no logging
		}
		catch (ArchiveException e){
			throw new VDocException(e);
//			MPA: no logging
		}
	}
	
	/**
	 * Returns an element describing the Definition Contexts
	 * @return
	 */
	public Element getDefinitionContexts()
	{
		return definitionContexts;
	}
	
	/**
	 * Checks that the the appropriate class is available and then creates an
	 * entry. Will not allow the same type to be registered twice.
	 * @param type				The definition type name
	 * @throws VDocException
	 */
	public void registerDefinitionType(String type) 
		throws 
			VDocException
	{
		if (Definition.isAvailable(type)){
			try{
				String _path = "definitionType[.=\"" + type + "\"]";
				XPath path = XPath.newInstance(_path);
				if (path.selectSingleNode(definitions) == null){
					Element definitionType = new Element("definitionType");
					definitionType.setText(type);
					definitions.addContent(definitionType);
					save();
				}
			}
			catch (JDOMException e){
				throw new VDocException(e);
//				MPA: no logging
			}
		}
		else{
			String className = "alma.archive.database.vdoc." + 
				type + "Definition";
			throw new VDocException("The class: " 
				+ className + "cannot be found");
		}
	}
	
	/**
	 * Un-registers a particular definition type, it will no longer be allowed
	 * for use by a user. This does not necessarily mean that the class is not
	 * available. Effectively removes the type from the list of options.
	 * @param type
	 * @throws VDocException
	 */
	public void unregisterDefinitionType(String type) 
		throws 
			VDocException
	{
		try{
			String _path = "definitionType[.=\"" + type + "\"]";
			XPath path = XPath.newInstance(_path);
			Element definitionType = 
				(Element)path.selectSingleNode(definitions); 
			if (definitionType != null){
				definitionType.detach();
				save();
			}
		}
		catch (JDOMException e){
			throw new VDocException(e);
//			MPA: no logging
		}
	}
	
	public Element getDefinitions()
	{
		return definitions;
	}
	
	/**
	 * Virtual Documents are created as an empty shell and stored in the
	 * Archive under a UID, this is held by the manager. As the document is 
	 * stored as XML it is easy to retrive the content and display it with 
	 * a transform.
	 * @param contextId
	 * @param user
	 * @param owner
	 * @param vdocSchema
	 * @return
	 * @throws VDocException
	 */
	public URI createVDoc(
			URI contextId, 
			String user, 
			String owner, 
			URI vdocSchema) 
		throws 
			VDocException
	{
		try
		{			
			//Create and store the new vdoc, need to be careful with the schema
			VDoc vdoc = new VDoc(contextId);
			URI id = getId();
			SchemaManager smanager = internal.getSchemaManager(user);
			String schemaName = smanager.getSchemaName(vdocSchema);
			smanager.close();
			internal.store(id,vdoc.getXml(),vdocSchema,
				schemaName,owner,new Permissions(),user,true);
			
			/* 
			 * Create an element that describes the location of the vdoc
			 * May possibly add more information in the future
			 */
			Element desc = new Element("vdocDesc");
			desc.setAttribute("location",id.toASCIIString());
			vdocs.addContent(desc);
			save();
			
			return id;
		}
		catch (ModuleCriticalException e){
			throw new VDocException(e);
//			MPA: no logging
		}
		catch (ArchiveException e){
			throw new VDocException(e);
//			MPA: no logging
		}
	}
	
	/**
	 * Returns a JDOM element with vdocDesc children, one for each VDoc. Each
	 * child has a location attribute that contans the URI of the VDoc
	 * @return
	 */
	public Element getVDocLocations()
	{
		return vdocs;
	}
	
	/**
	 * Returns a class that allows access to the Virtual Document, this class
	 * can then be displayed and modified outside by other processes.
	 * @param id
	 * @param user
	 * @return
	 * @throws VDocException
	 */
	public VDoc getVDoc(URI uri, String user) 
		throws 
			VDocException
	{
		try
		{
			String xml = internal.get(uri,user);
			return new VDoc(xml);
		}
		catch (ModuleCriticalException e){
			throw new VDocException(e);
//			MPA: no logging
		}
		catch (ArchiveException e){
			throw new VDocException(e);
//			MPA: no logging
		}
	}
	
	/**
	 * Used to update a VDOC that has been modified by an external process.
	 * @param uri
	 * @param vdoc
	 * @param user
	 * @throws VDocException
	 */
	public void updateVDoc(URI uri, VDoc vdoc, String user) 
		throws 
			VDocException
	{
		try
		{
			DocumentData dd = internal.status(uri,user);
			URI vdocSchema = dd.getSchema();
			ArchiveTimeStamp timeStamp = dd.getTimestamp();
			logger.finest("Updated  VDoc at: " + uri.toASCIIString());
			internal.update(uri,timeStamp,vdoc.getXml(),vdocSchema,true,user);
		}
		catch (ModuleCriticalException e){
			throw new VDocException(e);
//			MPA: no logging
		}
		catch (ArchiveException e){
			throw new VDocException(e);
//			MPA: no logging
		}
	}
}
