/*
 * 	  Created on 01-Mar-2005
 * 
 *    ALMA - Atacama Large Millimiter Array
 *    (c) European Southern Observatory, 2002
 *    Copyright by ESO (in the framework of the ALMA collaboration),
 *    All rights reserved
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License as published by the Free Software Foundation; either
 *    version 2.1 of the License, or (at your option) any later version.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
 *    MA 02111-1307  USA
 */
package alma.archive.database.vdoc;

import java.io.IOException;
import java.io.StringReader;
import java.io.StringWriter;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import javax.xml.transform.Result;
import javax.xml.transform.Source;
import javax.xml.transform.Templates;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerConfigurationException;
import javax.xml.transform.TransformerException;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.stream.StreamResult;
import javax.xml.transform.stream.StreamSource;

import org.jdom.Document;
import org.jdom.Element;
import org.jdom.JDOMException;
import org.jdom.input.SAXBuilder;
import org.jdom.output.Format;
import org.jdom.output.XMLOutputter;
import org.jdom.transform.JDOMResult;
import org.jdom.transform.JDOMSource;

import alma.archive.database.helpers.ArchiveResolver;
import alma.archive.database.interfaces.InternalIF;
import alma.archive.exceptions.ArchiveException;
import alma.archive.exceptions.ModuleCriticalException;
import alma.archive.exceptions.access.EntityDirtyException;
import alma.archive.exceptions.access.PermissionDeniedException;
import alma.archive.exceptions.general.DatabaseException;
import alma.archive.exceptions.general.EntityDoesNotExistException;
import alma.archive.exceptions.general.VDocException;
import alma.archive.exceptions.syntax.MalformedURIException;
import alma.archive.exceptions.user.UserDoesNotExistException;

/**
 * @author simon
 */
public class VDoc
{
	private URI contextid = null;
	private String description = "";
	private ArrayList dList = null;
	
	/**
	 * Build an empty VDoc
	 */
	public VDoc(URI contextid)
	{
		this.contextid = contextid;
		dList = new ArrayList();
	}
	
	
	public VDoc(String xml) throws VDocException
	{
		dList = new ArrayList();
		construct(xmlToElement(xml));
	}
	
	public VDoc(Element element) throws VDocException
	{
		dList = new ArrayList();
		construct(element);
	}
	
	private void construct(Element element) throws VDocException
	{
		try
		{
			contextid = new URI(element.getAttributeValue("contextid"));
			
			Element description_e = element.getChild("description");
			description = description_e.getText();
			
			List definitionList = element.getChildren("definition");
			Iterator iter = definitionList.iterator();
			while (iter.hasNext())
			{
				Element definition = (Element)iter.next();
				Definition def = Definition.instance(definition);
				addDefinition(def);
			}
		}
		catch (URISyntaxException e){
			throw new VDocException(e);
//			MPA: logging missing
		}
	}
	
	private Element xmlToElement(String xml) throws VDocException
	{
		SAXBuilder builder = new SAXBuilder();
		builder.setIgnoringElementContentWhitespace(true);
		try
		{
			Document doc = builder.build(new StringReader(xml));
			return doc.getRootElement();
		}
		catch (JDOMException e){
			throw new VDocException(e);
		}
		catch (IOException e){
			throw new VDocException(e);
		}
//		MPA: logging missing, 2 times
	}

	/**
	 * @return Returns the contextid.
	 */
	public URI getContextid()
	{
		return contextid;
	}
	/**
	 * @param contextid The contextid to set.
	 */
	public void setContextid(URI contextid)
	{
		this.contextid = contextid;
	}
	/**
	 * @return Returns the description.
	 */
	public String getDescription()
	{
		return description;
	}
	/**
	 * @param description The description to set.
	 */
	public void setDescription(String description)
	{
		this.description = description;
	}

	/**
	 * Add a definition to the end of the list
	 * @param definition
	 */
	public void addDefinition(Definition definition)
	{
		dList.add(definition);
	}
	
	/**
	 * Returns the Defintion from a particulr location
	 * @param location
	 * @return
	 */
	public Definition getDefinition(int location)
	{
		return (Definition)dList.get(location);
	}
	
	/**
	 * Insert a definition at a particular location, will move the occupying
	 * definition down the list.
	 * @param definition
	 * @param location
	 */
	public void insertDefinition(Definition definition, int location)
	{
		dList.add(location,definition);
	}
	
	/**
	 * removes a particular definition from the list, no gap is left.
	 * @param location
	 */
	public void removeDefinition(int location)
	{
		dList.remove(location);
	}
	
	/**
	 * Moves a definition from one place on the list to another
	 * @param from
	 * @param too
	 */
	public void moveDefinition(int from, int too)
	{
		Definition def = (Definition)dList.remove(from);
		dList.add(too,def);
	}
	
	/**
	 * Returns a JDOM Element for the entire VDoc.
	 * @return
	 * @throws VDocException
	 */
	public Element getElement() 
		throws 
			VDocException
	{
		Element vdoc = new Element("vdoc");
		vdoc.setAttribute("contextid",contextid.toASCIIString());
		Element desc = new Element("description");
		desc.addContent(description);
		vdoc.addContent(desc);
		
		Iterator iter = dList.iterator();
		while (iter.hasNext())
		{
			Definition def = (Definition)iter.next();
			vdoc.addContent(def.getElement());
		}	
		return vdoc;
	}
	
	/**
	 * Returns an xml String for the entire VDoc
	 * @return
	 * @throws VDocException
	 */
	public String getXml() 
		throws 
			VDocException
	{
		Element vdoc = getElement();
		XMLOutputter out = new XMLOutputter(Format.getRawFormat());
		return out.outputString(vdoc);
	}
	
	
	private Element transform(
			InternalIF internal, 
			String user,
			TransformerFactory factory,
			URI context, 
			Source source)
			throws
			VDocException,
			ModuleCriticalException
	{
		try
		{
			String defContext = internal.get(context,user);
			Templates template = factory.newTemplates(
				new StreamSource(
					new StringReader(defContext)));
			
			JDOMResult result = new JDOMResult();
			
			//The transform
			Transformer transformer = template.newTransformer();
			transformer.setURIResolver(
				new ArchiveResolver(internal,user));
			transformer.transform(source,result);

			Document doc = result.getDocument();
			return doc.detachRootElement();
		}
		catch (ArchiveException e){
			throw new VDocException(e);
//			MPA: logging missing
		}
		catch (TransformerConfigurationException e){
			throw new VDocException(e);
//			MPA: logging missing
		}
		catch (TransformerException e){
			throw new VDocException(e);	
		}
//		MPA: logging missing
	}
	
	public String process(InternalIF internal, String user) 
		throws  
			ModuleCriticalException,
			VDocException
	{
		System.setProperty("javax.xml.transform.TransformerFactory"
			,"org.apache.xalan.processor.TransformerFactoryImpl");
		TransformerFactory factory = TransformerFactory.newInstance();
		
		Element vdocElement = new Element("vdoc");
		Iterator iter = dList.iterator();
		while (iter.hasNext())
		{	
			Definition def = (Definition)iter.next();
			URI defContextid = def.getContextid();
			Element urilist = def.createList(internal);
			Source source = new JDOMSource(urilist);
			Element root = transform(internal,user,factory,defContextid,source);
			
			vdocElement.addContent(root);
		}
		
		Source source = new JDOMSource(vdocElement);
		Element finalresult = transform(internal,user,factory,contextid,source);
		
		XMLOutputter out = new XMLOutputter(Format.getRawFormat());
		return out.outputString(finalresult);
	}
}
