/*
 *    ALMA - Atacama Large Millimiter Array
 *    (c) European Southern Observatory, 2002
 *    Copyright by ESO (in the framework of the ALMA collaboration),
 *    All rights reserved
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License as published by the Free Software Foundation; either
 *    version 2.1 of the License, or (at your option) any later version.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
 *    MA 02111-1307  USA
 *
 *    Created on Oct 20, 2003
 *
 */

// $Author: hmeuss $
// $Date: 2009/07/17 14:37:32 $
// $Log: InternalIfImpl.java,v $
// Revision 1.26  2009/07/17 14:37:32  hmeuss
// removed restore functionality including references to xml_meta_history
//
// Revision 1.25  2009/07/02 13:14:59  hmeuss
// XMLtype is again constructed in client (DatabaseWriter.put), otherwise Oracle throws exception.
// Removed call to DatabaseReader.exist() in InternalIF, this is now done in one go in DatabaseWriter.put together with permission checking.
//
// Revision 1.24  2009/04/22 11:26:18  hmeuss
// implemented incremental updates for XMLstore
//
// Revision 1.23  2009/02/16 16:42:42  hmeuss
// Added environment in order to store STE information in log tables.
//
// Revision 1.22  2008/11/11 20:56:47  hmeuss
// storeLog() now has an explicit commit parameter that controls when a log is committed.
//
// Revision 1.21  2007/10/24 15:06:21  hmeuss
// Added retrieval methods for logs and some more infrastructure
//
// Revision 1.20  2007/10/24 09:56:42  hmeuss
// changed to new relational storage
//
// Revision 1.19  2007/07/12 14:46:51  hmeuss
// added method updateXML
//
// Revision 1.18  2007/02/08 16:52:07  hmeuss
// Added method for querying within given time intervals for asdmQuery
//
// Revision 1.17  2006/11/30 10:35:41  hmeuss
// improved log retrieval with an additional timestamp interval
//
// Revision 1.16  2006/10/26 15:24:25  hmeuss
// Added check, whether archive ID of incoming documents matches archive ID
//
// Revision 1.15  2006/09/21 08:53:31  hmeuss
// Implemented queryContent method, that correctly queries XPath expressions returning a non-node type (eg. ending with an attribute step)
//
// Revision 1.14  2006/09/15 14:12:56  hsommer
// new exceptions due to using the ACS-variant of Range
//
// Revision 1.13  2006/04/28 08:39:36  hmeuss
// Changed interface of queryRecent and implemented it for exist
//
// Revision 1.12  2006/04/25 09:34:12  hmeuss
// changed interface for timestamps of queryRecent
//
// Revision 1.11  2006/04/21 14:32:28  hmeuss
// New version, with test for new UIDLibrary
//
// Revision 1.10  2006/02/16 12:48:54  mpasquat
// added comments about exception handling problems
//
// Revision 1.9  2005/09/12 12:38:29  hmeuss
// Added argument to put in order to make storeLog working.
//
// Revision 1.8  2005/08/02 14:02:32  hmeuss
// added ping() method in internalIF and Administrative component
//
// Revision 1.7  2005/01/13 10:45:08  hmeuss
// - Constructed command line query interface for Archive.
// - Added functionality to init() for Oracle.
//
// Revision 1.6  2004/09/23 11:59:14  hmeuss
// Oracle DatabaseReader now creates a new connection object in every method call.
//
// Added ModuleCriticalException to more methods of the internal IF
//
// Revision 1.5  2004/09/14 09:40:55  hmeuss
// Added new init and close methods
//
// Revision 1.4  2004/08/19 14:58:08  hmeuss
// Changed behaviour of get in Oracle implementation
//
// Revision 1.3  2004/06/01 15:53:12  hmeuss
// Added treatment of virtual documents
//
// Revision 1.2  2004/05/28 07:37:27  hmeuss
// added dummy methods for setVirtual
//
// Revision 1.1  2004/04/05 13:59:29  hmeuss
// Internal IF implementation adapted to Oracle
//
// Revision 1.10  2004/03/18 09:49:33  hmeuss
// changed configuration to work with testStart, testEnd
//
// Revision 1.9  2004/02/11 16:20:09  hmeuss
// *** empty log message ***
//
// Revision 1.8  2004/01/29 14:17:05  hmeuss
// Adapted to the new interface
//
// Revision 1.7  2004/01/21 16:18:04  hmeuss
// Added class DBConfiguration that reads and parses config file
//
// Revision 1.6  2004/01/19 16:29:46  hmeuss
// Splitted functionality of Db2Database into Db2DatabaseReader, Db2DatabaseWriter, Db2DatabaseCache
//
// Revision 1.5  2004/01/13 17:09:28  hmeuss
// added method to change configuration
//
// Revision 1.4  2004/01/09 13:15:28  hmeuss
// 1) Refactored InternalIFFactory to return IdentifierArchive and OperationalArchive only.
// 2) Implemented a switching DB backend mechanism and configuration file
//
// Revision 1.3  2003/12/15 13:43:19  hmeuss
// *** empty log message ***
//
// Revision 1.2  2003/11/20 12:47:31  hmeuss
// *** empty log message ***
//
// Revision 1.1  2003/10/30 08:36:26  hmeuss
// *** empty log message ***
// 
package alma.archive.database.oracle;

import java.net.URI;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.logging.Logger;

import org.apache.xpath.axes.ChildIterator;
import org.jdom.Document;
import org.jdom.Element;

import alma.ArchiveIdentifierError.wrappers.AcsJArchiveIdentifierErrorEx;
import alma.archive.database.helpers.DatabaseHelper;
import alma.archive.database.interfaces.DBCursor;
import alma.archive.database.interfaces.InternalIF;
import alma.archive.database.interfaces.InternalIFFactory;
import alma.archive.database.interfaces.SchemaManager;
import alma.archive.database.interfaces.UserManager;
import alma.archive.exceptions.ArchiveException;
import alma.archive.exceptions.ArchiveGeneralException;
import alma.archive.exceptions.ModuleCriticalException;
import alma.archive.exceptions.access.EntityDirtyException;
import alma.archive.exceptions.access.PermissionDeniedException;
import alma.archive.exceptions.general.DatabaseException;
import alma.archive.exceptions.general.DocumentDoesNotExistException;
import alma.archive.exceptions.general.EntityAlreadyDeletedException;
import alma.archive.exceptions.general.EntityDoesNotExistException;
import alma.archive.exceptions.general.EntityExistsException;
import alma.archive.exceptions.general.EntityUndeletedException;
import alma.archive.exceptions.general.HistoryInconsistencyException;
import alma.archive.exceptions.general.IllegalTimestampException;
import alma.archive.exceptions.general.UndefinedNamespaceException;
import alma.archive.exceptions.general.UnknownSchemaException;
import alma.archive.exceptions.syntax.IllegalHistoryNumberException;
import alma.archive.exceptions.syntax.MalformedPermissionsException;
import alma.archive.exceptions.syntax.MalformedQueryException;
import alma.archive.exceptions.syntax.MalformedURIException;
import alma.archive.exceptions.syntax.MalformedXMLException;
import alma.archive.exceptions.syntax.MalformedXPointerException;
import alma.archive.exceptions.syntax.UnderspecifiedQueryException;
import alma.archive.exceptions.syntax.UnknownFlagException;
import alma.archive.exceptions.user.UserDoesNotExistException;
import alma.archive.wrappers.ArchiveTimeStamp;
import alma.archive.wrappers.DocumentData;
import alma.archive.wrappers.Permissions;

/**
 * @author hmeuss
 * 
 */

// TODO in internal IF:
// change all "throws DatabaseException/ArchiveException" to "throws
// GenericDatabaseException/GenericArchiveException"
public class InternalIfImpl implements InternalIF {

	protected Database database;

	protected DatabaseReader databaseReader;

	protected DatabaseWriter databaseWriter;

	// default logger is the global logger. ACS will set other logger. See
	// @addLogger
	protected Logger logger = Logger.global;

	// we use the singleton pattern
	protected static InternalIfImpl instance;

	public static InternalIfImpl instance() throws DatabaseException,
			ModuleCriticalException {
		if (instance == null) {
			instance = new InternalIfImpl();
		}
		return instance;
	}

	/**
	 * 
	 */
	private InternalIfImpl() throws DatabaseException, ModuleCriticalException {
		super();
		database = Database.instance();
		databaseReader = DatabaseReader.instance();
		databaseWriter = DatabaseWriter.instance();
	}

	/**
	 * @throws DatabaseException
	 * @throws ModuleCriticalException
	 * @see alma.archive.database.interfaces.InternalIF#init()
	 */
	public void init() throws DatabaseException, ModuleCriticalException {
		database.init();
	}

	/**
	 * @see alma.archive.database.interfaces.InternalIF#close()
	 */
	public void close() throws DatabaseException, ModuleCriticalException {
		database.close();

	}

	/**
	 * @see alma.archive.database.interfaces.InternalIF#store(java.net.URI,
	 *      java.lang.String, java.net.URI, java.lang.String, java.lang.String,
	 *      alma.archive.wrappers.Permissions, java.lang.String)
	 */
	public void store(URI uid, String xml, URI schema, String schemaName,
			String owner, Permissions permissions, String user, boolean emitLogs)
			throws ArchiveException, MalformedURIException,
			MalformedXMLException, UnknownSchemaException,
			UserDoesNotExistException, MalformedPermissionsException,
			DatabaseException, EntityExistsException, ModuleCriticalException {

		if (!DatabaseHelper.checkArchiveIdStoragePermission(uid,
				InternalIFFactory.getIdentifierManager(logger))) {
			// TODO define new exception for this case
			throw new ArchiveException("Cannot store document with UID "
					+ uid.toString()
					+ ": does not match with ArchiveId: "
					+ InternalIFFactory.getIdentifierManager(logger)
							.getArchiveId());
		}

		databaseWriter.put(uid, new ArchiveTimeStamp(), xml, schema,
				schemaName, owner, permissions, user, true, emitLogs);

	}

	/**
	 * @see alma.archive.database.interfaces.InternalIF#store(java.net.URI,
	 *      java.lang.String)
	 */
	public void store(URI uid, String xml) throws DatabaseException,
			ArchiveException, MalformedURIException, MalformedXMLException {
		// TODO Auto-generated method stub
		// don't know what to do with this
	}

	public void storeLog(String message, int level, String attributeValue,
			String attributeValue2, String attributeValue3,
			String attributeValue4, String attributeValue5,
			String attributeValue6, String attributeValue7,
			String attributeValue8, String attributeValue9,
			String attributeValue10, String attributeValue11,
			String attributeValue12, String attributeValue13,
			String attributeValue14, String attributeValue15,
			String[] dataStrings, String xml, boolean commit, String steEnv) throws DatabaseException {
		databaseWriter.storeLog(message, level, attributeValue,
				attributeValue2, attributeValue3, attributeValue4,
				attributeValue5, attributeValue6, attributeValue7,
				attributeValue8, attributeValue9, attributeValue10,
				attributeValue11, attributeValue12, attributeValue13,
				attributeValue14, attributeValue15, dataStrings, xml, commit, steEnv);
	}

	/**
	 * @see alma.archive.database.interfaces.InternalIF#update(java.net.URI,
	 *      alma.archive.wrappers.ArchiveTimeStamp, java.lang.String,
	 *      java.net.URI, java.lang.String)
	 */
	public void update(URI uid, ArchiveTimeStamp timeStamp, String xml,
			URI schema, boolean force, String user)
			throws PermissionDeniedException, ArchiveException,
			MalformedURIException, MalformedXMLException,
			UserDoesNotExistException, UnknownSchemaException,
			HistoryInconsistencyException, EntityDoesNotExistException,
			DatabaseException, ModuleCriticalException {

		// alternatively, we should keep a conversion list in main memory, at
		// least for the latest schema versions...
		String schemaName = URI2schemaString(schema);

		// check timestamp consistency and existence of entity (possibly throw
		// exception)
		if (!force) {
			databaseReader
					.checkTimestampConsistency(uid, schemaName, timeStamp);
		}

		databaseWriter.put(uid, new ArchiveTimeStamp(), xml, schema,
				schemaName, null, null, user, false, true);

	}

	/**
	 * @throws ModuleCriticalException
	 * @throws EntityDoesNotExistException
	 * @see alma.archive.database.interfaces.InternalIF#updateXML
	 */
	public void updateXML(URI uid, String schema, String newChild)
			throws DatabaseException, EntityDoesNotExistException,
			ModuleCriticalException {
		databaseWriter.updateXML(uid, schema, newChild);
	}
	
	/**
	 * Adds xmlElement as last child of the (first) element found by xPath in
	 * document uid (belonging to schema schema)
	 */
	public void addElement(URI uid, String schemaName, String xPath,
			String xmlElement, String user) throws EntityDoesNotExistException,
			DatabaseException, PermissionDeniedException,
			ModuleCriticalException {
		databaseWriter.addElement(uid, schemaName, xPath,
				xmlElement, user);
	}
		
	/**
	 * The (first) element pointed at by xPath in document uid (belonging to
	 * schema schema) will be replaced by element xmlElement.
	 */
	public void updateElement(URI uid, String schemaName, String xPath,
			String xmlElement, String user) throws EntityDoesNotExistException,
			DatabaseException, PermissionDeniedException,
			ModuleCriticalException {
				databaseWriter.updateElement(uid, schemaName, xPath,
						xmlElement, user);
			}

	/**
	 * The (first) element pointed at by xPath in document uid (belonging to
	 * schema schema) will be deleted.
	 */
	public void deleteElement(URI uid, String schemaName, String xPath,
			String user) throws EntityDoesNotExistException, DatabaseException,
			PermissionDeniedException, ModuleCriticalException {
				databaseWriter.deleteElement(uid, schemaName, xPath, user);
			}
		
	/**
	 * @see alma.archive.database.interfaces.InternalIF#get(java.net.URI)
	 */
	public String get(URI uri) throws DatabaseException, ArchiveException,
			MalformedURIException {
		throw new DatabaseException("get(URI) not supported.");
		// TODO
		// don't know what this is!
	}

	/**
	 * @see alma.archive.database.interfaces.InternalIF#get(java.net.URI,
	 *      boolean, java.lang.String)
	 */
	public String get(URI uid, String user) throws ArchiveException,
			UserDoesNotExistException, MalformedURIException,
			PermissionDeniedException, EntityDirtyException,
			ArchiveGeneralException, ModuleCriticalException {

		return databaseReader.get(uid, null, false, user);
	}

	/**
	 * @see alma.archive.database.interfaces.InternalIF#get(java.net.URI,
	 *      boolean, java.lang.String)
	 */
	public String getDirty(URI uid, String user) throws ArchiveException,
			UserDoesNotExistException, MalformedURIException,
			PermissionDeniedException, EntityDirtyException,
			ArchiveGeneralException, ModuleCriticalException {

		return databaseReader.get(uid, null, true, user);
	}

	/**
	 * @see alma.archive.database.interfaces.InternalIF#get(java.net.URI,
	 *      java.lang.String, java.util.HashMap, boolean, java.lang.String)
	 */
	public String[] get(URI uid, String xpath, HashMap namespaces, String user)
			throws EntityDirtyException, ArchiveException,
			MalformedURIException, IllegalHistoryNumberException,
			UserDoesNotExistException, MalformedXPointerException,
			PermissionDeniedException, MalformedXMLException,
			DatabaseException, MalformedQueryException,
			ArchiveGeneralException, ModuleCriticalException {

		return databaseReader.queryGet(uid, xpath, namespaces, true, user);
	}

	/**
	 * @see alma.archive.database.interfaces.InternalIF#get(java.net.URI,
	 *      alma.archive.wrappers.ArchiveTimeStamp, java.lang.String)
	 */
	public String get(URI uid, ArchiveTimeStamp timestamp, String user)
			throws ArchiveException, MalformedURIException,
			UserDoesNotExistException, PermissionDeniedException,
			EntityDirtyException, ArchiveGeneralException,
			ModuleCriticalException {
		return databaseReader.get(uid, timestamp, true, user);
	}

	/**
	 * @see alma.archive.database.interfaces.InternalIF#delete(java.net.URI,
	 *      java.lang.String)
	 */
	public void delete(URI uid, String user) throws PermissionDeniedException,
			EntityDoesNotExistException, DatabaseException, ArchiveException,
			MalformedURIException, UserDoesNotExistException,
			EntityAlreadyDeletedException {
		databaseWriter.un_delete(uid, user, true);
	}

	/**
	 * @see alma.archive.database.interfaces.InternalIF#undelete(java.net.URI,
	 *      java.lang.String)
	 */
	public void undelete(URI uid, String user)
			throws PermissionDeniedException, DatabaseException,
			ArchiveException, MalformedURIException,
			EntityDoesNotExistException, UserDoesNotExistException,
			EntityUndeletedException {
		databaseWriter.un_delete(uid, user, false);
	}

	/**
	 * @see alma.archive.database.interfaces.InternalIF#status(java.net.URI,
	 *      java.lang.String)
	 */
	public DocumentData status(URI uid, String user) throws ArchiveException,
			ModuleCriticalException {
		// timestamp set to null
		return databaseReader.status(uid, null, user);
	}

	/**
	 * @see alma.archive.database.interfaces.InternalIF#allStatus(java.net.URI,
	 *      java.lang.String)
	 */
	public DocumentData[] allStatus(URI uid, String user)
			throws ArchiveException, ModuleCriticalException {
		return databaseReader.allStatus(uid, user);
	}

	/**
	 * @see alma.archive.database.interfaces.InternalIF#status(java.net.URI,
	 *      alma.archive.wrappers.ArchiveTimeStamp, java.lang.String)
	 */
	public DocumentData status(URI uid, ArchiveTimeStamp timestamp, String user)
			throws ArchiveException, ModuleCriticalException {
		return databaseReader.status(uid, timestamp, user);
	}

	/**
	 * @see alma.archive.database.interfaces.InternalIF#queryRecent(java.lang.String,
	 *      java.lang.String, java.lang.String)
	 */
	public URI[] queryRecent(ArchiveTimeStamp timestamp, String schema,
			String user) throws ModuleCriticalException,
			PermissionDeniedException, DatabaseException, ArchiveException,
			MalformedQueryException, UnderspecifiedQueryException,
			UnknownSchemaException, UserDoesNotExistException,
			UndefinedNamespaceException {
		return databaseReader.queryRecent(timestamp, schema, user);
	}

	public URI[] queryInterval(ArchiveTimeStamp timeFrom,
			ArchiveTimeStamp timeTo, String schema, String XPathQuery,
			String user) throws ModuleCriticalException, ArchiveException {
		return databaseReader.queryInterval(timeFrom, timeTo, schema,
				XPathQuery, user);
	}

	/**
	 * @see alma.archive.database.interfaces.InternalIF#query(java.lang.String,
	 *      java.lang.String, java.util.HashMap, boolean, java.lang.String)
	 */
	public DBCursor query(String query, String schema, HashMap namespaces,
			boolean dirtyRead, String user) throws PermissionDeniedException,
			ArchiveException, UnderspecifiedQueryException,
			UnknownSchemaException, UserDoesNotExistException,
			UndefinedNamespaceException, DatabaseException,
			MalformedQueryException, ModuleCriticalException {
		return databaseReader.query(query, schema, namespaces, dirtyRead, user,
				null, null);
	}

	/**
	 * @see alma.archive.database.interfaces.InternalIF#queryContent(java.lang.String,
	 *      java.lang.String, java.util.HashMap, boolean, java.lang.String)
	 */
	public DBCursor queryContent(String query, String schema,
			HashMap namespaces, boolean dirtyRead, String user)
			throws PermissionDeniedException, ArchiveException,
			UnderspecifiedQueryException, UnknownSchemaException,
			UserDoesNotExistException, UndefinedNamespaceException,
			DatabaseException, MalformedQueryException, ModuleCriticalException {
		return databaseReader.queryContent(query, schema, namespaces,
				dirtyRead, user, null, null);
	}

	// same as query, but optimizes queries to logs.
	// no longer used, replaced by getLog (relational storage of logs)
	public DBCursor queryLog(String query, String schema,
	// not a URI, because we are independent of schema versions!
			HashMap namespaces, String user, String timeFrom, String timeTo)
			throws ModuleCriticalException, PermissionDeniedException,
			DatabaseException, ArchiveException, MalformedQueryException,
			UnderspecifiedQueryException, UnknownSchemaException,
			UserDoesNotExistException, UndefinedNamespaceException {
		return databaseReader.queryContent(query, schema, namespaces, false,
				user, timeFrom, timeTo);
	}

	public Collection<String> getLog(String timeFrom, String timeTo,
			short minType, short maxType, String routine, String source,
			String process, int maxRow) throws ArchiveException, ModuleCriticalException {
		return databaseReader.queryLog(timeFrom, timeTo, minType, maxType,
				routine, source, process, maxRow);
	}

	/**
	 * @see alma.archive.database.interfaces.InternalIF#queryIDs(java.lang.String,
	 *      java.lang.String, java.util.HashMap, boolean, java.lang.String)
	 */
	public URI[] queryIDs(String query, String schema, HashMap namespaces,
			boolean dirtyRead, String user) throws ArchiveException,
			ModuleCriticalException {
		return databaseReader.queryIDs(query, schema, namespaces, dirtyRead,
				false, user);
	}

	/**
	 * @see alma.archive.database.interfaces.InternalIF#setFlag(java.net.URI,
	 *      java.lang.String, boolean, java.lang.String)
	 */
	public void setFlag(URI uid, String flagName, boolean flagValue, String user)
			throws PermissionDeniedException, DatabaseException,
			ArchiveException, MalformedURIException,
			EntityDoesNotExistException, UserDoesNotExistException,
			UnknownFlagException {
		databaseWriter.setFlag(uid, flagName, flagValue, user);
	}

	/**
	 * @see alma.archive.database.interfaces.InternalIF#getSchemaManager(java.lang.String)
	 */
	public SchemaManager getSchemaManager(String user) throws ArchiveException,
			PermissionDeniedException, UserDoesNotExistException,
			DatabaseException, ModuleCriticalException {
		return new SchemaManagerImpl(user);
	}

	/**
	 * @see alma.archive.database.interfaces.InternalIF#getUserManager(java.lang.String)
	 */
	public UserManager getUserManager(String user) throws ArchiveException,
			PermissionDeniedException, UserDoesNotExistException {
		return new UserManagerImpl(user);
	}

	/**
	 * @see alma.archive.database.interfaces.InternalIF#remove(java.net.URI,
	 *      boolean, java.lang.String)
	 */
	public void remove(URI uid, boolean keepHead, String user)
			throws ArchiveException, MalformedURIException,
			UserDoesNotExistException, EntityDoesNotExistException,
			DatabaseException, PermissionDeniedException,
			ModuleCriticalException {
		databaseWriter.remove(uid, keepHead, user);
	}

	/**
	 * @see alma.archive.database.interfaces.InternalIF#setPermission(java.net.URI,
	 *      alma.archive.wrappers.Permissions, java.lang.String)
	 */
	public void setPermission(URI uid, Permissions permissions, String user)
			throws ArchiveException, MalformedURIException,
			UserDoesNotExistException, MalformedPermissionsException,
			EntityDoesNotExistException, DatabaseException,
			PermissionDeniedException, ModuleCriticalException {
		databaseWriter.setPermission(uid, permissions, user);
	}

	/**
	 * @see alma.archive.database.interfaces.InternalIF#restoreState(alma.archive.wrappers.ArchiveTimeStamp,
	 *      java.lang.String)
	 */
	public void restoreState(ArchiveTimeStamp timestamp, String user)
			throws PermissionDeniedException, DatabaseException,
			ArchiveException, UserDoesNotExistException,
			IllegalTimestampException {
		throw new ArchiveException("restoreState() no longer supported in Archive SW.");
		//databaseWriter.restoreState(timestamp, user);

	}

	/**
	 * @see alma.archive.database.interfaces.InternalIF#notifyConfigChange()
	 */
	public void notifyConfigChange() throws DatabaseException,
			ModuleCriticalException {
		database.notifyConfigChange();
	}

	/**
	 * @see alma.archive.database.interfaces.InternalIF#changeOwner(java.net.URI,
	 *      java.lang.String, java.lang.String)
	 */
	public void changeOwner(URI uid, String newOwner, String user)
			throws ArchiveException, MalformedURIException,
			UserDoesNotExistException, EntityDoesNotExistException,
			DatabaseException, PermissionDeniedException,
			ModuleCriticalException {
		databaseWriter.changeOwner(uid, newOwner, user);
	}

	/**
	 * @see alma.archive.database.interfaces.InternalIF#cleanTestArea(java.lang.String)
	 */
	public void cleanTestArea(String user) throws PermissionDeniedException,
			DatabaseException, ArchiveException, UserDoesNotExistException,
			AcsJArchiveIdentifierErrorEx {
		databaseWriter.cleanTestArea();
	}

	/**
	 * @see alma.archive.database.interfaces.InternalIF#dirty(java.net.URI,
	 *      java.lang.String)
	 */
	public void dirty(URI uid, String user) throws PermissionDeniedException,
			EntityDoesNotExistException, DatabaseException, ArchiveException,
			MalformedURIException, UserDoesNotExistException {
		databaseWriter.setFlag(uid, DBConfig.flagNameDirty, true, user);
	}

	/**
	 * @see alma.archive.database.interfaces.InternalIF#clean(java.net.URI,
	 *      java.lang.String)
	 */
	public void clean(URI uid, String user) throws PermissionDeniedException,
			DatabaseException, ArchiveException, MalformedURIException,
			EntityDoesNotExistException, UserDoesNotExistException,
			EntityUndeletedException {
		databaseWriter.setFlag(uid, DBConfig.flagNameDirty, false, user);

	}

	/**
	 * @see alma.archive.database.interfaces.InternalIF#hidden(java.net.URI,
	 *      java.lang.String)
	 */
	public void hidden(URI uid, String user) throws PermissionDeniedException,
			EntityDoesNotExistException, DatabaseException, ArchiveException,
			MalformedURIException, UserDoesNotExistException {
		databaseWriter.setFlag(uid, DBConfig.flagNameHidden, true, user);

	}

	/**
	 * @see alma.archive.database.interfaces.InternalIF#visible(java.net.URI,
	 *      java.lang.String)
	 */
	public void visible(URI uid, String user) throws PermissionDeniedException,
			DatabaseException, ArchiveException, MalformedURIException,
			EntityDoesNotExistException, UserDoesNotExistException,
			EntityUndeletedException {
		databaseWriter.setFlag(uid, DBConfig.flagNameHidden, false, user);

	}

	/**
	 * @see alma.archive.database.interfaces.InternalIF#queryIDsAll(java.lang.String,
	 *      java.lang.String, java.util.HashMap, java.lang.String)
	 */
	public URI[] queryIDsAll(String query, String schema, HashMap namespaces,
			String user) throws ArchiveException, ModuleCriticalException {
		return databaseReader.queryIDs(query, schema, namespaces, true, true,
				user);
	}

	/**
	 * @see alma.archive.database.interfaces.InternalIF#setVirtual(java.net.URI,
	 *      java.lang.String, boolean)
	 */
	public void setVirtual(URI uid, String user, boolean virtual)
			throws ArchiveException, MalformedURIException,
			UserDoesNotExistException, EntityDoesNotExistException,
			DatabaseException, PermissionDeniedException,
			ModuleCriticalException {
		databaseWriter.setVirtual(uid, user, virtual);

	}

	/**
	 * @see alma.archive.database.interfaces.InternalIF#ping()
	 */
	public boolean ping() {
		try {
			return DatabaseReader.instance().ping();
		} catch (Exception e) {
			logger.warning("Exception: " + e.toString());
			// MPA: better to use logger.log (Level.WARNING,"ddd"+e.toString,e)
			// otherwise the stack trace is lost
			return false;
		}
	}

	// ////////////////////////////////////////////////////////////////////
	// functional helpers //
	// ////////////////////////////////////////////////////////////////////

	/**
	 * 
	 * retrieve the sche name for a schema version given in the form of a URI
	 * 
	 * @param schema
	 * @return
	 */
	private String URI2schemaString(URI schema) throws UnknownSchemaException,
			DatabaseException, ModuleCriticalException {

		return databaseReader.getSchemaName(schema);

	}

	// ////////////////////////////////////////////////////////////////////
	// ACS and other helpers //
	// ////////////////////////////////////////////////////////////////////

	public void setLogger(Logger log) {
		logger = log;
		database.setLogger(log);
	}

}