/*
 *    ALMA - Atacama Large Millimiter Array
 *    (c) European Southern Observatory, 2002
 *    Copyright by ESO (in the framework of the ALMA collaboration),
 *    All rights reserved
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License as published by the Free Software Foundation; either
 *    version 2.1 of the License, or (at your option) any later version.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
 *    MA 02111-1307  USA
 *
 *    Created on Oct 31, 2003
 *
 */

// $Author: hmeuss $
// $Date: 2009/10/21 08:45:45 $
// $Log: IdentifierArchive.java,v $
// Revision 1.27.2.1  2009/10/21 08:45:45  hmeuss
// corrected bug in compatibility with old-style archive ids
//
// Revision 1.27  2009/07/02 12:46:58  hmeuss
// archID is now cached.
//
// Revision 1.26  2009/06/25 08:43:32  hmeuss
// Improvement for new archiveID format
//
// Revision 1.25  2009/06/19 08:50:32  hmeuss
// archiveId may now be an arbitrary string.
//
// Revision 1.24  2007/07/05 13:27:01  hmeuss
// Changed ID storage in Oracle: Now sequences are used
//
// Revision 1.23  2007/03/01 15:54:31  hmeuss
// Added tests for consistency of Oracle tables and repair methods
//
// Revision 1.22  2007/02/22 14:39:35  hmeuss
// replaced ugly conversion of Archive ID, which was wrong, but not harming.
//
// Revision 1.21  2007/01/31 17:28:22  hmeuss
// Corrected bug in UID handling introduced by last bugfix:-(
//
// Revision 1.20  2007/01/31 14:31:45  hmeuss
// Added warning message to setArchiveId()
//
// Revision 1.19  2007/01/30 18:40:25  hmeuss
// Fixed problem with first part of UIDs (The ArchiveID)
//
// Revision 1.18  2007/01/30 18:36:49  hmeuss
// first part of UIDs were not hex. Fixed this.
//
// Revision 1.17  2007/01/18 13:12:24  hmeuss
// Refactoring of UID handling
//
// Revision 1.16  2007/01/18 13:00:39  hmeuss
// fixed bug in UID handling
//
// Revision 1.15  2007/01/12 13:59:38  hmeuss
// getIdNamespace returned old style UIDs. Corrected this although it is deprecated.
//
// Revision 1.14  2006/10/26 15:36:52  hmeuss
// Added check, whether archive ID of incoming documents matches archive ID
//
// Revision 1.13  2006/04/07 14:46:44  hmeuss
// First version using the new UID scheme
//
// Revision 1.12  2006/03/09 08:47:00  hmeuss
// UIDs are read every time from DB. Necessary if multiple instances of Archive SW are accessing the same DB backend.
//
// Revision 1.11  2005/09/14 15:57:36  sfarrow
// Altered the operation of the Identifier manager, uses an alternative architecture
//
// Revision 1.10  2005/07/21 12:29:48  hmeuss
// Changed design of test area
//
// Revision 1.9  2005/05/30 11:10:20  hmeuss
// added cleanDB for deleting onconsistent tables
// changed behaviour of IdentiferArchive when re-initing
//
// Revision 1.8  2004/11/29 10:11:14  hmeuss
// Added log messages for UID initialization
//
// Revision 1.7  2004/10/06 14:17:47  hmeuss
// Changed the way schemas are stored (now using XMLtype!)
//
// Revision 1.6  2004/10/01 08:06:51  hmeuss
// Changed initialization of IdentifierArchive
//
// Revision 1.5  2004/09/30 16:14:59  hmeuss
// *** empty log message ***
//
// Revision 1.4  2004/09/23 11:59:14  hmeuss
// Oracle DatabaseReader now creates a new connection object in every method call.
//
// Added ModuleCriticalException to more methods of the internal IF
//
// Revision 1.3  2004/09/20 12:50:59  hmeuss
// Every time a new ID is fetched, the state is saved
//
// Revision 1.2  2004/06/29 14:18:48  hmeuss
// Changed config file from XML to Java properties
//
// Revision 1.1  2004/04/05 13:59:29  hmeuss
// Internal IF implementation adapted to Oracle
//
// Revision 1.6  2004/03/19 09:10:30  hmeuss
// changed stupid representation of ID counter from int to long
//
// Revision 1.5  2004/03/18 09:49:33  hmeuss
// changed configuration to work with testStart, testEnd
//
// Revision 1.4  2004/01/21 16:18:04  hmeuss
// Added class DBConfiguration that reads and parses config file
//
// Revision 1.3  2004/01/19 16:29:46  hmeuss
// Splitted functionality of Database into DatabaseReader, DatabaseWriter, DatabaseCache
//
// Revision 1.2  2003/11/26 13:26:43  hmeuss
// Migrated to ACS 3.0
//
// Revision 1.1  2003/11/20 13:03:41  hmeuss
// Implemented new internal IF
// 
package alma.archive.database.oracle;

import java.net.URI;
import java.net.URISyntaxException;
import java.util.logging.Logger;

import alma.archive.database.helpers.DBConfiguration;
import alma.archive.database.interfaces.IdentifierManager;
import alma.archive.database.interfaces.InternalIFFactory;
import alma.archive.exceptions.ArchiveException;
import alma.archive.exceptions.ModuleCriticalException;
import alma.archive.exceptions.general.DatabaseException;

/**
 * @author hmeuss
 * 
 */
public class IdentifierArchive extends IdentifierManager {

	protected DBConfiguration config; // set by Database in
										// notifyConfigChange()

	String archID=null;
	/* For using the singleton pattern: */
	protected static IdentifierArchive instance;

	Logger logger = Logger.global;

	Database db;

	DatabaseReader dbReader;

	DatabaseWriter dbWriter;

	/**
	 * 
	 */
	protected IdentifierArchive() throws DatabaseException,
			ModuleCriticalException {
		super();
		db = Database.instance();
		dbReader = DatabaseReader.instance();
		dbWriter = DatabaseWriter.instance();
		if (config == null) {
			// make sure that config is initialized:
			config = InternalIFFactory.dbConfig;
		}
	}

	/**
	 * 
	 */
	public static IdentifierArchive instance() throws DatabaseException,
			ModuleCriticalException {
		if (instance == null) {
			instance = new IdentifierArchive();
		}
		return instance;
	}

	public void setLogger(Logger log) {
		logger = log;
	}

	/**
	 * DEPRECATED
	 * 
	 * @see alma.archive.database.interfaces.IdentifierManager#getIdNamespace()
	 */
	public synchronized URI getIdNamespace() throws ArchiveException,
			DatabaseException, ModuleCriticalException {

		URI out = constructUIDstring(getNextUID());
		logger.fine("Assigning UID " + out);
		return out;
	}

	/**
	 * 
	 * @return next UID to use
	 * @throws DatabaseException
	 * @throws ModuleCriticalException
	 * @throws ArchiveException
	 */
	private long getNextUID() throws DatabaseException,
			ModuleCriticalException, ArchiveException {
		
		return Long.parseLong(dbReader.getLastUIDstored());
	}

	/**
	 * @see alma.archive.database.interfaces.IdentifierManager#close()
	 */
	public void close() throws DatabaseException {
	}

	/**
	 *  Converts an int into a UID. The actID is managed here in the
	 * form of an int. This method is used to give UIDs to the outside world.
	 * 
	 * @param actID
	 * @return
	 */
	protected URI constructUIDstring(long actID) throws ArchiveException {
		StringBuffer out = new StringBuffer("uid://");

		String archiveID;
		try {
			archiveID = getArchiveId();
		} catch (ModuleCriticalException e1) {
			// throw ArchiveException, not really correct, but the critical
			// exception will be discovered afterwards
			throw new ArchiveException(e1.getCause());
		}
		// archiveID
		out.append(archiveID);
		// global ID
		out.append("/X");
		out.append(Long.toHexString(actID));
		// local ID set to 1, 0 is reserved for Range docs.
		out.append("/X1");
		try {
			return new URI(out.toString());
		} catch (URISyntaxException e) {
			logger
					.severe("URI conversion error (long --> URI). long: "
							+ actID);
			throw new ArchiveException("URI conversion error: " + actID);
		}
	}


	/**
	 * Initializes Identifier. Possible necessary for re-initing the system.
	 * only performed when IdentifierArchive was already "alive" before
	 * 
	 * @throws ModuleCriticalException
	 * @throws DatabaseException
	 * 
	 */
	public void init() throws DatabaseException, ModuleCriticalException {
		// do nothing
	}

	/**
	 * Return the Archive ID value stored in the Database, if no value is set
	 * return "" not null.
	 * 
	 * @return
	 * @throws DatabaseException
	 * @throws ModuleCriticalException
	 */

	public String getArchiveId() throws DatabaseException,
			ModuleCriticalException {

		if (archID==null) {
		try {
		archID=dbReader
		.getMetaParamValue(DBConfig.paramName_archiveID);
		logger.fine("Fetched archiveID: "+archID);
		} catch (DatabaseException e) {
			throw new ModuleCriticalException(e, (short) 0,
					"Could not retrieve essential DB parameter: "
							+ DBConfig.tableName_metaInf + ", parameter: "
							+ DBConfig.paramName_archiveID);
		}
		}
		try {
		return "X"+String.valueOf(Long.parseLong(archID));
		} catch (NumberFormatException e) {
			// new style archiveId starting with a character:
			return archID;
		}
	}

	/**
	 * Store the Archive ID in the Database, calling set will overwrite any
	 * previous value.
	 * Only possible in test mode!
	 * 
	 * @param archiveId
	 * @throws DatabaseException
	 * @throws ModuleCriticalException
	 * @throws DatabaseException 
	 */
	protected void setArchiveId(String archiveId) throws ModuleCriticalException, DatabaseException {

		if (!config.testMode) {
			throw new DatabaseException("Can only set range ID in test mode!");
		}
		try {
			logger.warning("Setting archiveID to "+archiveId+". This is not a normal operation and can severely disrupt Archive operations!");
			dbWriter.setMetaParamValue(DBConfig.paramName_archiveID, archiveId);
		} catch (DatabaseException e) {
			throw new ModuleCriticalException(e, (short) 0,
					"Could not modify essential DB parameter: "
							+ DBConfig.tableName_metaInf + ", parameter: "
							+ DBConfig.paramName_archiveID);
		}

	}

	/**
	 * Return the Range ID value stored in the Database, if no value is set
	 * return 0 not null;
	 * 
	 * @return
	 * @throws DatabaseException
	 */
	protected long getRangeId() throws DatabaseException,
			ModuleCriticalException {
		try {
			return getNextUID();
		} catch (ArchiveException e) {
			throw new DatabaseException(e);
		}
	}

	/**
	 * This method makes no sense in the Oracle architecture. It stays empty.
	 * @see alma.archive.database.interfaces.IdentifierManager#setRangeId(long)
	 */
	protected void setRangeId(long rangeId) throws DatabaseException, ModuleCriticalException {
		// nothing
	}

}