/*
 *    ALMA - Atacama Large Millimiter Array
 *    (c) European Southern Observatory, 2002
 *    Copyright by ESO (in the framework of the ALMA collaboration),
 *    All rights reserved
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License as published by the Free Software Foundation; either
 *    version 2.1 of the License, or (at your option) any later version.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
 *    MA 02111-1307  USA
 *
 *    Created on Jan 16, 2004
 *
 */

// $Author: hmeuss $
// $Date: 2010/05/25 09:14:11 $
// $Log: DatabaseWriter.java,v $
// Revision 1.93.8.1  2010/05/25 09:14:11  hmeuss
// changed visibility of reinit to public
//
// Revision 1.93  2009/09/30 15:50:40  hmeuss
// The close method does not throw exceptions anymore, not necessary
//
// Revision 1.92  2009/09/25 14:17:28  hmeuss
// corrected minor bug in closing connections
//
// Revision 1.91  2009/07/17 14:36:40  hmeuss
// removed restore functionality including references to xml_meta_history
//
// Revision 1.90  2009/07/02 13:14:59  hmeuss
// XMLtype is again constructed in client (DatabaseWriter.put), otherwise Oracle throws exception.
// Removed call to DatabaseReader.exist() in InternalIF, this is now done in one go in DatabaseWriter.put together with permission checking.
//
// Revision 1.89  2009/06/15 12:38:44  hmeuss
// Improved SQL command for storing documents.
//
// Revision 1.88  2009/05/05 07:47:47  hmeuss
// Corrected typo
//
// Revision 1.87  2009/05/04 13:39:33  hmeuss
// Brought main method back to old status (doing consistency check)
//
// Revision 1.86  2009/04/30 15:32:03  hmeuss
// improved namespace handling for add/update/deleteElement
//
// Revision 1.85  2009/04/22 11:26:18  hmeuss
// implemented incremental updates for XMLstore
//
// Revision 1.84  2009/04/07 15:03:13  hmeuss
// close() also closes logConn now.
//
// Revision 1.83  2009/03/24 14:42:47  hmeuss
// storeLog throws more exceptions in error case instead of silently returning.
//
// Revision 1.82  2009/02/16 16:42:42  hmeuss
// Added environment in order to store STE information in log tables.
//
// Revision 1.81  2008/12/03 15:52:59  hmeuss
// In case of init/reinit Database.java now calls DatabaseWriter.reinit (instead of init) what is absolutely necessary in order to refresh references to Oracle connections.
//
// Revision 1.80  2008/11/11 20:56:47  hmeuss
// storeLog() now has an explicit commit parameter that controls when a log is committed.
//
// Revision 1.79  2008/10/08 15:56:54  hmeuss
// Removed the getAcsLogger command
//
// Revision 1.78  2008/06/24 15:44:37  hmeuss
// improved reinit functionality
//
// Revision 1.77  2008/03/04 10:56:54  hmeuss
// Added PURGE to drop table  statements
//
// Revision 1.76  2008/02/20 12:41:25  hmeuss
// added method for testing infrastructure for logs which was missing last checkin
//
// Revision 1.75  2008/02/06 21:06:27  hmeuss
// corrected bug for schemaLoader with names being too long
//
// Revision 1.68.4.2  2008/01/22 16:11:28  hmeuss
// corrected error (file was conforming to HEAD, not ACS 7.0)
//
// Revision 1.68.4.1  2008/01/22 14:35:08  hmeuss
// logs are now committed every 1000 times, or if a log arrives with level at least CRITICAL.
//
// Revision 1.70  2008/01/22 14:31:19  hmeuss
// logs are now committed every 1000 times, or if a log arrives with level at least CRITICAL.
//
// Revision 1.69  2008/01/02 09:17:46  hmeuss
// Added test in cleanTestArea that aborts when archiveID is null. Hopefully this prevents corruption of DB info in parallel executed tests.
//
// Revision 1.68  2007/10/24 15:06:21  hmeuss
// Added retrieval methods for logs and some more infrastructure
//
// Revision 1.67  2007/10/24 09:56:42  hmeuss
// changed to new relational storage
//
// Revision 1.66  2007/08/03 07:41:51  hmeuss
// added name for primary key on schema tables
//
// Revision 1.65  2007/08/02 15:52:22  hmeuss
// The xml_SCHEMANAME_entities tables are now created with a primary key on UID.
//
// Revision 1.64  2007/07/12 14:46:51  hmeuss
// added method updateXML
//
// Revision 1.63  2007/07/05 13:27:01  hmeuss
// Changed ID storage in Oracle: Now sequences are used
//
// Revision 1.62  2007/04/20 09:24:13  hmeuss
// improved sanity check and the test for it
//
// Revision 1.61  2007/04/18 15:44:39  hmeuss
// corrected wrong return value in checkNamespaces and checkSchemaConsistency
//
// Revision 1.60  2007/04/18 15:37:31  hmeuss
// corrected wrong return value in checkHistory
//
// Revision 1.59  2007/04/16 09:54:53  hmeuss
// improved logging
//
// Revision 1.58  2007/04/05 08:57:34  hmeuss
// history table now gets also deleted when all schemas are removed. This fixes the bug of archiveLoadSchema -c corrupting the Database.
//
// Revision 1.57  2007/03/22 13:38:39  hmeuss
// added command line tool for archiveSanityCheck
//
// Revision 1.56  2007/03/01 15:54:31  hmeuss
// Added tests for consistency of Oracle tables and repair methods
//
// Revision 1.55  2007/02/16 13:22:29  hmeuss
// imrpoved error handling
//
// Revision 1.54  2006/11/24 08:52:02  hmeuss
// added workaround for removing schemaLocation from XML documents to be stored.
// MUST be removed soon!
//
// Revision 1.53  2006/11/23 10:16:15  hmeuss
// improved connection pool debugging, so that not always logs are emitted
//
// Revision 1.52  2006/11/16 09:45:37  hmeuss
// In Oracle, no metadata history entries are done for log messages anymore.
//
// Revision 1.51  2006/11/16 09:23:08  hmeuss
// removed bug of unclosed connection, changed connection architecture, added loads of logs for connection inspection.
//
// Revision 1.50  2006/11/09 13:01:00  hmeuss
// Improved resource handling
//
// Revision 1.49  2006/10/11 09:33:42  hmeuss
// IdenitiferRange.xsd is loaded into DB after removing all schemas.
//
// Revision 1.47  2006/10/10 12:43:28  hmeuss
// IdentifierRange.xsd is loaded into DB after removing all schemas (eg. calling archiveLoadSchema -c).
// Not tested yet, but should work without problems.
//
// Revision 1.46  2006/09/19 15:07:30  hmeuss
// Corrected bug in cleanTestArea (IdentifierRange entities were not removed).
// But there is still something wrong here!!!!
//
// Revision 1.45  2006/09/15 14:13:26  hsommer
// uses ACS-variant of Range and its exceptions
//
// Revision 1.44  2006/08/24 12:56:29  hsommer
// cleaned imports (which even had a xindice reference)
//
// Revision 1.43  2006/07/31 14:53:30  hmeuss
// removed bug (connections weren't closed) that prevented schema loader from working properly
//
// Revision 1.42  2006/07/07 09:46:36  hmeuss
// cleanDB also removes entries from history table in case of inconsistencies. This solves SPR2006077.
//
// Revision 1.41  2006/07/05 08:15:05  hmeuss
// uses now connection pooling
//
// Revision 1.40  2006/07/04 14:35:28  hmeuss
// All schemas are checkde now for consistency in init()
//
// Revision 1.39  2006/06/08 15:21:29  hmeuss
// fixed bug for the  -c option of the schema loader
//
// Revision 1.38  2006/04/25 08:40:44  hmeuss
// Changed handling of lastUID storage in Oracle
//
// Revision 1.37  2006/04/21 14:32:28  hmeuss
// New version, with test for new UIDLibrary
//
// Revision 1.36  2006/04/07 14:46:44  hmeuss
// First version using the new UID scheme
//
// Revision 1.35  2006/03/28 12:51:44  hmeuss
// added comment
//
// Revision 1.34  2006/03/03 14:47:12  hmeuss
// Removed bug in schema loading, which deleted all entries in Oracle.
// Could not check yet, whether it's working for eXist, too, but it was necessary to check in for Optical Pointing
//
// Revision 1.33  2006/03/03 12:46:38  hmeuss
// Fixed bug that schema tables are *always* created when new schema is loaded. Problem was that then all old XML entities were los.
//
// Revision 1.32  2006/02/16 12:48:44  mpasquat
// added comments about exception handling problems
//
// Revision 1.31  2005/12/09 14:07:09  hmeuss
// moved functionality from init() to constructor.
//
// Revision 1.30  2005/12/07 15:42:44  hmeuss
// cleanTest can now only be executed in test mode!
//
// Revision 1.29  2005/12/07 15:05:23  hmeuss
// SQL script for creating test DB is now installed into database jar file.
//
// Revision 1.28  2005/09/12 12:38:29  hmeuss
// Added argument to put in order to make storeLog working.
//
// Revision 1.27  2005/07/22 13:56:52  hmeuss
// removed bug in executeSQLscript
//
// Revision 1.26  2005/07/21 13:48:12  hmeuss
// replaced DB2 in log messages by Oracle
//
// Revision 1.25  2005/07/21 13:39:36  hmeuss
// changed path of SQL script, so that other modules can find it, too.
//
// Revision 1.24  2005/07/21 12:29:48  hmeuss
// Changed design of test area
//
// Revision 1.23  2005/07/19 13:37:46  hmeuss
// first steps for new test area
//
// Revision 1.22  2005/06/02 16:38:56  hmeuss
// INSERT of XMLType modified, this should solve the problem with XML docs being too big finally
//
// Revision 1.21  2005/05/31 09:36:14  hmeuss
// small changes to get the archiveQueryClient running for Oracle again, also trying to get ArchiveManager running for Oracle again
//
// Revision 1.20  2005/05/30 12:11:24  hmeuss
// changed treamtment of xml content of schema files
//
// Revision 1.19  2005/05/30 11:10:20  hmeuss
// added cleanDB for deleting onconsistent tables
// changed behaviour of IdentiferArchive when re-initing
//
// Revision 1.18  2005/05/27 14:19:28  hmeuss
// removed System.out.println from debugs
//
// Revision 1.17  2005/05/27 12:21:07  hmeuss
// Oracle table names not longer than 30 characters
//
// Revision 1.16  2005/04/20 16:06:48  hmeuss
// changed behaviour for removal of documents
//
// Revision 1.15  2005/01/13 10:45:08  hmeuss
// - Constructed command line query interface for Archive.
// - Added functionality to init() for Oracle.
//
// Revision 1.14  2004/10/08 14:09:00  hmeuss
// *** empty log message ***
//
// Revision 1.13  2004/10/06 14:17:47  hmeuss
// Changed the way schemas are stored (now using XMLtype!)
//
// Revision 1.12  2004/09/23 11:59:14  hmeuss
// Oracle DatabaseReader now creates a new connection object in every method call.
//
// Added ModuleCriticalException to more methods of the internal IF
//
// Revision 1.11  2004/07/20 14:46:58  hmeuss
// corrected some errors
//
// Revision 1.10  2004/07/15 09:46:14  hmeuss
// Removedf superfluous jar files
//
// Revision 1.9  2004/06/29 14:18:48  hmeuss
// Changed config file from XML to Java properties
//
// Revision 1.8  2004/06/01 15:53:12  hmeuss
// Added treatment of virtual documents
//
// Revision 1.7  2004/06/01 08:44:10  hmeuss
// Using JDBC timestamp instead of String representation in SQL statements.
// Changed ArchiveTimeStamp to use java.sql.Timestamp instead of Date.
// Removed references to DB2 from InternalIFFactory.
//
// Revision 1.6  2004/05/28 09:20:45  hmeuss
// Throws NamespaceDefinedException, when a namespace prefix is registered that already exists
//
// Revision 1.5  2004/05/27 11:07:10  hmeuss
// Added oracleLocation to config file and DBConfiguration.
// Made DBConfiguration a singleton class.
//
// Revision 1.4  2004/05/27 09:04:00  hmeuss
// dummy adaption to new DocumentData
//
// Revision 1.3  2004/05/07 08:52:50  hmeuss
// Adapted to ISO timestamps. schemas are stored as CLOBs again (not XMLTYPE).
//
// Revision 1.2  2004/05/05 08:45:55  hmeuss
// *** empty log message ***
//
// Revision 1.1  2004/04/05 13:59:29  hmeuss
// Internal IF implementation adapted to Oracle
//
// Revision 1.8  2004/03/18 09:49:33  hmeuss
// changed configuration to work with testStart, testEnd
//
// Revision 1.7  2004/02/17 14:37:32  hmeuss
// Added some schema treatment
//
// Revision 1.6  2004/02/11 16:20:09  hmeuss
// *** empty log message ***
//
// Revision 1.5  2004/01/29 14:17:05  hmeuss
// Adapted to the new interface
//
// Revision 1.4  2004/01/29 10:21:24  hmeuss
// *** empty log message ***
//
// Revision 1.3  2004/01/23 15:11:45  hmeuss
// Added permission check in DB2CursorImpl
//
// Revision 1.2  2004/01/21 16:18:04  hmeuss
// Added class DBConfiguration that reads and parses config file
//
// Revision 1.1  2004/01/19 16:29:46  hmeuss
// Splitted functionality of Db2Database into Db2DatabaseReader, Db2DatabaseWriter, Db2DatabaseCache
// 
package alma.archive.database.oracle;

import java.io.BufferedReader;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.sql.Timestamp;
import java.sql.Types;
import java.util.HashMap;
import java.util.Iterator;
import java.util.HashSet;
import java.util.Set;
import java.util.Vector;
import java.util.logging.Level;
import java.util.logging.Logger;

import oracle.jdbc.OraclePreparedStatement;
import oracle.sql.ARRAY;
import oracle.sql.ArrayDescriptor;
import oracle.xdb.XMLType;

import alma.ArchiveIdentifierError.wrappers.AcsJArchiveIdentifierErrorEx;
import alma.ArchiveIdentifierError.wrappers.AcsJIdentifierUnexpectedEx;
import alma.ArchiveIdentifierError.wrappers.AcsJRangeExhaustedEx;
import alma.archive.database.helpers.DBConfiguration;
import alma.archive.database.helpers.DatabaseHelper;
import alma.archive.database.interfaces.InternalIFFactory;
import alma.archive.exceptions.ArchiveException;
import alma.archive.exceptions.ArchiveGeneralException;
import alma.archive.exceptions.ArchiveSyntaxException;
import alma.archive.exceptions.ModuleCriticalException;
import alma.archive.exceptions.access.PermissionDeniedException;
import alma.archive.exceptions.general.DatabaseException;
import alma.archive.exceptions.general.EntityAlreadyDeletedException;
import alma.archive.exceptions.general.EntityDoesNotExistException;
import alma.archive.exceptions.general.EntityExistsException;
import alma.archive.exceptions.general.EntityUndeletedException;
import alma.archive.exceptions.general.NamespaceDefinedException;
import alma.archive.exceptions.user.RoleAlreadyExistsException;
import alma.archive.exceptions.user.RoleDoesNotExistException;
import alma.archive.exceptions.user.RoleNotAssignedException;
import alma.archive.exceptions.user.UserAlreadyExistsException;
import alma.archive.exceptions.user.UserDoesNotExistException;
import alma.acs.container.archive.Range;
import alma.acs.logging.ClientLogManager;
import alma.archive.wrappers.ArchiveTimeStamp;
import alma.archive.wrappers.Permissions;

import com.cosylab.logging.engine.log.LogTypeHelper;

/**
 * 
 * class containing primitives for write access to Oracle
 * 
 * @author hmeuss
 * 
 */
public class DatabaseWriter {

	private DatabaseCache dbCache;

	// TODO close all connections in finally blocks

	Connection readConn; // for readOnly accesses

	DatabaseConnectionPool ocpds; // data source for pooling connections

	/* Should be set by setLogger() */
	private Logger logger = Logger.getAnonymousLogger();

	// ClientLogManager.getAcsLogManager().getLoggerForApplication("DatabaseWriter",
	// false);// Logger.global;

	/* for the singleton pattern: */
	private static DatabaseWriter database;

	/* PreparedStatement for log entries: */
	PreparedStatement logStmt = null;

	Connection logConn = null;

	// ////////////////////////////////////////////////////////////////////
	// copy table name definitions from DBConfig //
	// ////////////////////////////////////////////////////////////////////

	/* some definitions copied from DBConfig: */

	/* table names for general tables */
	protected static final String tableName_schemas = DBConfig.tableName_schemas;

	/* table name suffixes for schema specific tables */
	protected static final String tableName_xml_history = DBConfig.tableName_xml_history;

	//protected static final String tableName_meta_history = DBConfig.tableName_meta_history;

	protected static final String tableNameSuffix_side_unique = DBConfig.tableNameSuffix_side_unique;

	protected static final String tableNameSuffix_side = DBConfig.tableNameSuffix_side;

	protected static String tableName_metaInf = DBConfig.tableName_metaInf;

	protected static String tableName_users = DBConfig.tableName_users;

	protected static String tableName_roles = DBConfig.tableName_roles;

	protected static String tableName_userRoles = DBConfig.tableName_userRoles;

	protected static String tableName_schemaNamespaces = DBConfig.tableName_schemaNamespaces;

	protected static String tableName_namespaces = DBConfig.tableName_namespaces;

	/* column names for all tables */
	protected static final String colName_uid = DBConfig.colName_uid;

	protected static final String colName_schemaName = DBConfig.colName_schemaName;

	protected static final String colName_version = DBConfig.colName_version;

	protected static final String colName_schemaContents = DBConfig.colName_schemaContents;

	protected static final String colName_timestamp = DBConfig.colName_timestamp;

	protected static final String colName_xml = DBConfig.colName_xml;

	protected static final String colName_schemaUid = DBConfig.colName_schemaUid;

	protected static final String colName_owner = DBConfig.colName_owner;

	protected static final String colName_deleted = DBConfig.colName_deleted;

	protected static final String colName_readPermissions = DBConfig.colName_readPermissions;

	protected static final String colName_writePermissions = DBConfig.colName_writePermissions;

	protected static final String colName_hidden = DBConfig.colName_hidden;

	protected static final String colName_dirty = DBConfig.colName_dirty;

//	protected static final String colName_metaColumn = DBConfig.colName_metaColumn;

//	protected static final String colName_newValue = DBConfig.colName_newValue;

	protected static String colName_paramName = DBConfig.colName_paramName;

	protected static String colName_paramValue = DBConfig.colName_paramValue;

	protected static String colName_roleName = DBConfig.colName_roleName;

	protected static String colName_userName = DBConfig.colName_userName;

	protected static String colName_prefix = DBConfig.colName_prefix;

	protected static String colName_namespace = DBConfig.colName_namespace;

	protected static String colName_virtual = DBConfig.colName_virtual;

	/*
	 * values for columnNames in meta_history table:
	 */
	/*
	protected static final short metaHist_schemaUID = DBConfig.metaHist_schemaUID;

	protected static final short metaHist_owner = DBConfig.metaHist_owner;

	protected static final short metaHist_deletedHiddenDirty = DBConfig.metaHist_deletedHiddenDirty;

	protected static final short metaHist_permissions = DBConfig.metaHist_permissions;
*/
	
	// needed for Arrays in Oracle
	ArrayDescriptor logArrayDesc;

	// ////////////////////////////////////////////////////////////////////
	// Class organization //
	// ////////////////////////////////////////////////////////////////////

	/**
	 * 
	 */
	private DatabaseWriter() throws DatabaseException {
		super();
		dbCache = DatabaseCache.instance();

		try {
			ocpds = DatabaseConnectionPool.instance(logger);
		} catch (SQLException e) {
			throw new DatabaseException("Could not initialize connection pool"
					+ e);
		}

		reinit();
	}

	// used by archiveSanityCheck
	public static void main(String[] args) throws ModuleCriticalException,
			DatabaseException, IOException {
		// for debugging and testing only
		/*
		 * DatabaseWriter.instance().init();
		 * 
		 * DatabaseWriter.instance().logger = Logger.global;
		 * 
		 * DatabaseWriter.instance().executeSQLscript(
		 * "../scripts/OracleSetup/AlmaTestDbCreateTables.sql");
		 */

		DatabaseReader.instance().init();
		DatabaseWriter writer = DatabaseWriter.instance();
	
		
		 boolean delete = false; boolean repair = false; if (args.length > 0 &&
		 args[0].equals("repair")) { repair = true; if (args.length > 1 &&
		 args[1].equals("delete")) { delete = true; } }
		 
		 writer.checkHistory(delete, true); 
		 // writer.checkLastUID(repair,true); 
		 writer.checkNamespaces(repair, true);
		 writer.checkSchemaConsistency(delete, true);
		 
	}

	protected void init() throws DatabaseException {
		// unused in the moment, everything went to reinit

	}

	public synchronized void reinit() throws DatabaseException {
		// probably we have to close already open connections:
		try {
			if (readConn != null) {
				ocpds.close(readConn);
			}
		} catch (Exception e) {
			logger.fine("Ignoring SQL exception during reinit: " + e);
		}
		try {
			if (logStmt != null) {
				logStmt.close();
			}
		} catch (Exception e) {
			logger.fine("Ignoring SQL exception during reinit: " + e);
		}

		try {
			if (logConn != null) {
				ocpds.close(logConn);
			}
		} catch (Exception e) {
			logger.fine("Ignoring SQL exception during reinit: " + e);
		}

		// Connect to the database
		readConn = null;
		try {
			readConn = ocpds.getConnection("WRITER(rdConn)");
			logger.info("Created Oracle read connection");
		} catch (SQLException e) {
			logger.severe("Could not initialize JDBC connection to Oracle");
			throw new DatabaseException(e);
		}

		// create preparedStatement for log entries (Is it correct here? What
		// happens if it "breaks" sometime?)
		String sql = "INSERT INTO " + DBConfig.tableName_logging + " ("
				+ DBConfig.colName_log_level + ", "
				+ DBConfig.colName_log_TimeStamp + ", "
				+ DBConfig.colName_log_File + ", " + DBConfig.colName_log_Line
				+ ", " + DBConfig.colName_log_Routine + ", "
				+ DBConfig.colName_log_SourceObject + ", "
				+ DBConfig.colName_log_Host + ", "
				+ DBConfig.colName_log_Process + ", "
				+ DBConfig.colName_log_Context + ", "
				+ DBConfig.colName_log_Thread + ", "
				+ DBConfig.colName_log_LogId + ", "
				+ DBConfig.colName_log_Priority + ", "
				+ DBConfig.colName_log_Uri + ", "
				+ DBConfig.colName_log_Audience + ", "
				+ DBConfig.colName_log_Array + ", "
				+ DBConfig.colName_log_Antenna + ", "
				+ DBConfig.colName_log_dataStrings + ", "
				+ DBConfig.colName_log_message + ", "
				+ DBConfig.colName_log_xml + ", " + DBConfig.colName_log_ste
				+ ") VALUES (?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?)";
		logger.finest("Constructing PreparedStatement for Log entries: " + sql);
		try {
			logConn = ocpds.getConnection("WR, writeLogs", false);
			// for logs, we don't cxommit every insert
			logConn.setAutoCommit(false);
			logStmt = logConn.prepareStatement(sql);
			logArrayDesc = ArrayDescriptor.createDescriptor("STRING_VARRAY",
					logConn);
		} catch (SQLException e) {
			logger
					.severe("Could not create PreparedStatement for Log entries. NO LOGS WILL BE STORED IN ORACLE!");
		}

		checkSchemaConsistency(false, false);
	}

	/**
	 * For implementing the singleton pattern.
	 * 
	 * @return the instance of this class.
	 */
	public static DatabaseWriter instance() throws DatabaseException {
		if (database == null) {
			database = new DatabaseWriter();
		}
		return database;
	}

	/**
	 * @param log
	 */
	public void setLogger(Logger log) {
		logger = log;
	}

	// ////////////////////////////////////////////////////////////////////
	// Primitives for directly implementing methods from Internal IF //
	// ////////////////////////////////////////////////////////////////////

	// adds an XML string as the last child of the root node to an existing
	// document
	// just a test version in the moment
	public void updateXML(URI uid, String schema, String newChild)
			throws EntityDoesNotExistException, DatabaseException,
			ModuleCriticalException {
		if (!DatabaseReader.instance().exists(uid)) {
			throw new EntityDoesNotExistException(uid.toString());
		}
		String sql = "UPDATE " + DBConfig.schemaTabName(schema) + " SET "
				+ colName_xml + " =  appendChildXML(xml, '/*', XMLType('"
				+ newChild + "')) WHERE " + colName_uid + "='" + uid + "'";
		logger.finest("Executing SQL: " + sql);
		// System.out.println("========== SQL: "+sql);
		Statement stmt = null;
		Connection conn = null;
		PreparedStatement pstmt = null;

		try {
			conn = ocpds.getConnection("WR, updateXML(" + uid + ")", true);
			stmt = conn.createStatement();
			stmt.execute(sql);
			

			// and now the history table:
			// In the moment, the xml column in the history table is NOT
			// XMLTYPE... Therefore we have to go the hard way
			// TODO: improve this

			/* add to xml_history */
			sql = "INSERT ALL INTO " + tableName_xml_history + "("
					+ colName_uid + ", " + colName_timestamp + ", "
					+ colName_xml + ", " + colName_schemaName + ") VALUES ('"
					+ uid + "', ?, xmlclob, '" + schema
					+ "') SELECT xmltype.getclobval(entTab." + colName_xml
					+ ") xmlclob FROM " + DBConfig.schemaTabName(schema)
					+ " entTab WHERE " + colName_uid + "='" + uid + "'";
			logger.finest("Executing SQL: " + sql);
			pstmt = (OraclePreparedStatement) conn.prepareStatement(sql);
			pstmt.setTimestamp(1, new ArchiveTimeStamp().getTimestamp());
			pstmt.executeUpdate();
			conn.commit();
			
		} catch (SQLException e) {
			throw new DatabaseException("Could not update document " + uid
					+ ": " + e.toString());
		} finally {
			try {
				stmt.close();
				pstmt.close();
				ocpds.close(conn);
			} catch (SQLException e) {
				logger.fine("Ignoring SQL Exception when closing resources: "
						+ e.toString());
			}
		}
	}

	/**
	 * Stores logs in Oracle. The list of parameters (15 attribute values) is
	 * the following:
	 * "timeStamp","file","Line","Routine","SourceObject","Host","Process","Context","Thread","LogId","Priority","Uri","Audience","Array","Antenna"
	 * timestamp is the only required value, the others might be null. The first
	 * parameter is the log message itself, while the second parameter is the
	 * integer value of the log level. The parameter dataStrings is an array of
	 * data strings as defined in the log messages.
	 * 
	 * steEnv identifies the STE we are working on and is stored in a separate
	 * column.
	 * 
	 * only if commit=true, the transaction is committed.
	 * 
	 * @throws DatabaseException
	 */
	public void storeLog(String message, int level, String attributeValue,
			String attributeValue2, String attributeValue3,
			String attributeValue4, String attributeValue5,
			String attributeValue6, String attributeValue7,
			String attributeValue8, String attributeValue9,
			String attributeValue10, String attributeValue11,
			String attributeValue12, String attributeValue13,
			String attributeValue14, String attributeValue15,
			String[] dataStrings, String xml, boolean commit, String steEnv)
			throws DatabaseException {
		if (attributeValue == null) {
			throw new DatabaseException(
					"Could not store this log, no timestamp provided: " + xml);
		}
		if (logStmt == null) {
			throw new DatabaseException(
					"Could not store this log, PreparedStatement for logs is null:"
							+ xml);
		}

		// // System.out.println("dbWriter");

		try {
			logStmt.setInt(1, level);

			// System.out.println("1");

			if (attributeValue != null) {
				logStmt.setTimestamp(2, new ArchiveTimeStamp(attributeValue)
						.getTimestamp());
			} else {
				logStmt.setNull(2, Types.TIMESTAMP);
			}
			// System.out.println("2");
			if (attributeValue2 != null) {
				logStmt.setString(3, attributeValue2);
			} else {
				logStmt.setNull(3, Types.VARCHAR);
			}
			// System.out.println("3");
			if (attributeValue3 != null) {
				logStmt.setInt(4, Integer.parseInt(attributeValue3));
			} else {
				logStmt.setNull(4, Types.INTEGER);
			}
			// System.out.println("4");
			if (attributeValue4 != null) {
				logStmt.setString(5, attributeValue4);
			} else {
				logStmt.setNull(5, Types.VARCHAR);
			}
			// System.out.println("5");
			if (attributeValue5 != null) {
				logStmt.setString(6, attributeValue5);
			} else {
				logStmt.setNull(6, Types.VARCHAR);
			}
			// System.out.println("6");
			if (attributeValue6 != null) {
				logStmt.setString(7, attributeValue6);
			} else {
				logStmt.setNull(7, Types.VARCHAR);
			}
			// System.out.println("7");
			if (attributeValue7 != null) {
				logStmt.setString(8, attributeValue7);
			} else {
				logStmt.setNull(8, Types.VARCHAR);
			}
			// System.out.println("8");
			if (attributeValue8 != null) {
				logStmt.setString(9, attributeValue8);
			} else {
				logStmt.setNull(9, Types.VARCHAR);
			}
			// System.out.println("9");
			if (attributeValue9 != null) {
				logStmt.setString(10, attributeValue9);
			} else {
				logStmt.setNull(10, Types.VARCHAR);
			}
			// System.out.println("0");
			if (attributeValue10 != null) {
				logStmt.setString(11, attributeValue10);
			} else {
				logStmt.setNull(11, Types.VARCHAR);
			}
			// System.out.println("1");
			if (attributeValue11 != null) {
				logStmt.setString(12, attributeValue11);
			} else {
				logStmt.setNull(12, Types.VARCHAR);
			}
			// System.out.println("2");
			if (attributeValue12 != null) {
				logStmt.setString(13, attributeValue12);
			} else {
				logStmt.setNull(13, Types.VARCHAR);
			}
			// System.out.println("3");
			if (attributeValue13 != null) {
				logStmt.setString(14, attributeValue13);
			} else {
				logStmt.setNull(14, Types.VARCHAR);
			}
			// System.out.println("4");
			if (attributeValue14 != null) {
				logStmt.setString(15, attributeValue14);
			} else {
				logStmt.setNull(15, Types.VARCHAR);
			}
			// System.out.println("5");
			if (attributeValue15 != null) {
				logStmt.setString(16, attributeValue15);
			} else {
				logStmt.setNull(16, Types.VARCHAR);
			}
			// System.out.println("6");
			// if (dataStrings.length>0) {
			ARRAY dataArray = new ARRAY(logArrayDesc, logConn, dataStrings);
			((OraclePreparedStatement) logStmt).setArray(17, dataArray);
			// } else {
			// logStmt.setNull(17, Types.ARRAY);
			// }
			// System.out.println("7");
			if (message != null) {
				logStmt.setString(18, message);
			} else {
				logStmt.setNull(18, Types.VARCHAR);
			}
			if (xml != null) {
				logStmt.setString(19, xml);
			} else {
				logStmt.setNull(19, Types.VARCHAR);
			}
			if (steEnv != null) {
				logStmt.setString(20, steEnv);
			} else {
				logStmt.setNull(20, Types.VARCHAR);
			}
			// System.out.println("update");
			logStmt.execute();
			// System.out.println("finshed");
			// logStmt.clearParameters();

			// and probably commit:
			if (commit || level >= LogTypeHelper.CRITICAL.acsCoreLevel.value) {
				logConn.commit();
			}

		} catch (SQLException e) {
			throw new DatabaseException("Could not store log. " + e.toString());
		}

	}

	/**
	 * 
	 * store document with all necessary information.
	 * 
	 * If newEntity is true, the document to be stored is new, else it is
	 * already existing and updated.
	 * 
	 * If owner or permissions are null, they are not modified in the stored
	 * metadata
	 * 
	 * If emitLogs is false, no logs are emitted *and* the metadata is not
	 * stored in the metadata histort table.
	 * 
	 * @param uid
	 * @param xml
	 * @param schema
	 * @param schemaName
	 * @param owner
	 * @param permissions
	 * @param user
	 * @param stamp
	 */
	public synchronized void put(URI uid, ArchiveTimeStamp stamp, String xml,
			URI schema, String schemaName, String owner,
			Permissions permissions, String user, boolean newEntity,
			boolean emitLogs) throws EntityDoesNotExistException,EntityExistsException,
			DatabaseException, PermissionDeniedException,
			ModuleCriticalException {

		String sql = null;
		OraclePreparedStatement pstmt = null;
		Connection conn = null;

		// workaround for documents which specify a schema location, hopefully
		// not necessary, since it
		// requires partial parsing of the document
		xml = removeSchemaLocation(xml);

		try {
		conn = ocpds.getConnection("WR, put(" + uid + ")", emitLogs);
		} catch (Throwable e) {
			logger.severe("Could not create DB connection.");
			throw new DatabaseException("Could not create DB connection: \n"+e);
		}
		
		try {
			/* check permissions (only if entity already exists) */
			if (!newEntity) {

				/* check permissions, will throw exception */
				Database.instance().dbReader.checkPermission(uid, schemaName,
						user, true);
				// last argument: write = true.

			}

			// check permissions and existence of entity:
			boolean alreadyExists=true;
			try{
			Database.instance().dbReader.checkPermission(uid, schemaName,
					user, true);
			// last argument: write = true.
			} catch (EntityDoesNotExistException e) {
				alreadyExists=false;
				if (!newEntity) {
					throw new EntityDoesNotExistException(uid.toString());
				}
			}
			if (newEntity&&alreadyExists) {
				throw new EntityExistsException(uid.toString());
			}
			
			/*
			 * If the entitity didn't exist before, we don't delete something.
			 * Also, the INSERT into _entities is different, and metadata
			 * history is updated.
			 */
			if (newEntity) {
				/* Add entity to _entities. */
				// default entity values for new entities:
				int deleted = 0; // false
				int hidden = 0; // false
				int dirty = 0; // false
				int virtual = 0; // false
				try {
					sql = "INSERT INTO " + DBConfig.schemaTabName(schemaName)
					+ " (" + colName_uid + ", " + colName_timestamp
					+ ", " + colName_xml + ", " + colName_schemaUid
					+ ", " + colName_owner + ", " + colName_deleted
					+ ", " + colName_readPermissions + ", "
					+ colName_writePermissions + ", " + colName_hidden
					+ ", " + colName_dirty
					+ ", "
					+ colName_virtual
					+ ") VALUES ('"
					+ uid
					//+ "', ?, ?"
					//+ ", '"
					+ "', ?, ?" + ",'"  // we cannot use XMLTYPE, throws exception with docs>4k
					+ schema + "', '" + owner + "', " + deleted + ", '"
					+ permissions.getRead() + "', '"
					+ permissions.getWrite() + "', " + hidden + ", "
					+ dirty + ", " + virtual + ")";
								
					if (emitLogs) {
						logger.finest("Executing SQL: " + sql);
					}
					pstmt = (OraclePreparedStatement) conn
							.prepareStatement(sql);
					pstmt.setTimestamp(1, stamp.getTimestamp());
					
					//pstmt.setStringForClob(2, xml);
					// pstmt.setCharacterStream(2, new StringReader(xml),
					// xml.length());
					
					
					XMLType xmlInsert = XMLType.createXML(conn, xml);


					xml=null;  // please, Mr. Garbage Collector...

					pstmt.setObject(2, xmlInsert);

					pstmt.executeUpdate();
					xmlInsert=null;
					
					// add entity metadata to meta history table.
					/*
					if (emitLogs) {
						metaHistAddNewEnt(uid, schema, owner, deleted, hidden,
								dirty, permissions, conn);
					}
					*/

				} catch (SQLException e) {
					if (emitLogs) {
						logger.warning("Oracle error: " + e.toString());
					}
					try {
						conn.rollback();
					} catch (SQLException e1) {
						if (emitLogs) {
							logger
									.warning("Could not rollback transaction. CHECK UID "
											+ uid);
						}
					}
					throw new DatabaseException(e);
				} finally {
					try {
						pstmt.close();
					} catch (Throwable e) {
						if (emitLogs) {
							logger
									.fine("Ignoring SQL exception when closing resources "
											+ e.toString());
						}
					}
				}
			} else {
				// the entity already existed in the DB
				try {
					/* update entity (timestamp, xml) in _entities. */
					sql = "UPDATE " + DBConfig.schemaTabName(schemaName)
							+ " SET " + colName_timestamp + "= ?, "
							+ colName_xml + "= ? WHERE " + colName_uid
							// + colName_xml + "=XMLTYPE(?) WHERE " +
							// colName_uid
							+ "='" + uid + "'";
					pstmt = (OraclePreparedStatement) conn
							.prepareStatement(sql);
					pstmt.setTimestamp(1, stamp.getTimestamp());
					// pstmt.setStringForClob(2, xml);

					XMLType xmlInsert = XMLType.createXML(conn, xml);
					pstmt.setObject(2, xmlInsert);

					if (emitLogs) {
						logger.finer("Executing SQL: " + sql);
					}
					pstmt.executeUpdate();
				} catch (SQLException e) {
					if (emitLogs) {
						logger.warning("SQL command raised exception: "
								+ e.toString());
					}
					try {
						conn.rollback();
					} catch (SQLException e1) {
						if (emitLogs) {
							logger
									.warning("Could not rollback transaction. CHECK UID "
											+ uid);
						}
					}
					throw new DatabaseException(e);
				} finally {
					try {
						pstmt.close();
					} catch (SQLException e) {
						if (emitLogs) {
							logger
									.info("Ignoring SQL exception when closing resources: "
											+ e.toString());
						}
					}
				}
			}
			
			try {
				

				/* add to xml_history */
				
				// This version reads the XML from the entities table, less space consuming for the JDBC client here.
				sql = "INSERT ALL INTO " + tableName_xml_history + "("
				+ colName_uid + ", " + colName_timestamp + ", "
				+ colName_xml + ", " + colName_schemaName + ") VALUES ('"
				+ uid + "', ?, xmlclob, '" + schemaName
				+ "') SELECT xmltype.getclobval(entTab." + colName_xml
				+ ") xmlclob FROM " + DBConfig.schemaTabName(schemaName)
				+ " entTab WHERE " + colName_uid + "='" + uid + "'";

//				The following version constructed the SQL containing the actual XML: Too space consuming for huge XML files.
//				
//				sql = "INSERT INTO " + tableName_xml_history + "("
//						+ colName_uid + ", " + colName_timestamp + ", "
//						+ colName_xml + ", " + colName_schemaName
//						+ ") VALUES ('" + uid + "', ?, ?, '" + schemaName
//						+ "')";
				if (emitLogs) {
					logger.finest("Executing SQL: " + sql);
				}

				pstmt = (OraclePreparedStatement) conn.prepareStatement(sql);
				pstmt.setTimestamp(1, stamp.getTimestamp());

				//pstmt.setStringForClob(2, xml);   // was needed for the original version
				pstmt.execute();
			} catch (SQLException e) {
				if (emitLogs) {
					logger.warning("SQL command raised exception: "
							+ e.toString());
				}
				try {
					conn.rollback();
				} catch (SQLException e1) {
					if (emitLogs) {
						logger
								.warning("Could not rollback transaction. CHECK UID "
										+ uid);
					}
				}
				throw new DatabaseException(e);
			} finally {
				try {
					pstmt.close();
				} catch (SQLException e) {
					if (emitLogs) {
						logger
								.fine("Ignoring SQL Exception when closing resources: "
										+ e.toString());
					}
				}
			}

			
			//finally commit:
			try {
				conn.commit();
			} catch (SQLException e1) {
				try {
					if (emitLogs) {
						logger
								.info("Could not commit last statements, trying rollback...");
					}
					conn.rollback();
				} catch (SQLException e) {
					if (emitLogs) {
						logger
								.warning("Could not rollback transaction. CHECK UID "
										+ uid);
					}
				}
				throw new DatabaseException("put failed, could not commit.");
			}
		} finally {
			try {
				ocpds.close(conn, emitLogs);
			} catch (Throwable e1) {
				if (emitLogs) {
					logger.warning("Could not close connection..."
							+ e1.getMessage());
				}
			}
		}
	}

	/**
	 * 
	 * workaorund for documents which specify a schema location. These documents
	 * are modified in the schema location info is removed
	 * 
	 * @param xml
	 *            XML string, which is possibly modified
	 */
	private static String removeSchemaLocation(String xml) {
		return xml.replaceFirst(
				"xsi:((noNamespaceS)|s)chemaLocation=[\"'].*?[\"']", "");
		// return
		// xml.replace("xsi:[(noNamespaceS)s]chemaLocation=[\"'].*?[\"']", "");
	}

	/**
	 * Set the deleted flag of specified entity to newDelete.
	 * 
	 * Possibly we should just call setFlag, since the implementation is already
	 * there.
	 * 
	 * @param uid
	 * @param user
	 * @param newDelete
	 */
	public synchronized void un_delete(URI uid, String user, boolean newDelete)
			throws ArchiveGeneralException, PermissionDeniedException {
		Statement stmt = null;
		String sql;
		ResultSet rs = null;
		Connection conn = null;

		// String/shortint encoding for new value of deleted flag
		String newDel = newDelete ? "1" : "0";

		String schemaName;

		int docDeleted, docHidden, docDirty;
		String docOwner, docPermissions;

		try {
			conn = ocpds.getConnection("WR, un_delete(" + uid + ")");
			stmt = conn.createStatement();

			/* first fetch schema name from xml_history table */
			sql = "SELECT " + colName_schemaName + " FROM "
					+ tableName_xml_history + " WHERE " + colName_uid + "='"
					+ uid + "'";
			rs = stmt.executeQuery(sql);

			if (rs.next()) {
				schemaName = rs.getString(colName_schemaName);
			} else {
				throw new EntityDoesNotExistException();
			}

			/* now, fetch visibility information from _entities table */
			sql = "SELECT " + colName_owner + ", " + colName_deleted + ", "
					+ colName_dirty + ", " + colName_hidden + ", "
					+ colName_writePermissions + " FROM "
					+ DBConfig.schemaTabName(schemaName) + " WHERE "
					+ colName_uid + "='" + uid + "'";
			logger.finest("Executing SQL: " + sql);
			rs = stmt.executeQuery(sql);
			if (rs.next()) {
				docDeleted = rs.getBoolean(colName_deleted) ? 1 : 0;
				docHidden = rs.getBoolean(colName_hidden) ? 1 : 0;
				docDirty = rs.getBoolean(colName_dirty) ? 1 : 0;
				docOwner = rs.getString(colName_owner);
				docPermissions = rs.getString(colName_writePermissions);
			} else {
				logger.warning("Entity inconsistency: " + uid
						+ "does not exist in table "
						+ DBConfig.schemaTabName(schemaName)
						+ ", but in history table.");
				throw new ArchiveGeneralException("Entity inconsistency: "
						+ uid + "does not exist in table "
						+ DBConfig.schemaTabName(schemaName)
						+ ", but in history table.");
			}

			if ((docDeleted == 1) == newDelete) {
				if (newDelete)
					throw new EntityAlreadyDeletedException(uid.toString());
				else
					throw new EntityUndeletedException(uid.toString());
			}

			if (!DatabaseHelper.checkAccessPermissions(user, docPermissions,
					docOwner)) {
				logger.info("Permission for user " + user
						+ " to un/delete document " + uid + " denied");
				throw new PermissionDeniedException(uid.toString());
			}

			/* now set deleted flag in _entities table */
			sql = "UPDATE " + DBConfig.schemaTabName(schemaName) + " SET "
					+ colName_deleted + "=" + newDel + "  WHERE " + colName_uid
					+ "='" + uid + "'";
			stmt.executeUpdate(sql);

			/* add entry to meta_history table */
			/*
			metaAdd(uid, new ArchiveTimeStamp(), metaHist_deletedHiddenDirty,
					DBConfig.metaHist_DHDEncoding(newDelete ? 1 : 0, docHidden,
							docDirty));
		*/

			conn.commit();
		} catch (SQLException e) {
			logger.warning("SQL raised exception: " + e.toString());
			try {
				conn.rollback();
			} catch (SQLException e1) {
				logger.warning("Could not rollback transaction. CHECK UID "
						+ uid);
			}
			throw new DatabaseException(e);
		} finally {
			try {
				ocpds.close(conn);
				rs.close();
				stmt.close();
			} catch (SQLException e1) {
				// ignore
				logger.warning("Exception raised while closing SQL resources.");
			}
		}
	}

	/**
	 * 
	 * Set a document flag (deleted, hidden, dirty) to a a new value. If other
	 * flags are used, this method has to be changed.
	 * 
	 * 
	 * @param uid
	 * @param flagName
	 * @param flagValue
	 * @param user
	 */
	public synchronized void setFlag(URI uid, String flagName,
			boolean flagValue, String user) throws DatabaseException,
			EntityDoesNotExistException, PermissionDeniedException,
			ArchiveSyntaxException {
		Statement stmt = null;
		String sql = null;
		ResultSet rs = null;
		Connection conn = null;
		String schemaName;

		try {
			conn = ocpds.getConnection("WR, setFlag(" + uid + ")");
			stmt = conn.createStatement();
			/* first, retrieve schema name: */
			sql = "SELECT " + colName_schemaName + " FROM "
					+ tableName_xml_history + " WHERE " + colName_uid + "='"
					+ uid + "'";
			rs = stmt.executeQuery(sql);
			if (rs.next()) {
				schemaName = rs.getString(colName_schemaName);
			} else {
				throw new EntityDoesNotExistException(uid.toString());
			}

			/*
			 * now, fetch the values for other flags in order to store
			 * everything together in meta table. necessary, if the flag is one
			 * of deleted hidden, dirty. In addition, fetch owner an
			 * writePermissions to check.
			 */
			sql = "SELECT " + colName_deleted + ", " + colName_hidden + ", "
					+ colName_dirty + ", " + colName_owner + ", "
					+ colName_writePermissions + " FROM "
					+ DBConfig.schemaTabName(schemaName) + " WHERE "
					+ colName_uid + "='" + uid + "'";
			logger.finest("Executing SQL: " + sql);
			rs = stmt.executeQuery(sql);
			if (rs.next()) {
				int del = rs.getInt(colName_deleted);
				int dirt = rs.getInt(colName_dirty);
				int hid = rs.getInt(colName_hidden);
				String docOwner = rs.getString(colName_owner);
				String writePermissions = rs
						.getString(colName_writePermissions);

				/* check the permissions */
				if (!DatabaseHelper.checkAccessPermissions(user,
						writePermissions, docOwner)) {
					throw new PermissionDeniedException(uid.toString());
				}

				/* change value of column according to flagName: */
				if (flagName.equals(DBConfig.flagNameDeleted)) {
					del = (flagValue ? 1 : 0);
				} else if (flagName.equals(DBConfig.flagNameHidden)) {
					hid = (flagValue ? 1 : 0);
				} else if (flagName.equals(DBConfig.flagNameDirty)) {
					dirt = (flagValue ? 1 : 0);
				} else {
					throw new ArchiveSyntaxException("Unknown Flag: "
							+ flagName);
				}

				/* now, update the meta history: */
				/*
				metaAdd(uid, new ArchiveTimeStamp(),
						metaHist_deletedHiddenDirty, DBConfig
								.metaHist_DHDEncoding(del, hid, dirt));
								*/
			} else {
				logger.warning("Database inconsistency: "
						+ tableName_xml_history + " and "
						+ DBConfig.schemaTabName(schemaName)
						+ " have no matching entries for: " + uid);
				throw new DatabaseException(tableName_xml_history + " and "
						+ DBConfig.schemaTabName(schemaName)
						+ " have no matching entries for: " + uid);
			}

			/* finally, add flag to _entities table: */
			sql = "UPDATE " + DBConfig.schemaTabName(schemaName) + " SET "
					+ flagName + "=" + (flagValue ? "1" : "0") + " WHERE "
					+ colName_uid + "='" + uid + "'";
			logger.finest("Executing SQL: " + sql);
			stmt.executeUpdate(sql);

			conn.commit();

		} catch (SQLException e) {
			logger.warning("SQL raised exception: " + sql);
			try {
				conn.rollback();
			} catch (SQLException e1) {
				logger.warning("Could not rollback transaction. CHECK UID "
						+ uid);
			}
			throw new DatabaseException(e);
		} finally {
			try {
				ocpds.close(conn);
				rs.close();
				stmt.close();
			} catch (SQLException e1) {
				// ignore
				logger
						.info("Ignoring SQL exception while removing Oracle resources: "
								+ e1.toString());
			}
		}

	}

	/**
	 * 
	 * delete everything (including all history information) about an entity. If
	 * keepHead is true, the head is kept in the entitie and xml_history table.
	 * 
	 * @param uid
	 * @param keepHead
	 * @param user
	 */
	public synchronized void remove(URI uid, boolean keepHead, String user)
			throws EntityDoesNotExistException, DatabaseException,
			PermissionDeniedException, ModuleCriticalException {
		String sql;
		Statement stmt = null;
		ResultSet rs = null;
		Connection conn = null;
		String schemaName;

		try {
			conn = ocpds.getConnection("WR, remove(" + uid + ")");
			stmt = conn.createStatement();
			/* retrieve schema name */
			sql = "SELECT " + colName_schemaName + " FROM "
					+ tableName_xml_history + " WHERE " + colName_uid + "='"
					+ uid + "'";
			logger.finest("Executing SQL: " + sql);
			rs = stmt.executeQuery(sql);
			if (!rs.next()) {
				throw new EntityDoesNotExistException();
			}
			schemaName = rs.getString(colName_schemaName);

			/* check permissions, will throw exception */
			Database.instance().dbReader.checkPermission(uid, schemaName, user,
					true);
			// last argument: write = true.

			/* delete from xml_history table: */
			sql = "DELETE FROM " + tableName_xml_history + " WHERE "
					+ colName_uid + "='" + uid + "'";
			logger.finest("Executing SQL: " + sql);
			stmt.executeUpdate(sql);

			/* delete from meta_history table: */
			/*
			sql = "DELETE FROM " + tableName_meta_history + " WHERE "
					+ colName_uid + "='" + uid + "'";
			logger.finest("Executing SQL: " + sql);
			stmt.executeUpdate(sql);
			*/
			if (keepHead) {
				/*
				 * copy meta information from _entities table to meta_history
				 * table
				 */
				/*
				// TODO do it in SQL or Java? Here, I decided for Java
				sql = "SELECT " + colName_schemaUid + ", " + colName_deleted
						+ ", " + colName_hidden + ", " + colName_dirty + ", "
						+ colName_owner + ", " + colName_readPermissions + ", "
						+ colName_writePermissions + " FROM "
						+ DBConfig.schemaTabName(schemaName) + " WHERE "
						+ colName_uid + "='" + uid + "'";
				logger.finest("Executing SQL: " + sql);
				rs = stmt.executeQuery(sql);
				if (!rs.next()) {
					logger.warning("Entity " + uid
							+ "existed in history table, but not in "
							+ DBConfig.schemaTabName(schemaName) + " table.");
					try {
						conn.rollback();
					} catch (SQLException e1) {
						logger
								.warning("Could not rollback transaction. CHECK UID "
										+ uid);
					}

					throw new DatabaseException("Entity " + uid
							+ "existed in history table, but not in "
							+ DBConfig.schemaTabName(schemaName) + " table.");
				}
				int deleted = rs.getInt(colName_deleted);
				int hidden = rs.getInt(colName_hidden);
				int dirty = rs.getInt(colName_dirty);
				Permissions permissions = new Permissions(rs
						.getString(colName_readPermissions), rs
						.getString(colName_writePermissions));
				URI schemaURI = null;
				String owner = rs.getString(colName_owner);

				try {
					schemaURI = new URI(rs.getString(colName_schemaUid));
				} catch (URISyntaxException e1) {
					logger.warning("Schema URI " + schemaURI.toString()
							+ " for document " + uid
							+ " not conforming to URI syntax.");
					try {
						conn.rollback();
					} catch (SQLException e2) {
						logger
								.warning("Could not rollback transaction. CHECK UID "
										+ uid);
					}
					throw new DatabaseException("Schema URI "
							+ schemaURI.toString() + " for document " + uid
							+ " not conforming to URI syntax.");
				}

				metaHistAddNewEnt(uid, schemaURI, owner, deleted, hidden,
						dirty, permissions, conn);
				*/
				
				/* copy document from _entities table to xml_history table */
				sql = "INSERT INTO " + tableName_xml_history + " SELECT "
						+ schemaName + " AS " + colName_schemaName + ", "
						+ colName_uid + ", " + colName_timestamp + ", "
						+ colName_xml + " FROM "
						+ DBConfig.schemaTabName(schemaName) + " WHERE "
						+ colName_uid + "='" + uid + "'";
				logger.finest("Executing SQL: " + sql);
				stmt.executeUpdate(sql);
			} else {
				/* delete from entities table: */
				sql = "DELETE FROM " + DBConfig.schemaTabName(schemaName)
						+ " WHERE " + colName_uid + "='" + uid + "'";
				logger.finest("Executing SQL: " + sql);
				stmt.executeUpdate(sql);
			}
			conn.commit();
		} catch (SQLException e) {
			logger.warning("SQL raised exception: " + e.toString());
			try {
				conn.rollback();
			} catch (SQLException e1) {
				logger.warning("Could not rollback transaction. CHECK UID "
						+ uid);
			}
			throw new DatabaseException(e);
		} finally {
			try {
				ocpds.close(conn);
				rs.close();
				stmt.close();
			} catch (SQLException e1) {
				logger
						.info("Ignoring SQL excpetion while closing Oracle resources: "
								+ e1.toString());
			}
		}

	}

	/**
	 * @param uid
	 * @param permissions
	 * @param user
	 */
	public synchronized void setPermission(URI uid, Permissions permissions,
			String user) throws EntityDoesNotExistException, DatabaseException,
			PermissionDeniedException, ModuleCriticalException {
		Statement stmt = null;
		String sql = null;
		ResultSet rs = null;
		Connection conn = null;

		String schemaName;
		try {
			conn = ocpds.getConnection("WR, setPermission(" + uid + ")");
			stmt = conn.createStatement();

			/* first fetch schema name from xml_history table */
			sql = "SELECT " + colName_schemaName + " FROM "
					+ tableName_xml_history + " WHERE " + colName_uid + "='"
					+ uid + "'";
			logger.finest("Executing SQL: " + sql);
			rs = stmt.executeQuery(sql);

			if (rs.next()) {
				schemaName = rs.getString(colName_schemaName);
			} else {
				try {
					rs.close();
					stmt.close();
				} catch (SQLException e1) {
					logger
							.info("Ignoring SQL exception while closing DB resources.");
				}
				throw new EntityDoesNotExistException();
			}

			/* check permissions, will throw exception */
			Database.instance().dbReader.checkPermission(uid, schemaName, user,
					true);
			// last argument: write = true.

			sql = "UPDATE " + DBConfig.schemaTabName(schemaName) + " SET "
					+ colName_readPermissions + "='" + permissions.getRead()
					+ "', " + colName_writePermissions + "= '"
					+ permissions.getWrite() + "'  WHERE " + colName_uid + "='"
					+ uid + "'";
			logger.finest("Executing SQL: " + sql);
			stmt.execute(sql);

			/*
			metaAdd(uid, new ArchiveTimeStamp(), metaHist_permissions, DBConfig
					.metaHist_permEncoding(permissions));
					*/

			conn.commit();

		} catch (SQLException e) {
			logger.warning("SQL raised exception: " + sql);
			try {
				conn.rollback();
			} catch (SQLException e1) {
				logger.warning("Could not rollback transaction. CHECK UID "
						+ uid);
			}
			throw new DatabaseException(e);
		} finally {
			try {
				ocpds.close(conn);
				rs.close();
				stmt.close();
			} catch (SQLException e1) {
				logger
						.info("Ignoring SQL exception while closing DB resources.");
			}
		}
	}

	/**
	 * @param uid
	 * @param user
	 * @param virtual
	 */
	public void setVirtual(URI uid, String user, boolean virtual)
			throws EntityDoesNotExistException, DatabaseException,
			PermissionDeniedException, ModuleCriticalException {
		String sql;
		Statement stmt = null;
		ResultSet rs = null;
		Connection conn = null;
		String schemaName;

		try {
			conn = ocpds.getConnection("WR, setVirtual(" + uid + ")");
			stmt = conn.createStatement();
			// get schema name from history:
			sql = "SELECT " + colName_schemaName + " FROM "
					+ tableName_xml_history + " WHERE " + colName_uid + " ='"
					+ uid + "'";
			logger.finer("Executing SQL: " + sql);
			rs = stmt.executeQuery(sql);
			if (rs.next()) {
				schemaName = rs.getString(colName_schemaName);
			} else {
				throw new EntityDoesNotExistException(uid.toString());
			}

			// check permission:
			DatabaseReader.instance().checkPermission(uid, schemaName, user,
					true);

			// set virtual flag:
			sql = "UPDATE " + DBConfig.schemaTabName(schemaName) + " SET "
					+ colName_virtual + " = " + (virtual ? "1" : "0");
			logger.finer("Executing SQL: " + sql);
			stmt.executeUpdate(sql);
			conn.commit();
		} catch (SQLException e) {
			logger.warning("SQL exception: " + e.toString());
			throw new DatabaseException(e);
		} finally {
			try {
				rs.close();
				stmt.close();
				ocpds.close(conn);
			} catch (SQLException e1) {
				logger
						.warning("Ignoring SQL exception while closing DB resources: "
								+ e1.toString());
			}
		}
	}

	/**
	 * close the connection to database
	 */
	public void close() throws DatabaseException, ModuleCriticalException {
		saveState();
		try {
			ocpds.close(readConn);
		} catch (SQLException e) {
			logger.warning("Problems while closing read connection to DB: "
					+ e.toString());
			//throw new DatabaseException(e);
		}
		try {
			if (logConn != null) {
				ocpds.close(logConn);
			}
		} catch (Exception e) {
			logger.warning("Problems while closing log connection to DB: "
					+ e.toString());
		}

	}


	/**
	 * Adds xmlElement as last child of the (first) element found by xPath in
	 * document uid (belonging to schema schema)
	 */
	public void addElement(URI uid, String schemaName, String xPath,
			String xmlElement, String user) throws EntityDoesNotExistException,
			DatabaseException, PermissionDeniedException,
			ModuleCriticalException {
		String sql;
		PreparedStatement stmt = null;
		Connection conn = null;

		// find out element name:
		 // we go until blank or '>', to cover case if namespace is defined
		int endName = xmlElement.indexOf(' ')<xmlElement.indexOf('>')?xmlElement.indexOf(' '):xmlElement.indexOf('>');
		String elName = xmlElement.substring(xmlElement.indexOf('<') + 1,
				endName); 

		ArchiveTimeStamp timestamp = new ArchiveTimeStamp(); // time now

		try {
			/* check permissions, will throw exception */
			Database.instance().dbReader.checkPermission(uid, schemaName, user,
					true);

			conn = ocpds.getConnection("WR,insertchild(" + uid + ")");

			sql = "UPDATE "
					+ DBConfig.schemaTabName(schemaName)
					+ " SET "
					+ colName_xml
					+ "= INSERTCHILDXML("
					+ colName_xml
					+ ", '"
					+ xPath
					+ "', '"
					+ elName
					+ "', XMLTYPE('"
					+ xmlElement
					+ "'), '"
					+ DatabaseReader.instance().constructNamespaceString(
							DatabaseReader.instance().getSchemaNamespaces(
									dbCache.schemaName2uri.get(schemaName)))
					+ "'), " + colName_timestamp + "=? WHERE " + colName_uid
					+ "='" + uid + "'";
			stmt = conn.prepareStatement(sql);
			stmt.setTimestamp(1, timestamp.getTimestamp());
			/* update entities table */
			logger.finest("Executing SQL: " + sql);
			stmt.executeUpdate();

			// and now the history table:
			// In the moment, the xml column in the history table is NOT
			// XMLTYPE... Therefore we have to go the hard way
			// TODO: improve this

			/* add to xml_history */
			sql = "INSERT ALL INTO " + tableName_xml_history + "("
					+ colName_uid + ", " + colName_timestamp + ", "
					+ colName_xml + ", " + colName_schemaName + ") VALUES ('"
					+ uid + "', ?, xmlclob, '" + schemaName
					+ "') SELECT xmltype.getclobval(entTab." + colName_xml
					+ ") xmlclob FROM " + DBConfig.schemaTabName(schemaName)
					+ " entTab WHERE " + colName_uid + "='" + uid + "'";
			logger.finest("Executing SQL: " + sql);
			stmt = (OraclePreparedStatement) conn.prepareStatement(sql);
			stmt.setTimestamp(1, timestamp.getTimestamp());
			stmt.executeUpdate();
			conn.commit();
		} catch (Exception e) {
			logger.warning("UpdateXML for "+uid+" failed.");
			logger.info("SQL command raised exception: " + e.toString());
			try {
				conn.rollback();
			} catch (SQLException e1) {
				logger.warning("Could not rollback transaction. CHECK UID "
						+ uid);
			}
			throw new DatabaseException(e);
		} finally {
			try {
				stmt.close();
				ocpds.close(conn);
			} catch (SQLException e) {
				logger.fine("Ignoring SQL Exception when closing resources: "
						+ e.toString());
			}
		}

	}


	/**
	 * The (first) element pointed at by xPath in document uid (belonging to
	 * schema schema) will be replaced by element xmlElement.
	 */
	public void updateElement(URI uid, String schemaName, String xPath,
			String xmlElement, String user) throws EntityDoesNotExistException,
			DatabaseException, PermissionDeniedException,
			ModuleCriticalException {
		String sql;
		PreparedStatement stmt = null;
		Connection conn = null;

		ArchiveTimeStamp timestamp = new ArchiveTimeStamp(); // time now

		try {
			/* check permissions, will throw exception */
			Database.instance().dbReader.checkPermission(uid, schemaName, user,
					true);

			conn = ocpds.getConnection("WR,updateXML(" + uid + ")");
			
			sql = "UPDATE "
					+ DBConfig.schemaTabName(schemaName)
					+ " SET "
					+ colName_xml
					+ "= UPDATEXML("
					+ colName_xml
					+ ", '"
					+ xPath
					+ "', XMLTYPE('"
					+ xmlElement
					+ "'), '"
					+ DatabaseReader.instance().constructNamespaceString(
							DatabaseReader.instance().getSchemaNamespaces(
									dbCache.schemaName2uri.get(schemaName)))
					+ "'), " + colName_timestamp + "=? WHERE " + colName_uid
					+ "='" + uid + "'";
			stmt = conn.prepareStatement(sql);
			stmt.setTimestamp(1, timestamp.getTimestamp());
			/* update entities table */
			logger.finest("Executing SQL: " + sql);
			stmt.executeUpdate();

			// and now the history table:
			// In the moment, the xml column in the history table is NOT
			// XMLTYPE... Therefore we have to go the hard way
			// TODO: improve this

			/* add to xml_history */
			sql = "INSERT ALL INTO " + tableName_xml_history + "("
					+ colName_uid + ", " + colName_timestamp + ", "
					+ colName_xml + ", " + colName_schemaName + ") VALUES ('"
					+ uid + "', ?, xmlclob, '" + schemaName
					+ "') SELECT xmltype.getclobval(entTab." + colName_xml
					+ ") xmlclob FROM " + DBConfig.schemaTabName(schemaName)
					+ " entTab WHERE " + colName_uid + "='" + uid + "'";
			logger.finest("Executing SQL: " + sql);
			stmt = (OraclePreparedStatement) conn.prepareStatement(sql);
			stmt.setTimestamp(1, timestamp.getTimestamp());
			stmt.executeUpdate();
			conn.commit();
		} catch (Exception e) {
			logger.warning("UpdateXML for "+uid+" failed.");
			logger.info("SQL command raised exception: " + e.toString());
			try {
				conn.rollback();
			} catch (SQLException e1) {
				logger.warning("Could not rollback transaction. CHECK UID "
						+ uid);
			}
			throw new DatabaseException(e);
		} finally {
			try {
				stmt.close();
				ocpds.close(conn);
			} catch (SQLException e) {
				logger.fine("Ignoring SQL Exception when closing resources: "
						+ e.toString());
			}
		}

	}


	/**
	 * The (first) element pointed at by xPath in document uid (belonging to
	 * schema schema) will be deleted.
	 */
	public void deleteElement(URI uid, String schemaName, String xPath,
			String user) throws EntityDoesNotExistException, DatabaseException,
			PermissionDeniedException, ModuleCriticalException {
		String sql;
		PreparedStatement stmt = null;
		Connection conn = null;

		ArchiveTimeStamp timestamp = new ArchiveTimeStamp(); // time now

		try {
			/* check permissions, will throw exception */
			Database.instance().dbReader.checkPermission(uid, schemaName, user,
					true);

			conn = ocpds.getConnection("WR,deleteXML(" + uid + ")");
			
			sql = "UPDATE "
					+ DBConfig.schemaTabName(schemaName)
					+ " SET "
					+ colName_xml
					+ "= DELETEXML("
					+ colName_xml
					+ ", '"
					+ xPath
					+ "', '"
					+ DatabaseReader.instance().constructNamespaceString(
							DatabaseReader.instance().getSchemaNamespaces(
									dbCache.schemaName2uri.get(schemaName)))
					+ "'), " + colName_timestamp + "=? WHERE " + colName_uid
					+ "='" + uid + "'";
			stmt = conn.prepareStatement(sql);
			stmt.setTimestamp(1, timestamp.getTimestamp());
			/* update entities table */
			logger.finest("Executing SQL: " + sql);
			stmt.executeUpdate();

			// and now the history table:
			// In the moment, the xml column in the history table is NOT
			// XMLTYPE... Therefore we have to go the hard way
			// TODO: improve this

			/* add to xml_history */
			sql = "INSERT ALL INTO " + tableName_xml_history + "("
					+ colName_uid + ", " + colName_timestamp + ", "
					+ colName_xml + ", " + colName_schemaName + ") VALUES ('"
					+ uid + "', ?, xmlclob, '" + schemaName
					+ "') SELECT xmltype.getclobval(entTab." + colName_xml
					+ ") xmlclob FROM " + DBConfig.schemaTabName(schemaName)
					+ " entTab WHERE " + colName_uid + "='" + uid + "'";
			logger.finest("Executing SQL: " + sql);
			stmt = (OraclePreparedStatement) conn.prepareStatement(sql);
			stmt.setTimestamp(1, timestamp.getTimestamp());
			stmt.executeUpdate();
			conn.commit();
		} catch (Exception e) {
			logger.warning("UpdateXML for "+uid+" failed.");
			logger.info("SQL command raised exception: " + e.toString());
			try {
				conn.rollback();
			} catch (SQLException e1) {
				logger.warning("Could not rollback transaction. CHECK UID "
						+ uid);
			}
			throw new DatabaseException(e);
		} finally {
			try {
				stmt.close();
				ocpds.close(conn);
			} catch (SQLException e) {
				logger.fine("Ignoring SQL Exception when closing resources: "
						+ e.toString());
			}
		}

	}

	/**
	 * 
	 * Restore state of archive as has been on timestamp, dangerous!!! All newer
	 * data is deleted!!! Schemas are left as they are!!!
	 * 
	 * @param timestamp
	 * @param user
	 */
	/*
	public synchronized void restoreState(ArchiveTimeStamp timestamp,
			String user) throws DatabaseException {
		logger.warning("===================== RESTORING ARCHIVE STATE TO "
				+ timestamp.toISOString() + " =============================");
		String sql = null;
		Statement stmt = null;
		ResultSet rs = null, rs1;
		Connection conn = null;
		OraclePreparedStatement pstmt = null;

		Vector schemas = new Vector();
		try {
			conn = ocpds.getConnection("WR, restoreState");
			stmt = conn.createStatement();

			// find all schema names 
			sql = "SELECT " + colName_schemaName + " FROM " + tableName_schemas;
			logger.finest("Executing SQL: " + sql);
			rs = stmt.executeQuery(sql);
			boolean found = false;
			while (rs.next()) {
				found = true;
				schemas.add(rs.getString(colName_schemaName));
			}

			// remove everything from all entities tables 

			// here, we could be much more efficient, in keeping all docs that
			// have neither changes in history nor meta_history
			for (Iterator iter = schemas.iterator(); iter.hasNext();) {
				String schema = (String) iter.next();
				// do not delete schemas (not really necessary, since they are
				// not part of the iterator)
				if (!DBConfig.schemaSchemaName.equals(schema)) {
					sql = "DELETE FROM " + DBConfig.schemaTabName(schema);
					logger.finest("Executing SQL: " + sql);
					stmt.execute(sql);
				}
			}

			
			// find latest docs for every uid in every history table. Schemas
			// are not treated!!!
			 
			sql = "WITH actDocs AS ( SELECT " + colName_uid + " actUid, MAX("
					+ colName_timestamp + ") actTime FROM "
					+ tableName_xml_history + " WHERE " + colName_timestamp
					+ "<= ? GROUP BY " + colName_uid
					+ ") SELECT actUid, actTime, " + colName_xml + ", "
					+ colName_schemaName + " FROM actDocs, "
					+ tableName_xml_history + " WHERE actUid=" + colName_uid
					+ " and actTime=" + colName_timestamp + " AND "
					+ colName_schemaName + "!='" + DBConfig.schemaSchemaName
					+ "'";
			logger.finest("Executing SQL: " + sql);
			pstmt = (OraclePreparedStatement) conn.prepareStatement(sql);
			pstmt.setTimestamp(1, timestamp.getTimestamp());
			rs = pstmt.executeQuery();

			while (rs.next()) {
				String uid = rs.getString("actUid");
				try {
					Timestamp time = rs.getTimestamp("actTime");
					String xml = rs.getString(colName_xml);
					String schemaName = rs.getString(colName_schemaName);
					String owner = null, schemaUID = null;
					int delHidDirty = 0;
					Permissions permissions = null;

					
					// for every uid, collect latest meta information before
					// timestamp
					 
					sql = "WITH actMeta AS ( SELECT " + colName_metaColumn
							+ " actCol, MAX(" + colName_timestamp
							+ ") actTime FROM " + tableName_meta_history
							+ " WHERE " + colName_uid + "='" + uid + "' AND "
							+ colName_timestamp + "<= ? GROUP BY "
							+ colName_metaColumn + ") SELECT actCol, "
							+ colName_newValue + " FROM actMeta, "
							+ tableName_meta_history + " WHERE actCol="
							+ colName_metaColumn + " and actTime="
							+ colName_timestamp + " AND " + colName_uid + "='"
							+ uid + "'";
					logger.finest("Executing SQL: " + sql);
					OraclePreparedStatement pstmt1 = (OraclePreparedStatement) readConn
							.prepareStatement(sql);
					pstmt1.setTimestamp(1, timestamp.getTimestamp());
					rs1 = pstmt1.executeQuery();
					while (rs1.next()) {
						short col = rs1.getShort("actCol");
						String val = rs1.getString(colName_newValue);
						switch (col) {
						case metaHist_schemaUID:
							schemaUID = val;
							break;
						case metaHist_owner:
							owner = val;
							break;
						case metaHist_deletedHiddenDirty:
							delHidDirty = Integer.parseInt(val);
							break;
						case metaHist_permissions:
							permissions = DBConfig
									.metaHist_permEncoding2Perm(val);
						}
					}

					rs1.close();
					pstmt1.close();

					// now, write the dataset to the appropriate _entities table 
					sql = "INSERT INTO "
							+ DBConfig.schemaTabName(schemaName)
							+ " ("
							+ colName_uid
							+ ", "
							+ colName_timestamp
							+ ", "
							+ colName_xml
							+ ", "
							+ colName_schemaUid
							+ ", "
							+ colName_owner
							+ ", "
							+ colName_deleted
							+ ", "
							+ colName_readPermissions
							+ ", "
							+ colName_writePermissions
							+ ", "
							+ colName_hidden
							+ ", "
							+ colName_dirty
							+ ") VALUES ('"
							+ uid
							+ "', ?, ?, '"
							+ schemaUID
							+ "', '"
							+ owner
							+ "', "
							+ DBConfig
									.metaHist_DHDEncoding2DelBool(delHidDirty)
							+ ", '"
							+ permissions.getRead()
							+ "', '"
							+ permissions.getWrite()
							+ "', "
							+ DBConfig
									.metaHist_DHDEncoding2HidBool(delHidDirty)
							+ ", "
							+ DBConfig
									.metaHist_DHDEncoding2DirBool(delHidDirty)
							+ ")";
					logger.finest("Executing SQL: " + sql);
					pstmt1 = (OraclePreparedStatement) conn
							.prepareStatement(sql);
					pstmt1.setTimestamp(1, time);
					pstmt1.setStringForClob(2, xml);
					pstmt1.execute();

					pstmt1.close();
				} catch (Exception e) {
					logger.warning("Problems with entity " + uid
							+ ". Please CHECK");
					logger.info(e.toString());
				}
			}

			// remove everything newer than timestamp from xml_history 
			sql = "DELETE FROM " + tableName_xml_history + " WHERE "
					+ colName_timestamp + "> ?";
			logger.finest("Executing SQL: " + sql);
			pstmt = (OraclePreparedStatement) conn.prepareStatement(sql);
			pstmt.setTimestamp(1, timestamp.getTimestamp());
			pstmt.executeUpdate();

			// remove everything newer than timestamp from schemas 
			sql = "DELETE FROM " + tableName_schemas + " WHERE "
					+ colName_timestamp + "> ?";
			logger.finest("Executing SQL: " + sql);
			pstmt = (OraclePreparedStatement) conn.prepareStatement(sql);
			pstmt.setTimestamp(1, timestamp.getTimestamp());
			pstmt.executeUpdate();

			// remove everything newer than timestamp from meta_history 
			sql = "DELETE FROM " + tableName_meta_history + " WHERE "
					+ colName_timestamp + "> ?";
			logger.finest("Executing SQL: " + sql);
			pstmt = (OraclePreparedStatement) conn.prepareStatement(sql);
			pstmt.setTimestamp(1, timestamp.getTimestamp());
			pstmt.executeUpdate();

			conn.commit();

		} catch (SQLException e) {
			// TODO until we know something better:
			logger.warning("SQL raised exception: " + sql);
			try {
				conn.rollback();
			} catch (SQLException e1) {
				logger
						.severe("Could not rollback transaction. CHECK SYSTEM STATUS!");
			}
			throw new DatabaseException(e);
		} finally {
			try {
				ocpds.close(conn);
				rs.close();
				stmt.close();
				pstmt.close();
			} catch (SQLException e1) {
				// ignore
				logger.info("SQL exception while closing DB resources: "
						+ e1.toString());
			}
		}
	}
	*/

	// ////////////////////////////////////////////////////////////////////
	// schema helpers //
	// ////////////////////////////////////////////////////////////////////

	/**
	 * add the schema to the schemas table and update uri2schemaName, create all
	 * necessary table for schema
	 * 
	 * @param schemaName
	 * @return
	 */
	protected synchronized void addSchemaAndTables(URI schemaURI,
			String schemaName, String schemaContents, String dadFile)
			throws DatabaseException {
		Connection conn = null;
		try {
			conn = ocpds.getConnection("WR, addSchemaAndTables(" + schemaURI
					+ ")");
		} catch (SQLException e1) {
			logger.warning("Could not fetch connection from connenction pool");
			throw new DatabaseException(
					"Could not fetch connection from database connection pool.");
		}

		// TODO add rollback, if error is thrown.
		try {
			// create all tables for this schema
			createTablesForSchema(schemaName, dadFile, conn);

			// add schema to schemas table (version number 1)
			addSchema(schemaURI, schemaName, 1, schemaContents, dadFile, conn);
		} catch (DatabaseException e) {
			try {
				ocpds.close(conn);
			} catch (SQLException e2) {
				logger.warning("Could not close connection..." + e2.toString());
			}
			throw e;
			// MPA: is it correct to rethrow e without e txt?
		}
		// commit changes
		try {
			conn.commit();
		} catch (SQLException e) {
			logger.warning("Could not commit schema changes for schema "
					+ schemaURI + "! Check it!");
		}
		try {
			ocpds.close(conn);
		} catch (SQLException e1) {
			logger.warning("Could not close connection..." + e1.getMessage());
		}
	}

	/**
	 * 
	 * create all necessary tables for the schema. Caution: Does not commit. Use
	 * conn.commit() thereafter.
	 * 
	 * @param schemaName
	 * @param dadFile:
	 *            content od DAD file
	 */
	protected void createTablesForSchema(String schemaName, String dadFile,
			Connection conn) throws DatabaseException {
		String sql;
		Statement stmt = null;
		ResultSet rs = null;

		try {
			stmt = conn.createStatement();
			// create _entities table *only* if it does not exist yet!!!

			// unfortunately, the following does not work (returned resultSet is
			// always empty). JDBC problem when accessing system.tab????
			sql = "SELECT tname FROM system.tab WHERE tname='"
					+ DBConfig.schemaTabName(schemaName).toUpperCase() + "'";
			logger.finest("Executing SQL: " + sql);
			rs = stmt.executeQuery(sql);
			if (!rs.next()) {
				logger.info("Creating table "
						+ DBConfig.schemaTabName(schemaName));
				// System.out.println("Schema tab not existing. Creating...");
				sql = "CREATE TABLE "
						+ DBConfig.schemaTabName(schemaName)
						+ " ( "
						+ colName_uid
						+ " VARCHAR(33) NOT NULL, "
						+ colName_timestamp
						+ " TIMESTAMP NOT NULL, "
						+ colName_xml
						+ " xmlTYPE, "
						+ colName_schemaUid
						+ " VARCHAR(33) NOT NULL, "
						+ colName_owner
						+ " VARCHAR(32), "
						+ colName_deleted
						+ " NUMBER(1), "
						+ colName_readPermissions
						+ " VARCHAR(8), "
						+ colName_writePermissions
						+ " VARCHAR(8), "
						+ colName_hidden
						+ " NUMBER(1), "
						+ colName_dirty
						+ " NUMBER(1), "
						+ colName_virtual
						+ " NUMBER(1), CONSTRAINT "
						+ DBConfig.schemaTabName(schemaName)
								.substring(
										0,
										(DBConfig.schemaTabName(schemaName)
												.length()) - 9)
						+ "_uid_pk PRIMARY KEY (" + colName_uid + "))";
				logger.fine("Executing: " + sql);
				stmt.executeUpdate(sql);

				/* create index on uid */
				// TODO necessary? Since uid is primary key, Oracle does it
				// // itself...
				// sql = "CREATE UNIQUE INDEX "
				// + DBConfig.schemaIndName(schemaName) + " ON "
				// + DBConfig.schemaTabName(schemaName) + " ("
				// + colName_uid + ")";
				// logger.finest("Executing SQL: " + sql);
				// stmt.executeUpdate(sql);
				// create and index side tables, use information from DAD file
				stmt.close();

				// the following belongs to the non-working access to system.tab
			}
		} catch (SQLException e) {
			logger.severe("Oracle error: " + e.getMessage());
			try {
				conn.rollback();
			} catch (SQLException e1) {
				logger.severe("Could not rollback transaction. CHECK SCHEMA "
						+ schemaName);
			}
			throw new DatabaseException(e);
		} finally {
			try {
				rs.close();
				stmt.close();
			} catch (SQLException e) {
				logger
						.info("Ignoring SQL exception while closing Oracle resources: "
								+ e.toString());
			}
		}

	}

	/**
	 * add the schema to the schemas table and update uri2schemaName. Use
	 * conn.commit() thereafter. dadFile is ignored here...
	 * 
	 * @param schemaName
	 * @return
	 */
	protected synchronized void addSchema(URI schemaURI, String schemaName,
			int schemaVersion, String schemaContents, String dadFile,
			Connection conn) throws DatabaseException {
		OraclePreparedStatement pstmt = null;

		logger.info("Inserting schema " + schemaName + " into table "
				+ tableName_schemas + " with UID " + schemaURI);

		try {
			// first add schema and dad information to schema table:

			// Oracle seems to be quite stupid with LOB handling. We first
			// insert am empty CLOB and then updathe the thing...
			String sql = "INSERT INTO " + tableName_schemas + "(" + colName_uid
					+ "," + colName_schemaName + "," + colName_version + ","
					+ colName_timestamp + "," + colName_schemaContents + ","
					+ colName_deleted + "," + colName_hidden + ","
					+ colName_dirty + ", " + colName_schemaUid + ") VALUES ('"
					+ schemaURI + "','" + schemaName + "', " + schemaVersion
					+ ", ?, " +
					// "empty_clob(), "
					// "XMLTYPE(?), "
					"?, " + "0,0,0, '" + DBConfig.schemaSchemaUri + "')";
			logger.fine("Executing: " + sql);
			pstmt = (OraclePreparedStatement) conn.prepareStatement(sql);
			pstmt.setTimestamp(1, new ArchiveTimeStamp().getTimestamp());

			// When XMLType is used for schema contents:
			pstmt.setStringForClob(2, schemaContents);
			// XMLType xmlTypeO = new XMLType(conn, schemaContents);
			// pstmt.setObject(2, xmlTypeO);

			pstmt.executeUpdate();

			// ///////////////////////////////////////////////////////////
			// the following must be used when we store XML as CLOB
			// // get the CLOB
			// // TODO check timestamp!!!!!
			// sql =
			// "SELECT "
			// + colName_xml
			// + " FROM "
			// + tableName_schemas
			// + " WHERE "
			// + colName_uid
			// + "='"
			// + schemaURI
			// + "' FOR UPDATE NOWAIT";
			// Statement stmt = conn.createStatement();
			// logger.fine("Executing: " + sql);
			// ResultSet rs = stmt.executeQuery(sql);
			// if (rs.next()) {
			// CLOB clob = (CLOB) ((OracleResultSet) rs).getClob(colName_xml);
			//
			// // write schema contents into the CLOB
			// Writer outstream = clob.setCharacterStream(1L);
			// Reader reader = new StringReader(schemaContents);
			//
			// // we read in chunks of data:
			// int size = clob.getBufferSize();
			// char[] buffer = new char[size];
			// int length = -1;
			//
			// try {
			// while ((length = reader.read(buffer, 0, size)) != -1) {
			// outstream.write(buffer, 0, length);
			// }
			// reader.close();
			// outstream.close();
			// } catch (IOException e1) {
			// rs.close();
			// logger.severe(
			// "Could not transfer schema contents to Database CLOB.");
			// throw new DatabaseException(
			// "Could not insert schema data: " + schemaURI);
			// }
			//
			// } else {
			// rs.close();
			// logger.severe(
			// "Problems when inserting LOB data of schema " + schemaURI);
			// throw new DatabaseException(
			// "Could not insert schema data: " + schemaURI);
			// }
			// rs.close();
			// stmt.close();
			// ///////////////////////////////////////////////////////////////////////////////////////////////////////

			// then, add schema to xml_history table
			sql = "INSERT INTO " + tableName_xml_history + "(" + colName_uid
					+ ", " + colName_timestamp + ", " + colName_xml + ", "
					+ colName_schemaName + ") VALUES ('" + schemaURI
					+ "', ?, ?,'" + DBConfig.schemaSchemaName + "')";
			logger.finest("Executing SQL: " + sql);
			pstmt = (OraclePreparedStatement) conn.prepareStatement(sql);
			pstmt.setTimestamp(1, new ArchiveTimeStamp().getTimestamp());
			pstmt.setStringForClob(2, schemaContents);
			pstmt.executeUpdate();

			// add new schemaURI to URI table and remove older version:
			dbCache.uri2schemaName.remove(dbCache.schemaName2uri
					.get(schemaName));
			dbCache.uri2schemaName.put(schemaURI, schemaName);
			// DatabaseCache.schemaName2uri.remove(schemaName); not necessary,,
			// will be overwritten
			dbCache.schemaName2uri.put(schemaName, schemaURI);

			/* add schema to meta_history */
			// TODO what to do with newer versions?
			/*
			if (schemaVersion == 1) {
				try {
					metaHistAddNewEnt(schemaURI, new URI(
							DBConfig.schemaSchemaUri), DBConfig.schemaOwner, 0,
							0, 0, new Permissions(), conn);
				} catch (URISyntaxException e1) {
					// this should not happen
					logger
							.warning("Illformed (hardcoded) URI for schemaSchema in DBConfig!");
				}
			}
			*/

		} catch (SQLException e) {
			logger.severe("SQL raised exception: " + e.toString());
			try {
				conn.rollback();
			} catch (SQLException e1) {
				logger.severe("Could not rollback transaction. CHECK SCHEMA "
						+ schemaURI);
			}
			throw new DatabaseException("Could not insert schema " + schemaURI);
		} finally {
			try {
				pstmt.close();
			} catch (SQLException e1) {
				logger
						.info("Ignoring SQL exception while closing Oracle resources: "
								+ e1.toString());
			}
		}
	}

	/**
	 * 
	 * convenience method for the abobve that commits (used by SchemaManager)
	 * 
	 * @param schemaURInew
	 * @param schemaName
	 * @param schemaContents
	 * @param dadFileLocation
	 */
	public void addSchema(URI schemaURInew, String schemaName,
			int schemaVersion, String schemaContents, String dadFile)
			throws DatabaseException {
		Connection conn = null;
		try {
			conn = ocpds.getConnection("WR, addSchema(" + schemaURInew + ")");
			addSchema(schemaURInew, schemaName, schemaVersion, schemaContents,
					dadFile, conn);
			conn.commit();
		} catch (SQLException e) {
			logger.warning("Addition of schema failed" + e.getMessage());
			throw new DatabaseException(e);
		} finally {
			try {
				ocpds.close(conn);
			} catch (SQLException e) {
				logger.warning("Could not close connection...");
			}
		}
	}

	/**
	 * This method removes all entities belonging to a schema. The schema itself
	 * is not removed. TODO A lot has to be reworked here: disable columns,
	 * remove namespaces? etc. TODO maybe we should call cleanSchema that does
	 * most of the necessary things
	 * 
	 * @param schemaName
	 */
	public synchronized void removeSchema(String schemaName)
			throws DatabaseException {
		Statement stmt;
		String sql;
		Connection conn = null;
		ResultSet rs = null;
		String uid;

		try {
			conn = ocpds.getConnection("WR, removeScheme(" + schemaName + ")");
			stmt = conn.createStatement();

			// first: get schema UID and delete associated namespaces
			uid = ((URI) dbCache.schemaName2uri.get(schemaName)).toString();
			// now delete namespace associations:
			// TODO delete namespaces, too?
			sql = "DELETE FROM " + tableName_schemaNamespaces + " WHERE "
					+ colName_schemaUid + "='" + uid + "'";
			logger.finest("Executing SQL: " + sql);
			stmt.executeUpdate(sql);

			sql = "DELETE FROM " + tableName_schemas + " WHERE "
					+ colName_schemaName + "='" + schemaName + "'";
			logger.finest("Executing SQL: " + sql);
			stmt.executeUpdate(sql);

			sql = "DROP TABLE " + DBConfig.schemaTabName(schemaName) + " PURGE";
			logger.finest("Executing SQL: " + sql);
			stmt.executeUpdate(sql);
			conn.commit();

			// TODO: delete entries from history table:?????

			// propagate removal to dbCache
			dbCache.uri2schemaName.remove(dbCache.schemaName2uri
					.get(schemaName));
			dbCache.schemaName2uri.remove(schemaName);

		} catch (SQLException e) {
			logger.warning("SQL raised exception: " + e.toString());
			throw new DatabaseException(e);
		} finally {
			try {
				ocpds.close(conn);
			} catch (SQLException e1) {
				logger.warning("Could not close connection...");
			}
		}

	}

	// ////////////////////////////////////////////////////////////////////
	// Namespace helpers //
	// ////////////////////////////////////////////////////////////////////

	/**
	 * Register a namespace preix in the DB. A prefix is unique (constraint in
	 * DB). In case this constraint is violated a NamespaceDefinedException is
	 * thrown.
	 * 
	 * @param prefix
	 * @param namespace
	 */
	public void registerNamespace(String prefix, URI namespace)
			throws NamespaceDefinedException, DatabaseException {
		Statement stmt = null;
		String sql;
		Connection conn = null;
		ResultSet rs = null;

		try {
			conn = ocpds.getConnection("WR, registerNamespace(" + prefix + ","
					+ namespace + ")");
			stmt = conn.createStatement();
			sql = "INSERT INTO " + tableName_namespaces + " (" + colName_prefix
					+ ", " + colName_namespace + ") VALUES ('" + prefix
					+ "', '" + namespace.toString() + "')";
			logger.finest("Executing SQL: " + sql);
			stmt.executeUpdate(sql);
		} catch (SQLException e) {
			if (e.getErrorCode() == 1) {
				// namespace was already defined!

				// before throwing an exception, we check whether the target
				// namespace is equal:
				sql = "SELECT " + colName_namespace + " FROM "
						+ tableName_namespaces + " WHERE " + colName_prefix
						+ "='" + prefix + "'";
				logger.finest("Executing SQL: " + sql);
				try {
					rs = stmt.executeQuery(sql);
					if (rs.next()) {
						String defNamespace = rs.getString(colName_namespace);
						if (!defNamespace.equals(namespace.toString())) {
							throw new NamespaceDefinedException(prefix + ": "
									+ defNamespace);
						}
					}
				} catch (SQLException e1) {
					throw new DatabaseException(e);
				} finally {
					try {
						rs.close();
					} catch (SQLException e1) {
						logger
								.warning("Ignoring SQL exception while closing DB resources: "
										+ e1.toString());
					}
				}
			} else {
				logger.warning("Could not insert namespace and prefix: "
						+ namespace + ", " + prefix);
				throw new DatabaseException(e);
			}
		} finally {
			try {
				stmt.close();
				ocpds.close(conn);
			} catch (SQLException e1) {
				logger
						.warning("Ignoring SQL exception while closing DB resources: "
								+ e1.toString());
			}
		}
	}

	/**
	 * @param prefix
	 */
	public void removeNamespace(String prefix) throws DatabaseException {
		Statement stmt = null;
		String sql;
		Connection conn = null;

		try {
			conn = ocpds.getConnection("WR, removeNamespace(" + prefix + ")");
			stmt = conn.createStatement();
			sql = "DELETE FORM " + tableName_namespaces + " WHERE "
					+ colName_prefix + "='" + prefix + "'";
			logger.finest("Executing SQL: " + sql);
			stmt.executeUpdate(sql);
		} catch (SQLException e) {
			logger.warning("Could not remove namespace prefix: " + prefix);
			throw new DatabaseException(e);
		} finally {
			try {
				stmt.close();
				ocpds.close(conn);
			} catch (SQLException e1) {
				logger
						.warning("Ignoring SQL exception while closing DB resources: "
								+ e1.toString());
			}
		}
	}

	/**
	 * removes all namespaces
	 */
	public void removeNamespaces() throws DatabaseException {
		Statement stmt = null;
		String sql;
		Connection conn = null;

		try {
			conn = ocpds.getConnection("WR, removeNamespaces");
			stmt = conn.createStatement();
			sql = "DELETE FROM " + tableName_namespaces;
			logger.finest("Executing SQL: " + sql);
			stmt.executeUpdate(sql);
		} catch (SQLException e) {
			logger.warning("Could not remove namespaces.");
			throw new DatabaseException(e);
		} finally {
			try {
				stmt.close();
			} catch (SQLException e1) {
				logger
						.warning("Ignoring SQL exception while closing DB resources: "
								+ e1.toString());
			}
		}
		try {
			stmt = conn.createStatement();
			sql = "DELETE FROM " + tableName_schemaNamespaces;
			logger.finest("Executing SQL: " + sql);
			stmt.executeUpdate(sql);
		} catch (SQLException e) {
			logger.warning("Could not remove namespaces.");
			throw new DatabaseException(e);
		} finally {
			try {
				stmt.close();
				ocpds.close(conn);
			} catch (SQLException e1) {
				logger
						.warning("Ignoring SQL exception while closing DB resources: "
								+ e1.toString());
			}
		}
	}

	/**
	 * @param prefix
	 * @param schemaUri
	 */
	public void assignNamespace(String prefix, URI schemaUri)
			throws DatabaseException {
		Statement stmt = null;
		String sql;
		Connection conn = null;

		try {
			conn = ocpds.getConnection("WR, assignNamespace(" + prefix + ","
					+ schemaUri + ")");
			stmt = conn.createStatement();
			sql = "INSERT INTO " + tableName_schemaNamespaces + " ( "
					+ colName_schemaUid + "," + colName_prefix + ") VALUES ('"
					+ schemaUri + "', '" + prefix + "')";
			logger.finest("Executing SQL: " + sql);
			stmt.executeUpdate(sql);
		} catch (SQLException e) {
			logger.warning("Could not assign namespace prefix " + prefix
					+ " to schema " + schemaUri);
			throw new DatabaseException(e);
		} finally {
			try {
				stmt.close();
				ocpds.close(conn);
			} catch (SQLException e1) {
				logger
						.warning("Ignoring SQL exception while closing DB resources: "
								+ e1.toString());
			}
		}
	}

	/**
	 * @param prefix
	 * @param schemaUri
	 */
	public void withdrawNamespace(String prefix, URI schemaUri)
			throws DatabaseException {
		Statement stmt = null;
		String sql;
		Connection conn = null;

		try {
			conn = ocpds.getConnection("WR, withdrawNamespace(" + prefix + ","
					+ schemaUri + ")");
			stmt = conn.createStatement();
			sql = "DELETE FROM " + tableName_schemaNamespaces + " WHERE "
					+ colName_schemaUid + "='" + schemaUri + "' AND "
					+ colName_prefix + "='" + prefix + "'";
			logger.finest("Executing SQL: " + sql);
			stmt.executeUpdate(sql);
		} catch (SQLException e) {
			logger.warning("Could not remove namespace prefix: " + prefix
					+ " from schema " + schemaUri);
			throw new DatabaseException(e);
		} finally {
			try {
				stmt.close();
				ocpds.close(conn);
			} catch (SQLException e1) {
				logger
						.warning("Ignoring SQL exception while closing DB resources: "
								+ e1.toString());
			}
		}
	}

	// ////////////////////////////////////////////////////////////////////
	// user helpers //
	// ////////////////////////////////////////////////////////////////////

	/**
	 * @see alma.archive.database.interfaces.UserManager#addUser(java.lang.String)
	 */
	public void addUser(String user) throws DatabaseException,
			ArchiveException, UserAlreadyExistsException {

		// MPA: excpetion UserAlreadyExists never thrown. to be checked
		String sql = null;
		Statement stmt = null;
		Connection conn = null;
		try {
			conn = ocpds.getConnection("WR, addUser(" + user + ")");
			stmt = conn.createStatement();
			sql = "INSERT INTO " + tableName_users + " (" + colName_userName
					+ ") VALUES ('" + user + "')";
			logger.finest("Executing SQL: " + sql);
			stmt.executeUpdate(sql);
			conn.commit();
		} catch (SQLException e) {
			logger.info("SQL raised exception: " + e.toString());
			throw new DatabaseException(e);
		} finally {
			try {
				stmt.close();
				ocpds.close(conn);
			} catch (SQLException e1) {
				// ignore
				logger
						.info("Ignoring SQL exception while closing Oracle resources: "
								+ e1.toString());
			}
		}

	}

	/**
	 * @see alma.archive.database.interfaces.UserManager#deleteUser(java.lang.String)
	 */
	public void deleteUser(String user) throws DatabaseException,
			ArchiveException, UserDoesNotExistException {

		// MPA: excpetion UserDoesNotExists never thrown. to be checked
		String sql = null;
		Statement stmt = null;
		Connection conn = null;
		try {
			conn = ocpds.getConnection("WR, deleteUser(" + user + ")");
			stmt = conn.createStatement();
			sql = "DELETE FROM " + tableName_users + " WHERE "
					+ colName_userName + "='" + user + "'";
			logger.finest("Executing SQL: " + sql);
			stmt.executeUpdate(sql);
			conn.commit();
		} catch (SQLException e) {
			logger.info("SQL raised exception: " + e.toString());
			throw new DatabaseException(e);
		} finally {
			try {
				ocpds.close(conn);
				stmt.close();
			} catch (SQLException e1) {
				// ignore
				logger
						.info("Ignoring SQL exception while closing Oracle resources: "
								+ e1.toString());
			}
		}

	}

	/**
	 * @see alma.archive.database.interfaces.UserManager#addRole(java.lang.String)
	 */
	public void addRole(String role) throws DatabaseException,
			ArchiveException, RoleAlreadyExistsException {

		// MPA: excpetion RoleALreadyExist never thrown. to be checked
		String sql = null;
		Statement stmt = null;
		Connection conn = null;
		try {
			conn = ocpds.getConnection("WR, addRole(" + role + ")");
			stmt = conn.createStatement();
			sql = "INSERT INTO " + tableName_roles + " (" + colName_roleName
					+ ") VALUES ('" + role + "')";
			logger.finest("Executing SQL: " + sql);
			stmt.executeUpdate(sql);
			conn.commit();
		} catch (SQLException e) {
			logger.info("SQL raised exception: " + e.toString());
			throw new DatabaseException(e);
		} finally {
			try {
				ocpds.close(conn);
				stmt.close();
			} catch (SQLException e1) {
				// ignore
				logger
						.info("Ignoring SQL exception while closing Oracle resources: "
								+ e1.toString());
			}
		}

	}

	/**
	 * @see alma.archive.database.interfaces.UserManager#deleteRole(java.lang.String)
	 */
	public void deleteRole(String role) throws DatabaseException,
			ArchiveException, RoleDoesNotExistException {
		// MPA: excpetion RoleDoesNotExists never thrown. to be checked
		String sql = null;
		Statement stmt = null;
		Connection conn = null;
		try {
			conn = ocpds.getConnection("WR, deleteRole(" + role + ")");
			stmt = conn.createStatement();
			sql = "DELETE FROM " + tableName_roles + " WHERE "
					+ colName_roleName + "='" + role + "'";
			logger.finest("Executing SQL: " + sql);
			stmt.executeUpdate(sql);
			conn.commit();
		} catch (SQLException e) {
			logger.info("SQL raised exception: " + e.toString());
			throw new DatabaseException(e);
		} finally {
			try {
				ocpds.close(conn);
				stmt.close();
			} catch (SQLException e1) {
				// ignore
				logger
						.info("Ignoring SQL exception while closing Oracle resources: "
								+ e1.toString());
			}
		}
	}

	/**
	 * @see alma.archive.database.interfaces.UserManager#assignRole(java.lang.String,
	 *      java.lang.String)
	 */
	public void assignRole(String user, String role) throws DatabaseException,
			ArchiveException, RoleDoesNotExistException,
			UserDoesNotExistException {
		// TODO check user/role existence
		// MPA: last 2 exception never thrown
		String sql = null;
		Statement stmt = null;
		Connection conn = null;
		try {
			conn = ocpds.getConnection("WR, assignRole(" + user + "," + role
					+ ")");
			stmt = conn.createStatement();
			sql = "INSERT INTO " + tableName_userRoles + " ("
					+ colName_userName + ", " + colName_roleName
					+ ") VALUES ('" + user + "', '" + role + "')";
			logger.finest("Executing SQL: " + sql);
			stmt.executeUpdate(sql);
			conn.commit();
		} catch (SQLException e) {
			logger.info("SQL raised exception: " + e.toString());
			throw new DatabaseException(e);
		} finally {
			try {
				ocpds.close(conn);
				stmt.close();
			} catch (SQLException e1) {
				// ignore
				logger
						.info("Ignoring SQL exception while closing Oracle resources: "
								+ e1.toString());
			}
		}

	}

	/**
	 * @see alma.archive.database.interfaces.UserManager#withdrawRole(java.lang.String,
	 *      java.lang.String)
	 */
	public void withdrawRole(String user, String role)
			throws DatabaseException, ArchiveException,
			RoleDoesNotExistException, UserDoesNotExistException,
			RoleNotAssignedException {

		// MPA: last 3 exceptions never thrown
		String sql = null;
		Statement stmt = null;
		Connection conn = null;
		try {
			conn = ocpds.getConnection("WR, withdrawRole(" + user + "," + role
					+ ")");
			stmt = conn.createStatement();
			sql = "DELETE FROM " + tableName_userRoles + " WHERE "
					+ colName_userName + "='" + user + "' AND "
					+ colName_roleName + "= '" + role + "'";
			logger.finest("Executing SQL: " + sql);
			stmt.executeUpdate(sql);
			conn.commit();
		} catch (SQLException e) {
			logger.info("SQL raised exception: " + e.toString());
			throw new DatabaseException(e);
		} finally {
			try {
				ocpds.close(conn);
				stmt.close();
			} catch (SQLException e1) {
				// ignore
				logger
						.info("Ignoring SQL exception while closing Oracle resources: "
								+ e1.toString());
			}
		}

	}

	/**
	 * 
	 * Assign ownership of entity uid to user newOwner
	 * 
	 * @param uid
	 * @param newOwner
	 * @param user
	 */
	public void changeOwner(URI uid, String newOwner, String user)
			throws EntityDoesNotExistException, DatabaseException,
			PermissionDeniedException, ModuleCriticalException {
		Statement stmt = null;
		String sql = null;
		ResultSet rs = null;
		Connection conn = null;

		String schemaName;
		try {
			conn = ocpds.getConnection("WR, changeOwner(" + uid + ")");
			stmt = conn.createStatement();

			/* first fetch schema name from xml_history table */
			sql = "SELECT " + colName_schemaName + " FROM "
					+ tableName_xml_history + " WHERE " + colName_uid + "='"
					+ uid + "'";
			logger.finest("Executing SQL: " + sql);
			rs = stmt.executeQuery(sql);

			if (rs.next()) {
				schemaName = rs.getString(colName_schemaName);
			} else {
				try {
					rs.close();
					stmt.close();
				} catch (SQLException e1) {
					logger
							.info("Ignoring SQL exception while closing DB resources.");
				}
				throw new EntityDoesNotExistException();
			}

			/* check permissions, will throw exception */
			Database.instance().dbReader.checkPermission(uid, schemaName, user,
					true);
			// last argument: write = true.

			sql = "UPDATE " + DBConfig.schemaTabName(schemaName) + " SET "
					+ colName_owner + "= '" + newOwner + "'  WHERE "
					+ colName_uid + "='" + uid + "'";
			logger.finest("Executing SQL: " + sql);
			stmt.execute(sql);


			//metaAdd(uid, new ArchiveTimeStamp(), metaHist_owner, newOwner);

			conn.commit();

		} catch (SQLException e) {
			logger.warning("SQL raised exception: " + sql);
			try {
				conn.rollback();
			} catch (SQLException e1) {
				logger.warning("Could not rollback transaction. CHECK UID "
						+ uid);
			}
			throw new DatabaseException(e);
		} finally {
			try {
				ocpds.close(conn);
				rs.close();
				stmt.close();
			} catch (SQLException e1) {
				logger
						.info("Ignoring SQL exception while closing DB resources.");
			}
		}
	}

	// ////////////////////////////////////////////////////////////////////
	// Various helpers //
	// ////////////////////////////////////////////////////////////////////

	/*
	 * remove all entries in test area. Done by calling first a script that
	 * drops all standard tables, and generates them newly (empty). After that,
	 * some _entities tables, which have no corresponding entry in the
	 * xml_schema_entities table, are deleted.
	 * 
	 */
	protected void cleanTestArea() throws DatabaseException,
			AcsJArchiveIdentifierErrorEx {

		// first check permissions
		DBConfiguration config = DBConfiguration.instance(logger);
		if (!"test".equals(config.get("archive.db.mode"))
				|| !"almatest".equals(config.get("archive.oracle.user"))) {
			throw new DatabaseException(
					"Must use connection to test database for cleaning. Check file dbConfig.properties!");
		}

		logger.info("Removing all entries from test area.");

		// First read archiveID for xml_metaInfo, remember it and insert it
		// again in the end:
		String archiveID;
		try {
			archiveID = DatabaseReader.instance().getMetaParamValue(
					DBConfig.paramName_archiveID);
		} catch (ModuleCriticalException e) {
			throw new DatabaseException(e.getCause());
		}
		logger.severe("================ARCHIVE ID " + archiveID);
		if (archiveID.equals("null")) {
			logger
					.severe("Found null archiveID in DB. Either another parallel tets is running or something wrong. Aborting...");
			return;
		}
		InputStream SQLscript;

		// read SQL script from classpath
		// this script drops basic tables and creates new ones. It is
		// important to call cleanDB thereafter in order to remove all
		// entities tables
		ClassLoader loader = DBConfiguration.class.getClassLoader();
		URL propsFile = loader.getResource("AlmaTestDbCreateTables.sql");
		if (propsFile == null) {
			logger
					.severe("Could not find SQL script for cleaning test DB. Assumed location: AlmaTestDbCreateTables.sql in archive_database.jar");
			throw new DatabaseException(
					"Could not find SQL script for cleaning test DB. Assumed location: AlmaTestDbCreateTables.sql in archive_database.jar");
		} else {
			logger.info("-------- Loading " + propsFile);
			try {
				SQLscript = propsFile.openStream();
			} catch (IOException e1) {
				logger
						.severe("Problems when opening SQL script for cleaning test DB. Location: AlmaTestDbCreateTables.sql in archive_database.jar");
				throw new DatabaseException(
						"Problems when opening SQL script for cleaning test DB. Location: AlmaTestDbCreateTables.sql in archive_database.jar");
			}
		}

		try {
			executeSQLscript(SQLscript);
		} catch (IOException e) {
			logger
					.severe("Problems when reading SQL script for cleaning test DB. Location: AlmaTestDbCreateTables.sql in archive_database.jar");
			throw new DatabaseException(e);
		}

		checkSchemaConsistency(true, false);

		// Finally, insert archiveID again into xml_metaInfo:
		setMetaParamValue(DBConfig.paramName_archiveID, archiveID);

		loadIdentifierRangeSchema();

	}

	/**
	 * loads the identifierRange.xsd into database. Necessary after cleaning the
	 * database.
	 */
	protected void loadIdentifierRangeSchema() throws DatabaseException,
			AcsJIdentifierUnexpectedEx, AcsJRangeExhaustedEx {
		// // read IdentifierRange.xsd from jar file:
		// changed, it is read from $ACSROOT/idl now
		ClassLoader loader = DBConfiguration.class.getClassLoader();
		StringBuffer UIDschema = new StringBuffer();
		BufferedReader UIDschemaReader;
		URL UIDschemaFile = loader.getResource("IdentifierRange.xsd");
		if (UIDschemaFile == null) {
			logger
					.severe("Could not find IdentifierRange.xsd. Assumed location: IdentifierRange.xsd in archive_xmlstore_if.jar");
			throw new DatabaseException(
					"Could not find IdentifierRange.xsd. Assumed location: IdentifierRange.xsd in archive_xmlstore_if.jar");
		}
		logger.info("-------- Loading " + UIDschemaFile);
		try {
			UIDschemaReader = new BufferedReader(new InputStreamReader(
					UIDschemaFile.openStream()));
		} catch (IOException e1) {
			logger
					.severe("Problems when opening schema file for UID range. Location: IndentifierRange.xsd in archive_xmlstore_if.jar");
			throw new DatabaseException(
					"Problems when opening schema file for UID range. Location: IndentifierRange.xsd in archive_xmlstore_if.jar");
		}
		try {
			while (UIDschemaReader.ready()) {
				UIDschema.append(UIDschemaReader.readLine());
			}
		} catch (IOException e) {
			logger
					.severe("Problems when opening schema file for UID range. Location: IndentifierRange.xsd in archive_xmlstore_if.jar");
			throw new DatabaseException(
					"Problems when opening schema file for UID range. Location: IndentifierRange.xsd in archive_xmlstore_if.jar");
		}

		// get new UID. (The one inserted before.)
		URI idRangeURI;
		try {
			idRangeURI = new Range(InternalIFFactory.getIdentifierManager(
					logger).getNewRange()).next();
		} catch (ArchiveException e) {
			throw new DatabaseException(e);
		} catch (ModuleCriticalException e) {
			throw new DatabaseException(e.getCause());
		}
		logger.info("Storing IdentifierRange.xsd under UID: " + idRangeURI);
		addSchemaAndTables(idRangeURI, "IdentifierRange", UIDschema.toString(),
				"");
	}

	/**
	 * 
	 * Add all metadata of a new entity to metadata history. Caution!!! This
	 * procedure does *not* commit transactions, must be done by caller! (Use
	 * conn.commit())
	 * 
	 * @param uid
	 * @param schema
	 * @param owner
	 * @param deleted
	 * @param hidden
	 * @param dirty
	 * @param permissions
	 */
	/*
	private void metaHistAddNewEnt(URI uid, URI schema, String owner,
			int deleted, int hidden, int dirty, Permissions permissions,
			Connection conn) throws SQLException {
		String sql;
		OraclePreparedStatement pstmt = null;
		ArchiveTimeStamp stamp = new ArchiveTimeStamp();

		// Update _meta_history table 
		sql = "INSERT INTO " + tableName_meta_history + " (" + colName_uid
				+ ", " + colName_timestamp + ", " + colName_metaColumn + ", "
				+ colName_newValue + ") VALUES ('" + uid + "', ?, ?, ?)";
		logger.finest("Executing SQL: " + sql);

		pstmt = (OraclePreparedStatement) conn.prepareStatement(sql);
		pstmt.setTimestamp(1, stamp.getTimestamp());

		// schema UID
		pstmt.setShort(2, metaHist_schemaUID);
		pstmt.setStringForClob(3, schema.toString());
		pstmt.executeUpdate();

		// Owner
		pstmt.setShort(2, metaHist_owner);
		pstmt.setString(3, owner);
		pstmt.executeUpdate();

		// Deleted+hidden+dirty
		pstmt.setShort(2, metaHist_deletedHiddenDirty);
		pstmt.setString(3, DBConfig
				.metaHist_DHDEncoding(deleted, hidden, dirty));
		pstmt.executeUpdate();

		// Permissions
		pstmt.setShort(2, metaHist_permissions);
		pstmt.setString(3, DBConfig.metaHist_permEncoding(permissions));
		pstmt.executeUpdate();
		try {
			pstmt.close();
		} catch (SQLException e) {
			logger.info("Ignoring exception while closing Oracle resources.");
		}
	}
	*/

	/**
	 * 
	 * add the change information of column colName with new value newValue to
	 * the meta_history table. Caution!!! This procedure does *not* commit
	 * transactions, must be done by caller! (Use conn.commit())
	 * 
	 * @param uid
	 * @param stamp
	 * @param colName_deleted
	 * @param newDel
	 */
	/*
	private void metaAdd(URI uid, ArchiveTimeStamp stamp, short colName,
			String newValue) throws DatabaseException {
		PreparedStatement pstmt = null;
		Connection conn = null;
		String sql;
		sql = "INSERT INTO " + tableName_meta_history + " (" + colName_uid
				+ ", " + colName_timestamp + ", " + colName_metaColumn + ", "
				+ colName_newValue + ") VALUES ('" + uid + "', ?, " + colName
				+ ", '" + newValue + "')";
		logger.finest("Executing SQL: " + sql);
		try {
			conn = ocpds.getConnection("WR, metaAdd(" + uid + ")");
			pstmt = conn.prepareStatement(sql);
			pstmt.setTimestamp(1, stamp.getTimestamp());
			pstmt.executeUpdate();
		} catch (SQLException e) {
			logger.warning("SQL raised exception: " + e.toString());
			try {
				conn.rollback();
			} catch (SQLException e1) {
				logger.severe("Could not rollback transaction. CHECK " + uid);
			}
			throw new DatabaseException(e);
		} finally {
			try {
				ocpds.close(conn);
				pstmt.close();
			} catch (SQLException e) {
				logger
						.info("Ignoring exception while closing Oracle resources.");
			}
		}

	}
	*/

	/*
	 * save all actual information to Oracle. Used before shutting down Oracle.
	 * 
	 */
	protected void saveState() throws DatabaseException,
			ModuleCriticalException {
	}

	/**
	 * 
	 * set a new meta parameter
	 * 
	 * @param paramName
	 * @param paramValue
	 */
	protected void setMetaParamValue(String paramName, String paramValue)
			throws DatabaseException {
		Statement stmt = null;
		Connection conn = null;
		String sql;
		try {
			conn = ocpds.getConnection("WR, setMetaParamValue(" + paramName
					+ ")");

			// first we delete all rows with that parameter:
			sql = "DELETE FROM " + tableName_metaInf + " WHERE "
					+ colName_paramName + "='" + paramName + "'";
			logger.finest("Executing SQL: " + sql);
			stmt = conn.createStatement();
			stmt.executeUpdate(sql);

			sql = "INSERT INTO " + tableName_metaInf + " ( "
					+ colName_paramName + "," + colName_paramValue
					+ ") VALUES ('" + paramName + "', '" + paramValue + "')";
			logger.finest("Executing SQL: " + sql);
			stmt = conn.createStatement();
			stmt.executeUpdate(sql);
			conn.commit();
		} catch (SQLException e) {
			logger.warning("Oracle error during storage of meta parameter: "
					+ e.toString());
			try {
				conn.rollback();
			} catch (SQLException e1) {
				logger
						.warning("Could not rollback transaction! CHECK parameter "
								+ paramName
								+ " in table "
								+ tableName_metaInf
								+ "! " + e1.toString());
			}
			throw new DatabaseException(e);
		} finally {
			try {
				ocpds.close(conn);
				stmt.close();
			} catch (SQLException e) {
				logger.info(e.toString());
			}
		}

	}

	/**
	 * 
	 * update a meta parameter
	 * 
	 * @param paramName
	 * @param paramValue
	 */
	protected void updateMetaParamValue(String paramName, String paramValue)
			throws DatabaseException {
		Statement stmt = null;
		Connection conn = null;
		String sql;
		try {
			conn = ocpds.getConnection("WR, updateMetaParamValue(" + paramName
					+ ")");
			sql = "UPDATE " + tableName_metaInf + " SET " + colName_paramValue
					+ "='" + paramValue + "' WHERE " + colName_paramName + "='"
					+ paramName + "'";
			logger.finest("Executing SQL: " + sql);
			stmt = conn.createStatement();
			stmt.executeUpdate(sql);

			// we also track the change in the meta_history table (might not be
			// absolutely correct, if the archiveID is changed, it will also
			// appear here, undistinguishable from lastUID value changes):
			/*
			try {
				metaAdd(new URI("uid://X00/X0/X0"), new ArchiveTimeStamp(),
						(short) 11, paramValue);
			} catch (URISyntaxException e) {
				// Cannot happen
				e.printStackTrace();
			}
			*/
			conn.commit();
		} catch (SQLException e) {
			logger.warning("Oracle error during storage of meta parameter: "
					+ e.toString());
			throw new DatabaseException(e);
		} finally {
			try {
				ocpds.close(conn);
				stmt.close();
			} catch (SQLException e) {
				logger.info(e.toString());
			}
		}

	}

	// ////////////////////////////////////////////////////////////////////
	// helpers for tests //
	// ////////////////////////////////////////////////////////////////////

	/**
	 * For testing purposes only! Removes all data and tables belonging to
	 * schema schemaName.
	 * 
	 * Use carefully, since this method does not throw an exception!
	 * 
	 */
	public void cleanSchema(String schemaName) throws DatabaseException,
			ModuleCriticalException {
		logger.info("Removing infrastructure related to schema " + schemaName);
		Statement stmt = null;
		Connection conn = null;
		String sql;
		try {
			conn = ocpds.getConnection("WR, cleanSchema(" + schemaName + ")");
			stmt = conn.createStatement();

			// remove schema from schema table
			sql = "DELETE FROM " + tableName_schemas + " WHERE "
					+ colName_schemaName + "='" + schemaName + "'";
			logger.finest("Executing SQL: " + sql);
			stmt.executeUpdate(sql);
		} catch (SQLException e) {
			logger.warning("Problems while dropping schema: " + e.getMessage());
		}

		// remove documents belonging to schema from hhistory table
		try {
			sql = "DELETE FROM " + tableName_xml_history + " WHERE "
					+ colName_schemaName + "='" + schemaName + "'";
			logger.finest("Executing SQL: " + sql);
			stmt.executeUpdate(sql);
		} catch (SQLException e) {
			logger.warning("Problems while dropping schema: " + e.getMessage());
		}

		// TODO remove entries from meta_history?

		// drop entity table itself
		try {
			sql = "DROP TABLE " + DBConfig.schemaTabName(schemaName) + " PURGE";
			logger.finest("Executing SQL: " + sql);
			stmt.executeUpdate(sql);

			conn.commit();

			// update cache
			DatabaseReader.instance().initSchemaMaps();
		} catch (SQLException e) {
			logger.warning("Problems while dropping schema: " + e.getMessage());
		} finally {
			try {
				ocpds.close(conn);
				stmt.close();
			} catch (SQLException e) {
				logger.info("Ignoring SQL exception while closing resources: "
						+ e.toString());
			}
		}

	}

	/**
	 * For testing purposes only! Removes all data from meta and xml history
	 * tables
	 * 
	 */
	protected void cleanDBhistory() {
		logger.info("Removing history information.");
		Statement stmt = null;
		Connection conn = null;
		try {
			conn = ocpds.getConnection("WR, cleanDBhistory");
			stmt = conn.createStatement();
			/*
			String sql = "DELETE FROM " + tableName_meta_history;
			logger.finest("Executing SQL: " + sql);
			stmt.executeUpdate(sql);
		*/

			String sql = "DELETE FROM " + tableName_xml_history;
			logger.finest("Executing SQL: " + sql);
			stmt.executeUpdate(sql);
			conn.commit();

		} catch (SQLException e) {
			logger.info("Ignoring " + e.toString());
		} finally {
			try {
				ocpds.close(conn);
				stmt.close();
			} catch (SQLException e) {
				logger.info(e.toString());
			}
		}

	}

	/**
	 * Checks whether all schemas associated with namespaces in
	 * xml_schemanamespaces also appear in schema table and are the most recent
	 * ones. If replace is set to 1, entries in xml_schemanamespaces are
	 * replaced (in the moment, only older schemas are replaced with newer
	 * schemas). if set to 0, only a warning is printed. We also check whether
	 * every namespace in xml_namespaces is associated with a schema in
	 * xml_schemanamespaces. If not, and replace is set to true, they are
	 * removed. Only if print is set to true, info is displayed on console.
	 * 
	 * TODO update schema handling and this method accordingly.
	 * 
	 * @throws ModuleCriticalException
	 * @throws DatabaseException
	 * 
	 */
	protected boolean checkNamespaces(boolean replace, boolean print)
			throws DatabaseException, ModuleCriticalException {
		boolean inconsistencyFound = false;
		Statement stmt = null, stmt2 = null;
		Connection conn = null;
		ResultSet rs = null, rs2 = null;

		logger.info("Checking namespace consistency in Oracle.");
		if (replace) {
			logger
					.warning("Replacing incosistent references to schemas (older schemas) with correct ones (newer schema versions).");
		}
		if (print) {
			System.out.println("Checking namespace consistency in Oracle.");
		}
		DatabaseReader.instance().initSchemaMaps();
		// System.out.println(dbCache.uri2schemaName);

		// loop through schemas
		// take care: in the update statement below, *all* namespaces
		// referencing schemaUID are replaced, not only the updated one
		try {
			// go through all entries in xml_schemanamespaces
			String sql = "SELECT " + colName_schemaUid + " FROM "
					+ tableName_schemaNamespaces;
			logger.finest("Executing SQL: " + sql);

			URI schema = null;

			conn = ocpds.getConnection("WR, checkNS");
			stmt = conn.createStatement();
			stmt2 = conn.createStatement();
			rs = stmt.executeQuery(sql);
			while (rs.next()) {
				// check whether referenced schema is the newest version
				try {
					schema = new URI(rs.getString(colName_schemaUid));
					if (!dbCache.uri2schemaName.containsKey(schema)) {
						// bad case: URI is not the newest one
						inconsistencyFound = true;
						logger
								.warning("Schema "
										+ schema
										+ " is not the newest version. Queries will fail!");
						if (print) {
							System.out
									.println("Schema "
											+ schema
											+ " is not the newest version. Queries will fail!");
						}
						if (replace) {
							logger.info("Replacing outdated schema reference "
									+ schema + " in "
									+ tableName_schemaNamespaces);
							if (print) {
								System.out
										.println("Replacing outdated schema reference "
												+ schema
												+ " in "
												+ tableName_schemaNamespaces);
							}
							// fetch newest version of schema in question
							sql = "SELECT " + colName_schemaName + " FROM "
									+ tableName_schemas + " WHERE "
									+ colName_uid + "='" + schema + "'";
							logger.finest("Executing SQL: " + sql);
							rs2 = stmt2.executeQuery(sql);

							if (rs2.next()) {
								String schemaName = rs2
										.getString(colName_schemaName);
								// update xml_schemanamespaces
								sql = "UPDATE "
										+ tableName_schemaNamespaces
										+ " SET "
										+ colName_schemaUid
										+ "='"
										+ dbCache.schemaName2uri
												.get(schemaName) + "' WHERE "
										+ colName_schemaUid + "='" + schema
										+ "'";
								logger.finest("Executing SQL: " + sql);
								stmt2.executeUpdate(sql);
							} else {
								// this means we have a UID in the namespace
								// table not associated with a schema:
								// remove it

								// the following is unnecessary, it is done in
								// the next loop (chec references from
								// xml_namespaces...)
								// first we remove entries in namespaces, that
								// will be dangling
								// find prefix:
								// sql="SELECT "+colName_prefix+" FROM
								// "+tableName_schemaNamespaces+" WHERE
								// "+colName_schemaUid+"='"+schema+"'";
								// logger.finest("Executing SQL: "+sql);
								// rs2=stmt2.executeQuery(sql);
								// Statement stmt3=conn2.createStatement();
								// ResultSet rs3;
								// while (rs2.next()) {
								// String prefix =
								// rs2.getString(colName_prefix);
								// sql="SELECT "+colName_prefix+" FROM
								// "+tableName_schemaNamespaces+" WHERE
								// "+colName_schemaUid+"!='"+schema+"'"+" AND
								// "+colName_prefix+"='"+prefix+"'";
								// logger.finest("Executing SQL: "+sql);
								// rs3 = stmt3.executeQuery(sql);
								// if (!rs3.next()) {
								// // no results --> this entry will be dangling
								// sql="DELETE FROM "+tableName_namespaces+"
								// WHERE "+colName_prefix+"='"+prefix+"'";
								// logger.finest("Executing SQL: "+sql);
								// logger.info("Removing dangling entry for
								// "+prefix+" from "+tableName_namespaces);
								// stmt3.execute(sql);
								// }
								// rs3.close();
								// }

								// now remove the entry from schema_namespaces:
								sql = "DELETE FROM "
										+ tableName_schemaNamespaces
										+ " WHERE " + colName_schemaUid + "='"
										+ schema + "'";
								logger.finest("Executing SQL: " + sql);
								logger.warning("Removing reference to schema "
										+ schema + " from "
										+ tableName_schemaNamespaces
										+ " since this schema does not exist!");
								stmt2.executeUpdate(sql);
							}
							rs2.close();
						}
					}
				} catch (URISyntaxException e) {
					logger.severe("Found incorrect UID in Database: " + schema);
				}

			}

			// check references from xml_namespaces to xml_schemanamespaces:
			sql = "SELECT " + colName_prefix + " FROM " + tableName_namespaces;
			logger.finest("Executing SQL: " + sql);
			rs = stmt.executeQuery(sql);
			while (rs.next()) {
				String prefix = rs.getString(colName_prefix);
				sql = "SELECT * FROM " + tableName_schemaNamespaces + " WHERE "
						+ colName_prefix + "='" + prefix + "'";
				logger.finest("Executing SQL: " + sql);
				rs2 = stmt2.executeQuery(sql);
				// we expect at least one result:
				if (!rs2.next()) {
					inconsistencyFound = true;
					logger.warning("Prefix " + prefix
							+ " is not associated with a schema");
					if (print) {
						System.out.println("Prefix " + prefix
								+ " is not associated with a schema");
					}
					if (replace) {
						logger.info("Removing namespace prefix " + prefix
								+ " from " + tableName_namespaces);
						sql = "DELETE FROM " + tableName_namespaces + " WHERE "
								+ colName_prefix + "='" + prefix + "'";
						logger.finest("Executing SQL: " + sql);
						rs2 = stmt2.executeQuery(sql);
					}
				}
				rs2.close();
			}

		} catch (SQLException e) {
			throw new DatabaseException("Caught SQL Exception: " + e.toString());
		} finally {
			try {
				rs.close();
			} catch (Exception e) {
				logger.info("Ignoring SQL exception while closing resources: "
						+ e.toString());
			}
			try {
				stmt.close();
			} catch (Exception e) {
				logger.info("Ignoring SQL exception while closing resources: "
						+ e.toString());
			}
			try {
				stmt2.close();
			} catch (Exception e) {
				logger.info("Ignoring SQL exception while closing resources: "
						+ e.toString());
			}
			try {
				ocpds.close(conn);
			} catch (Exception e) {
				logger.info("Ignoring SQL exception while closing resources: "
						+ e.toString());
			}
		}

		if (print) {
			if (inconsistencyFound) {
				if (replace) {
					System.out.println("OK: Inconsistencies repaired.");
				} else {
					System.out
							.println("NOT OK: Found namespace inconsistencies.");
				}
			} else {
				System.out.println("OK");
			}
		}
		return !inconsistencyFound;
	}

	/**
	 * Checks whether all documents in history table also appear in schema table
	 * and vice versa If delete is set to 1, lonely documents are removed from
	 * the respective table. if set to 0, only a warning is printed. Only if
	 * print is set to true, info is displayed on console.
	 * 
	 * @throws ModuleCriticalException
	 * @throws DatabaseException
	 * 
	 * 
	 * 
	 */
	protected boolean checkHistory(boolean delete, boolean print)
			throws DatabaseException, ModuleCriticalException {
		boolean inconsistencyFound = false;
		Statement stmt = null, stmt2 = null;
		Connection conn = null, conn2 = null;
		ResultSet rs = null, rs2 = null;

		logger.info("Checking history entries consisytency in Oracle.");
		if (delete) {
			logger
					.warning("Checking history entries consisytency in Oracle and removing entries in case of incosistencies.");
		}
		if (print) {
			System.out
					.println("Checking history entries consistency in Oracle.");
		}

		// go through all schema tabs, check whether there is a corresponding
		// entry in history tab
		// fetch all schemas
		DatabaseReader.instance().initSchemaMaps();
		Set<String> schemas = dbCache.schemaName2uri.keySet();

		// loop through schemas
		for (Iterator schemaIt = schemas.iterator(); schemaIt.hasNext();) {
			String schema = (String) schemaIt.next();

			// loop through documents (possibly the loop should be executed in
			// the database, but in the moment,
			// performance is not os important here.)
			String sql = "SELECT " + colName_uid + ", " + colName_timestamp
					+ " FROM " + DBConfig.schemaTabName(schema);
			logger.finest("Submitting SQL: " + sql);
			try {
				conn = ocpds.getConnection("WR, getallDocs");
				stmt = conn.createStatement();
			} catch (SQLException e) {
				throw new DatabaseException("Oracle connection problem: "
						+ e.toString());
			}
			try {
				rs = stmt.executeQuery(sql);
			} catch (SQLException e) {
				throw new DatabaseException("Failure in executing query: "
						+ e.toString());
			}

			try {
				conn2 = ocpds.getConnection("WR, checkHistory1");
				stmt2 = conn2.createStatement();
				while (rs.next()) {
					// check whether doc has entry in history table
					ArchiveTimeStamp time = new ArchiveTimeStamp(rs
							.getTimestamp(colName_timestamp));
					String uid = rs.getString(colName_uid);
					sql = "SELECT " + colName_uid + " FROM "
							+ tableName_xml_history + " WHERE " + colName_uid
							+ "='" + uid + "' AND " + colName_timestamp + "='"
							+ time.toSQLString() + "'";
					logger.finest("Submitting SQL: " + sql);
					rs2 = stmt2.executeQuery(sql);

					if (!rs2.next()) {
						// bad case: no entry in history table found!
						inconsistencyFound = true;
						logger.warning("Document " + uid + " with timestamp "
								+ time.toISOString()
								+ " has no matching entry in history table.");
						if (print) {
							System.out
									.println("Document "
											+ uid
											+ " with timestamp "
											+ time.toISOString()
											+ " has no matching entry in history table.");
						}
						if (delete) {
							logger.info("Removing " + uid + " from "
									+ DBConfig.schemaTabName(schema));
							if (print) {
								System.out.println("Removing " + uid + " from "
										+ DBConfig.schemaTabName(schema));
							}
							sql = "DELETE FROM "
									+ DBConfig.schemaTabName(schema)
									+ " WHERE " + colName_uid + "='" + uid
									+ "'";
							stmt2.executeUpdate(sql);
						}
					}
					rs2.close();
				}
			} catch (Exception e) {
				throw new DatabaseException("Caught SQL Exception: "
						+ e.toString());
			} finally {
				try {
					rs.close();
				} catch (SQLException e) {
					logger
							.info("Ignoring SQL exception while closing resources: "
									+ e.toString());
				}
				try {
					stmt2.close();
				} catch (Exception e) {
					logger
							.info("Ignoring SQL exception while closing resources: "
									+ e.toString());
				}
				try {
					ocpds.close(conn2);
				} catch (Exception e) {
					logger
							.info("Ignoring SQL exception while closing resources: "
									+ e.toString());
				}
				try {
					stmt.close();
				} catch (Exception e) {
					logger
							.info("Ignoring SQL exception while closing resources: "
									+ e.toString());
				}
				try {
					ocpds.close(conn);
				} catch (Exception e) {
					logger
							.info("Ignoring SQL exception while closing resources: "
									+ e.toString());
				}
			}
		}

		// go through history tab, check whether there is a corresponding entry
		// in entities tab
		try {
			conn = ocpds.getConnection("WR, checkHistory3");
			stmt = conn.createStatement();
		} catch (SQLException e) {
			throw new DatabaseException("Oracle connection problem: "
					+ e.toString());
		}

		// loop through documents in history table
		String sql = "SELECT " + colName_uid + ", MAX(" + colName_timestamp
				+ "), " + colName_schemaName + " FROM " + tableName_xml_history
				+ " WHERE " + colName_schemaName + "<>'"
				+ DBConfig.schemaSchemaName + "' GROUP BY " + colName_uid
				+ ", " + colName_schemaName;
		logger.finest("Submitting SQL: " + sql);
		try {
			rs = stmt.executeQuery(sql);
		} catch (SQLException e) {
			throw new DatabaseException("Caught SQL Exception: " + e.toString());
		}

		try {
			conn2 = ocpds.getConnection("WR, checkHistory2");
			stmt2 = conn2.createStatement();
		} catch (SQLException e) {
			throw new DatabaseException("Oracle connection problem: "
					+ e.toString());
		}
		try {
			while (rs.next()) {
				// check whether there is an entry in corresponding entities
				// table
				String schema = rs.getString(colName_schemaName);
				String uid = rs.getString(colName_uid);
				// timestamp is in second column (see SQL aboive)
				ArchiveTimeStamp time = new ArchiveTimeStamp(rs.getTimestamp(2));

				sql = "SELECT " + colName_uid + " FROM "
						+ DBConfig.schemaTabName(schema) + " WHERE "
						+ colName_uid + "='" + uid + "' AND "
						+ colName_timestamp + "='" + time.toSQLString() + "'";
				logger.finest("Submitting SQL: " + sql);
				rs2 = stmt2.executeQuery(sql);
				if (!rs2.next()) {
					// bad case: no entry in entities table found
					inconsistencyFound = true;
					logger.warning("Document " + uid + " with timestamp "
							+ time.toISOString()
							+ " has no matching entry in entities table "
							+ DBConfig.schemaTabName(schema));
					if (print) {
						System.out.println("Document " + uid
								+ " with timestamp " + time.toISOString()
								+ " has no matching entry in entities table "
								+ DBConfig.schemaTabName(schema));
					}
					if (delete) {
						logger.info("Removing " + uid + " from history table");
						if (print) {
							System.out.println("Removing " + uid
									+ " from history table");
						}
						sql = "DELETE FROM " + tableName_xml_history
								+ " WHERE " + colName_uid + "='" + uid + "'";
						stmt2.executeUpdate(sql);
					}
				}
				rs2.close();
			}
		} catch (SQLException e) {
			throw new DatabaseException("Caught SQL Exception: " + e.toString());
		} finally {
			try {
				rs.close();
			} catch (Exception e) {
				logger.info("Ignoring SQL exception while closing resources: "
						+ e.toString());
			}
			try {
				stmt.close();
			} catch (Exception e) {
				logger.info("Ignoring SQL exception while closing resources: "
						+ e.toString());
			}
			try {
				ocpds.close(conn);
			} catch (Exception e) {
				logger.info("Ignoring SQL exception while closing resources: "
						+ e.toString());
			}
			try {
				stmt2.close();
			} catch (Exception e) {
				logger.info("Ignoring SQL exception while closing resources: "
						+ e.toString());
			}
			try {
				ocpds.close(conn2);
			} catch (Exception e) {
				logger.info("Ignoring SQL exception while closing resources: "
						+ e.toString());
			}
		}
		if (print) {
			if (inconsistencyFound) {
				if (delete) {
					System.out.println("OK: Inconsistent entries deleted.");
				} else {
					System.out
							.println("NOT OK: Found history inconsistencies.");
				}
			} else {
				System.out.println("OK");
			}
		}
		return !inconsistencyFound;
	}

	//
	// /**
	// * Checks whether lastUID stored is bigger than UIDs of documents in
	// history
	// * table. If delete is set to 1, lastUID is replaced by appropriate value,
	// * if set to 0, only a warning is printed. Only if print is set to true,
	// * info is displayed on console.
	// *
	// * @throws ModuleCriticalException
	// * @throws DatabaseException
	// *
	// *
	// *
	// */
	// protected boolean checkLastUID(boolean replace, boolean print)
	// throws DatabaseException, ModuleCriticalException {
	// boolean inconsistencyFound = false;
	// Statement stmt = null;
	// Connection conn = null;
	// ResultSet rs = null;
	//	
	// logger.info("Checking lastUID value in Oracle.");
	// if (replace) {
	// logger
	// .warning("Checking actual UID value in Oracle and replace it in case of
	// inconsistencies.");
	// }
	// if (print) {
	// System.out.println("Checking lastUID value in Oracle.");
	// }
	//
	// DatabaseReader dbReader = DatabaseReader.instance();
	// IdentifierArchive idArchive = IdentifierArchive.instance();
	//
	// Long lastUid = null;
	// try {
	// lastUid = Long.parseLong(dbReader
	// .getMetaParamValue(DBConfig.paramName_lastUID));
	// } catch (Throwable e) {
	// throw new DatabaseException(
	// "No lastUID parameter found in Oracle: " + e.toString());
	// }
	//
	// // get archive ID
	// String archiveID;
	// archiveID = Long.toHexString(idArchive.getArchiveId());
	//
	// if (archiveID.length() == 1) {
	// archiveID = "0" + archiveID;
	// }
	//
	// logger
	// .info("Checking actual UID value in Oracle. Only considering UIDs from
	// this Archive instance, ie. with archive ID "
	// + archiveID);
	//
	// String maxUID = null;
	// // find max UID in history table. Only UIDs with the correct archive ID
	// // are considered!
	// String sql = "SELECT max(to_number(replace(replace(" + colName_uid
	// + ", 'uid:/', ''), '/X', ''), 'XXXXXXXXXXXXXXXXXXXXXX')), "
	// + colName_uid + " FROM " + tableName_xml_history + " WHERE "
	// + colName_uid + " like 'uid://X" + archiveID + "%' GROUP by "
	// + colName_uid;
	// logger.finest("Submitting SQL: " + sql);
	// try {
	// conn = ocpds.getConnection("WR, getMaxUID");
	// stmt = conn.createStatement();
	// } catch (SQLException e) {
	// throw new DatabaseException("Oracle connection problem: "
	// + e.toString());
	// }
	// try {
	// rs = stmt.executeQuery(sql);
	// } catch (SQLException e) {
	// throw new DatabaseException("Caught SQL Exception: " + e.toString());
	// }
	// try {
	// if (rs.next()) {
	// maxUID = rs.getString(colName_uid);
	// } else {
	// throw new DatabaseException("No UIDs found in xml_history.");
	// }
	// } catch (SQLException e) {
	// // TODO Auto-generated catch block
	// e.printStackTrace();
	// } finally {
	// try {
	// rs.close();
	// } catch (Exception e) {
	// logger.info("Ignoring SQL exception while closing resources: "
	// + e.toString());
	// }
	// try {
	// stmt.close();
	// } catch (Exception e) {
	// logger.info("Ignoring SQL exception while closing resources: "
	// + e.toString());
	// }
	// try {
	// ocpds.close(conn);
	// } catch (Exception e) {
	// logger.info("Ignoring SQL exception while closing resources: "
	// + e.toString());
	// }
	// }
	//
	// // find global part
	// String global = maxUID.substring(maxUID.indexOf("/X", 6) + 2, maxUID
	// .lastIndexOf("/X"));
	// // to decimal:
	// Long maxGlob = Long.parseLong(global, 16);
	//
	// if (maxGlob.longValue() > lastUid.longValue()) {
	// // bad case
	// inconsistencyFound = true;
	// logger
	// .severe("lastUID value is to small. In the moment, must be at least "
	// + maxGlob);
	// if (print) {
	// System.out
	// .println("lastUID value is to small. In the moment, must be at least "
	// + maxGlob);
	// }
	// if (replace) {
	// logger.warning("Changing value of lastUID to " + maxGlob);
	// setMetaParamValue("lastUID", maxGlob.toString());
	// if (print) {
	// System.out
	// .println("lastUID value was to small. Changed to "
	// + maxGlob);
	// }
	// }
	// if (print) {
	// if (replace) {
	// System.out.println("OK: Inconsistency repaired.");
	// } else {
	// System.out.println("NOT OK: lastUID value too small.");
	// }
	//
	// }
	// return false;
	// } else {
	// if (print) {
	// System.out.println("OK");
	// }
	// // good case
	// return true;
	// }
	//
	// }
	//
	/**
	 * Removes inconsistent tables. If delete is set to 1, tables are removed,
	 * if set to 0, only a list if inconsisten tables is printed. Only if print
	 * is set to true, info is displayed on console.
	 * 
	 * 
	 * 
	 */
	protected boolean checkSchemaConsistency(boolean delete, boolean print) {
		boolean foundInconsitency = false;
		if (delete) {
			logger.warning("Removing inconsistent tables and schema entries.");
		}
		logger.info("Searching for inconsistent XML schema tables....");

		if (print) {
			System.out.println("Searching for inconsistent XML schema tables.");
		}
		Statement stmt = null;
		Connection conn = null;
		ResultSet rs = null;
		HashSet tableNames = new HashSet();
		// stores table names int lowercase and their entries as written int
		// xml_schema_entities table
		HashMap schemaEntries = new HashMap();
		try {
			// get all schemas
			DatabaseReader.instance().initSchemaMaps();
			for (Iterator it = dbCache.schemaName2uri.keySet().iterator(); it
					.hasNext();) {
				String schemaEntry = (String) it.next();
				schemaEntries.put((schemaEntry.length() < 17 ? schemaEntry
						.toLowerCase() : schemaEntry.toLowerCase().substring(0,
						17)), schemaEntry);
			}

			// get all xml tables (with name 'XML_%_ENTITIES')
			String sql = "SELECT TNAME FROM SYSTEM.TAB WHERE TNAME LIKE '"
					+ DBConfig.tabPrefix.toUpperCase() + "%"
					+ DBConfig.tableNameSuffix_entities.toUpperCase() + "'";
			logger.finest("Executing SQL: " + sql);
			conn = ocpds.getConnection("WR, cleanDB");
			stmt = conn.createStatement();
			rs = stmt.executeQuery(sql);
			while (rs.next()) {
				tableNames.add(rs.getString("TNAME"));
			}

			// schema names treated differently!!!
			tableNames.remove(tableName_schemas.toUpperCase());

			// print inconsistencies and remove them in case delete=1

			String schemaName, tName;
			// cycle through tableNames
			for (Iterator it = tableNames.iterator(); it.hasNext();) {
				tName = ((String) it.next());
				schemaName = tName.replaceFirst(
						DBConfig.tableNameSuffix_entities.toUpperCase(), "");
				schemaName = schemaName.replaceFirst(
						DBConfig.tabPrefix.toUpperCase(), "").toLowerCase();
				// System.out.println(schemaName);
				if (!schemaEntries.containsKey(schemaName)) {
					if (print) {
						logger.severe("Inconsistent table " + tName);
					}
					foundInconsitency = true;
					if (delete) {
						sql = "DROP TABLE " + tName + " PURGE";
						logger.info("Dropping table " + tName);
						logger.fine("Executing SQL: " + sql);
						stmt.executeUpdate(sql);

						// also remove entries from history table:
						sql = "DELETE FROM " + tableName_xml_history
								+ " WHERE UPPER(" + colName_schemaName + ")='"
								+ schemaName.toUpperCase() + "'";
						logger.info("Removing entries belonging to "
								+ schemaName + " from table "
								+ tableName_xml_history);
						logger.finest("Executing SQL: " + sql);
						stmt.executeUpdate(sql);
					}
				}
			}

			// cycle through schemaEntries
			for (Iterator it = schemaEntries.keySet().iterator(); it.hasNext();) {
				schemaName = (String) it.next();
				tName = DBConfig.tabPrefix.toUpperCase()
						+ schemaName.toUpperCase()
						+ DBConfig.tableNameSuffix_entities.toUpperCase();
				if (!tableNames.contains(tName)) {
					if (print) {
						logger.severe("Inconsistent entry "
								+ schemaEntries.get(schemaName));
					}
					foundInconsitency = true;
					if (delete) {
						sql = "DELETE FROM " + tableName_schemas + " WHERE "
								+ colName_schemaName + "='"
								+ schemaEntries.get(schemaName) + "'";
						logger.info("Removing entry " + schemaName
								+ " from table " + tableName_schemas);
						logger.fine("Executing SQL: " + sql);
						stmt.executeUpdate(sql);

						// also remove entries from history table:
						sql = "DELETE FROM " + tableName_xml_history
								+ " WHERE UPPER(" + colName_schemaName + ")='"
								+ schemaName.toUpperCase() + "'";
						logger.info("Removing entries belonging to "
								+ schemaName + " from table "
								+ tableName_xml_history);
						logger.finest("Executing SQL: " + sql);
						stmt.executeUpdate(sql);
					}
				}
			}

			conn.commit();
		} catch (Exception e) {
			try {
				logger.warning(e.toString());
				conn.rollback();
			} catch (SQLException e1) {
				// ignoring
				logger.info("Ignoring exception during DB rollback: "
						+ e.toString());
			}
			// ignore exception
			logger
					.info("Ignoring exception during DB cleanup: "
							+ e.toString());
		} finally {
			try {
				rs.close();
				stmt.close();
				ocpds.close(conn);
			} catch (Exception e1) {
				logger
						.fine("Ignoring DB exception while closing DB ressources");
			}
		}
		if (print) {
			if (foundInconsitency) {
				if (delete) {
					// update cache
					try {
						DatabaseReader.instance().initSchemaMaps();
					} catch (Exception e) {
						throw new RuntimeException(e);
					}
					System.out.println("OK: Inconsistent tables deleted.");
				} else {
					System.out
							.println("NOT OK: Found schema table inconsistencies.");
				}
			} else {
				System.out.println("OK");
			}
		}

		// we also have to check namespaces now:
		if (foundInconsitency && delete) {
			try {
				checkNamespaces(true, print);
			} catch (Exception e) {
				throw new RuntimeException(e);
			}
		}

		return !foundInconsitency;

	}

	/**
	 * reads SQL script and executes it. Each SQL statement line must be
	 * finished with ';' followed by newline. Each statement is read and
	 * submitted to Oracle via JDBC.
	 * 
	 * @return
	 * @throws IOException
	 * @throws DatabaseException
	 */
	public void executeSQLscript(InputStream script) throws IOException,
			DatabaseException {
		Statement stmt = null;
		Connection conn = null;

		InputStreamReader reader = new InputStreamReader(script);

		int lineRead;
		StringBuffer line = null;

		try {
			conn = ocpds.getConnection("WR, executeSQL");
			stmt = conn.createStatement();
		} catch (SQLException e2) {
			logger.warning("Could not initialize SQL connection: "
					+ e2.toString());
			throw new DatabaseException(e2);
		}

		for (int read = reader.read(); read != -1; read = reader.read()) {
			// System.out.println(read+": #"+new Character((char)
			// read)+"#");
			try {
				// we search for ';'. This will then be recognized as a full
				// line and submitted to Oracle.
				line = new StringBuffer();
				for (lineRead = read; (lineRead != -1) && (lineRead != 59); lineRead = reader
						.read()) {
					if (lineRead != 10 && lineRead != 47) { // ignore new lines
						line.append(new Character((char) lineRead));
					}
				}
				if (lineRead == -1) {
					read = -1;
					// EOF, exit outer loop
				} else {
					// submit command
					logger.finest("Executing SQL: " + line);
					// System.out.println("Executing SQL: " + line);
					stmt.executeUpdate(line.toString());
				}
			} catch (SQLException e) {
				String lineOut = line.toString().trim();
				// ignore exception if it was a drop table command:
				if (lineOut.substring(0, 5).equalsIgnoreCase("drop ")) {
					// do nothing
				} else {
					logger.warning("SQL exception: " + e.toString());
					throw new DatabaseException(e);
				}
			}
		}
		try {
			conn.commit();
		} catch (SQLException e) {
			logger.warning("Could not commit connection: " + e.toString());
			throw new DatabaseException(e);
		} finally {
			try {
				ocpds.close(conn);
			} catch (SQLException e) {
				logger.info("Ignoring SQL Exception while closing resources: "
						+ e.toString());
			}
		}
	}

	public void testLog() throws DatabaseException {
		String sql = "SELECT tname FROM system.tab where tname='"
				+ DBConfig.tableName_logging.toUpperCase() + "'";
		Statement stmt = null;
		ResultSet rs = null;
		boolean tableMissing = false;

		// System.out.println("Exec: "+sql);
		try {
			stmt = logConn.createStatement();
			rs = stmt.executeQuery(sql);

			if (!rs.next()) {
				// Table not found
				tableMissing = true;
			}
		} catch (Exception e) {
			throw new DatabaseException(e);
		} finally {
			try {
				rs.close();
				stmt.close();
			} catch (Exception e) {
				// ignore
			}
		}
		if (tableMissing)
			throw new DatabaseException("Table " + DBConfig.tableName_logging
					+ " does not exist.");
	}

	/**
	 * Scans the DB for inconsistent entries: Every XML doc stored must have at
	 * least one entry in a entities table, in the history table and in the meta
	 * table. If this is not the case for a doc, its inconsistent parts should
	 * be deleted. This method generates a script doing this. It contains two
	 * parts, first "normal" docs that can be deleted with less danger, and
	 * second schemas that have to be deleted cery carefully.
	 * 
	 * Implementation not finished
	 * 
	 * @return Script with SQL delete statements
	 */
	protected String generateDBcleanerScript() {

		// We proceed as follows: Three sets of UIDs are generated: UIDs coming
		// from entities tables (including schema entities), UIDs from the meta
		// table and ones from the history table. They are compared to each
		// other
		// in order to detect UIDs being "isolated" i.e. having no entry in
		// either
		// of the three tables. For these isolated UIDs delete statements are
		// constructed.

		StringBuffer outScript = new StringBuffer();
		HashMap intersection = new HashMap();
		HashMap union = new HashMap();
		HashMap entities = new HashMap();
		HashMap history = new HashMap();
		HashMap metHist = new HashMap();

		return outScript.toString();
	}

	public void removeHistoryTab() throws DatabaseException {
		Statement stmt = null;
		String sql;
		Connection conn = null;

		try {
			conn = ocpds.getConnection("WR, removeHistoryTab");
			stmt = conn.createStatement();
			sql = "DELETE FROM " + tableName_xml_history;
			logger.finest("Executing SQL: " + sql);
			stmt.executeUpdate(sql);
		} catch (SQLException e) {
			logger.warning("Could not remove namespaces.");
			throw new DatabaseException(e);
		} finally {
			try {
				stmt.close();
				ocpds.close(conn);
			} catch (SQLException e1) {
				logger
						.warning("Ignoring SQL exception while closing DB resources: "
								+ e1.toString());
			}
		}
	}

}
