/*
 *    ALMA - Atacama Large Millimiter Array
 *    (c) European Southern Observatory, 2002
 *    Copyright by ESO (in the framework of the ALMA collaboration),
 *    All rights reserved
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License as published by the Free Software Foundation; either
 *    version 2.1 of the License, or (at your option) any later version.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
 *    MA 02111-1307  USA
 *
 *    Created on Jan 16, 2004
 *
 */

// $Author: hmeuss $
// $Date: 2009/11/10 09:15:09 $
// $Log: DatabaseReader.java,v $
// Revision 1.53.2.2  2009/11/10 09:15:09  hmeuss
// fixed bug in ping method
//
// Revision 1.53.2.1  2009/10/14 12:14:50  hmeuss
// ping functionality was partially broken by an optimization, fixed that now
//
// Revision 1.53  2009/07/17 14:37:09  hmeuss
// removed restore functionality including references to xml_meta_history
//
// Revision 1.52  2009/04/23 12:56:13  hmeuss
// implemented incremental updates for XMLstore
//
// Revision 1.51  2009/03/13 15:52:57  hmeuss
// removed debug output
//
// Revision 1.50  2009/03/13 15:39:29  hmeuss
// Added STE environment for retrieval
//
// Revision 1.49  2008/10/08 15:56:05  hmeuss
// Removed the getAcsLogger command
//
// Revision 1.48  2008/05/27 09:40:09  hmeuss
// Corrected wrong default initialization of logger
//
// Revision 1.47  2008/01/23 16:27:29  hmeuss
// Improved handling of loggers
//
// Revision 1.46  2007/10/24 15:25:58  hmeuss
// removed debug message
//
// Revision 1.45  2007/10/24 15:06:21  hmeuss
// Added retrieval methods for logs and some more infrastructure
//
// Revision 1.44  2007/07/05 13:27:01  hmeuss
// Changed ID storage in Oracle: Now sequences are used
//
// Revision 1.43  2007/03/22 13:38:39  hmeuss
// added command line tool for archiveSanityCheck
//
// Revision 1.42  2007/03/01 15:54:31  hmeuss
// Added tests for consistency of Oracle tables and repair methods
//
// Revision 1.41  2007/02/08 16:52:07  hmeuss
// Added method for querying within given time intervals for asdmQuery
//
// Revision 1.40  2007/01/31 14:10:06  hmeuss
// Improved ping method: It reads now the archiveId, which allows better debugging
//
// Revision 1.39  2007/01/22 15:20:20  hmeuss
// removed debug log message
//
// Revision 1.38  2007/01/03 10:51:37  hmeuss
// improved exception handling for non SQL exceptions
//
// Revision 1.37  2006/12/15 21:57:52  hmeuss
// fixed bug, that some connections were not closed
//
// Revision 1.36  2006/11/30 15:20:09  hmeuss
// improved SQL command generated
//
// Revision 1.35  2006/11/30 10:35:41  hmeuss
// improved log retrieval with an additional timestamp interval
//
// Revision 1.34  2006/11/16 09:23:08  hmeuss
// removed bug of unclosed connection, changed connection architecture, added loads of logs for connection inspection.
//
// Revision 1.33  2006/11/09 13:01:00  hmeuss
// Improved resource handling
//
// Revision 1.32  2006/09/21 13:16:27  hmeuss
// Implemented queryContent method, that correctly queries XPath expressions returning a non-node type (eg. ending with an attribute step)
//
// Revision 1.31  2006/09/06 10:13:46  hsommer
// generics;
// removed unused exceptions from method signatures
//
// Revision 1.30  2006/08/18 13:17:28  hmeuss
// fixed bug that prevented docs belonging to old schema versions from being retrieved
//
// Revision 1.29  2006/07/05 08:15:05  hmeuss
// uses now connection pooling
//
// Revision 1.28  2006/07/04 11:46:29  hmeuss
// initSchemaMaps is called in init(). Therefore archiveLoadSchema now works again with a running system.
//
// Revision 1.27  2006/05/04 15:42:13  hmeuss
// Fixed bug in query methods (replacement of ' chars)
//
// Revision 1.26  2006/04/28 08:39:36  hmeuss
// Changed interface of queryRecent and implemented it for exist
//
// Revision 1.25  2006/04/25 09:34:12  hmeuss
// changed interface for timestamps of queryRecent
//
// Revision 1.24  2006/04/21 14:32:28  hmeuss
// New version, with test for new UIDLibrary
//
// Revision 1.23  2006/03/03 14:47:13  hmeuss
// Removed bug in schema loading, which deleted all entries in Oracle.
// Could not check yet, whether it's working for eXist, too, but it was necessary to check in for Optical Pointing
//
// Revision 1.22  2006/02/16 12:47:59  mpasquat
// added comments about exception handling problems
//
// Revision 1.21  2005/12/09 14:07:09  hmeuss
// moved functionality from init() to constructor.
//
// Revision 1.20  2005/11/08 15:24:34  hmeuss
// fixed bug: replacing ' by " in XML queries now (otherwise Oracle parser gets confused)
//
// Revision 1.19  2005/09/09 13:39:15  hmeuss
// changed implementation of ping (less strict)
//
// Revision 1.18  2005/08/02 14:02:32  hmeuss
// added ping() method in internalIF and Administrative component
//
// Revision 1.17  2005/07/21 13:48:12  hmeuss
// replaced DB2 in log messages by Oracle
//
// Revision 1.16  2005/07/21 12:29:48  hmeuss
// Changed design of test area
//
// Revision 1.15  2005/06/02 11:40:59  hmeuss
// removed System.out.println()
//
// Revision 1.14  2005/06/01 14:21:41  hmeuss
// added special treatment of schemas in get()
//
// Revision 1.13  2005/05/31 09:36:14  hmeuss
// small changes to get the archiveQueryClient running for Oracle again, also trying to get ArchiveManager running for Oracle again
//
// Revision 1.12  2005/05/30 14:21:46  hmeuss
// added Oracles xmlparserv2.jar to classpath
//
// Revision 1.11  2005/01/13 10:45:08  hmeuss
// - Constructed command line query interface for Archive.
// - Added functionality to init() for Oracle.
//
// Revision 1.10  2004/09/23 11:59:14  hmeuss
// Oracle DatabaseReader now creates a new connection object in every method call.
//
// Added ModuleCriticalException to more methods of the internal IF
//
// Revision 1.9  2004/06/29 14:18:48  hmeuss
// Changed config file from XML to Java properties
//
// Revision 1.8  2004/06/01 15:53:12  hmeuss
// Added treatment of virtual documents
//
// Revision 1.7  2004/06/01 08:44:10  hmeuss
// Using JDBC timestamp instead of String representation in SQL statements.
// Changed ArchiveTimeStamp to use java.sql.Timestamp instead of Date.
// Removed references to DB2 from InternalIFFactory.
//
// Revision 1.6  2004/05/27 11:07:10  hmeuss
// Added oracleLocation to config file and DBConfiguration.
// Made DBConfiguration a singleton class.
//
// Revision 1.5  2004/05/27 09:04:00  hmeuss
// dummy adaption to new DocumentData
//
// Revision 1.4  2004/05/07 12:48:55  hmeuss
// get now throws entityDirtyException
//
// Revision 1.3  2004/05/07 08:52:50  hmeuss
// Adapted to ISO timestamps. schemas are stored as CLOBs again (not XMLTYPE).
//
// Revision 1.2  2004/05/05 08:45:55  hmeuss
// *** empty log message ***
//
// Revision 1.1  2004/04/05 13:59:29  hmeuss
// Internal IF implementation adapted to Oracle
//
// Revision 1.6  2004/03/18 10:20:19  hmeuss
// implemented test visibility for query and queryUids
//
// Revision 1.5  2004/02/17 14:37:32  hmeuss
// Added some schema treatment
//
// Revision 1.4  2004/02/11 16:20:09  hmeuss
// *** empty log message ***
//
// Revision 1.3  2004/01/29 14:17:05  hmeuss
// Adapted to the new interface
//
// Revision 1.2  2004/01/23 15:11:45  hmeuss
// Added permission check in DB2CursorImpl
//
// Revision 1.1  2004/01/19 16:29:46  hmeuss
// Splitted functionality of Db2Database into Db2DatabaseReader, Db2DatabaseWriter, Db2DatabaseCache
// 
package alma.archive.database.oracle;

import java.net.URI;
import java.net.URISyntaxException;
import java.sql.Clob;
import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.sql.Timestamp;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Vector;
import java.util.logging.Logger;

import org.jaxen.BaseXPath;
import org.jaxen.expr.DefaultAbsoluteLocationPath;
import org.jaxen.expr.LocationPath;
import org.jaxen.expr.Step;
import org.jaxen.jdom.JDOMXPath;
import org.jaxen.JaxenException;

import com.cosylab.logging.engine.log.LogTypeHelper;

import oracle.jdbc.pool.OracleConnectionCacheImpl;
import oracle.xdb.XMLType;

import alma.acs.logging.ClientLogManager;
import alma.archive.database.helpers.DBConfiguration;
import alma.archive.database.helpers.DatabaseHelper;
import alma.archive.database.interfaces.DBCursor;
import alma.archive.exceptions.ArchiveException;
import alma.archive.exceptions.ArchiveGeneralException;
import alma.archive.exceptions.ModuleCriticalException;
import alma.archive.exceptions.access.EntityDirtyException;
import alma.archive.exceptions.access.PermissionDeniedException;
import alma.archive.exceptions.cursor.CursorClosedException;
import alma.archive.exceptions.general.DatabaseException;
import alma.archive.exceptions.general.DocumentDoesNotExistException;
import alma.archive.exceptions.general.EntityDoesNotExistException;
import alma.archive.exceptions.general.HistoryInconsistencyException;
import alma.archive.exceptions.general.UndefinedNamespaceException;
import alma.archive.exceptions.general.UnknownSchemaException;
import alma.archive.exceptions.syntax.MalformedQueryException;
import alma.archive.exceptions.syntax.MalformedXMLException;
import alma.archive.exceptions.syntax.UnderspecifiedQueryException;
import alma.archive.exceptions.user.UserDoesNotExistException;
import alma.archive.wrappers.ArchiveTimeStamp;
import alma.archive.wrappers.DocumentData;
import alma.archive.wrappers.Permissions;

import alma.alarmsystem.source.ACSAlarmSystemInterfaceFactory;
import alma.alarmsystem.source.ACSAlarmSystemInterface;
import alma.alarmsystem.source.ACSFaultState;
/**
 * 
 * This class collects all DB communication methods for read access. It uses
 * DatabaseCache, DatabaseHelper.
 * 
 * @author hmeuss
 *  
 */
public class DatabaseReader {

	private DatabaseCache dbCache;

	/* data source for fetching the connections: */
	DatabaseConnectionPool ds = null;

	/* Should be set by setLogger() */
	private Logger logger = Logger.getAnonymousLogger();
		//ClientLogManager.getAcsLogManager().getLoggerForApplication("DatabaseWriter", false);

	/* for the singleton pattern: */
	private static DatabaseReader database;

	//////////////////////////////////////////////////////////////////////
	//          copy table name definitions from DBConfig //
	//////////////////////////////////////////////////////////////////////

	/* some definitions copied from DBConfig: */

	/* table names for general tables */
	protected static final String tableName_schemas = DBConfig.tableName_schemas;

	/* table name suffixes for schema specific tables */
	protected static final String tableName_xml_history = DBConfig.tableName_xml_history;

	//protected static final String tableName_meta_history = DBConfig.tableName_meta_history;

	protected static final String tableNameSuffix_side_unique = DBConfig.tableNameSuffix_side_unique;

	protected static final String tableNameSuffix_side = DBConfig.tableNameSuffix_side;

	protected static String tableName_metaInf = DBConfig.tableName_metaInf;

	protected static String tableName_users = DBConfig.tableName_users;

	protected static String tableName_roles = DBConfig.tableName_roles;

	protected static String tableName_userRoles = DBConfig.tableName_userRoles;

	protected static String tableName_schemaNamespaces = DBConfig.tableName_schemaNamespaces;

	protected static String tableName_namespaces = DBConfig.tableName_namespaces;

	/* column names for all tables */
	protected static final String colName_uid = DBConfig.colName_uid;

	protected static final String colName_schemaName = DBConfig.colName_schemaName;

	protected static final String colName_version = DBConfig.colName_version;

	protected static final String colName_schemaContents = DBConfig.colName_schemaContents;

	protected static final String colName_timestamp = DBConfig.colName_timestamp;

	protected static final String colName_xml = DBConfig.colName_xml;

	protected static final String colName_schemaUid = DBConfig.colName_schemaUid;

	protected static final String colName_owner = DBConfig.colName_owner;

	protected static final String colName_deleted = DBConfig.colName_deleted;

	protected static final String colName_readPermissions = DBConfig.colName_readPermissions;

	protected static final String colName_writePermissions = DBConfig.colName_writePermissions;

	protected static final String colName_hidden = DBConfig.colName_hidden;

	protected static final String colName_dirty = DBConfig.colName_dirty;

	//protected static final String colName_metaColumn = DBConfig.colName_metaColumn;

	//protected static final String colName_newValue = DBConfig.colName_newValue;

	protected static String colName_paramName = DBConfig.colName_paramName;

	protected static String colName_paramValue = DBConfig.colName_paramValue;

	protected static String colName_roleName = DBConfig.colName_roleName;

	protected static String colName_userName = DBConfig.colName_userName;

	protected static String colName_prefix = DBConfig.colName_prefix;

	protected static String colName_namespace = DBConfig.colName_namespace;

	protected static String colName_virtual = DBConfig.colName_virtual;

	/*
	 * values for columnNames in meta_history table:
	 */
	//protected static final short metaHist_schemaUID = DBConfig.metaHist_schemaUID;

	//protected static final short metaHist_owner = DBConfig.metaHist_owner;

	//protected static final short metaHist_deletedHiddenDirty = DBConfig.metaHist_deletedHiddenDirty;

	//protected static final short metaHist_permissions = DBConfig.metaHist_permissions;

	//////////////////////////////////////////////////////////////////////
	//                Class organization //
	//////////////////////////////////////////////////////////////////////

	/**
	 *  
	 */
	private DatabaseReader() throws DatabaseException, ModuleCriticalException {
		super();
		dbCache = DatabaseCache.instance();
		
		try {
		ds = DatabaseConnectionPool.instance(logger);
		} catch (SQLException e) {
			throw new DatabaseException("Could not initialize connection pool"+e);
		}
		initSchemaMaps();
	}

	protected void init() throws ModuleCriticalException, DatabaseException {
		initSchemaMaps();
	}

	/**
	 * 
	 * @return a connection obeject (for read-only access)
	 */
	private Connection getConn(String connInfo) throws ModuleCriticalException {
		// Connect to the database
		Connection readConn = null;
		try {
			readConn = ds.getConnection(connInfo);
			// logger.finest("Created Oracle read connection");
			// System.out.println("Active: "+occi.getActiveSize()+" Cache: "+occi.getCacheSize());
			return readConn;
		} catch (SQLException e) {
			logger.severe("Could not initialize JDBC connection to Oracle");
			throw new ModuleCriticalException(new DatabaseException(e),
					(short) 1, "Could not initialize JDBC connection to Oracle");
		}
	}

	/* for testing purposes only.... */
	public static void main(String[] args) throws ModuleCriticalException {
		try {
			DatabaseReader test = new DatabaseReader();
		} catch (DatabaseException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
			//MPA: no logging, no action. is it correct?
		}
	}

	/**
	 * For implementing the singleton pattern.
	 * 
	 * @return the instance of this class.
	 */
	protected static DatabaseReader instance() throws DatabaseException,
			ModuleCriticalException {
		if (database == null) {
			database = new DatabaseReader();
		}
		return database;
	}

	/**
	 * @param log
	 */
	public void setLogger(Logger log) {
		logger = log;
		ds.setLogger(log);
	}

	//////////////////////////////////////////////////////////////////////
	//  Primitives for directly implementing methods from Internal IF //
	//////////////////////////////////////////////////////////////////////

	/**
	 * 
	 * get document with necessary information if time==null return head of
	 * history if dirtyRead=true return doc even if dirty flag is set
	 * 
	 * @param uid
	 * @param i
	 * @param dirtyRead
	 * @param user
	 * @return xlm document string
	 */
	public String get(URI uid, ArchiveTimeStamp time, boolean dirtyRead,
			String user) throws ArchiveGeneralException,
			PermissionDeniedException, EntityDirtyException,
			ModuleCriticalException {

		Connection readConn = getConn("RD, get("+uid+")");

		Statement stmt = null;
		String sql;
		ResultSet rs = null;

		// TODO: throw exception even if user has no read rights, or doc is
		// deleted etc...???

		/* various document properties */
		String xmlResult = null;
		String schemaName;
		String docPermissions = null;
		String docOwner = null;

		boolean docDirty = false;
		boolean docDeleted = false;
		boolean docHidden = false;
		String[] result; //MPA: never used
		try {
			stmt = readConn.createStatement();
			/* read head of history */
			if (time == null) {
				/*
				 * First read any version of specified doc for determining the
				 * schema.
				 */
				sql = "SELECT " + colName_schemaName + " FROM "
						+ tableName_xml_history + " WHERE " + colName_uid
						+ "='" + uid + "'";
				logger.finest("Executing SQL: " + sql);
				rs = stmt.executeQuery(sql);
				if (rs.next()) {
					schemaName = rs.getString(colName_schemaName);
				} else {
					throw new EntityDoesNotExistException();
				}

				/* Now read the doc from schema specific table: */
				sql = "SELECT " + colName_xml + ", " + colName_owner + ", "
						+ colName_deleted + ", " + colName_dirty + ", "
						+ colName_hidden + ", " + colName_readPermissions
						+ " FROM " + DBConfig.schemaTabName(schemaName)
						+ " WHERE " + colName_uid + "='" + uid + "'" + " AND "
						+ colName_deleted + "=0 AND " + colName_hidden + "=0";
				logger.finest("Executing SQL: " + sql);
				rs = stmt.executeQuery(sql);
				// TODO Normally I should have exception handling after each
				// call to provide more specific error messages...
				if (rs.next()) {
					// schemas are stored as clobs not xmltypes
					if (schemaName.equals(DBConfig.schemaSchemaName)) {
						Clob schemaClob = rs.getClob(colName_xml);
						xmlResult = schemaClob.getSubString(1, (int) schemaClob
								.length());
						// System.out.println(xmlResult);
						//logger.warning(xmlResult);
					} else {
						xmlResult = ((XMLType) rs.getObject(colName_xml))
								.getStringVal();
					}
					docDeleted = rs.getBoolean(colName_deleted);
					docHidden = rs.getBoolean(colName_hidden);
					docDirty = rs.getBoolean(colName_dirty);
					docOwner = rs.getString(colName_owner);
					docPermissions = rs.getString(colName_readPermissions);
				} else {
					throw new EntityDoesNotExistException(uid + " invisible.");
				}

				// check dirty documents:
				if (!dirtyRead && docDirty) {
					throw new EntityDirtyException(uid.toString());
				}

			}
			/* read history document */
			else {
				/* First read schema name and doc from history table. */
				sql = "SELECT " + colName_schemaName + ", " + colName_xml
						+ " FROM " + tableName_xml_history + " WHERE "
						+ colName_uid + "='" + uid + "'" + " AND "
						+ colName_timestamp + "= ?";
				logger.finest("Executing SQL: " + sql);
				PreparedStatement pstmt = readConn.prepareStatement(sql);
				pstmt.setTimestamp(1, time.getTimestamp());
				rs = pstmt.executeQuery();

				if (rs.next()) {
					schemaName = rs.getString(colName_schemaName);
					// TODO is it better to retrieve a CLOB (instead of string)
					// here?
					xmlResult = rs.getString(colName_xml);
				} else {
					// TODO in fact, we should differentiate between doc and ent
					// not exist exception.
					throw new EntityDoesNotExistException();
				}

				pstmt.close();

				/* Now read the metainfo from schema specific table: */
				sql = "SELECT " + colName_owner + ", " + colName_deleted + ", "
						+ colName_dirty + ", " + colName_hidden + ", "
						+ colName_readPermissions + " FROM "
						+ DBConfig.schemaTabName(schemaName) + " WHERE "
						+ colName_uid + "='" + uid + "'" + " AND "
						+ colName_deleted + "=0 AND " + colName_hidden + "=0"
						+ (dirtyRead ? "" : (" AND " + colName_dirty + "=0"));
				logger.finest("Executing SQL: " + sql);
				rs = stmt.executeQuery(sql);
				if (rs.next()) {
					docDeleted = rs.getBoolean(colName_deleted);
					docHidden = rs.getBoolean(colName_hidden);
					docDirty = rs.getBoolean(colName_dirty);
					docOwner = rs.getString(colName_owner);
					docPermissions = rs.getString(colName_readPermissions);
				} else {
					logger.info("Document " + uid + " invisible. Deleted: "
							+ docDeleted + ", hidden: " + docHidden
							+ ", dirty: " + docDirty);
					throw new EntityDoesNotExistException(uid + " invisible.");
				}
			}
		} catch (SQLException e) {
			logger.warning("SQL command raised exception: " + e.toString());
			throw new DatabaseException(e);
		} finally {
			try {
				ds.close(readConn);
				stmt.close();
				rs.close();
			} catch (Exception e) {
				// ignored
				logger.warning(e.toString());
			}
		}
		if (!DatabaseHelper.checkAccessPermissions(user, docPermissions,
				docOwner)) {
			logger.info("Permission for user " + user + " to get document "
					+ uid + " denied");
			throw new PermissionDeniedException(uid.toString());
		}
		return xmlResult;
	}

	/**
	 * 
	 * get document fragments as specified by xpath
	 * 
	 * @param uid
	 * @param xpath
	 * @param namespaces
	 * @param dirtyRead
	 * @param user
	 * @return
	 */
	public String[] queryGet(URI uid, String query, HashMap<String, String> namespaces, boolean dirtyRead, String user) 
		throws PermissionDeniedException, ArchiveGeneralException, DatabaseException, ModuleCriticalException {
		
		
		Connection readConn = getConn("RD, queryGet("+uid+")");

		String sql;
		Statement stmt = null;
		ResultSet rs = null;

		String schemaName;

		Vector<String> out;

		logger.finer("Executing query " + query + " on entity " + uid);

		// obsolete, due to new design of test area
		// in case we are in test mode and visibility is set to test area only,
		// we have to filter the docs:
		//		String testVisibCompPattern = "";
		//		DBConfiguration config = DBConfiguration.instance(logger);
		//		if (config.testMode && config.visibilityTestOnly) {
		//			// comparison pattern for finding elements in the test area:
		//			testVisibCompPattern = " AND " + colName_uid + ">='"
		//					+ config.testStartUid + "' AND " + colName_uid + "<='"
		//					+ config.testEndUid + "'";
		//		}

		// we have to replace ' in query with ", otherwise we will have a
		// parsing problem
		query = query.replace('\'', '"');

		try {
			stmt = readConn.createStatement();
			// TODO check permissions in Database!!!!

			/*
			 * First read any version of specified doc for determining the
			 * schema.
			 */
			sql = "SELECT " + colName_schemaName + " FROM "
					+ tableName_xml_history + " WHERE " + colName_uid + "='"
					+ uid + "'";
			logger.finest("Executing SQL: " + sql);
			rs = stmt.executeQuery(sql);
			if (rs.next()) {
				schemaName = rs.getString(colName_schemaName);
			} else {
				throw new EntityDoesNotExistException();
			}

			String oracleNSstring = constructNamespaceString(namespaces);

			/* construct sequence of XML fragments matching XPath */
			sql = "SELECT value(xmlres) " + colName_xml + ", " + colName_owner
					+ ", " + colName_readPermissions + " FROM "
					+ DBConfig.schemaTabName(schemaName)
					+ ", table(xmlsequence(extract(" + colName_xml + ", '"
					+ query + "', '" + oracleNSstring + "'))) xmlres WHERE "
					+ colName_uid + "='" + uid + "' AND " + colName_deleted
					+ "=0 AND " + colName_hidden + "=0"
					+ (dirtyRead ? "" : (" AND " + colName_dirty + "=0"));
			//+ testVisibCompPattern;
			logger.finest("Executing SQL: " + sql);
			rs = stmt.executeQuery(sql);

			if (!rs.next()) {
				return new String[0];
			}

			/* check permissions */
			String readPermissions = rs.getString(colName_readPermissions);
			String owner = rs.getString(colName_owner);
			if (!DatabaseHelper.checkAccessPermissions(user, readPermissions,
					owner)) {
				logger.info("Permission for user " + user + " to get document "
						+ uid + " denied");
				throw new PermissionDeniedException(uid.toString());
			}

			/* collect all fragments and return them */
			out = new Vector<String>();
			out.add(((XMLType) rs.getObject(colName_xml)).getStringVal());
			while (rs.next()) {
				out.add(((XMLType) rs.getObject(colName_xml)).getStringVal());
			}
		} catch (SQLException e) {
			logger.warning("SQL raised exception: " + e);
			throw new DatabaseException(e);
		} finally {
			try {
				ds.close(readConn);
				stmt.close();
				rs.close();
			} catch (Exception e) {
				logger
						.warning("Ignoring SQL exception while closing DB ressources: "
								+ e.toString());
			}
		}

		return out.toArray(new String[0]);
	}

	/**
	 * TODO remove this method, could be covered by status
	 * 
	 * @param uid
	 * @param user
	 * @return
	 */
	public DocumentData[] allStatus(URI uid, String user)
			throws ArchiveException, ModuleCriticalException {

		Connection readConn = getConn("RD, allStatus("+uid+")");

		Statement stmt = null;
		ResultSet rsHist = null, rsEnt = null;
		String sql;

		DocumentData docData = null;
		DocumentData[] allDocData;
		Vector<Timestamp> allTimestamps = new Vector<Timestamp>();

		Timestamp timestamp = null;
		String schemaUid = null, owner, readPermissions, writePermissions;
		boolean deleted, dirty, hidden, virtual;

		String schemaName = null;

		try {
			try {
				stmt = readConn.createStatement();

				/* first, read schema from xml_history table */
				sql = "SELECT " + colName_schemaName + ", " + colName_timestamp
						+ " FROM " + tableName_xml_history + " WHERE "
						+ colName_uid + "='" + uid + "' ORDER BY "
						+ colName_timestamp;
				logger.finest("Executing SQL: " + sql);
				rsHist = stmt.executeQuery(sql);
				boolean found = false;
				while (rsHist.next()) {
					found = true;
					schemaName = rsHist.getString(colName_schemaName);
					allTimestamps.add(rsHist.getTimestamp(colName_timestamp));
				}
				if (!found) {
					throw new EntityDoesNotExistException(uid.toString());
				}
			} catch (SQLException e) {
				logger.warning("SQL raised exception: " + e.toString());
				try {
					stmt.close();
				} catch (SQLException e1) {
					logger
							.info("Ignoring SQL exception while closing DB resources.");
				}
				throw new DatabaseException(e);
			} finally {
				try {
					rsHist.close();
				} catch (SQLException e1) {
					// ignore
					logger
							.warning("SQL exception while closing Oracle resources: "
									+ e1.toString());
				}
			}
			allDocData = new DocumentData[allTimestamps.size()];
			try {
				/* then, read _entities tabel: */
				sql = "SELECT " + colName_schemaUid + ", " + colName_owner
						+ ", " + colName_readPermissions + ", "
						+ colName_writePermissions + ", " + colName_deleted
						+ ", " + colName_dirty + ", " + colName_hidden + ", "
						+ colName_virtual + " FROM "
						+ DBConfig.schemaTabName(schemaName) + " WHERE "
						+ colName_uid + "='" + uid + "'";
				logger.finest("Executing SQL: " + sql);
				rsEnt = stmt.executeQuery(sql);
				if (rsEnt.next()) {
					schemaUid = rsEnt.getString(colName_schemaUid);
					owner = rsEnt.getString(colName_owner);
					readPermissions = rsEnt.getString(colName_readPermissions);
					writePermissions = rsEnt
							.getString(colName_writePermissions);
					deleted = rsEnt.getBoolean(colName_deleted);
					dirty = rsEnt.getBoolean(colName_dirty);
					hidden = rsEnt.getBoolean(colName_hidden);
					virtual = rsEnt.getBoolean(colName_virtual);
				} else {
					logger.warning("Entity inconsistency: " + uid
							+ "does not exist in table "
							+ DBConfig.schemaTabName(schemaName)
							+ ", but in history table.");
					throw new ArchiveGeneralException("Entity inconsistency: "
							+ uid + " does not exist in table "
							+ DBConfig.schemaTabName(schemaName)
							+ ", but in history table.");
				}

				if (!DatabaseHelper.checkAccessPermissions(user,
						readPermissions, owner)) {
					logger.info("Permission for user " + user
							+ " to get status of entity" + uid + " denied");
					throw new PermissionDeniedException(uid.toString());
				}

				int i = 0;
				for (Iterator iter = allTimestamps.iterator(); iter.hasNext();) {
					timestamp = (Timestamp) iter.next();
					// TODO added dummy parameter vor virtual docs!!!
					docData = new DocumentData(new ArchiveTimeStamp(timestamp),
							new URI(schemaUid), owner, new Permissions(
									readPermissions, writePermissions), hidden,
							dirty, deleted, virtual, "ADMIN");
					allDocData[i] = docData;
					i++;
				}
			} catch (URISyntaxException e) {
				logger.warning("Invalid uid format in entitiy " + uid
						+ "refering to schema: " + schemaUid);
				throw new ArchiveException("Invalid uid format in entitiy "
						+ uid + "refering to schema: " + schemaUid);
			} catch (SQLException e) {
				logger.warning("SQL raised exception: " + e.toString());
				throw new DatabaseException(e);
			} finally {
				try {
					rsEnt.close();
					stmt.close();
				} catch (Exception e1) {
					// ignore
					logger
							.info("SQL exception while closing Oracle resources: "
									+ e1.toString());
				}
			}
		} finally {
			try {
				ds.close(readConn);
			} catch (Exception e) {
				logger.warning("Could not close read connection to Oracle.");
				// ignore
			}
		}
		return allDocData;

	}

	/**
	 * @param uid
	 * @param object
	 * @param user
	 * @return metadata of specified doc. If timestamp is null, return latest
	 *         doc metadata.
	 */
	public DocumentData status(URI uid, ArchiveTimeStamp stamp, String user)
			throws ArchiveException, ModuleCriticalException {

		Connection readConn = getConn("RD, status("+uid+")");

		Statement stmt = null;
		ResultSet rs = null;
		String sql;

		DocumentData docData;

		Timestamp timestamp = null;

		String schemaUid, owner, readPermissions, writePermissions;
		boolean deleted, dirty, hidden, virtual;

		String schemaName;

		try {

			/* first, read schema from xml_history table */
			sql = "SELECT " + colName_schemaName + " FROM "
					+ tableName_xml_history + " WHERE " + colName_uid + "='"
					+ uid + "'";

			if (timestamp != null) {
				sql = sql + " AND " + colName_timestamp + "= ?";
			}
			logger.finer("Executing SQL: " + sql);
			PreparedStatement pstmt = readConn.prepareStatement(sql);
			if (timestamp != null) {
				pstmt.setTimestamp(1, timestamp);
			}
			rs = pstmt.executeQuery();
			if (rs.next()) {
				schemaName = rs.getString(colName_schemaName);
			} else {
				throw new EntityDoesNotExistException(uid.toString());
			}

			pstmt.close();

			stmt = readConn.createStatement();
			/* then, read _entities tabel: */
			sql = "SELECT " + colName_timestamp + ", " + colName_schemaUid
					+ ", " + colName_owner + ", " + colName_readPermissions
					+ ", " + colName_writePermissions + ", " + colName_deleted
					+ ", " + colName_dirty + ", " + colName_hidden + ", "
					+ colName_virtual + " FROM "
					+ DBConfig.schemaTabName(schemaName) + " WHERE "
					+ colName_uid + "='" + uid + "'";
			logger.finer("Executing SQL: " + sql);
			rs = stmt.executeQuery(sql);
			if (rs.next()) {
				timestamp = rs.getTimestamp(colName_timestamp);
				schemaUid = rs.getString(colName_schemaUid);
				owner = rs.getString(colName_owner);
				readPermissions = rs.getString(colName_readPermissions);
				writePermissions = rs.getString(colName_writePermissions);
				deleted = rs.getBoolean(colName_deleted);
				dirty = rs.getBoolean(colName_dirty);
				hidden = rs.getBoolean(colName_hidden);
				virtual = rs.getBoolean(colName_virtual);
			} else {
				logger.warning("Entity inconsistency: " + uid
						+ "does not exist in table "
						+ DBConfig.schemaTabName(schemaName)
						+ ", but in history table.");
				throw new ArchiveGeneralException("Entity inconsistency: "
						+ uid + "does not exist in table"
						+ DBConfig.schemaTabName(schemaName)
						+ ", but in history table.");
			}

			if (!DatabaseHelper.checkAccessPermissions(user, readPermissions,
					owner)) {
				logger.info("Permission for user " + user
						+ " to get status of entity" + uid + " denied");
				throw new PermissionDeniedException(uid.toString());
			}

			try {
				docData = new DocumentData(new ArchiveTimeStamp(timestamp),
						new URI(schemaUid), owner, new Permissions(
								readPermissions, writePermissions), hidden,
						dirty, deleted, virtual, "ADMIN");
			} catch (URISyntaxException e1) {
				logger.warning("Invalid uid format in entitiy " + uid
						+ "refering to schema: " + schemaUid);
				throw new ArchiveException("Invalid uid format in entitiy "
						+ uid + "refering to schema: " + schemaUid);
			}

		} catch (SQLException e) {
			logger.warning("SQL raised exception: " + e.toString());
			throw new DatabaseException(e);
		} finally {
			try {
				ds.close(readConn);
				stmt.close();
				rs.close();
			} catch (Exception e1) {
				// ignore
				logger
						.info("Ignoring SQL exception while closing Oracle resources");
			}
		}
		return docData;
	}

	// Note: Here, the namespaces are created in the query method itself. We should change the global architecture and do the same everywhere!
	public URI[] queryInterval(ArchiveTimeStamp timeFrom, ArchiveTimeStamp timeTo, String schema, String pathQuery, String user) throws ModuleCriticalException, ArchiveException {
		
		Vector<URI> uids = new Vector();
		
		// get results from database
		DBCursor cursor = queryContent(pathQuery, schema, getSchemaNamespaces(dbCache.schemaName2uri.get(schema)), false, user, timeFrom.toISOString(), timeTo.toISOString());
		
		while (cursor.hasNext()) {
			uids.add(cursor.next().getUri());
		}
		cursor.close();

		return uids.toArray(new URI[0]);
		
		// put results from cursor into uids:
		
		/*
		 * The following uses code from queryRecent, but this does not work for content queries (attributes etc.)
		 * Therefore I delegated the call to queryContent
		Connection readConn = getConn("RD, queryInterval("+timeFrom.toISOString()+"-"+timeTo.toISOString()+")");

		Statement stmt = null;
		String sql;
		ResultSet rs = null;
		Vector<URI> out = new Vector<URI>();
		String uid = null;

		try {
			stmt = readConn.createStatement();
			

			String oracleNSstring = constructNamespaceString(getSchemaNamespaces(dbCache.schemaName2uri.get(schema)));

			
			sql = "SELECT " + colName_uid + ", value(xmlres) " + colName_xml + " FROM "
					+ DBConfig.schemaTabName(schema)
					+ ", table(xmlsequence(extract(" + colName_xml + ", '"
					+ pathQuery + "', '" + oracleNSstring + "'))) xmlres WHERE "
					+ colName_deleted + "=0 AND " + colName_hidden + "=0"
					+ " AND " + colName_dirty + "=0 AND " + colName_timestamp
					+ " > '" + timeFrom.toSQLString()+"'" + " AND " + colName_timestamp
					+ " < '" + timeTo.toSQLString()+"'";
			logger.finest("Executing SQL: " + sql);
			rs = stmt.executeQuery(sql);
			
			while (rs.next()) {
					uid = rs.getString(colName_uid);
					try {
						out.add(new URI(uid));
					} catch (URISyntaxException e) {
						logger.warning("Found incorrect UID in table for schema "+schema+":"+uid);	
					}
				}
		} catch (SQLException e) {
			logger.info("SQL raised exception: " + e.toString());
			throw new DatabaseException(e);
		} finally {
			try {
				rs.close();
				stmt.close();
				ds.close(readConn);
			} catch (SQLException e1) {
				// ignore
				logger
						.info("Ignoring exception while closing Oracle resources.");
			}
		}

			return out.toArray(new URI[0]);
			*/
	}


	/**
	 * @param timestamp
	 * @param schema
	 * @param user
	 */
	//	 queries only heads of history, later than timestamp
	public URI[] queryRecent(ArchiveTimeStamp timestamp, String schema, String user)
			throws ModuleCriticalException, DatabaseException, ArchiveException {

		Connection readConn = getConn("RD, queryRecent("+timestamp+")");

		Statement stmt = null;
		String sql;
		ResultSet rs = null;
		Vector<URI> out = new Vector<URI>();
		String uid = null;

		try {
			stmt = readConn.createStatement();
			// TODO check permissions in Database!!!!
			// I construct the sequence of fragments in Oracle
			sql = "SELECT " + colName_uid + " FROM "
					+ DBConfig.schemaTabName(schema) + " WHERE "
					+ colName_deleted + "=0 AND " + colName_hidden + "=0"
					+ " AND " + colName_dirty + "=0 AND " + colName_timestamp
					+ " > '" + timestamp.toSQLString()+"'";
			logger.finest("Executing SQL: " + sql);
			rs = stmt.executeQuery(sql);

			while (rs.next()) {
					uid = rs.getString(colName_uid);
					try {
						out.add(new URI(uid));
					} catch (URISyntaxException e) {
						logger.warning("Found incorrect UID in table for schema "+schema+":"+uid);	
					}
				}
		} catch (SQLException e) {
			logger.info("SQL raised exception: " + e.toString());
			throw new DatabaseException(e);
		} finally {
			try {
				rs.close();
				stmt.close();
				ds.close(readConn);
			} catch (SQLException e1) {
				// ignore
				logger
						.info("Ignoring exception while closing Oracle resources.");
			}
		}

			return out.toArray(new URI[0]);
	}


	/**
	 * same as query, but handles content queries correctly. Does not always work
	 * for non-content queries. 
	 * 
	 * Optimized for queries to log entries: The two timestamps are used as kind of interval, only docs lying in the interval are returned. 
	 * If timeFrom or timeTo are null, they are not considered. 
	 * 
	 * @param query
	 * @param schema
	 * @param namespaces
	 * @param dirtyRead
	 * @param user
	 */
	public DBCursor queryContent(String query, String schema, HashMap namespaces,
			boolean dirtyRead, String user, String timeFrom, String timeTo) throws DatabaseException,
			MalformedQueryException, ModuleCriticalException {

		// 1) find out last step in query
		BaseXPath  xpath;
		try {
        xpath = new JDOMXPath(query);
		} catch (JaxenException e) {
			throw new MalformedQueryException("Could not parse query: "+query);
		}
        LocationPath locPath = (LocationPath) (xpath.getRootExpr());        
        
        List locSteps = 
            locPath.getSteps();
        
        Step lastStep = (Step) locSteps.get(locSteps.size()-1);
		
		// 2) add upward step to query.
        query=query+"/..";
		
		// submit modified query to Oracle
		CursorImpl cursor = (CursorImpl) query (query, schema, namespaces, dirtyRead, user, timeFrom, timeTo); 
        
		// set returned cursor attribute reevaluateStep (meaning that 
		// before returning single query results, they have to be modified.
		
		cursor.reevaluateLastStep(lastStep);
		
		// return cursor, rest is done by cursor implementation
		return cursor;
		
	}
	
	/**
	 *  
	 * 
	 * Optimized for queries to log entries: The two timestamps are used as kind of interval, only docs lying in the interval are returned. 
	 * If timeFrom or timeTo are null, they are not considered. 
	 * 
	 * @param query
	 * @param schema
	 * @param namespaces
	 * @param dirtyRead
	 * @param user
	 */
	public DBCursor query(String query, String schema, HashMap namespaces,
			boolean dirtyRead, String user, String timeFrom, String timeTo) throws DatabaseException,
			MalformedQueryException, ModuleCriticalException {
		// TODO include namespace treatment!!!
		// TODO include permissions

		Connection readConn = getConn("RD, query("+query+")");

		Statement stmt = null;
		String sql;
		ResultSet rs = null;

		logger.finer("Executing query " + query);

		// obsolete due to new design
		// in case we are in test mode and visibility is set to test area only,
		// we have to filter the docs:
		//		String testVisibCompPattern = "";
		//		DBConfiguration config = DBConfiguration.instance(logger);
		//		if (config.testMode && config.visibilityTestOnly) {
		//			// comparison pattern for finding elements in the test area:
		//			testVisibCompPattern = " AND " + colName_uid + ">='"
		//					+ config.testStartUid + "' AND " + colName_uid + "<='"
		//					+ config.testEndUid + "'";
		//		}

		// we have to replace ' in query with ", otherwise we will have a
		// parsing problem
		query = query.replace('\'', '"');

		try {
			stmt = readConn.createStatement();

			String oracleNSstring = constructNamespaceString(namespaces);

			// TODO check permissions in Database!!!!
			// I construct the sequence of fragments in Oracle
			sql = "SELECT " + colName_uid + ", value(xmlres) " + colName_xml
					+ ", " + colName_owner + ", " + colName_readPermissions
					+ " FROM " + DBConfig.schemaTabName(schema)
					+ ", table(xmlsequence(extract(" + colName_xml + ", '"
					+ query + "', '" + oracleNSstring + "'))) xmlres WHERE "
					+ colName_deleted + "=0 AND " + colName_hidden + "=0"
					+ (dirtyRead ? "" : (" AND " + colName_dirty + "=0"));
			// + testVisibCompPattern;
			// TODO construct SQL query with side table access
			
			// if timeFrom and timeTo are != null, use them in the SQL statement:
			if (timeFrom!=null && timeTo!=null) {
				sql = sql + " AND (timestamp BETWEEN '"+new ArchiveTimeStamp(timeFrom).toSQLString()+"' AND '"+new ArchiveTimeStamp(timeTo).toSQLString()+"')";
			}
			
			logger.finest("Executing SQL: " + sql);
			rs = stmt.executeQuery(sql);
		} catch (SQLException e) {
			logger.info("SQL raised exception: " + e.toString());
			try {
				rs.close();
				stmt.close();
			} catch (Exception e1) {
				// ignore
				logger.info("Ignoring exception while closing Oracle resources. "+e.toString());
			}
			throw new DatabaseException(e);
		}

		return new CursorImpl(rs, readConn, user);
		// Note: the resources must not be closed, since Db2Cursor needs to
		// access them alive.
	}

	/**
	 * @param query
	 * @param schema
	 * @param namespaces
	 * @param dirtyRead
	 * @param user
	 * @return
	 */
	public URI[] queryIDs(String query, String schema, HashMap namespaces,
			boolean dirtyRead, boolean readAll, String user)
			throws ArchiveException, ModuleCriticalException {
		// TODO include namespace treatment!!!
		// TODO include permissions

		Connection readConn = getConn("RD, queryIDs("+query+")");

		Statement stmt = null;
		String sql;
		ResultSet rs = null;
		Vector<URI> out = new Vector<URI>();
		String uid = null;

		logger.finer("Executing query " + query);

		// obsolete due to new design
		// in case we are in test mode and visibility is set to test area only,
		// we have to filter the docs:
		//		String testVisibCompPattern = "";
		//		DBConfiguration config = DBConfiguration.instance(logger);
		//		if (config.testMode && config.visibilityTestOnly) {
		//			// comparison pattern for finding elements in the test area:
		//			testVisibCompPattern = " AND " + colName_uid + ">='"
		//					+ config.testStartUid + "' AND " + colName_uid + "<='"
		//					+ config.testEndUid + "'";
		//		}

		// we have to replace ' in query with ", otherwise we will have a
		// parsing problem
		query = query.replace('\'', '"');

		try {
			stmt = readConn.createStatement();

			String oracleNSstring = constructNamespaceString(namespaces);

			// simple solution fetch all docs, scan and filter them:
			// TODO construct SQL query with side table access
			sql = "SELECT "
					+ colName_uid
					+ ", "
					+ colName_owner
					+ ", "
					+ colName_readPermissions
					+ " FROM "
					+ DBConfig.schemaTabName(schema)
					+ " WHERE existsNode("
					+ colName_xml
					+ ", '"
					+ query
					+ "', '"
					+ oracleNSstring
					+ "')=1"
					+ (readAll ? "" : (" AND " + colName_deleted + "=0 AND "
							+ colName_hidden + "=0" + (dirtyRead ? ""
							: (" AND " + colName_dirty + "=0"))));
			// + testVisibCompPattern;
			logger.finest("Executing SQL: " + sql);
			rs = stmt.executeQuery(sql);
			while (rs.next()) {
				String readPermissions = rs.getString(colName_readPermissions);
				String owner = rs.getString(colName_owner);
				if (DatabaseHelper.checkAccessPermissions(user,
						readPermissions, owner)) {
					uid = rs.getString(colName_uid);
					out.add(new URI(uid));
				}
			}
		} catch (SQLException e) {
			logger.info("SQL raised exception: " + e.toString());
			throw new DatabaseException(e);
		} catch (URISyntaxException e) {
			logger.warning("Found uid not conforming to URI syntax: " + uid);
			throw new ArchiveException(e);
		} finally {
			try {
				ds.close(readConn);
				stmt.close();
				rs.close();
			} catch (Exception e1) {
				// ignore
				logger
						.info("Ignoring SQL exception while closing DB resources");
			}
		}
		return out.toArray(new URI[0]);
	}

	/**
	 * timeFrom and timeTo restrict the time interval of returned logs.
	 * type restrict the (numerical) types of logs. 
	 * routine, source and process can be set to * in order to denote don't care. 
	 * maxRow is the maximal number of rows returned in the collection (if there are more logs, they are ignored).
	 * Logs are returned in timestamp order, starting with the lowest timestamp.
	 * 
	 * This method will check the UNIX environment variable LOCATION. If it is set, only logs written from that STE will 
	 * be returned (using column STEENV)
	 * 
	 * @param timeFrom
	 * @param timeTo
	 * @param minType
	 * @param maxType
	 * @param routine
	 * @param source
	 * @param process
	 * @param maxRow
	 * @return
	 * @throws ArchiveException
	 * @throws ModuleCriticalException 
	 */
	public Collection<String> queryLog(String timeFrom, String timeTo,
			short minType, short maxType, String routine, String source,
			String process, int maxRow) throws ArchiveException, ModuleCriticalException {
		Statement stmt = null;
		String sql;
		ResultSet rs = null;
		Vector<String> out = new Vector<String>();
//
//		// until we have a numeric value for the types available, we have to do a stupid comparison:
//		Vector<String> types = new Vector<String>();
//		for (int i=minType; i<=maxType; i++) {
//			types.add(LogTypeHelper.getAllTypesDescriptions()[i]);
//		}
		
		// get STE variable
		String steEnv = System.getenv("LOCATION");
		
		sql = "SELECT "+DBConfig.colName_log_xml+" FROM xml_logEntries WHERE '"+new ArchiveTimeStamp(timeFrom).toSQLString()+"'<"+DBConfig.colName_log_TimeStamp+" AND "+DBConfig.colName_log_TimeStamp+"<'"+new ArchiveTimeStamp(timeTo).toSQLString()+"' AND ("+DBConfig.colName_log_level+" BETWEEN '"+minType+"' AND '"+maxType+"')";
		if (routine!=null && !routine.equals("*")) {
			sql = sql+" AND "+DBConfig.colName_log_Routine+" = '"+routine+"'";
		}
		if (source!=null && !source.equals("*")) {
			sql = sql+" AND "+DBConfig.colName_log_SourceObject+" = '"+source+"'";
		}
		if (process!=null && !process.equals("*")) {
			sql = sql+" AND "+DBConfig.colName_log_Process+" = '"+process+"'";
		}
		if (steEnv!=null && !steEnv.equals("")) {
			sql = sql+" AND "+DBConfig.colName_log_ste+" = '"+steEnv+"'";
		}
		sql = sql +  " ORDER BY "+DBConfig.colName_log_TimeStamp;
		
		Connection readConn=null;
		try {
			readConn = getConn("RD, getLogs");

			stmt = readConn.createStatement();
			//System.out.println("Executing LOG SQL "+sql);// TODO remove
			rs = stmt.executeQuery(sql);
			
			for (int logCount = 0; logCount<maxRow && rs.next(); logCount++) {
				out.add(rs.getString(1));
			}
			
		} catch (SQLException e) {
			throw new DatabaseException(e);
		} finally {
			try {
				rs.close();
				stmt.close();
				ds.close(readConn);
			} catch (Exception e1) {
				// ignore
			}
		}
		
		
		return out;
	}

	/**
	 * 
	 * retrieves schema name (string) from database given a specific version
	 * (URI) of a schema.
	 * 
	 * @param schema
	 * @return
	 */
	public String getSchemaName(URI schema) throws UnknownSchemaException,
			DatabaseException, ModuleCriticalException {

		Statement stmt = null;
		String sql;
		ResultSet rs = null;

		// first have a look in the cached schema names:
		if (dbCache.uri2schemaName.get(schema) != null) {
			return (String) dbCache.uri2schemaName.get(schema);
		}

		Connection readConn = getConn("RD, getSchemaName("+schema+")");

		try {
			stmt = readConn.createStatement();
			sql = "SELECT " + colName_schemaName + " FROM " + tableName_schemas
					+ " WHERE " + colName_uid + "='" + schema + "'";
			logger.finer("Executing SQL: " + sql);
			rs = stmt.executeQuery(sql);
			if (rs.next()) {
				String ret = rs.getString(colName_schemaName);
				return ret;
			} else {
				throw new UnknownSchemaException(schema.toString());
			}

		} catch (SQLException e) {
			logger.warning("SQL raised exception: " + e.toString());
			throw new DatabaseException(e);
		} finally {
			try {
				ds.close(readConn);
				stmt.close();
				rs.close();
			} catch (Exception e1) {
				// ignore
				logger.warning("SQL Exception while closing Oracle resources: "
						+ e1.toString());
			}
		}
	}

	/**
	 * @param schemaUri
	 * @return integer version number of schema URI in schema history, or 0, if
	 *         URI is not found
	 */
	public int getSchemaVersion(URI schemaUri) throws DatabaseException,
			ModuleCriticalException {

		Connection readConn = getConn("RD, getSchemaVersion("+schemaUri+")");

		Statement stmt = null;
		ResultSet rs = null;
		try {
			String sql = "SELECT " + colName_version + " FROM "
					+ tableName_schemas + " WHERE " + colName_uid + "='"
					+ schemaUri + "'";
			logger.finest("Executing SQL: " + sql);
			stmt = readConn.createStatement();
			rs = stmt.executeQuery(sql);
			if (rs.next()) {
				return rs.getInt(colName_version);
			} else {
				return 0;
			}
		} catch (SQLException e) {
			logger.warning("Could not access schema table: " + e.toString());
			throw new DatabaseException(e);
		} finally {
			try {
				ds.close(readConn);
				stmt.close();
				rs.close();
			} catch (Exception e1) {
				logger.warning("SQL exception while closing Oracle resources: "
						+ e1.getMessage());
			}
		}
	}

	/**
	 * @param schema
	 * @param version
	 * @return Schema URI of schema in given version
	 */
	public URI getSchemaURI(String schema, int version)
			throws UnknownSchemaException, DatabaseException,
			ModuleCriticalException {

		Connection readConn = getConn("RD, getSchemaURI("+schema+")");

		Statement stmt = null;
		ResultSet rs = null;
		try {
			String sql = "SELECT " + colName_uid + " FROM " + tableName_schemas
					+ " WHERE " + colName_schemaName + "='" + schema + "' AND "
					+ colName_version + "=" + version;
			logger.finest("Executing SQL: " + sql);
			stmt = readConn.createStatement();
			rs = stmt.executeQuery(sql);
			if (rs.next()) {
				String uid = rs.getString(colName_uid);
				try {
					return new URI(uid);
				} catch (URISyntaxException e1) {
					logger.warning("Invalid UID in database, table "
							+ tableName_schemas + ": " + uid);
					throw new DatabaseException(
							"Invalid UID in database, table "
									+ tableName_schemas + ": " + uid);
				}
			} else {
				logger.info("Requested non-existing version " + version
						+ " of schema " + schema);
				throw new UnknownSchemaException("Schema " + schema
						+ " does not exist in version " + version
						+ " or at all");
			}
		} catch (SQLException e) {
			logger.info("Could not retrieve schema uid for schema " + schema
					+ " version " + version);
			throw new DatabaseException(
					"Could not retrieve schema uid for schema " + schema
							+ " version " + version);
		} finally {
			try {
				ds.close(readConn);
				stmt.close();
				rs.close();
			} catch (Exception e1) {
				logger
						.info("Ignoring SQL exception while closing DB resources."
								+ e1.toString());
			}
		}
	}

	/**
	 * @return true iff database backend is alive.
	 * 
	 */
	public boolean ping() {
		// We test sanity of backend by reading the archiveID.
		
		try {
			return (getMetaParamValue(DBConfig.paramName_archiveID)!=null);
		} catch (Throwable e) {
			// error!!!
			logger.severe("Could not read archive ID. "+e.toString());
		
			return false;
		} 
		
		// all went fine
		//return true;
		
		/*
		try {
			ods_l = DatabaseConnectionPool.instance(logger);
			readConn_l = ods_l.getConnection("RD, ping");

			stmt = readConn_l.createStatement();
			String sql = "SELECT " + colName_schemaUid + " FROM "
					+ tableName_schemas;
			logger.finest("Executing SQL: " + sql);
			rs = stmt.executeQuery(sql);

			//			// There must be at least one schema:
			//			if (rs.next()) {
			//				return true;
			//			} else {
			//				return false;
			//			}
			return true; // no exception --> ok
		} catch (Exception e) {
			logger.warning(e.toString());
			//MPA: i would prefer the other logging format
			return false;
		} finally {
			try {
				// close ressources
				rs.close();
			} catch (SQLException e1) {
				logger.warning(e1.toString());
			}
			try {
				stmt.close();
			} catch (SQLException e2) {
				logger.warning(e2.toString());
			}
			try {
				ods_l.close(readConn_l);
			} catch (SQLException e3) {
				logger.warning(e3.toString());
			}
		}
		*/

	}
	
	//////////////////////////////////////////////////////////////////////
	//                   namespace helpers //
	//////////////////////////////////////////////////////////////////////

	/**
	 * 
	 * @param namespaces:
	 *            Key/value pairs where key is the namespace prefix and value
	 *            the namespace.
	 * 
	 * @return string of the form xmlns:key="value"
	 */
	protected String constructNamespaceString(HashMap namespaces) {
		StringBuffer out = new StringBuffer();

		for (Iterator it = namespaces.entrySet().iterator(); it.hasNext();) {
			Map.Entry next = (Map.Entry) it.next();
			out.append(" xmlns:" + next.getKey() + "=\"" + next.getValue()
					+ "\"");
		}

		return out.toString();
	}

	/**
	 * @param schemaUri
	 * @return namespace registered for schemaUri. If schemaUri is null, return
	 *         all registered namespaces.
	 */
	public HashMap<String, String> getSchemaNamespaces(URI schemaUri) throws DatabaseException, ModuleCriticalException {
		// TODO buffer namespaces in cache???

		Connection readConn = getConn("RD, getSchemaNamespaces("+schemaUri+")");

		Statement stmt = null;
		ResultSet rs = null;
		HashMap<String, String> namespaces = new HashMap<String, String>();
		String sql = "SELECT ns."
				+ colName_prefix
				+ ", "
				+ colName_namespace
				+ " FROM "
				+ tableName_namespaces
				+ " ns"
				+ (schemaUri == null ? "" : (" , " + tableName_schemaNamespaces
						+ " sns WHERE ns." + colName_prefix + " = sns."
						+ colName_prefix + " AND " + colName_schemaUid + "='"
						+ schemaUri + "'"));
		logger.finest("Executing SQL: " + sql);
		try {
			stmt = readConn.createStatement();
			rs = stmt.executeQuery(sql);
			while (rs.next()) {
				String prefix = rs.getString(colName_prefix);
				String namespace = rs.getString(colName_namespace);
				//try {
				namespaces.put(prefix, namespace);
				//				} catch (URISyntaxException e1) {
				//					logger.warning(
				//						"Illformed namespace "
				//							+ namespace
				//							+ " for schema "
				//							+ schemaUri
				//							+ " and prefix "
				//							+ prefix);
				//				}
			}
		} catch (SQLException e) {
			logger.warning("Could not retrieve namespaces for schema "
					+ schemaUri);
			throw new DatabaseException(e);
		} finally {
			try {
				ds.close(readConn);
				stmt.close();
				rs.close();
			} catch (Exception e1) {
				logger
						.warning("Ignoring SQL exception while closing DB resources: "
								+ e1.toString());
			}
		}
		return namespaces;
	}

	/**
	 * @param namespace
	 * @return
	 */
	public boolean namespaceExists(URI namespace) throws DatabaseException,
			ModuleCriticalException {
		// TODO buffer namespaces in cache???

		Connection readConn = getConn("RD, namespaceExists("+namespace+")");

		Statement stmt = null;
		ResultSet rs = null;
		String sql = "SELECT " + colName_prefix + " FROM "
				+ tableName_namespaces + " WHERE " + colName_namespace + "='"
				+ namespace + "'";
		logger.finest("Executing SQL: " + sql);
		try {
			stmt = readConn.createStatement();
			rs = stmt.executeQuery(sql);
			if (rs.next()) {
				return true;
			} else {
				return false;
			}
		} catch (SQLException e) {
			logger.warning("Could not access namespace " + namespace);
			throw new DatabaseException(e);
		} finally {
			try {
				ds.close(readConn);
				stmt.close();
				rs.close();
			} catch (Exception e1) {
				logger.warning("Ignoring SQL exception while closing DB resources: " + e1.toString());
			}
		}
	}

	//////////////////////////////////////////////////////////////////////
	//                   Various helpers //
	//////////////////////////////////////////////////////////////////////

	/**
	 * close the connection to database
	 */
	public void close() throws DatabaseException {
		// empty in the moment...
	}

	/**
	 * 
	 * Check whether the latest version of schemaName is schemaURI.
	 * 
	 * @param schemaName
	 * @param schemaURIold
	 * @param schemaURInew
	 */
	protected boolean checkSchemaConsistency(String schemaName, URI schemaURI)
			throws DatabaseException, DocumentDoesNotExistException,
			ModuleCriticalException {

		Connection readConn = getConn("RD, checkSchemaConsistency("+schemaName+")");

		Statement stmt = null;
		ResultSet rs = null;
		String sql = null;

		try {
			stmt = readConn.createStatement();
			sql = "SELECT " + colName_timestamp + " FROM " + tableName_schemas
					+ " WHERE " + colName_uid + "='" + schemaURI.toString()
					+ "'";
			logger.finest("Executing SQL: " + sql);
			rs = stmt.executeQuery(sql);
			if (!rs.next()) {
				throw new DocumentDoesNotExistException(schemaURI.toString());
			}
			String time = new ArchiveTimeStamp(rs
					.getTimestamp(colName_timestamp)).toSQLString();
			sql = "SELECT " + colName_uid + " FROM " + tableName_schemas
					+ " WHERE " + colName_schemaName + "='" + schemaName
					+ "' AND " + colName_timestamp + "> '" + time + "'";
			logger.finest("Executing SQL: " + sql);
			rs = stmt.executeQuery(sql);
			boolean found = rs.next();
			return !found;

		} catch (SQLException e) {
			logger.info("SQL raised exception: " + sql);
			throw new DatabaseException(e.toString());
		} finally {
			try {
				ds.close(readConn);
				stmt.close();
				rs.close();
			} catch (Exception e1) {
				logger
						.info("SQL Exception while closing Oracle resources ignored: "
								+ e1.toString());
			}
		}

	}

	/**
	 * 
	 * check whether the latest version of entity uid has timestamp timeStamp
	 * check whether entity uid exists if not, throw exception
	 * 
	 * @param uid
	 * @param timeStamp
	 */
	protected void checkTimestampConsistency(URI uid, String schemaName,
			ArchiveTimeStamp timeStamp) throws HistoryInconsistencyException,
			EntityDoesNotExistException, DatabaseException,
			ModuleCriticalException {
		// TODO should we do the SELECT really here? Maybe the Timestamp is
		// already known...

		Connection readConn = getConn("RD, checkTimestampConsistency("+uid+")");

		Statement stmt = null;
		String sql = null;
		ResultSet rs = null;
		try {
			sql = "SELECT " + colName_timestamp + " FROM "
					+ DBConfig.schemaTabName(schemaName) + " WHERE "
					+ colName_uid + "='" + uid + "'";
			stmt = readConn.createStatement();
			rs = stmt.executeQuery(sql);
			if (!rs.next()) {
				throw new EntityDoesNotExistException(uid.toString());
			}
			Timestamp dbTimeStamp = rs.getTimestamp(colName_timestamp);
			if (!timeStamp.equals(dbTimeStamp)) {
				throw new HistoryInconsistencyException();
			}
		} catch (SQLException e) {
			throw new DatabaseException(e);
			//MPA: logging missing
		} finally {
			try {
				ds.close(readConn);
				stmt.close();
				rs.close();
			} catch (Exception e) {
				logger.info(e.toString());
			}
		}

	}

	/**
	 * Test whether user has permission to access doc (belonging to schema). If
	 * writePermision=true, then write access is tested, else read access.
	 *  
	 */

	protected void checkPermission(URI uid, String schemaName, String user,
			boolean writePermission) throws EntityDoesNotExistException,
			DatabaseException, PermissionDeniedException,
			ModuleCriticalException {

		Connection readConn = getConn("RD, checkPermission("+uid+")");

		String sql;
		String docOwner, docPermissions;

		ResultSet rs = null;
		Statement stmt = null;

		try {
			stmt = readConn.createStatement();

			sql = "SELECT "
					+ colName_owner
					+ ", "
					+ (writePermission ? colName_writePermissions
							: colName_readPermissions) + " FROM "
					+ DBConfig.schemaTabName(schemaName) + " WHERE "
					+ colName_uid + "='" + uid + "'";
			logger.finest("Executing SQL: " + sql);
			rs = stmt.executeQuery(sql);
			// TODO Normally I should have exception handling after each call to
			// provide more specific error messages...
			if (rs.next()) {
				docOwner = rs.getString(colName_owner);
				docPermissions = rs
						.getString(writePermission ? colName_writePermissions
								: colName_readPermissions);
			} else {
				throw new EntityDoesNotExistException(uid.toString());
			}
		} catch (SQLException e) {
			logger.warning("SQL raised exception: " + e.toString());
			throw new DatabaseException(e);
		} finally {
			try {
				ds.close(readConn);
				stmt.close();
				rs.close();
			} catch (Exception e) {
				// ignore
				logger
						.info("Ignoring SQL exception while closing DB resources.");
			}
		}

		if (!DatabaseHelper.checkAccessPermissions(user, docPermissions,
				docOwner)) {
			logger.info("Permission for user " + user + " to "
					+ (writePermission ? "write" : "read") + " document " + uid
					+ " denied");
			throw new PermissionDeniedException(uid.toString());
		}

	}

	/**
	 * Check whether the uid already exists in the database. In the actual table
	 * architecture schemas are not considerered, i.e. exists returns false,
	 * even if a schema with this uid exists in the schema table.
	 * 
	 * @param uid
	 * @return true if entity uid is already stored.
	 */
	public boolean exists(URI uid) throws DatabaseException,
			ModuleCriticalException {

		Connection readConn = getConn("RD, exists("+uid+")");

		Statement stmt = null;
		ResultSet rs = null;
		String sql = null;
		try {
			stmt = readConn.createStatement();
			// TODO only retrieve one result!
			sql = "SELECT " + colName_uid + " FROM " + tableName_xml_history
					+ " WHERE " + colName_uid + "='" + uid + "'";
			rs = stmt.executeQuery(sql);
			boolean exists = rs.next();
			return exists;
		} catch (SQLException e) {
			logger.warning("SQLException thrown when executing: " + sql);
			throw new DatabaseException(e);
		} finally {
			try {
				ds.close(readConn);
				stmt.close();
				rs.close();
			} catch (Exception e) {
				// Do nothing...
				logger.warning(e.toString());
			}
		}

	}

	/**
	 * check whether schema with a given name exists.
	 * 
	 * @param schemaName
	 * @return
	 */
	protected boolean existsSchema(String schemaName) {
		/* check whether schema exists in schemaName2uri */
		return dbCache.schemaName2uri.containsKey(schemaName);
	}

	/**
	 * initialize schema maps in DatabaseCache, i.e. fill variable
	 * schemaName2uri and uri2schemaName with pairs mapping URIs to schema names
	 * (Strings) and vice versa. only the latest versions of a schema are stored
	 * in the HashMap. It is updated, whenever a new version of a schema is
	 * inserted.
	 */
	protected void initSchemaMaps() throws DatabaseException,
			ModuleCriticalException {
		dbCache.uri2schemaName = new HashMap<URI, String>();
		dbCache.schemaName2uri = new HashMap<String, URI>();

		Connection readConn = getConn("RD, initSchemaMaps");

		String sql = null;
		Statement stmt = null;
		ResultSet rs = null;

		try {
			stmt = readConn.createStatement();
			sql = "WITH actSchemas AS ( SELECT " + colName_schemaName
					+ " actName" + ", MAX(" + colName_timestamp
					+ ") actTime FROM " + tableName_schemas + " GROUP BY "
					+ colName_schemaName + ") SELECT " + colName_uid
					+ ", actName FROM actSchemas, " + tableName_schemas
					+ " WHERE actName=" + colName_schemaName + " and actTime="
					+ colName_timestamp;
			logger.finest("Executing SQL: " + sql);
			rs = stmt.executeQuery(sql);

			String schemaName;
			String schemaURI = null;
			while (rs.next()) {
				schemaName = rs.getString("actName");
				try {
					schemaURI = rs.getString(colName_uid);
					dbCache.uri2schemaName.put(new URI(schemaURI), schemaName);
					dbCache.schemaName2uri.put(schemaName, new URI(schemaURI));
				} catch (URISyntaxException e1) {
					logger.severe("UID of schema " + schemaName
							+ " does not respect URI syntax: " + schemaURI);
				}
				logger.finest("Found schema: " + schemaName+" with UID "+schemaURI);
			}

		} catch (SQLException e) {
			logger.warning("SQL command raised exception: " + sql);
			logger.severe("Could not initialize schema name map.");
			throw new DatabaseException(e);
		} finally {
			try {
				ds.close(readConn);
				stmt.close();
				rs.close();
			} catch (Exception e) {
				logger.info(e.toString());
			}
		}

	}

	/**
	 * get value of meta param as stored in Oracle
	 * 
	 * @param paramName
	 * @return value stored in table metaInf in database. If no entry is found,
	 *         return null.
	 */
	protected String getMetaParamValue(String paramName)
			throws DatabaseException, ModuleCriticalException {

		Connection readConn = getConn("RD, getMetaParamValue("+paramName+")");

		Statement stmt = null;
		String sql;
		String out = null;
		ResultSet rs = null;
		try {
			sql = "SELECT " + colName_paramValue + " FROM " + tableName_metaInf
					+ " WHERE " + colName_paramName + "='" + paramName + "'";
			logger.finest("Executing SQL: " + sql);
			stmt = readConn.createStatement();
			rs = stmt.executeQuery(sql);
			if (rs.next()) {
				out = rs.getString(colName_paramValue);
			} else {
				logger.info("Could not find parameter value for parameter "
						+ paramName + " in table " + tableName_metaInf);
			}
		} catch (SQLException e) {
			logger.info("Oracle error: " + e.toString());
			throw new DatabaseException(e);
		} finally {
			try {
				ds.close(readConn);
				stmt.close();
				rs.close();
			} catch (Exception e) {
				logger.info(e.toString());
			}
		}

		return out;
	}
	
	/**
	 * get value of last UID as stored as a SEQUENCE in Oracle
	 * 
	 * @param paramName
	 * @return value stored in table metaInf in database. If no entry is found,
	 *         return null.
	 */
	protected String getLastUIDstored()
			throws DatabaseException, ModuleCriticalException {

		Connection readConn = getConn("RD, get(lastUID)");

		Statement stmt = null;
		String sql;
		String out = null;
		ResultSet rs = null;
		try {
			sql = "SELECT "+DBConfig.paramName_lastUIDstored+"."+DBConfig.colName_nextVal+" FROM DUAL";
			logger.finest("Executing SQL: " + sql);
			stmt = readConn.createStatement();
			rs = stmt.executeQuery(sql);
			if (rs.next()) {
				out = rs.getString(DBConfig.colName_nextVal);
				logger.finest("Retrieved "+out+" as decimal value for UIDs from SEQUENCE"+DBConfig.paramName_lastUIDstored);
			} else {
				logger.severe("Could not access SEQUENCE value for "+DBConfig.paramName_lastUIDstored);
			}
		} catch (SQLException e) {
			logger.info("Oracle error: " + e.toString());
			throw new DatabaseException(e);
		} finally {
			try {
				ds.close(readConn);
				stmt.close();
				rs.close();
			} catch (Exception e) {
				logger.info(e.toString());
			}
		}

		return out;
	}


	//////////////////////////////////////////////////////////////////////
	//                identifier helpers //
	//////////////////////////////////////////////////////////////////////

}
