/*
 *    ALMA - Atacama Large Millimiter Array
 *    (c) European Southern Observatory, 2002
 *    Copyright by ESO (in the framework of the ALMA collaboration),
 *    All rights reserved
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License as published by the Free Software Foundation; either
 *    version 2.1 of the License, or (at your option) any later version.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
 *    MA 02111-1307  USA
 *
 *    Created on Nov 14, 2006
 *
 */

// $Author: hmeuss $
// $Date: 2010/05/03 15:08:50 $
// $Log: DatabaseConnectionPool.java,v $
// Revision 1.16.8.3  2010/05/03 15:08:50  hmeuss
// removed prefix of serviceAlias, since archiveConfig is now supposed to contain a full URL starting jdbc:...
//
// Revision 1.16.8.2  2010/04/16 11:44:53  hmeuss
// The connection string from archiveConfig is now a real JDBC connection string (and not just the service alias from tnsNames).
//
// Revision 1.16.8.1  2010/03/22 10:57:36  hsommer
// Merge from TmcdbSwConfig-2009-11-B2: handling of oracle.net.tns_admin property
//
// Revision 1.16.6.1  2010/02/16 14:11:31  hmeuss
// The oracle.net.tns_admin property is now set in ArchiveConfiguration, so that all subsystems have it available (using the same VM).
// Removed logical error in DBConfiguration.java.
//
// Revision 1.16  2009/09/17 15:31:52  hmeuss
// added possibility for olf config files to work
//
// Revision 1.15  2009/09/01 15:00:26  hmeuss
// Working with archiveConfig file now
//
// Revision 1.14  2009/08/31 13:42:02  hmeuss
// Smaller corrections for more stability
//
// Revision 1.13  2008/12/11 13:02:32  hmeuss
// Addded try-catch around connection.close during reinit, otherwise reinit fails with an exception.
//
// Revision 1.12  2008/09/11 09:29:52  hmeuss
// Added the url property to dbConfig which overwrites locationn and name properties.
//
// Revision 1.11  2008/08/14 15:41:16  hmeuss
// Added the (hopefully last) bits and pieces of the reinit functionality
//
// Revision 1.10  2008/06/27 09:45:51  hmeuss
// Removed debug print
//
// Revision 1.9  2008/06/24 15:44:37  hmeuss
// improved reinit functionality
//
// Revision 1.8  2007/07/11 14:03:22  hmeuss
// added new conncetion string (as comment only) allowing to use multiple databases and load balancing
//
// Revision 1.7  2007/03/22 13:38:39  hmeuss
// added command line tool for archiveSanityCheck
//
// Revision 1.6  2006/11/23 15:32:17  hmeuss
// added method that returns number of open connections to Oracle
//
// Revision 1.5  2006/11/23 10:16:15  hmeuss
// improved connection pool debugging, so that not always logs are emitted
//
// Revision 1.3  2006/11/16 14:49:13  hmeuss
// getConn and close are now synchronized in order to avoid concurrent access to connMap
//
// Revision 1.2  2006/11/16 09:23:08  hmeuss
// removed bug of unclosed connection, changed connection architecture, added loads of logs for connection inspection.
//
// Revision 1.1.2.2  2006/11/15 15:24:40  hmeuss
// increased max number of connections to 100. Not necessary, but just to be sure.
//
// Revision 1.1.2.1  2006/11/15 14:48:28  hmeuss
// Created new architecture, that only uses 1 connection pool. In addition openeing and closing of connections is exactly monitored.
// 
package alma.archive.database.oracle;

import java.sql.Connection;
import java.sql.SQLException;
import java.util.HashMap;
import java.util.Iterator;
import java.util.logging.Logger;

import alma.archive.database.helpers.ArchiveConfiguration;
import alma.archive.database.helpers.DBConfiguration;
import alma.archive.exceptions.general.DatabaseException;

import oracle.jdbc.pool.OracleDataSource;

/**
 * @author hmeuss
 * 
 * A wrapper around OracleDataSource serving two purposes: 1) implemented as a
 * singleton pattern 2) adding debug messages for tracing connections
 * 
 */

public class DatabaseConnectionPool extends OracleDataSource {

	static private DatabaseConnectionPool _instance = null;

	static private Logger m_logger = null;

	static private HashMap connMap = new HashMap();

	static private int counter = 0;

	/**
	 * @throws SQLException
	 */
	private DatabaseConnectionPool() throws SQLException {
		super();
	}

	static public DatabaseConnectionPool instance(Logger logger)
			throws SQLException, DatabaseException {
		if (_instance == null) {
			_instance = new DatabaseConnectionPool();
			m_logger = logger;
			_instance.reinit();
		}
		return _instance;
	}

	public synchronized void reinit() throws DatabaseException, SQLException {

		// now close all open connections:
		for (Iterator it = ((HashMap) connMap.clone()).keySet().iterator(); it
				.hasNext();) {
			Connection conn = (Connection) it.next();
			try {
				close(conn);
			} catch (Throwable e) {
				m_logger.info("Problems while closing Connection "+conn+". Ignoring...");
			}
			conn = null;
		}
		// the following line seems to be very important, otherwise a change in
		// the URL is not recognized!
		_instance.close();

		// System.out.println("!!!!!!!!!!!!!!! "+connMap.size());

		/* initialize oracle datasource: */
		// example: String url =
		// "jdbc:oracle:thin:@//almadev2.hq.eso.org:1521/test";

		//		if (url == null || url.equals("")) {
//			// if the property above is given, the other properties are not
//			// used. Otherwise the connection string is built from the
//			// properties below:
//			url = "jdbc:oracle:thin:@//"
//					+ DBConfiguration.instance(m_logger).get(
//							"archive.oracle.location")
//					+ "/"
//					+ DBConfiguration.instance(m_logger).get(
//							"archive.oracle.name");
//		}

		// The following connection string allows multiple databases to be used.
		// If the listener of the first DB is switched off, then the second DB
		// is used.
		// url="jdbc:oracle:thin:@(DESCRIPTION=(LOAD_BALANCE=on)(ADDRESS_LIST=(ADDRESS=(PROTOCOL=TCP)(HOST=almadev1)
		// (PORT=1521))(ADDRESS=(PROTOCOL=TCP)(HOST=almadev2)(PORT=1521)))(CONNECT_DATA=(SERVICE_NAME=alma1)))";
		// initializeDS(url, DBConfiguration.instance(logger).get(
		// "archive.oracle.user"), "alma$dba");

		if (DBConfiguration.instance(m_logger) instanceof ArchiveConfiguration) {
			// for the case when using tnsnames.ora
			/* the following is now done in ArchiveConfiguration.java
		String tnsDir = DBConfiguration.instance(m_logger).get("archive.db.tnsFileDirectory");
		if (null==tnsDir || tnsDir.equals("")) {
			String oraHome = System.getenv("ORACLE_HOME");
			if (oraHome!=null&&!oraHome.equals("")) {
				tnsDir=oraHome+"/network/admin";
			} else {
				throw new DatabaseException("archiveConfig.properties does not contain value for archive.db.tnsFileDirectory and $ORACLE_HOME not defined. Cannot read tnsnames.ora, aborting...");
			}
		}
		m_logger.info("Using this tnsnames.ora for DB connection: " + tnsDir);
		System.setProperty("oracle.net.tns_admin", tnsDir);
		*/
		_instance = new DatabaseConnectionPool();
		String serviceAlias = DBConfiguration.instance(m_logger).get("archive.db.connection");
		//m_logger.info("Connecting to service alias "+serviceAlias+" in tnsnames.ora");
		//_instance.setTNSEntryName(serviceAlias);
		_instance.setURL(serviceAlias);
		m_logger.info("Using connection URL: "+serviceAlias);
		_instance.setDriverType("thin");
		_instance.setUser(DBConfiguration.instance(m_logger).get(
		"archive.oracle.user"));
		_instance.setPassword("alma$dba");
		} else {
			// we are using the dbConfig mechanism:
			String url = DBConfiguration.instance(m_logger).get(
			"archive.oracle.connectionString");
	if (url == null || url.equals("")) {
		// if the property above is given, the other properties are not
		// used. Otherwise the connection string is built from the
		// properties below:
		url = "jdbc:oracle:thin:@//"
				+ DBConfiguration.instance(m_logger).get(
						"archive.oracle.location")
				+ "/"
				+ DBConfiguration.instance(m_logger).get(
						"archive.oracle.name");
	}

	m_logger.info("Using this URL for DB connection: " + url);

	_instance.setUser(DBConfiguration.instance(m_logger).get(
			"archive.oracle.user"));
	_instance.setPassword("alma$dba");
	_instance.setURL(url);

		}

		//_instance.setURL(url);

		m_logger.info("SETTING UB DB CONN...");

		
		try {
			_instance.setConnectionCachingEnabled(true);
			java.util.Properties prop = new java.util.Properties();
			prop.setProperty("MinLimit", "5"); // the cache size is 5 at
			// least
			prop.setProperty("MaxLimit", "100");
			prop.setProperty("InitialLimit", "3"); // create 3 connections
			// at startup
			prop.setProperty("InactivityTimeout", "180"); // seconds
			// prop.setProperty("AbandonedConnectionTimeout", "90"); //
			// seconds
			prop.setProperty("MaxStatementsLimit", "10");
			prop.setProperty("PropertyCheckInterval", "60"); // seconds

			_instance.setConnectionCacheProperties(prop); // set properties
		} catch (Exception e) {
			throw new DatabaseException(e);
		}

		m_logger.info("DB connection pool initialized.");
	}

	/**
	 * version for debugging: will administrate connections in a HashMap and
	 * print many debug infos
	 * 
	 * @see javax.sql.DataSource#getConnection()
	 */
	public synchronized Connection getConnection(String callInfo)
			throws SQLException {
		Connection conn = super.getConnection();
		counter++;
		connMap.put(conn, +counter + ": " + callInfo);
		m_logger.finest("Opened connection: " + counter + ": " + callInfo);

		printLogs();

		return conn;
	}

	/**
	 * version for debugging: will administrate connections in a HashMap and
	 * print many debug infos, *if* emitLogs is set to true, otherwise goes
	 * silently.
	 * 
	 * @see javax.sql.DataSource#getConnection()
	 */
	public synchronized Connection getConnection(String callInfo,
			boolean emitLogs) throws SQLException {
		Connection conn = super.getConnection();
		counter++;
		connMap.put(conn, +counter + ": " + callInfo);
		if (emitLogs) {
			m_logger.finest("Opened connection: " + counter + ": " + callInfo);
			printLogs();
		}

		return conn;
	}

	/**
	 * version for debugging: will administrate connections in a HashMap and
	 * print many debug infos
	 * 
	 * @see javax.sql.DataSource#getConnection()
	 */
	public synchronized void close(Connection conn) throws SQLException {
		conn.commit();
		conn.close();

		m_logger.finest("Closed connection: " + connMap.get(conn));
		connMap.remove(conn);

		printLogs();

	}

	/**
	 * 
	 * @return number of open connections stored in this class.
	 */
	public synchronized int getConnectionNumber() {
		return connMap.size();
	}

	/**
	 * version for debugging: will administrate connections in a HashMap and
	 * print many debug infos, *if* emitLogs is set to true, otherwise goes
	 * silently.
	 * 
	 * @see javax.sql.DataSource#getConnection()
	 */
	public synchronized void close(Connection conn, boolean emitLogs)
			throws SQLException {
		conn.commit();
		conn.close();
		connMap.remove(conn);

		if (emitLogs) {
			m_logger.finer("Closed connection: " + connMap.get(conn));
			printLogs();
		}

	}

	/**
	 * 
	 */
	private void printLogs() {
		m_logger.finest("Open Connections: (" + connMap.size() + " in total)");
		for (Iterator it = connMap.values().iterator(); it.hasNext();) {
			m_logger.finest("---" + (String) it.next());
		}
	}

	protected void setLogger(Logger log) {
		m_logger = log;
	}

}
