/*
 *    ALMA - Atacama Large Millimiter Array
 *    (c) European Southern Observatory, 2002
 *    Copyright by ESO (in the framework of the ALMA collaboration),
 *    All rights reserved
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License as published by the Free Software Foundation; either
 *    version 2.1 of the License, or (at your option) any later version.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
 *    MA 02111-1307  USA
 *
 *    Created on Jan 19, 2004
 *
 */
 
  
// $Author: hmeuss $
// $Date: 2008/12/03 15:52:59 $
// $Log: Database.java,v $
// Revision 1.9  2008/12/03 15:52:59  hmeuss
// In case of init/reinit Database.java now calls DatabaseWriter.reinit (instead of init) what is absolutely necessary in order to refresh references to Oracle connections.
//
// Revision 1.8  2008/06/24 15:44:37  hmeuss
// improved reinit functionality
//
// Revision 1.7  2005/05/30 11:10:20  hmeuss
// added cleanDB for deleting onconsistent tables
// changed behaviour of IdentiferArchive when re-initing
//
// Revision 1.6  2005/01/13 10:45:08  hmeuss
// - Constructed command line query interface for Archive.
// - Added functionality to init() for Oracle.
//
// Revision 1.5  2004/10/01 08:06:51  hmeuss
// Changed initialization of IdentifierArchive
//
// Revision 1.4  2004/09/23 11:59:14  hmeuss
// Oracle DatabaseReader now creates a new connection object in every method call.
//
// Added ModuleCriticalException to more methods of the internal IF
//
// Revision 1.3  2004/06/01 15:53:12  hmeuss
// Added treatment of virtual documents
//
// Revision 1.2  2004/05/27 09:04:00  hmeuss
// dummy adaption to new DocumentData
//
// Revision 1.1  2004/04/05 13:59:29  hmeuss
// Internal IF implementation adapted to Oracle
//
// Revision 1.11  2004/03/18 09:49:33  hmeuss
// changed configuration to work with testStart, testEnd
//
// Revision 1.10  2004/02/11 16:20:09  hmeuss
// *** empty log message ***
//
// Revision 1.9  2004/01/21 16:18:04  hmeuss
// Added class DBConfiguration that reads and parses config file
//
// Revision 1.8  2004/01/19 16:29:46  hmeuss
// Splitted functionality of Db2Database into Db2DatabaseReader, Db2DatabaseWriter, Db2DatabaseCache
// 
 
package alma.archive.database.oracle;

import java.sql.SQLException;
import java.util.HashMap;
import java.util.Iterator;
import java.util.logging.Logger;

import alma.archive.database.helpers.DBConfiguration;
import alma.archive.database.interfaces.InternalIFFactory;
import alma.archive.exceptions.ModuleCriticalException;
import alma.archive.exceptions.general.DatabaseException;

/**
 * This class administrates three other classes: DatabaseReader, DatabaseWriter, DatabaseCache, IdentifierArchive.
 * Administration consists mainly of storing config params and setting the logger. Originally, the full functionality was in this class, 
 * but in order to reduce the complexity the functionality was divided into the three mentioned classes. 
 * 
 * 
 * @author hmeuss
 *
 */
public class Database {

    /* stores all configuration parameters. */
	protected DBConfiguration m_config;

	/* for implementing the singleton pattern */
	private static Database m_instance;
	
	protected DatabaseReader dbReader;
	protected DatabaseWriter dbWriter;
	protected DatabaseCache dbCache;

	private Logger m_logger=Logger.getLogger("ARCHIVE");

	/**
	 * 
	 */
	private Database() throws DatabaseException, ModuleCriticalException {
		super();

		dbReader=DatabaseReader.instance();
		dbWriter=DatabaseWriter.instance();
		dbCache=DatabaseCache.instance();
		
		
	}

	/**
	 *  for implementing the singleton pattern *
	 */
	public static Database instance() throws DatabaseException, ModuleCriticalException {
		if (m_instance==null) {
			m_instance=new Database();
		}
		return m_instance;
	}
	
	public void setLogger(Logger logger) {
		m_logger=logger;
		dbReader.setLogger(logger);
		dbWriter.setLogger(logger);
		dbCache.setLogger(logger);
	}

	// also serves as reinit()
	protected void init() throws DatabaseException, ModuleCriticalException {
		IdentifierArchive.instance().close(); // saves state according to dbConfig as it was before calling the init
		try {
			DatabaseConnectionPool.instance(m_logger).reinit();
		} catch (SQLException e) {
			throw new DatabaseException(e);
		}
		dbReader.init();
		dbWriter.reinit();
		dbCache.init();
		IdentifierArchive.instance().init(); // creates new UID counter according to new config
	}
	
	/**
	 * 
	 * set the config params to new values as provided in the argument.
	 * propagate them to other classes
	 * 
	 * @param configParams
	 */
	public void notifyConfigChange() throws DatabaseException, ModuleCriticalException {
		m_config=InternalIFFactory.dbConfig;
		m_logger.info(m_config.toString());
		
	}

	public void close() throws DatabaseException, ModuleCriticalException {
    	dbWriter.close();
    	dbReader.close();
	IdentifierArchive.instance().close();
    }

}
