/*
 *    ALMA - Atacama Large Millimiter Array
 *    (c) European Southern Observatory, 2002
 *    Copyright by ESO (in the framework of the ALMA collaboration),
 *    All rights reserved
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License as published by the Free Software Foundation; either
 *    version 2.1 of the License, or (at your option) any later version.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
 *    MA 02111-1307  USA
 *
 *    Created on Sep 3, 2003
 *
 */

// $Author: hsommer $
// $Date: 2006/09/06 10:11:42 $
// $Log: SchemaManager.java,v $
// Revision 1.18  2006/09/06 10:11:42  hsommer
// generics
//
// Revision 1.17  2004/10/28 09:03:32  hmeuss
// - Implementation of refresh in oracle.SchemaManager
// - Added ModuleCriticalException to refresh
//
// Revision 1.16  2004/10/26 10:04:44  sfarrow
// Added refresh tot he SchemaManager
//
// Revision 1.15  2004/09/17 08:50:38  hmeuss
// Added ModuleCriticalException that will be propagated upwards through all classes
//
// Revision 1.14  2004/05/28 09:20:45  hmeuss
// Throws NamespaceDefinedException, when a namespace prefix is registered that already exists
//
// Revision 1.13  2004/04/07 12:52:33  hmeuss
// Changed some comments in the namespace section
//
// Revision 1.12  2004/03/10 15:27:11  sfarrow
// *** empty log message ***
//
// Revision 1.11  2004/03/10 12:51:49  sfarrow
// Added to comments
//
// Revision 1.10  2004/01/29 12:36:12  hmeuss
// removed user, added DAD parameter in add/update schema
//
// Revision 1.9  2004/01/28 11:20:10  sfarrow
// Added javadoc to the interface
//
// Revision 1.8  2004/01/12 11:25:18  sfarrow
// Added the list schema method as previously discussed
//
// Revision 1.7  2003/12/11 12:08:07  sfarrow
// *** empty log message ***
//
// Revision 1.6  2003/12/01 12:23:11  sfarrow
// Added new stuff for namespaces
//
// Revision 1.5  2003/10/13 13:15:54  sfarrow
// Working operational interface
//
// Revision 1.4  2003/10/07 16:20:05  sfarrow
// *** empty log message ***
//
// Revision 1.3  2003/09/25 11:30:04  sfarrow
// *** empty log message ***
//
// Revision 1.2  2003/09/19 09:53:34  sfarrow
// Alterations to the Interface
//
// Revision 1.1  2003/09/03 14:30:22  hmeuss
// *** empty log message ***
// 

package alma.archive.database.interfaces;

import java.net.URI;
import java.util.HashMap;
import java.util.List;

import alma.archive.exceptions.ArchiveException;
import alma.archive.exceptions.ModuleCriticalException;
import alma.archive.exceptions.access.PermissionDeniedException;
import alma.archive.exceptions.general.DatabaseException;
import alma.archive.exceptions.general.NamespaceDefinedException;
import alma.archive.exceptions.general.UnknownSchemaException;
import alma.archive.exceptions.syntax.MalformedURIException;
import alma.archive.exceptions.syntax.MalformedXMLException;
import alma.archive.wrappers.Permissions;

/**
 * @author simon
 * @author hmusse
 * 
 * This interface defines the access to schema and namespaces in the archive.
 * It will needs to be seperatly implemented for each type of database as they
 * use schema in different ways.
 * 
 * It is important to define some terms at this point, an example is
 * probably the simplest way. A section of the schema GeneralIncludes is given below.
 * 
 * <code>GeneralIncludes.xsd</code>
 * <pre>
 *  &lt;?xml version="1.0" encoding="UTF-8"?&gt;
 *  &lt;xsd:schema targetNamespace="Alma/GeneralIncludes"
 *	xmlns:xsd="http://www.w3.org/2001/XMLSchema"
 *	xmlns:gen="Alma/GeneralIncludes"
 *	elementFormDefault="qualified" attributeFormDefault="unqualified" version="1"&gt;
 * </pre>
 * 
 * 
 * The <b>name</b> of this schema is <b>GeneralIncludes</b>
 * It uses the <b>namespace</b> <b>Alma/GeneralIncludes</b> 
 * which is associated with the tag <b>gen</b> several namespaces
 * can be associated with a particular schema. The SchedBlock schema 
 * includes four namespaces.
 * 
 * The Schema manager can be used to associate namespaces with a particular schema,
 * this allows them to be registered for use in parsing xml documents or perfoming XPath
 * queries.
 * 
 * The usage for this example would be:
 * 
 * 1) {@link #registerNamespace Register} namespace where the name is gen and the namespace is Alma/GeneralIncludes 
 * 
 * 2) {@link #addSchema Add} the Schema(the xml), this is stored under the location schemaURInewm most likely
 * a UID with the associated owner and permissions.
 * 
 * 3) The appropriate namespaces can then be {@link assignNamespace associated} with the Schema. This is done with
 * assignNamespace. The name is the name of the namespace e.g gen and the schemaUri is the UID
 * of the schema.
 * 
 * The Schema manager will also allow a form of versioning beyond that of the Archive, spacific versions
 * of a Schema can be accessed with their own URI(UID) all of the versioning information is retained by the
 * schema manager. The full definition of this versioning functionality is <b>TBD</b>.
 * 
 * <b>NOTE:</b> A special reserved namespace URI defined inside the manager
 * this is uid://test if at any point a schema name string of "test" is
 * passed then the test namespace is returned. This allows xml docs
 * without schema to be used. This is, st present, a testing feature.
 * When the namespaces for the null schema are requestes then an empty
 * hashmap is returned.
 */

public interface SchemaManager {

	void close() throws DatabaseException;

	/**
	 * This returns the URI of the latest version of a schema given its name
	 * @param schemaName	The name of the schema e.g. GeneralIncludes
	 * @return
	 * @throws DatabaseException
	 * @throws ArchiveException
	 * @throws UnknownSchemaException
	 */
	URI getSchemaURI(String schemaName)
		throws 
	ModuleCriticalException, DatabaseException, ArchiveException, UnknownSchemaException;
	
	/**
	 * Causes the SchemaManager to rebuild itself from the database.
	 * @throws DatabaseException
	 */
	public void refresh() 
	throws ModuleCriticalException, 
		DatabaseException;

	/**
	 * This returns the URI of a <b>specific</b> version of a schema given its name
	 * @param schema	The name of the schema e.g. GeneralIncludes
	 * @param version	The version to be retrived
	 * @return
	 * @throws DatabaseException
	 * @throws ArchiveException
	 * @throws UnknownSchemaException
	 */
	URI getSchemaURI(String schema, int version)
		throws 
	ModuleCriticalException, DatabaseException, ArchiveException, UnknownSchemaException;

	
	/**
	 * Given a particular URI(UID) this will return the name of the schema
	 * @param	schemaUri	The URI(UID) of the schema
	 * @return	The name of the schema 
	 * @throws DatabaseException
	 * @throws ArchiveException
	 * @throws UnknownSchemaException
	 * @throws MalformedURIException
	 */
	String getSchemaName(URI schemaUri)
		throws
	ModuleCriticalException,
			DatabaseException,
			ArchiveException,
			UnknownSchemaException,
			MalformedURIException;

	/**
	 * Given the URI(UID) of a paricular schema this will return the version
	 * @param schemaUri	The URI(UID) of the schema
	 * @return	The version of the schema
	 * @throws DatabaseException
	 * @throws ArchiveException
	 * @throws UnknownSchemaException
	 * @throws MalformedURIException
	 */
	int getSchemaVersion(URI schemaUri)
		throws ModuleCriticalException,
			DatabaseException,
			ArchiveException,
			UnknownSchemaException,
			MalformedURIException;

// TODO the following methods implicitly assume that namespace prefixes are unique across schemas. We should either explicitly state this or change the methods...

	/**
		 * This returns a HashMap containg the prefix and namespace pairs associated with
		 * a particular namespace.
		 * @param schemaUri	The URI(UID) of the schema
		 * @return
		 * @throws DatabaseException
		 * @throws ArchiveException
		 * @throws UnknownSchemaException
		 */
		HashMap<String, String> getSchemaNamespaces(URI schemaUri)
			throws ModuleCriticalException, DatabaseException, ArchiveException, UnknownSchemaException;

	/**
	 * Used to register a particular namespace with the manager
	 * @param prefix	The namespace prefix, sometimes refered to as the tag e.g gen
	 * @param namespace The actual namespace in the form of a URI e.g Alma/GeneralIncludes
	 * @throws DatabaseException
	 * @throws ArchiveException
	 * @throws MalformedURIException
	 */
	void registerNamespace(String prefix, URI namespace)
		throws ModuleCriticalException, NamespaceDefinedException, DatabaseException, ArchiveException, MalformedURIException;

	/**
	 * Removes a namespace and prefix pair from the list of available namespaces and also from
	 * all of the asscoiations with schema. 
	 * @param name	The namespace prefix to remove
	 * @throws DatabaseException
	 * @throws ArchiveException
	 * @throws MalformedURIException
	 */
	void removeNamespace(String prefix)
		throws 
	ModuleCriticalException, DatabaseException, ArchiveException, MalformedURIException;

	/**
	 * Removes all of the namespaces and schema
	 * @deprecated This is a temporary solution
	 * @throws DatabaseException
	 * @throws ArchiveException
	 * @throws MalformedURIException
	 */
	void removeAll()
		throws 
	ModuleCriticalException, DatabaseException, ArchiveException, MalformedURIException;

	/**
	 * Verifys the existence of a particular nameaspace
	 * @param namespace	The URI of a particular namespace
	 * @return	True if the namespace exists
	 * @throws ArchiveException
	 */
	boolean namespaceExists(URI namespace) throws ModuleCriticalException, ArchiveException;

	/**
	 * Associate a namespace with a particular schema
	 * @param name	The name (tag) of the namespace
	 * @param schemaUri	The URI(UID) of the schema
	 * @throws DatabaseException
	 * @throws ArchiveException
	 * @throws UnknownSchemaException
	 */
	void assignNamespace(String namespaceName, URI schemaUri)
		throws 
	ModuleCriticalException, DatabaseException, ArchiveException, UnknownSchemaException;

	/**
	 * Remove the namespace from a particular schema
	 * @param name
	 * @param schemaUri
	 * @throws DatabaseException
	 * @throws ArchiveException
	 * @throws UnknownSchemaException
	 */
	void withdrawNamespace(String name, URI schemaUri)
		throws 
	ModuleCriticalException, DatabaseException, ArchiveException, UnknownSchemaException;

	/**
	 * Returns a hashmap containing all of the namespaces that are registered
	 * @return
	 * @throws DatabaseException
	 * @throws ArchiveException
	 */
	HashMap<String, String> namespaces() throws 
	ModuleCriticalException, DatabaseException, ArchiveException;

	/**
	 * Returns a list of all of the registered schema
	 * @return
	 * @throws DatabaseException
	 * @throws ArchiveException
	 */
	List listSchema() throws 
	ModuleCriticalException, DatabaseException, ArchiveException;

	/**
	 * Used to add the first version of a schema to the manager
	 * @param schemaName	The name of the schema
	 * @param xml	The schema itself
	 * @param indexConfig Information on the index configuration for documents belonging to that schema (DAD file for DB2). 
	 * @param schemaURInew	The URI(UID) of that the schema will be stored under
	 * @param owner	The owner of the schema
	 * @param permissions	Permissions for the schema
	 * @throws DatabaseException
	 * @throws ArchiveException
	 * @throws MalformedURIException
	 * @throws UnknownSchemaException
	 * @throws PermissionDeniedException
	 * @throws MalformedXMLException
	 */
	void addSchema(
		String schemaName,
		String xml,
		String indexConfig,
		URI schemaURInew,
		String owner,
		Permissions permissions)
		throws
	ModuleCriticalException,
			DatabaseException,
			ArchiveException,
			MalformedURIException,
			UnknownSchemaException,
			PermissionDeniedException,
			MalformedXMLException;

	/**
	 * Used to update an existing schema, must be careful with the imlplementation not to
	 * allow two new Schema to refernce one schemaURIold as this would mean that there could
	 * be two different schema with the same version number.
	 * @param schemaName	The name of the schema
	 * @param xml	The schema itself
	 * @param indexConfig Information on the index configuration for documents belonging to that schema (DAD file for DB2).
	 * @param schemaURIold	The URI(UID) of the last version of the schema
	 * @param schemaURInew	The URI(UID) that this schema will be stored under
	 * @param owner	The owner of this schema
	 * @param permissions	Permissions for the schema
	 * @throws DatabaseException
	 * @throws ArchiveException
	 * @throws MalformedURIException
	 * @throws UnknownSchemaException
	 * @throws PermissionDeniedException
	 * @throws MalformedXMLException
	 */
	void updateSchema(
		String schemaName,
		String xml,
		String indexConfig,
		URI schemaURIold,
		URI schemaURInew,
		String owner,
		Permissions permissions)
		throws
	ModuleCriticalException, 
			DatabaseException,
			ArchiveException,
			MalformedURIException,
			UnknownSchemaException,
			PermissionDeniedException,
			MalformedXMLException;

	/**
	 * Only for testing purposes and admin
	 * @param schemaName
	 * @throws DatabaseException
	 * @throws ArchiveException
	 * @throws UnknownSchemaException
	 * @throws PermissionDeniedException
	 */
	void removeSchema(String schemaName)
		throws
	ModuleCriticalException, 
			DatabaseException,
			ArchiveException,
			UnknownSchemaException,
			PermissionDeniedException;
}
