/*
 * 	  Created on 23-Sep-2003
 * 
 *    ALMA - Atacama Large Millimiter Array
 *    (c) European Southern Observatory, 2002
 *    Copyright by ESO (in the framework of the ALMA collaboration),
 *    All rights reserved
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License as published by the Free Software Foundation; either
 *    version 2.1 of the License, or (at your option) any later version.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
 *    MA 02111-1307  USA
 */
package alma.archive.database.interfaces;

import java.util.logging.Logger;

import alma.archive.database.helpers.DBConfiguration;
import alma.archive.database.xmldb.XmldbIdentifierManager;
import alma.archive.database.xmldb.XmldbInternalIF;
import alma.archive.exceptions.ModuleCriticalException;
import alma.archive.exceptions.general.DatabaseException;

/**
 * @author simon
 */
public class InternalIFFactory {
	/*
	 * 
	 * @author simon
	 *
	 * In time this will go and find an xml file 
	 * with all of the configuration information
	 * for the various databases an example is given
	 * below
	 * 
	 * <archive_configuration>
	 * 	<database type=xindice>
	 * 		<xindice>
	 * 			<!-- one of the options below -->
	 * 			<path>an absolute path to to the database</path>
	 * 			<variable>a system variable with the path in</variable>
	 * 			...
	 * 		</xindice>
	 * 		<db2>
	 * 		...
	 * 		</dbz>
	 * 	</database>
	 * 	<!-- more configuration information -->
	 * </archive_configuration>
	 * 
	 */

	/**
	 * The configuration file, set by readConfigFile()
	 * HSO 2006-08: should we not better use a getter method and keep this field private?
	 */
	public static DBConfiguration dbConfig;

	public static InternalIF getInternalIF(Logger logger)
		throws DatabaseException, ModuleCriticalException {
		// check whether we have to propagate new configuration:
		boolean notifyConfig;
		notifyConfig = ensureConfigIsLoaded(logger);
		InternalIF internal = null;

		if (dbConfig.dbBackend.equals("xmldb")) {
			internal = XmldbInternalIF.instance(logger);
		} else if (dbConfig.dbBackend.equals("oracle")) {
			alma.archive.database.oracle.InternalIfImpl.instance().setLogger(logger);
			internal = alma.archive.database.oracle.InternalIfImpl.instance();
			((alma.archive.database.oracle.InternalIfImpl) internal).setLogger(logger);
		}

		if (internal != null) {
			if (notifyConfig) {
				internal.notifyConfigChange();
			}
			return internal;
		} else {
			logger.severe(
				"Unrecognized database "
					+ dbConfig.dbBackend
					+ " specified in config file");
			throw new DatabaseException(
				"Unrecognized database "
					+ dbConfig.dbBackend
					+ " specified in config file");
		}
	}

	public static IdentifierManager getIdentifierManager(Logger logger)
		throws DatabaseException, ModuleCriticalException {
		ensureConfigIsLoaded(logger);
		IdentifierManager imanager = null;
		if (dbConfig.dbBackend.equals("xmldb")) {
			imanager = XmldbIdentifierManager.instance(logger);
		} else if (dbConfig.dbBackend.equals("oracle")) {
			alma
				.archive
				.database
				.oracle
				.IdentifierArchive
				.instance()
				.setLogger(
				logger);
			imanager =
				alma.archive.database.oracle.IdentifierArchive.instance();
		}

		if (imanager != null) {
			return imanager;
		} else {
			logger.severe(
				"Unrecognized database "
					+ dbConfig.dbBackend
					+ " specified in config file");
			throw new DatabaseException(
				"Unrecognized database "
					+ dbConfig.dbBackend
					+ " specified in config file");
		}
	}

	/**
	 * @return true if {@link #dbConfig} had to be created
	 * @throws DatabaseException
	 */
	private static synchronized boolean ensureConfigIsLoaded(Logger logger) throws DatabaseException {
		if (dbConfig == null) {
			dbConfig = DBConfiguration.instance(logger);
			return true;
		}
		else {
			return false;
		}
	}
}
