/*
 *    ALMA - Atacama Large Millimiter Array
 *    (c) European Southern Observatory, 2002
 *    Copyright by ESO (in the framework of the ALMA collaboration),
 *    All rights reserved
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License as published by the Free Software Foundation; either
 *    version 2.1 of the License, or (at your option) any later version.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
 *    MA 02111-1307  USA
 *
 *    Created on Aug 26, 2003
 *
 */

// $Author: hmeuss $
// $Date: 2009/04/22 11:26:18 $
// $Log: InternalIF.java,v $
// Revision 1.34  2009/04/22 11:26:18  hmeuss
// implemented incremental updates for XMLstore
//
// Revision 1.33  2009/02/16 16:42:42  hmeuss
// Added environment in order to store STE information in log tables.
//
// Revision 1.32  2008/11/11 20:56:47  hmeuss
// storeLog() now has an explicit commit parameter that controls when a log is committed.
//
// Revision 1.31  2007/10/24 15:06:22  hmeuss
// Added retrieval methods for logs and some more infrastructure
//
// Revision 1.30  2007/10/18 15:41:18  hmeuss
// changed to new relational storage
//
// Revision 1.29  2007/07/12 14:46:51  hmeuss
// added method updateXML
//
// Revision 1.28  2007/02/08 16:52:07  hmeuss
// Added method for querying within given time intervals for asdmQuery
//
// Revision 1.27  2006/11/30 10:35:41  hmeuss
// improved log retrieval with an additional timestamp interval
//
// Revision 1.26  2006/11/16 09:45:37  hmeuss
// In Oracle, no metadata history entries are done for log messages anymore.
//
// Revision 1.25  2006/09/21 08:54:06  hmeuss
// Implemented queryContent method, that correctly queries XPath expressions returning a non-node type (eg. ending with an attribute step)
//
// Revision 1.24  2006/09/15 14:12:36  hsommer
// new exceptions due to using the ACS-variant of Range
//
// Revision 1.23  2006/04/28 08:39:36  hmeuss
// Changed interface of queryRecent and implemented it for exist
//
// Revision 1.22  2006/04/25 09:34:12  hmeuss
// changed interface for timestamps of queryRecent
//
// Revision 1.21  2005/09/12 11:47:32  hmeuss
// Added argument to put in order to make storeLog working.
//
// Revision 1.20  2005/08/02 14:02:32  hmeuss
// added ping() method in internalIF and Administrative component
//
// Revision 1.19  2004/09/23 11:59:15  hmeuss
// Oracle DatabaseReader now creates a new connection object in every method call.
//
// Added ModuleCriticalException to more methods of the internal IF
//
// Revision 1.18  2004/09/17 08:50:38  hmeuss
// Added ModuleCriticalException that will be propagated upwards through all classes
//
// Revision 1.17  2004/09/14 09:40:55  hmeuss
// Added new init and close methods
//
// Revision 1.16  2004/08/18 14:40:08  sfarrow
// changed the way that dirty flags are handled, added getDirty to the internal if and
// removed the dirty paramaeter from the other gets
//
// Revision 1.15  2004/05/26 12:55:53  sfarrow
// Added an extra function to the internal interface
//
// Revision 1.14  2004/03/23 14:20:04  sfarrow
// Added support for forceUpdate
//
// Revision 1.13  2004/03/19 15:25:46  sfarrow
// Added methods to the internal if
//
// Revision 1.12  2004/03/17 11:33:28  sfarrow
// Added the cleanTestArea method to the internal interface
//
// Revision 1.11  2004/01/28 16:39:05  sfarrow
// in InternalIF:
// - removed store(uid, xml)
// - changed comment in update: I don't think we should allow entities to
// change schemas
// - removed get(uid)
// - removed get(uid, historyNumber, dirtyRead, user)
// - added method change owner, that could also be used for schemas. But
// only for one version of a schema.....
//
// Revision 1.10  2004/01/21 16:18:04  hmeuss
// Added class DBConfiguration that reads and parses config file
//
// Revision 1.9  2004/01/13 17:09:28  hmeuss
// added method to change configuration
//
// Revision 1.8  2003/12/01 12:23:11  sfarrow
// Added new stuff for namespaces
//
// Revision 1.7  2003/10/08 15:33:20  sfarrow
// *** empty log message ***
//
// Revision 1.6  2003/10/07 16:20:05  sfarrow
// *** empty log message ***
//
// Revision 1.5  2003/10/06 14:50:08  sfarrow
// *** empty log message ***
//
// Revision 1.4  2003/09/24 13:46:31  sfarrow
// *** empty log message ***
//
// Revision 1.3  2003/09/19 09:53:34  sfarrow
// Alterations to the Interface
//
// Revision 1.2  2003/09/04 12:00:07  hmeuss
// *** empty log message ***
//
// Revision 1.1  2003/09/03 14:30:22  hmeuss
// *** empty log message ***
//
// Revision 1.3  2003/08/29 11:25:34  hmeuss
// Java internal interface updated
//
// Revision 1.2  2003/08/29 09:02:04  hmeuss
// Java internal interface updated
//
// Revision 1.1  2003/08/28 15:42:11  hmeuss
// A first proposal for the new Java internal interface
// 
package alma.archive.database.interfaces;

import java.net.URI;
import java.util.Collection;
import java.util.HashMap;

import alma.ArchiveIdentifierError.wrappers.AcsJArchiveIdentifierErrorEx;
import alma.archive.exceptions.ArchiveException;
import alma.archive.exceptions.ModuleCriticalException;
import alma.archive.exceptions.access.EntityDirtyException;
import alma.archive.exceptions.access.PermissionDeniedException;
import alma.archive.exceptions.general.DatabaseException;
import alma.archive.exceptions.general.EntityAlreadyDeletedException;
import alma.archive.exceptions.general.EntityDoesNotExistException;
import alma.archive.exceptions.general.EntityExistsException;
import alma.archive.exceptions.general.EntityUndeletedException;
import alma.archive.exceptions.general.HistoryInconsistencyException;
import alma.archive.exceptions.general.IllegalTimestampException;
import alma.archive.exceptions.general.UndefinedNamespaceException;
import alma.archive.exceptions.general.UnknownSchemaException;
import alma.archive.exceptions.syntax.IllegalHistoryNumberException;
import alma.archive.exceptions.syntax.MalformedPermissionsException;
import alma.archive.exceptions.syntax.MalformedQueryException;
import alma.archive.exceptions.syntax.MalformedURIException;
import alma.archive.exceptions.syntax.MalformedXMLException;
import alma.archive.exceptions.syntax.MalformedXPointerException;
import alma.archive.exceptions.syntax.UnderspecifiedQueryException;
import alma.archive.exceptions.syntax.UnknownFlagException;
import alma.archive.exceptions.user.UserDoesNotExistException;
import alma.archive.wrappers.ArchiveTimeStamp;
import alma.archive.wrappers.DocumentData;
import alma.archive.wrappers.Permissions;

/**
 * Defines the generic interface allowing access to the Archive XmlStore. A
 * unified interface to deliver XmlStore functionality. The inteface itself
 * should be implemented as a singleton class as it is controling access to the
 * Archvie.
 * 
 * @author hmeuss
 * @author simon
 */
public interface InternalIF {

	/**
	 * Checks whether database backend is alive. Throws *no* exception, just
	 * returns true, if backend is alive, and false otherwise.
	 */
	public boolean ping();

	/**
	 * Free up resources.
	 * 
	 * @throws DatabaseException
	 * @throws ModuleCriticalException
	 */
	public void close() throws DatabaseException, ModuleCriticalException;

	/**
	 * Initialize XMLstore
	 */
	public void init() throws DatabaseException, ModuleCriticalException;

	/**
	 * Used to create the first instance of an entity. Sets all of the start of
	 * day information for an entity, this includes the schema, owner and
	 * permissions.
	 * 
	 * If emitLogs==false, no logs are emitted *and* for Oracle, no entry in the
	 * metadata history is made!
	 * 
	 * @param uid
	 *            URI for this entity
	 * @param xml
	 *            Content
	 * @param schema
	 *            URI that refernces the schema
	 * @param schemaName
	 *            Name or type of the schema
	 * @param owner
	 *            Initial owner of the Entity
	 * @param permissions
	 *            Initial permissions for the entity
	 * @param user
	 *            User requesting this operation
	 * @param emitLogs
	 *            specifies, whether logging is activated
	 * @throws DatabaseException
	 * @throws ArchiveException
	 * @throws MalformedURIException
	 * @throws MalformedXMLException
	 * @throws EntityExistsException
	 * @throws UnknownSchemaException
	 * @throws UserDoesNotExistException
	 * @throws MalformedPermissionsException
	 */
	public void store(URI uid, String xml, URI schema, String schemaName,
			String owner, Permissions permissions, String user, boolean emitLogs)
			throws ModuleCriticalException, DatabaseException,
			ArchiveException, MalformedURIException, MalformedXMLException,
			EntityExistsException, UnknownSchemaException,
			UserDoesNotExistException, MalformedPermissionsException;

	/**
	 * Stores logs in the DB. eXist implementation will simply do nothing. The
	 * list of parameters (attribute values) is the following:
	 * "timeStamp","file","Line","Routine","SourceObject","Host","Process","Context","Thread","LogId","Priority","Uri","Audience","Array","Antenna"
	 *
	 * The parameter steEnv identifies the STE we are working on and is stored in the log table.
	 *
	 *We only commit, if commit==true, or if a CRITICAL log is stored.
	 *
	 * @param dataStrings is additional dataStrings contained in the log message
	 * @param xml is the full message in its original XML form.
	 * @throws DatabaseException
	 */
	public void storeLog(String message, int level, String attributeValue,
			String attributeValue2, String attributeValue3,
			String attributeValue4, String attributeValue5,
			String attributeValue6, String attributeValue7,
			String attributeValue8, String attributeValue9,
			String attributeValue10, String attributeValue11,
			String attributeValue12, String attributeValue13,
			String attributeValue14, String attributeValue15,
			String[] dataStrings, String xml, boolean commit, String steEnv) throws DatabaseException;

	/**
	 * The blint instrument version of the above, for internal use only.
	 * 
	 * @param uid
	 *            URI for this entity
	 * @param xml
	 *            Content
	 * @throws DatabaseException
	 * @throws ArchiveException
	 * @throws MalformedURIException
	 * @throws MalformedXMLException
	 */
	/*
	 * public void store(URI uid, String xml) throws DatabaseException,
	 * ArchiveException, MalformedURIException, MalformedXMLException;
	 */
	/**
	 * Updates an existing entity, must have the correct time stamp or it will
	 * be rejected.
	 * 
	 * @param uid
	 *            URI for the entity
	 * @param timeStamp
	 *            {@link ArchiveTimeStamp} for the previous version
	 * @param xml
	 *            Content
	 * @param schema
	 *            Allows for a change of schema
	 * @param force
	 *            Turns of timestamp checking
	 * @param user
	 *            Checks user permissions
	 * @throws PermissionDeniedException
	 * @throws DatabaseException
	 * @throws ArchiveException
	 * @throws MalformedURIException
	 * @throws HistoryInconsistencyException
	 * @throws MalformedXMLException
	 * @throws EntityDoesNotExistException
	 * @throws UserDoesNotExistException
	 * @throws UnknownSchemaException
	 */
	public void update(URI uid, ArchiveTimeStamp timeStamp, String xml,
	// the upper layer already modified the version number in the XML string
			URI schema, boolean force, String user)
			throws ModuleCriticalException, PermissionDeniedException,
			DatabaseException, ArchiveException, MalformedURIException,
			HistoryInconsistencyException, MalformedXMLException,
			EntityDoesNotExistException, UserDoesNotExistException,
			UnknownSchemaException;

	/**
	 * updates the XML document uid, so that newChild is appended as last child
	 * directly under the root element. No history information is updated: from
	 * the outside the document keeps the same version.
	 * 
	 * @param uid
	 * @param schema
	 * @param newChild
	 * @throws DatabaseException
	 * @throws EntityDoesNotExistException
	 * @throws MalformedXMLException
	 * @throws ModuleCriticalException
	 */
	public void updateXML(URI uid, String schema, String newChild)
			throws DatabaseException, EntityDoesNotExistException,
			MalformedXMLException, ModuleCriticalException;
	
	/**
	 * Adds xmlElement as last child of the (first) element found by xPath in
	 * document uid (belonging to schema schema)
	 */
	public void addElement(URI uid, String schemaName, String xPath,
			String xmlElement, String user) throws EntityDoesNotExistException,
			DatabaseException, PermissionDeniedException,
			ModuleCriticalException;
		
	/**
	 * The (first) element pointed at by xPath in document uid (belonging to
	 * schema schema) will be replaced by element xmlElement.
	 */
	public void updateElement(URI uid, String schemaName, String xPath,
			String xmlElement, String user) throws EntityDoesNotExistException,
			DatabaseException, PermissionDeniedException,
			ModuleCriticalException;

	/**
	 * The (first) element pointed at by xPath in document uid (belonging to
	 * schema schema) will be deleted.
	 */
	public void deleteElement(URI uid, String schemaName, String xPath,
			String user) throws EntityDoesNotExistException, DatabaseException,
			PermissionDeniedException, ModuleCriticalException;
		
	/**
	 * Returns the latest and greatest version of the Entity. TODO: Sort out
	 * whether or not dirty flags are required or relevant.
	 * 
	 * @param uid
	 *            URI of the entity
	 * @param dirtyRead
	 *            Allows a read even flagged dirty, irrelevant at present
	 * @param user
	 *            Checks user permissions
	 * @return Returns the XML content of the entity
	 * @throws PermissionDeniedException
	 * @throws EntityDoesNotExistException
	 * @throws DatabaseException
	 * @throws ArchiveException
	 * @throws UserDoesNotExistException
	 * @throws MalformedURIException
	 */
	public String get(URI uid, String user) throws ModuleCriticalException,
			PermissionDeniedException, EntityDirtyException,
			EntityDoesNotExistException, DatabaseException, ArchiveException,
			UserDoesNotExistException, MalformedURIException;

	/**
	 * Retrieves a specific subsection of an entity. The subsection is defined
	 * by an XPath query. TODO: It may be possible to combine this with the
	 * above method
	 * 
	 * @param uid
	 *            URI of the entity
	 * @param xpath
	 *            XPath query on the document
	 * @param namespaces
	 *            Namespaces required for the query
	 * @param dirtyRead
	 *            Allows a read even flagged dirty, irrelevant at present
	 * @param user
	 *            Checks user permissions
	 * @return Returns the subsection of the entity defined by the XPath
	 * @throws PermissionDeniedException
	 * @throws EntityDoesNotExistException
	 * @throws DatabaseException
	 * @throws ArchiveException
	 * @throws MalformedURIException
	 * @throws IllegalHistoryNumberException
	 * @throws UserDoesNotExistException
	 * @throws MalformedXPointerException
	 * @throws UndefinedNamespaceException
	 */
	public String[] get(URI uid, String xpath, HashMap namespaces, String user)
			throws ModuleCriticalException, PermissionDeniedException,
			EntityDirtyException, EntityDoesNotExistException,
			DatabaseException, ArchiveException, MalformedURIException,
			IllegalHistoryNumberException, UserDoesNotExistException,
			MalformedXPointerException, UndefinedNamespaceException;

	/**
	 * Rollback functionality, database image at a given point in time. This
	 * ignores all of the the deleted and dirty flags.
	 * 
	 * @param uid
	 *            URI of the entity
	 * @param timestamp
	 *            Time marking the valid document
	 * @param user
	 *            Check user permissions
	 * @return Returns the version valid at the point in time given
	 * @throws PermissionDeniedException
	 * @throws EntityDoesNotExistException
	 * @throws DatabaseException
	 * @throws ArchiveException
	 * @throws MalformedURIException
	 * @throws UserDoesNotExistException
	 * @throws IllegalTimestampException
	 */
	public String get(URI uid, ArchiveTimeStamp timestamp, String user)
			throws ModuleCriticalException, PermissionDeniedException,
			EntityDirtyException, EntityDoesNotExistException,
			DatabaseException, ArchiveException, MalformedURIException,
			UserDoesNotExistException, IllegalTimestampException;

	/**
	 * Returns the latest and greatest version of the Entity. TODO: Sort out
	 * whether or not dirty flags are required or relevant.
	 * 
	 * @param uid
	 *            URI of the entity
	 * @param dirtyRead
	 *            Allows a read even flagged dirty, irrelevant at present
	 * @param user
	 *            Checks user permissions
	 * @return Returns the XML content of the entity
	 * @throws PermissionDeniedException
	 * @throws EntityDoesNotExistException
	 * @throws DatabaseException
	 * @throws ArchiveException
	 * @throws UserDoesNotExistException
	 * @throws MalformedURIException
	 */
	public String getDirty(URI uid, String user)
			throws ModuleCriticalException, PermissionDeniedException,
			EntityDoesNotExistException, DatabaseException, ArchiveException,
			UserDoesNotExistException, MalformedURIException;

	/**
	 * Sets the delete flag on the specified entity.
	 * 
	 * @param uid
	 *            URI of the entity
	 * @param user
	 *            Check user permissions
	 * @throws PermissionDeniedException
	 * @throws EntityDoesNotExistException
	 * @throws DatabaseException
	 * @throws ArchiveException
	 * @throws MalformedURIException
	 * @throws UserDoesNotExistException
	 * @throws EntityAlreadyDeletedException
	 */
	public void delete(URI uid, String user) throws ModuleCriticalException,
			PermissionDeniedException, EntityDoesNotExistException,
			DatabaseException, ArchiveException, MalformedURIException,
			UserDoesNotExistException, EntityAlreadyDeletedException;

	/**
	 * Unsets the delete flag on the specified entity
	 * 
	 * @param uid
	 *            URI of the entity
	 * @param user
	 *            Check user permissions
	 * @throws PermissionDeniedException
	 * @throws DatabaseException
	 * @throws ArchiveException
	 * @throws MalformedURIException
	 * @throws EntityDoesNotExistException
	 * @throws UserDoesNotExistException
	 * @throws EntityUndeletedException
	 */
	public void undelete(URI uid, String user)
			throws PermissionDeniedException, DatabaseException,
			ArchiveException, MalformedURIException,
			EntityDoesNotExistException, UserDoesNotExistException,
			EntityUndeletedException;

	/**
	 * Sets the dirty flag on the specified entity.
	 * 
	 * @param uid
	 *            URI of the entity
	 * @param user
	 *            Check user permissions
	 * @throws PermissionDeniedException
	 * @throws EntityDoesNotExistException
	 * @throws DatabaseException
	 * @throws ArchiveException
	 * @throws MalformedURIException
	 * @throws UserDoesNotExistException
	 * @throws EntityAlreadyDeletedException
	 */
	public void dirty(URI uid, String user) throws ModuleCriticalException,
			PermissionDeniedException, EntityDoesNotExistException,
			DatabaseException, ArchiveException, MalformedURIException,
			UserDoesNotExistException;

	/**
	 * Unsets the dirty flag on the specified entity
	 * 
	 * @param uid
	 *            URI of the entity
	 * @param user
	 *            Check user permissions
	 * @throws PermissionDeniedException
	 * @throws DatabaseException
	 * @throws ArchiveException
	 * @throws MalformedURIException
	 * @throws EntityDoesNotExistException
	 * @throws UserDoesNotExistException
	 * @throws EntityUndeletedException
	 */
	public void clean(URI uid, String user) throws ModuleCriticalException,
			PermissionDeniedException, DatabaseException, ArchiveException,
			MalformedURIException, EntityDoesNotExistException,
			UserDoesNotExistException, EntityUndeletedException;

	/**
	 * Sets the hidden flag on the specified entity.
	 * 
	 * @param uid
	 *            URI of the entity
	 * @param user
	 *            Check user permissions
	 * @throws PermissionDeniedException
	 * @throws EntityDoesNotExistException
	 * @throws DatabaseException
	 * @throws ArchiveException
	 * @throws MalformedURIException
	 * @throws UserDoesNotExistException
	 * @throws EntityAlreadyDeletedException
	 */
	public void hidden(URI uid, String user) throws PermissionDeniedException,
			EntityDoesNotExistException, DatabaseException, ArchiveException,
			MalformedURIException, UserDoesNotExistException;

	/**
	 * Unsets the hidden flag on the specified entity
	 * 
	 * @param uid
	 *            URI of the entity
	 * @param user
	 *            Check user permissions
	 * @throws PermissionDeniedException
	 * @throws DatabaseException
	 * @throws ArchiveException
	 * @throws MalformedURIException
	 * @throws EntityDoesNotExistException
	 * @throws UserDoesNotExistException
	 * @throws EntityUndeletedException
	 */
	public void visible(URI uid, String user) throws ModuleCriticalException,
			PermissionDeniedException, DatabaseException, ArchiveException,
			MalformedURIException, EntityDoesNotExistException,
			UserDoesNotExistException, EntityUndeletedException;

	/**
	 * Allows the virtual flag to be set on a particular document
	 * 
	 * @param uid
	 *            URI of the entity
	 * @param user
	 *            Check user permissions
	 * @param virtual
	 *            The value for the virtual flag
	 * @throws PermissionDeniedException
	 * @throws DatabaseException
	 * @throws ArchiveException
	 * @throws MalformedURIException
	 * @throws EntityDoesNotExistException
	 * @throws UserDoesNotExistException
	 */
	public void setVirtual(URI uid, String user, boolean virtual)
			throws ModuleCriticalException, PermissionDeniedException,
			DatabaseException, ArchiveException, MalformedURIException,
			EntityDoesNotExistException, UserDoesNotExistException;

	/**
	 * Retrives {@link DocumentData} for the latest version of the entity
	 * 
	 * @param uid
	 *            URI of the entity
	 * @param user
	 *            Check user permissions
	 * @return Returns Latest {@link DocumentData}
	 * @throws PermissionDeniedException
	 * @throws EntityDoesNotExistException
	 * @throws DatabaseException
	 * @throws ArchiveException
	 * @throws UserDoesNotExistException
	 * @throws IllegalHistoryNumberException
	 * @throws MalformedURIException
	 */
	DocumentData status(URI uid, String user) throws ModuleCriticalException,
			PermissionDeniedException, EntityDoesNotExistException,
			DatabaseException, ArchiveException, UserDoesNotExistException,
			IllegalHistoryNumberException, MalformedURIException;

	/**
	 * The full history of the entity, oldest first.
	 * 
	 * @param uid
	 *            URI of the entity
	 * @param user
	 *            Check user permissions
	 * @return An array of {@link DocumentData} the first in the array is the
	 *         oldest, last is newest
	 * @throws PermissionDeniedException
	 * @throws EntityDoesNotExistException
	 * @throws DatabaseException
	 * @throws ArchiveException
	 * @throws UserDoesNotExistException
	 * @throws MalformedURIException
	 */
	DocumentData[] allStatus(URI uid, String user)
			throws ModuleCriticalException, PermissionDeniedException,
			EntityDoesNotExistException, DatabaseException, ArchiveException,
			UserDoesNotExistException, MalformedURIException;

	// returns document data that was valid at a given point of time.
	DocumentData status(URI uid, ArchiveTimeStamp timestamp, String user)
			throws ModuleCriticalException, PermissionDeniedException,
			EntityDoesNotExistException, DatabaseException, ArchiveException,
			MalformedURIException, UserDoesNotExistException,
			IllegalTimestampException;

	// queries heads of history in between timeFrom and timeTo.
	URI[] queryInterval(ArchiveTimeStamp timeFrom, ArchiveTimeStamp timeTo,
			String schema, String XPathQuery, String user)
			throws DatabaseException, ArchiveException, ModuleCriticalException;

	// queries only heads of history, later than timestamp
	URI[] queryRecent(ArchiveTimeStamp timestamp, String schema,
	// not a URI, because we are independent of schema versions!
			String user) throws ModuleCriticalException,
			PermissionDeniedException, DatabaseException, ArchiveException,
			MalformedQueryException, UnderspecifiedQueryException,
			UnknownSchemaException, UserDoesNotExistException,
			UndefinedNamespaceException;

	// queries only heads of history. Works not for content queries in Oracle,
	// use
	// queryContent in this case.
	DBCursor query(String query, String schema,
	// not a URI, because we are independent of schema versions!
			HashMap namespaces, boolean dirtyRead,
			// are dirty documents integrated into result or not
			String user) throws ModuleCriticalException,
			PermissionDeniedException, DatabaseException, ArchiveException,
			MalformedQueryException, UnderspecifiedQueryException,
			UnknownSchemaException, UserDoesNotExistException,
			UndefinedNamespaceException;

	// same as query, but works for content queries. Proper behaviour for
	// non-content queries is not guaranteed.
	DBCursor queryContent(String query, String schema,
	// not a URI, because we are independent of schema versions!
			HashMap namespaces, boolean dirtyRead,
			// are dirty documents integrated into result or not
			String user) throws ModuleCriticalException,
			PermissionDeniedException, DatabaseException, ArchiveException,
			MalformedQueryException, UnderspecifiedQueryException,
			UnknownSchemaException, UserDoesNotExistException,
			UndefinedNamespaceException;

	/**
	 * timeFrom and timeTo restrict the time interval of returned logs.
	 * type restrict the (numerical) types of logs. 
	 * routine, source and process can be set to * in order to denote don't care. 
	 * maxRow is the maximal number of rows returned in the collection (if there are more logs, they are ignored).
	 * Logs are returned in timestamp order, starting with the lowest timestamp.
	 * @param timeFrom
	 * @param timeTo
	 * @param minType
	 * @param maxType
	 * @param routine
	 * @param source
	 * @param process
	 * @param maxRow
	 * @return
	 * @throws ArchiveException
	 */
	public Collection<String> getLog(String timeFrom, String timeTo,
			short minType, short maxType, String routine, String source,
			String process, int maxRow) throws ArchiveException, ModuleCriticalException;

	// same as query, but returns the metadata only:
	URI[] queryIDs(String query, String schema, HashMap namespaces,
			boolean dirtyRead,
			// are dirty documents integrated into result or not
			String user) throws ModuleCriticalException,
			PermissionDeniedException, DatabaseException, ArchiveException,
			MalformedQueryException, UnderspecifiedQueryException,
			UnknownSchemaException, UserDoesNotExistException,
			UndefinedNamespaceException;

	// same as query, but returns the metadata only, returns all data regardless
	// of flags
	URI[] queryIDsAll(String query, String schema, HashMap namespaces,
			String user) throws ModuleCriticalException,
			PermissionDeniedException, DatabaseException, ArchiveException,
			MalformedQueryException, UnderspecifiedQueryException,
			UnknownSchemaException, UserDoesNotExistException,
			UndefinedNamespaceException;

	// -------------------------------------------------------------------------

	// -------------------------------------------------------------------------
	// Admin flags (special flags for entities, like the dirty flag):
	void setFlag(URI uid, String flagName, boolean flagValue, String user)
			throws ModuleCriticalException, PermissionDeniedException,
			DatabaseException, ArchiveException, MalformedURIException,
			EntityDoesNotExistException, UserDoesNotExistException,
			UnknownFlagException;

	// values of admin flags are retruned by status

	// Allows the owner of an entity to be changed
	void changeOwner(URI uid, String newOwner, String user)
			throws ModuleCriticalException, PermissionDeniedException,
			DatabaseException, ArchiveException, MalformedURIException,
			EntityDoesNotExistException, UserDoesNotExistException;

	// -------------------------------------------------------------------------

	// -------------------------------------------------------------------------
	// Schema administration:
	SchemaManager getSchemaManager(String user) throws ModuleCriticalException,
			ArchiveException, PermissionDeniedException,
			UserDoesNotExistException;

	// -------------------------------------------------------------------------

	// -------------------------------------------------------------------------
	// User administration:
	UserManager getUserManager(String user) throws ModuleCriticalException,
			ArchiveException, PermissionDeniedException,
			UserDoesNotExistException;

	// -------------------------------------------------------------------------

	// -------------------------------------------------------------------------
	// Privileged methods:

	// physically remove entities and documents. If keepHead=false, the full
	// entity is removed with all its metadata. If keepHead=true, the head of
	// history is kept.
	// remove makes restore partially invalid!
	// -------------------------------------------------------------------------

	void remove(URI uid, boolean keepHead, String user)
			throws ModuleCriticalException, PermissionDeniedException,
			DatabaseException, ArchiveException, MalformedURIException,
			UserDoesNotExistException, EntityDoesNotExistException;

	void setPermission(URI uid, Permissions permissions, String user)
			throws ModuleCriticalException, PermissionDeniedException,
			DatabaseException, ArchiveException, MalformedURIException,
			UserDoesNotExistException, MalformedPermissionsException;

	// restore the archive state of a given time
	void restoreState(ArchiveTimeStamp timestamp, String user)
			throws ModuleCriticalException, PermissionDeniedException,
			DatabaseException, ArchiveException, UserDoesNotExistException,
			IllegalTimestampException;

	/**
	 * Cleans the test area after a test is completed. This will totally erase
	 * the section of the archive that was defined inside the test boundaries.
	 * 
	 * @throws PermissionDeniedException
	 * @throws DatabaseException
	 * @throws ArchiveException
	 * @throws UserDoesNotExistException
	 */
	void cleanTestArea(String user) throws ModuleCriticalException,
			PermissionDeniedException, DatabaseException, ArchiveException,
			UserDoesNotExistException, AcsJArchiveIdentifierErrorEx;

	/**
	 * 
	 * set database configuration. Configuration is stored in XML document
	 * InternalIFFactory.dbConfig
	 * 
	 * @param dbConfig
	 */
	public void notifyConfigChange() throws ModuleCriticalException,
			DatabaseException;

};
