/**
 * Copyright European Southern Observatory 2010
 */

package alma.archive.database.helpers.wrappers;

import java.util.logging.Logger;

import alma.archive.database.helpers.ArchiveConfigurationOld;
import alma.archive.database.helpers.DBConfiguration;

/**
 * An abstract helper class for loading archiveConfig.properties
 *
 * @author rkurowsk, April 30, 2010
 * @version $Revision: 1.1.2.2 $
 */

// $Id: AbstractDbConfig.java,v 1.1.2.2 2010/05/03 09:36:09 rkurowsk Exp $

public abstract class AbstractDbConfig {
	
	public static final String HSQLDB_DRIVER 	= "org.hsqldb.jdbcDriver";
	public static final String ORACLE_DRIVER 	= "oracle.jdbc.driver.OracleDriver";
    
    private static final String JDBC_HSQLDB 	= "jdbc:hsqldb";
    private static final String HSQLDB_DIALECT 	= "org.hibernate.dialect.HSQLDialect";

    private static final String JDBC_ORACLE 	= "jdbc:oracle";
    private static final String ORACLE_DIALECT 	= "org.hibernate.dialect.Oracle10gDialect";

    private Logger logger;
    private DBConfiguration dbConfiguration;
    
    private String connectionUrl;
    private String user;
    private String password;
    private String driver;
    private String dialect;
    
    /** 
     * Constructor which attempts to load dbConfiguration and 
     * hence the archiveConfig.properties file. 
     * 
     * @param logger
     * @throws DbConfigException
     * @throws IllegalArgumentException
     */
    public AbstractDbConfig(Logger logger ) throws DbConfigException {
        
    	if( logger == null ) {
            throw new IllegalArgumentException( "Null logger passed to AbstractDbConfig" );
        }
        
        this.logger = logger;
        
        try {
        	
        	dbConfiguration = DBConfiguration.instance( logger );
        	
        	// ArchiveConfigurationOld means that archiveConfig.properties wasn't found
        	if(dbConfiguration instanceof ArchiveConfigurationOld){
        		
        		String archiveConfigFileSysProp = System.getProperty("archive.configFile");
        		String acsDataSysProp = System.getProperty("ACS.data");
        		throw new DbConfigException("Could not find archiveConfig.properties. " +
        				"\nJava System property 'archive.configFile' = " + archiveConfigFileSysProp + 
        				"\nJava System property 'ACS.data' = " + acsDataSysProp + 
        				"\nEither point to archiveConfig.properties directly " +
        				"with the Java system property: 'archive.configFile' " +
        				"\nor define the Java system property 'ACS.data' and " +
        				"put archiveConfig.properties in ACS.data/config.");
        	}
        	
        	extractParams();
        	
        }catch( Exception e ) {
            throw new DbConfigException( e.getMessage(), e );
        }

    }
    
    /**
     * Extracts the required parameters from archiveConfig.properties using 
     * the DBConfiguration class. Implementations of this method should call:
     * extractParam and configureConnectionUrl.
     * 
     * @throws DbConfigException
     */
    protected abstract void extractParams() throws DbConfigException;

    /**
     * Exctracts the connection url and sets the driver & dialect
     * 
     * @param connectionParamName
     * @throws DbConfigException
     */
    protected final void configureConnectionUrl(String connUrl) throws DbConfigException {
    	
    	setConnectionUrl(connUrl);
    	
    	// work out DB driver and dialect (only oracle and hsqldb are currently supported)
    	if(getConnectionUrl().contains(JDBC_ORACLE)){
        	setDriver(ORACLE_DRIVER);
        	setDialect(ORACLE_DIALECT);
        }else if( getConnectionUrl().contains(JDBC_HSQLDB)){
        	setDriver(HSQLDB_DRIVER);
        	setDialect(HSQLDB_DIALECT);
        }else{
        	String msg = "Unsupported db url: " + connUrl;
        	logger.severe(msg);
        	throw new RuntimeException(msg);
        }
    }
    
    /**
     * This method extracts the given parameter from the dbConfiguration.
     * It throws a DbConfigException if the parmeter is null.
     * 
     * @param paramName
     * @return
     * @throws DbConfigException
     */
    protected final String extractParam(String paramName) throws DbConfigException {
    	
    	String paramValue = dbConfiguration.get(paramName);
    	if(paramValue == null){
    		throw new DbConfigException( "Error in archiveConfig.properties: "  +  
    				paramName + " is not defined" ); 
    	}
        logger.finer( paramName + " = " + paramValue );
        
        return paramValue;

    }
    
    public String getPassword() {
        return password;
    }

    public String getUsername() {
       return user;
    }

    public String getDialect() {
    	return dialect;
    }

    public String getConnectionUrl() {
        return connectionUrl;
    }

    public String getDriver(){
    	return driver;
    }

	public Logger getLogger() {
		return logger;
	}
    
	protected void setConnectionUrl(String connectionUrl) {
		this.connectionUrl = connectionUrl;
	}

	protected void setUser(String user) {
		this.user = user;
	}

	protected void setPassword(String password) {
		this.password = password;
	}

	protected void setDriver(String driver) {
		this.driver = driver;
	}

	protected void setDialect(String dialect) {
		this.dialect = dialect;
	}

}