/*
 * 	  Created on 16-Nov-2004
 * 
 *    ALMA - Atacama Large Millimiter Array
 *    (c) European Southern Observatory, 2002
 *    Copyright by ESO (in the framework of the ALMA collaboration),
 *    All rights reserved
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License as published by the Free Software Foundation; either
 *    version 2.1 of the License, or (at your option) any later version.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
 *    MA 02111-1307  USA
 */
package alma.archive.database.helpers;

import java.io.IOException;
import java.io.Reader;
import java.io.StringReader;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.Iterator;
import java.util.List;
import java.util.logging.Logger;

import org.jdom.Document;
import org.jdom.Element;
import org.jdom.JDOMException;
import org.jdom.input.SAXBuilder;
import org.jdom.output.Format;
import org.jdom.output.XMLOutputter;
import org.jdom.xpath.XPath;

import alma.archive.database.interfaces.InternalIF;
import alma.archive.database.interfaces.InternalIFFactory;
import alma.archive.exceptions.ArchiveException;
import alma.archive.exceptions.ModuleCriticalException;
import alma.archive.exceptions.general.EntityDoesNotExistException;

/**
 * @author simon
 * 
 * This class is used to build an overview of a project from a link to a part 
 * of that project. The XPath used to find links is //*[@entityId]. It's
 * impossible to find the parent documents though so the whole hierarchy cannot
 * be discovered.
 */
public class Hierarchy
{
	private URI start = null;
	private String user = null;
	private InternalIF internal = null;
	
	public Hierarchy(URI uid, String user)
		throws 
			ArchiveException, 
			ModuleCriticalException
	{
		this.start = uid;
		this.user = user;
		
		// TODO This must be changed: try to get the ACS logger and use that for getting the internalIF
		internal = InternalIFFactory.getInternalIF(Logger.getAnonymousLogger());
	}
	
	public Reader generate() throws ModuleCriticalException, ArchiveException
	{
		Element entityGroup = new Element("EntityGroup");
		Element child = children(start,"");
		entityGroup.addContent(child);
		printElement(entityGroup);
		
		XMLOutputter out = new XMLOutputter();
		StringReader sr = new StringReader(out.outputString(entityGroup));
		
		return sr;
	}
	
	/**
	 * Returns null if there are no children
	 * @param uid
	 * @return
	 * @throws ArchiveException
	 * @throws ModuleCriticalException
	 */
	private Element children(URI uid,String entityTypeName) 
		throws 
			ModuleCriticalException, 
			ArchiveException
	{
		try
		{
			String xml = internal.get(uid,user);
			
			Element entity = new Element("Entity");
			entity.setAttribute("entityId",uid.toASCIIString());
			entity.setAttribute("exist","True");
			entity.setAttribute("entityTypeName",entityTypeName);
			
			SAXBuilder builder = new SAXBuilder();
			builder.setIgnoringElementContentWhitespace(true);
			try
			{
				Document doc = builder.build(new StringReader(xml));
				Element root = doc.getRootElement();
				
				String _path = "//*[@entityId]";
				XPath path = XPath.newInstance(_path);
				
				List children = path.selectNodes(root);
				Iterator iter = children.iterator();
				while (iter.hasNext())
				{
					Element child = (Element)iter.next();
					URI childUri = 
						new URI((String)child.getAttributeValue("entityId"));
					String childEntityTypeName = 
						child.getAttributeValue("entityTypeName");
					if (!childUri.equals(uid))
					{
						Element childEntity = 
							children(childUri,childEntityTypeName);
						entity.addContent(childEntity);
					}
					else
					{
						entity.setAttribute(
							"entityTypeName",childEntityTypeName);
					}
				}
				return entity;
			}
			catch (JDOMException e)
			{
				throw new ArchiveException("Problems parsing the document",e);
			}
			catch (IOException e)
			{
				throw new ArchiveException("Problems parsing the document",e);
			}
			catch (URISyntaxException e)
			{
				throw new ArchiveException("Problems parsing the document",e);
			}
		}
		catch (EntityDoesNotExistException e)
		{
			Element empty = new Element("Entity");
			empty.setAttribute("entityId",uid.toASCIIString());
			empty.setAttribute("exist","False");
			empty.setAttribute("entityTypeName",entityTypeName);
			return empty;
		}
	}
	
	private void printElement(Element e)
	{

		XMLOutputter out = new XMLOutputter(Format.getPrettyFormat());
		// XMLOutputter out = new XMLOutputter("  ",true,"UTF-8");
		System.out.println(out.outputString(e));
	}
	
	public static void main(String[] args) 
		throws 
			URISyntaxException, 
			ArchiveException, 
			ModuleCriticalException
	{
		URI uid = new URI("uid://X0000000000000089/X00000000");
		Hierarchy h = new Hierarchy(uid,"user"); 
		h.generate();
	}
}
