/*
 *    ALMA - Atacama Large Millimiter Array
 *    (c) European Southern Observatory, 2002
 *    Copyright by ESO (in the framework of the ALMA collaboration),
 *    All rights reserved
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License as published by the Free Software Foundation; either
 *    version 2.1 of the License, or (at your option) any later version.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
 *    MA 02111-1307  USA
 *
 *    Created on Jan 21, 2004
 *
 */

// $Author: hmeuss $
// $Date: 2010/03/30 15:26:12 $
// $Log: DBConfiguration.java,v $
// Revision 1.31.8.3  2010/03/30 15:26:12  hmeuss
// corrected bug
//
// Revision 1.31.8.2  2010/03/30 14:56:49  hmeuss
// added method getConnectionURL
//
// Revision 1.31.8.1  2010/03/22 10:58:29  hsommer
// Merge from TmcdbSwConfig-2009-11-B2: "else" fix
//
// Revision 1.31.6.1  2010/02/16 14:13:01  hmeuss
// The oracle.net.tns_admin property is now set in ArchiveConfiguration, so that all subsystems have it available (using the same VM).
// Removed logical error in DBConfiguration.java.
//
// Revision 1.31  2009/08/26 16:00:49  hmeuss
// Changed to new dbConfig mode
//
// Revision 1.30  2008/10/08 07:48:58  hmeuss
// Corrected misleading error message.
//
// Revision 1.29  2008/06/24 15:44:37  hmeuss
// improved reinit functionality
//
// Revision 1.28  2008/05/09 13:14:27  hmeuss
// Added fileLocation as public visible variable.
//
// Revision 1.27  2008/05/09 11:25:06  hmeuss
// Added fileLocation as public visible variable.
//
// Revision 1.26  2008/04/09 12:25:55  hmeuss
// made log output more specific
//
// Revision 1.25  2006/10/26 15:25:34  hmeuss
// only one config file is read, instead of many which were combined before
//
// Revision 1.24  2006/09/06 10:07:58  hsommer
// generics, improved singleton accessor
//
// Revision 1.23  2005/10/27 12:24:41  hmeuss
// added reinit method, yet to be tested.
//
// Revision 1.22  2005/07/21 12:29:48  hmeuss
// Changed design of test area
//
// Revision 1.21  2005/05/13 09:33:46  hmeuss
// replaced deprecated XMLOutputter constructor by new one.
//
// Revision 1.20  2005/04/27 15:11:06  hmeuss
// Now, properties starting with "archive." are considered
//
// Revision 1.19  2004/07/20 14:46:58  hmeuss
// corrected some errors
//
// Revision 1.18  2004/07/20 14:16:14  hmeuss
// Corrected some inconsistencies with property names.
//
// Revision 1.17  2004/07/15 17:21:30  sfarrow
// Added new functionality to the query form and configuration
//
// Revision 1.16  2004/07/15 13:06:29  hmeuss
// Added new location for config file: $ACSDATA/config
//
// Revision 1.15  2004/06/30 08:12:07  hmeuss
// *** empty log message ***
//
// Revision 1.14  2004/06/29 14:55:10  hmeuss
// Only properties starting with archive.db. are considered
//
// Revision 1.13  2004/06/29 14:18:48  hmeuss
// Changed config file from XML to Java properties
//
// Revision 1.12  2004/05/27 11:07:10  hmeuss
// Added oracleLocation to config file and DBConfiguration.
// Made DBConfiguration a singleton class.
//
// Revision 1.11  2004/05/07 12:54:43  hmeuss
// do not throw exception when ACS.tmp is not set as expected.
//
// Revision 1.10  2004/05/05 08:45:55  hmeuss
// *** empty log message ***
//
// Revision 1.9  2004/03/19 09:10:30  hmeuss
// changed stupid representation of ID counter from int to long
//
// Revision 1.8  2004/03/18 09:49:33  hmeuss
// changed configuration to work with testStart, testEnd
//
// Revision 1.7  2004/02/27 13:42:23  hmeuss
// *** empty log message ***
//
// Revision 1.6  2004/02/27 13:04:07  hmeuss
// *** empty log message ***
//
// Revision 1.5  2004/02/27 12:40:12  hmeuss
// *** empty log message ***
//
// Revision 1.4  2004/02/27 10:25:20  hmeuss
// now using ACS.tmp for locating $ACSDATA/config
//
// Revision 1.3  2004/02/17 14:37:32  hmeuss
// Added some schema treatment
//
// Revision 1.2  2004/02/11 16:20:22  hmeuss
// *** empty log message ***
//
// Revision 1.1  2004/01/21 16:18:04  hmeuss
// Added class DBConfiguration that reads and parses config file
// 

package alma.archive.database.helpers;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Properties;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.jdom.Element;
import org.jdom.output.Format;
import org.jdom.output.XMLOutputter;

import alma.archive.exceptions.general.DatabaseException;

/**
 * 
 * A class holding all configuration parameters handed over by the config file. Parameters are 
 * set when InternalIFFactory is called for the first time, or when a new configuration is handed over by ArchiveAdministration.
 * 
 * Other classes should use the method get or the public variable configParams to access the properties. Some of them are also 
 * stored in dedicated variables for convenience (eg. testMode).
 * 
 * The parameters are taken from Java system properties. These are taken from (in this order): 
 * 1) The file dbConfig.properties located in the classpath (a fallback is provided in archive_database.jar)
 * 2) The file dbConfig.properties.jar located in $ACSDATA/config
 * 2) The file dbConfig.properties located in the current working directory
 * 3) Java command line properties passed with -DpropName=propValue
 * 
 * Command line properties overwrite properties from the properties file in the current directory, which themselves 
 * overwrite properties taken from the properties file in $ACSDATA/config and the classpath, resp.
 * 
 * <b>Only</b> properties with prefix archive. are considered, all others are ignored!!!
 *  
 * 
 * The idea and some code are taken from ObsPreps PropertyHandler 
 * 
 * The properties used are the following (others may also be used):
 *  MANDATORY:
 * - archive.db.backend
 * OPTIONAL: 
 * - archive.db.testStart
 * - archive.db.testEnd
 * - archive.db.mode
 * - archive.db.visibility
 * - archive.db.idStart
 * BACKEND SPECIFIC (SEMI-OPTIONAL)
 * - archive.db.oracleLocation
 * - archive.db.xindiceLocation
 * 
 * Other properties can be added and used without changing the code. These properties must 
 * start with the prefix alma.archive., otherwise they won't be recognized.
 * 
 * @author hmeuss
 *
 */
public abstract class DBConfiguration {

	private static DBConfiguration instance;

	/** 
	 * default name of the config file
	 */
	public static String defaultConfigFileName = "archiveConfig.properties";

	/* default values will be overwritten if a correspinding parameter is set in the config file */

	/* database backend used, eg. "db2" or "xindice" */
	public String dbBackend;
	/* running in test mode? */
	public boolean testMode = false;
	/* all parameters from the config file in a HashMap. In addition, all 
	 * Java properties are also stored here.   */
	public HashMap<String, String> configParams = new HashMap<String, String>();
	/**
	 * Actual location where the dbConfig file was found.
	 */
	public String fileLocation;
	
	/**
	 * re-reads information from config file
	 */
	public abstract void reinit(Logger logger) throws DatabaseException;
	
	/**
	 * Singleton accessor
	 */
	public static synchronized DBConfiguration instance(Logger logger)
		throws DatabaseException {
		
		// TODO: check whether we use the new ArchiveConfiguration implementation class, or the old ArchiveConfigurationOld class.
		// This is determined by the existence of the archiveConfig file.
		
		if (instance == null) {

			// if archive.configFile is defined, we use the new class:
			if (System.getProperty("archive.configFile")!=null&&!System.getProperty("archive.configFile").equals("")) {
				instance = new ArchiveConfiguration(logger);
			} else
			
			// then check, whether archiveConfig.properties exists in $ACSDATA:
			if (System.getProperty("ACS.data")!=null&&new File(System.getProperty("ACS.data")+"/config/"+defaultConfigFileName).exists()) {
				instance = new ArchiveConfiguration(logger);
			}
			
			// else, we take the old one:
			if (instance == null) {
				instance = new ArchiveConfigurationOld(logger);
			}
		} 
		return instance;
	}

	/* returns string representation of the configuration, i.e. the parameter name/value pairs */
	public abstract String toString();
	
	/*
	 * <dbconfiguration>
	 * 	<config name="" value="">
	 * 	...
	 * </dbconfiguration>
	 */
	public abstract Element toElement();
	
	public abstract String toXmlString();
	
	/* returns value of parameter, if defined. Otherwise returns null */
	public abstract String get(String paramName);


	// Oracle case: returns a JDBC URL based on the service alias specified in property name
	public abstract String getConnectionURL(String propertyName);

	
}
