package alma.archive.database.helpers;
import java.util.logging.Level;
import java.util.logging.Logger;

import alma.archive.exceptions.general.DatabaseException;

import com.cosylab.cdb.jdal.hibernate.plugin.HibernateWDALConfigurationPlugin;
import com.cosylab.cdb.jdal.hibernate.DBUtil;

/**
 * Plugin implementation for HibernateDAL.
 * It provides ARCHIVE configuration mechanism to HibernateDAL.
 * To enable it set "cdb_rdb.plugins.configuration" JVM property when starting HibernateDAL.
 * @author msekoranja
 */
public class ArchiveHibernateWDALConfigurationPlugin implements
		HibernateWDALConfigurationPlugin {

	private DBConfiguration archiveConfig = null;
	
	/* (non-Javadoc)
	 * @see com.cosylab.cdb.jdal.hibernate.plugin.HibernateWDALConfigurationPlugin#initialize(java.util.logging.Logger)
	 */
	public void initialize(Logger logger) {
		try {
			archiveConfig = ArchiveConfiguration.instance(logger);
		} catch (DatabaseException de) {
			logger.log(Level.FINE, "Exception from ArchiveConfiguration ", de);
			throw new RuntimeException("Failed to obtain ArchiveConfiguration instance.", de);
		}
	}

	/* (non-Javadoc)
	 * @see com.cosylab.cdb.jdal.hibernate.plugin.HibernateWDALConfigurationPlugin#getName()
	 */
	public String getName() {
		return "ArchiveConfiguration plugin";
	}

	/* (non-Javadoc)
	 * @see com.cosylab.cdb.jdal.hibernate.plugin.HibernateWDALConfigurationPlugin#get(java.lang.String, java.lang.String)
	 */
	public String get(String name, String defaultValue) {
		String retVal = archiveConfig.get(name);
		if (retVal != null)
			return retVal;
		else
			return defaultValue;
	}

	/**
	 * The Alma Archive classes provide {@link DBConfiguration#dbBackend}
	 * which gets parsed out from the connection string given in property <code>archive.db.connection</code>
	 * (see {@link alma.archive.database.helpers.ArchiveConfiguration#createConfig}).
	 * <p>
	 * However the Archive (XML etc parts of archive, other than TMCDB) only supports "xmldb" and "oracle".
	 * Therefore the TMCDB must use a different mechanism if it wants to support "hsqldb" or other databases as well.
	 * In this method we encapsulate these differences, and allow the TMCDB to fully support HSQLDB.
	 * <p>
	 * Implementation: We parse <code>archive.tmcdb.connection</code> (see {@link #getURL()}) for "hsqldb"
	 * and for a match return the String constant {@link DBUtil#HSQLDB_BACKEND_NAME}.
	 * Otherwise we return {@link DBConfiguration#dbBackend}.
	 * 
	 * @see com.cosylab.cdb.jdal.hibernate.plugin.HibernateWDALConfigurationPlugin#getBackend()
	 */
	public String getBackend() {
		String dbConnection = getURL();
		if (dbConnection.indexOf("hsqldb") > 0) {
			return DBUtil.HSQLDB_BACKEND_NAME;
		}
		else {
			return archiveConfig.dbBackend;
		}
	}

	/* (non-Javadoc)
	 * @see com.cosylab.cdb.jdal.hibernate.plugin.HibernateWDALConfigurationPlugin#getURL()
	 */
	public String getURL() {
		return archiveConfig.get("archive.tmcdb.connection");
	}

	/* (non-Javadoc)
	 * @see com.cosylab.cdb.jdal.hibernate.plugin.HibernateWDALConfigurationPlugin#getUserName()
	 */
	public String getUserName() {
		return archiveConfig.get("archive.tmcdb.user");
	}

	/* (non-Javadoc)
	 * @see com.cosylab.cdb.jdal.hibernate.plugin.HibernateWDALConfigurationPlugin#getPassword()
	 */
	public String getPassword() {
		return archiveConfig.get("archive.tmcdb.passwd");
	}
}
