/*
 *    ALMA - Atacama Large Millimiter Array
 *    (c) European Southern Observatory, 2002
 *    Copyright by ESO (in the framework of the ALMA collaboration),
 *    All rights reserved
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License as published by the Free Software Foundation; either
 *    version 2.1 of the License, or (at your option) any later version.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
 *    MA 02111-1307  USA
 *
 *    Created on Feb 6, 2007
 *
 */

// $Author: hmeuss $
// $Date: 2010/01/05 09:36:48 $
// $Log: ASDMQueryClient.java,v $
// Revision 1.3.20.1  2010/01/05 09:36:48  hmeuss
// improved error message for non-existing schemas
//
// Revision 1.4  2009/12/16 17:05:15  hmeuss
// Improved error message for unknown schemas
//
// Revision 1.3  2008/05/28 13:25:01  hmeuss
// Changed log output of CLIs
//
// Revision 1.2  2008/02/22 12:20:49  hmeuss
// changed ACS_LOGSTDOUT value from 31 to 99
//
// Revision 1.1  2007/02/08 16:52:07  hmeuss
// Added method for querying within given time intervals for asdmQuery
// 
package alma.archive.client;

import java.io.FileWriter;
import java.io.IOException;
import java.net.URI;
import java.util.logging.Logger;
import alma.acs.logging.ClientLogManager;
import alma.archive.database.interfaces.InternalIF;
import alma.archive.database.interfaces.InternalIFFactory;
import alma.archive.exceptions.ArchiveException;
import alma.archive.exceptions.ModuleCriticalException;
import alma.archive.exceptions.access.EntityDirtyException;
import alma.archive.exceptions.access.PermissionDeniedException;
import alma.archive.exceptions.general.DatabaseException;
import alma.archive.exceptions.general.EntityDoesNotExistException;
import alma.archive.exceptions.general.UnknownSchemaException;
import alma.archive.exceptions.syntax.MalformedURIException;
import alma.archive.exceptions.user.UserDoesNotExistException;
import alma.archive.wrappers.ArchiveTimeStamp;

/**
 * @author hmeuss
 * 
 * used for querying the ASDM in various ways Usage: see method printHelp
 * 
 */
public class ASDMQueryClient {

	static Logger m_logger;

	static String schema, attributeName, attributeValue;

	static ArchiveTimeStamp timeFrom, timeTo;

	static FileWriter out;

	static int parsePosition = 0; // actual position when reading command line

	// args

	/**
	 * @param args
	 * @throws IOException
	 * @throws ModuleCriticalException
	 * @throws DatabaseException
	 */
	public static void main(String[] args) throws IOException,
			DatabaseException, ModuleCriticalException, ArchiveException {
		m_logger = ClientLogManager.getAcsLogManager().getLoggerForApplication(
				"ASDMQueryClient", false);

		// parse input
		if (args.length < 1) {
			inputError("Schema name must be specified.");
			return;
		}

		if (args[parsePosition].equals("-i")) {
			printHelp();
			return;
		}

		schema = args[parsePosition];
		parsePosition++;

		if (args.length > 2 && args[1].equals("-w")) { // we write to a file:
			parsePosition++;
			if (args.length < 3) {
				inputError("Switch -w must be followed by filename.");
				return;
			}
			out = new FileWriter(args[parsePosition]);
			System.out.println("Writing matching XML files contents to file "
					+ args[parsePosition]);
			parsePosition++;
		}

		if (args.length <= parsePosition) {
			// input string finished, submit query
			query();
			return;
		}

		// continue reading input args

		try {
			timeFrom = new ArchiveTimeStamp(args[parsePosition]);
			parsePosition++;
		} catch (Exception e) {
			timeFrom = null;
			if (args[parsePosition].equals("0")) {
				parsePosition++;
				// query from start of history:
				try {
					timeFrom = new ArchiveTimeStamp("2000-01-01T12:00:00.000");
				} catch (DatabaseException e1) {
					// TODO Auto-generated catch block
					e1.printStackTrace();
				}
			} else {
				// arg is not a timestamp
			}

		}

		if (args.length <= parsePosition) {
			// input string finished, submit query
			query();
			return;
		}

		// continue reading input args

		if (timeFrom != null) {
			// try whether we also have timeTo
			try {
				timeTo = new ArchiveTimeStamp(args[parsePosition]);
				parsePosition++;
			} catch (Exception e) {
				// query until end of history:
				timeTo = new ArchiveTimeStamp();
			}
		}

		if (args.length <= parsePosition) {
			// input string finished, submit query
			query();
			return;
		}

		// continue reading input args

		attributeName = args[parsePosition];
		parsePosition++;

		if (args.length <= parsePosition) {
			// input string finished, submit query
			query();
			return;
		}

		// continue reading input args until no more are left, all are added to
		// attributeValue
		attributeValue = "";
		for (int i = parsePosition; i < args.length - 1; i++) {
			attributeValue = attributeValue + args[i] + " ";
		}
		attributeValue = attributeValue + args[args.length - 1];

		// forward query to InternalIF
		query();
		return;

	}

	private static void query() throws DatabaseException,
			ModuleCriticalException, IOException, ArchiveException {

		// defaults:
		if (timeFrom == null)
			timeFrom = new ArchiveTimeStamp("2000-01-01T12:00:00.000");
		if (timeTo == null)
			timeTo = new ArchiveTimeStamp();

		System.out.println("Querying observation data for schema " + schema
				+ " from " + timeFrom.toISOString() + " to "
				+ timeTo.toISOString() + ".");
		if (attributeName != null)
			System.out.println("Querying for attribute "
					+ attributeName
					+ ((attributeValue == null) ? "" : " with value "
							+ attributeValue) + ".");

		String query;
		if (attributeName == null) {
			query = "/*";
		} else if (attributeName.startsWith("@")) {
			// deal with a real attribute
			query = "//*[attribute::"
					+ attributeName.substring(1)
					+ ((attributeValue == null) ? ""
							: ("='" + attributeValue + "'")) + "]";
		} else {
			// attributeName is an element name:
			query = "//" + attributeName;
			if (attributeValue != null) {
				// content specified
				query = query + "[contains(., \"" + attributeValue + "\")]";
			}
		}

		InternalIF internal = InternalIFFactory.getInternalIF(m_logger);

		try {
			internal.getSchemaManager("asdmQuery").getSchemaURI(schema);
		} catch (UnknownSchemaException e) {
			System.out.println("Schema "+schema+" is unknown. Please check your query or the database connection used.");
			return;
		} 
		
		m_logger.info("Constructed XPath query: " + query);

		URI[] uids = internal.queryInterval(timeFrom, timeTo, schema, query, "asdmQueryClient");

		for (int i = 0; i < uids.length; i++) {
			if (attributeName != null) {
				System.out.print(attributeName);
				if (attributeValue != null) {
					System.out.print("='" + attributeValue + "'");
				}
				System.out.print(": ");
			}
			System.out.println(uids[i]);

			if (out != null) {
				// write result to files, one could also write them to single
				// files with UID as filename
				try {
					out.write(internal.get(uids[i], "asdmQueryClient"));
				} catch (Exception e) {
					System.out.println("ERROR: Could not retrieve matching "
							+ uids[i] + " from XMLstore!");
				}
			}
		}

		System.out.println("Found "+uids.length+" matching results.");
		
		if (out != null) {
			out.close();
		}

		//		
		// if (out!=null) System.out.println("Filename: "+out.toString());
		// System.out.println("Schema: "+schema);
		// if (timeFrom!=null) System.out.println("TimeFrom:
		// "+timeFrom.toISOString());
		// if (timeTo!=null) System.out.println("TimeTo:
		// "+timeTo.toISOString());
		// System.out.println("AttributeName: "+attributeName);
		// System.out.println("AttributeValue: "+attributeValue);

	}

	public static void inputError(String message) {
		System.out.println("ERROR: " + message);
		System.out.println("");
		printHelp();
	}

	public static void printHelp() {
		System.out
				.println("Usage: asdmQuery schemaName (-w filename)? timeFrom? timeTo? (name value?)?");
		System.out
				.println("Retrieves documents belonging to a given schema (schemaName) in a given time interval (timeFrom and timeTo, if either is not specified, beginning or end of history is used).");
		System.out
				.println("Retrieved results can be restricted to ones having a given attribute or element (name) with a value or content (value). If value is specified, ");
		System.out
				.println("name must also be specified. If name is prefixed with @, then it specifies an attribute, otherwise an element.");
		System.out
				.println("Documents returned will be the ones who have the specified attribute or element at any place in the document structure, with the specified value contained.");
		System.out
				.println("Values may contain blanks, but values containing symbols like ( or ) must be enclosed in \". Values must not contain \" or '.");
		System.out
				.println("If timeTo is given, timeFrom must also be specified. If you want to query all data until a given time, you can use 0 for timeFrom.");
		System.out
				.println("Timestamps have to be specified in ISO format with milliseconds, eg.: 2006-12-31T23:59:59.000, and refer to database *storage* time, not creation time.");
		System.out
				.println("Names have to be given with namespace prefix, eg. prj, if necessary. If specifying an attribute the @ symbol must precede the namespace prefix.");
		System.out
				.println("If -w is specified, the full contents (not just the matching parts) of all matching XML documents will be written to the specified filename.");
		System.out
				.println("Logging can be drastically reduced by setting ACS_LOG_STDOUT to 99");

		System.out.println("");
		
		System.out
				.println("EXAMPLE: Query for all Holography (truncated, test) projects until 2006:");
		System.out
				.println("   asdmQuery ObsProject 0 2006-12-31T23:59:59.000 prj:projectName Holography Testing - truncated");
		System.out
				.println("EXAMPLE: Query for projects using km/s as velocity unit:");
		System.out.println("   asdmQuery ObsProject @unit km/s");
		System.out
				.println("EXAMPLE: Query for all projects starting with 2006 referencing a given part ID and concatenate the full XML files in file testfile.txt:");
		System.out
				.println("   asdmQuery ObsProject -w testfile.txt 2006-01-01T00:00:00.000 @prj:entityPartId 1164141831796/29263776 1606933650");
		System.out.println("IMPORTANT NOTE: asdmQuery will always use dbConfig.properties files in your working directory or in $ACSDATA/config");
	}

}
