/*
 *    ALMA - Atacama Large Millimiter Array
 *    (c) European Southern Observatory, 2002
 *    Copyright by ESO (in the framework of the ALMA collaboration),
 *    All rights reserved
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License as published by the Free Software Foundation; either
 *    version 2.1 of the License, or (at your option) any later version.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
 *    MA 02111-1307  USA
 *
 *    Created on Apr 21, 2008
 *
 */

// $Author: hmeuss $
// $Date: 2010/05/11 15:57:41 $
// $Log: APDMQuery.java,v $
// Revision 1.9.26.2  2010/05/11 15:57:41  hmeuss
// COMP-4432 and COMP-4364: fix.
//
// Revision 1.9.26.1  2010/03/31 12:39:02  hmeuss
// merged fixes from HEAD to branch
//
// Revision 1.12  2010/03/31 12:35:09  hmeuss
// removed bug
//
// Revision 1.11  2010/02/26 10:39:57  awicenec
// Corrected XPath query to retrieve ASDM refereneces. [AW]
//
// Revision 1.10  2010/02/09 13:28:42  hmeuss
// Improved according to JIRA tickets  COMP-3993 and COMP-3996: option handling, illegal characters
//
// Revision 1.9  2008/07/10 12:07:53  hmeuss
// Improved help text according to Debras comments.
//
// Revision 1.8  2008/06/20 14:46:43  hmeuss
// Improved help page.
//
// Revision 1.7  2008/06/20 14:40:55  hmeuss
// Corrected bug for
// projectQuery ObsProject=
// that found no results (wrong XPath query)
//
// Revision 1.6  2008/05/23 16:23:01  hmeuss
// improved command line parameter parsing
//
// Revision 1.5  2008/05/23 14:17:05  hmeuss
// Corrected small bug in help page.
//
// Revision 1.4  2008/05/14 14:32:01  hmeuss
// Single apostrophes (') in XPath expressions are now escaped before handing over to SQL.
//
// Revision 1.3  2008/05/14 14:17:55  hmeuss
// All sources are printed now instead of only the first one.
//
// Revision 1.2  2008/05/09 13:14:45  hmeuss
// Improved CLI
//
// Revision 1.1  2008/05/02 13:59:38  hmeuss
// Added CLI projectQuery
// 
package alma.archive.client;

import java.io.FileNotFoundException;
import java.io.FileWriter;
import java.io.IOException;
import java.io.PrintStream;
import java.io.StringReader;
import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.StringTokenizer;
import java.util.Map.Entry;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.CommandLineParser;
import org.apache.commons.cli.GnuParser;
import org.apache.commons.cli.Options;
import org.apache.commons.cli.ParseException;
import org.apache.commons.cli.PosixParser;
import org.jdom.Element;
import org.jdom.JDOMException;
import org.jdom.Namespace;
import org.jdom.filter.ElementFilter;
import org.jdom.filter.Filter;
import org.jdom.input.SAXBuilder;
import org.jdom.output.Format;
import org.jdom.output.XMLOutputter;

import oracle.xdb.XMLType;

import alma.archive.database.helpers.DBConfiguration;
import alma.archive.database.oracle.DBConfig;
import alma.archive.database.oracle.Database;
import alma.archive.database.oracle.DatabaseConnectionPool;
import alma.archive.database.oracle.DatabaseReader;
import alma.archive.database.oracle.DatabaseWriter;
import alma.archive.database.oracle.InternalIfImpl;
import alma.archive.exceptions.ModuleCriticalException;
import alma.archive.exceptions.general.DatabaseException;

/**
 * 
 * Evaluates queries to ObsProjects stored in Oracle
 * 
 * command line arguments as follows: projectQuery [-i|-h|-e] [-w filename]
 * [[param=[=]]?value]*
 * 
 * -i: Info about DB backend -h: help -e: detailed examples -w: write output to
 * filename param: parameter name value: desired value, could be enclosed in
 * some kind of quotes. If two = are given the match is exact, otherwise
 * containment.
 * 
 * More than one param/values pairs are evaluated conjunctive. Values without
 * parameter names are evaluated as unbound textual searches (containment). If a
 * parameter is not contained in the list of parameters, it is tried to match as
 * element name literally.
 * 
 * About special parameters: RaDec: no idea! RestFreq: The interval is specified
 * by '-', ie.: RestFreq=345-350. Unit is GHz.
 * 
 * TODO: values containing blanks!
 * 
 * @author hmeuss
 * 
 */
public class APDMQuery {
	static// some DB connection stuff:
	DatabaseConnectionPool ocpds;

	static Connection readConn;

	private static PrintStream writeFile = null;

	private static DBConfiguration dbConfig = null;

	// basic XPath query:
	static String basicXpath = "/prj:ObsProject";

	// namespaces:
	static String namespaces = "xmlns:prj=\"Alma/ObsPrep/ObsProject\"";

	// table name of obsProjects:
	static String obsProjTab = "xml_obsproject_entities";

	static String execblockTab = "xml_execblocktable_entities";

	// The allowed parameters are mapped to XPath subqueries in the following
	// way with two arrays:
	// The parameter names (all in lower case):
	static final List<String> paramsList = Arrays.asList("projname", "pi",
			"projcode", "obstype", "sourcename", "linespecies", "pol",
			"restfreq");

	// and how we reach the relevant parameters with an XPath expression:
	static final List<String> xPathList = Arrays.asList("./prj:projectName",
			"./prj:pI", "./prj:code",
			".//prj:DataProcessingParameters/attribute::projectType",
			".//prj:sourceName", ".//prj:transitionName",
			".//prj:polarisation", ".//restFrequency");

	// comments to the above:
	// - not sure about //code equals projcode, was using mail by Alan Bridger,
	// 23.4.08: Yes, it is the ObsProject "code" element.
	// - <DataProcessingParameters almatype="APDM::DataProcessingParameters"
	// projectType="Continuum"> specifies the obstype (Alan Bridger)
	// Polarisation has values 1,2 or 4. Found in SchedBlock??? Meaning?

	// static String projNameQuery = "/prj:projectName";
	// static String piQuery = "/prj:pI";
	// static String projCodeQuery = "/prj:";
	// static String obsTypeQuery = "/prj:";
	// static String sourceNameQuery = "/prj:";
	// static String LineSpeciesQuery = "/prj:";
	// static String PolQuery = "/prj:";

	private static Logger m_logger = Logger.getAnonymousLogger();

	static private SAXBuilder builder = new SAXBuilder();

	static String xmlFile = null; // if filename for XML output is specified.

	// If no filename is specified, but XML
	// output is desired, filenames follow the
	// UID structure.

	static PrintStream xmlStream = null;

	static List<String> xmlFileList = new ArrayList<String>();

	/**
	 * 
	 */
	public APDMQuery() {
		// TODO Auto-generated constructor stub
	}

	/**
	 * @param args
	 * @throws ParseException
	 * @throws DatabaseException
	 * @throws SQLException
	 * @throws IOException
	 * @throws JDOMException
	 * @throws ModuleCriticalException
	 */
	public static void main(String[] args) throws ParseException {
		// Logger logger = Logger.global;

		m_logger.setLevel(Level.OFF);

		// InternalIfImpl.instance().setLogger(logger);
		// DatabaseWriter.instance().setLogger(logger);
		// Database.instance().setLogger(logger);

		boolean detailedOut = false; // should screem output contain detailed
		// information? (File output always
		// does.)
		boolean xmlOut = false; // XML output?
		boolean caseIns = false; // case insensitive output (values only!)
		List<String> elementSearch = new ArrayList<String>(); // if the
		// parameter
		// name did not
		// match an
		// entry in the
		// parameter
		// list, we
		// perform a
		// literal
		// search on the
		// element names
		// directly.
		// This variable collects all xpath expressions, so that we can later
		// perform an unconstrained (by values) element search, in order to
		// infor the user
		// if the element doesn't exist at all.

		// possible options: [-h] [-e] [-c] [-w <filename>] [-d] [-x|-xf
		// <filename>] [-i]
		Options options = new Options();
		options.addOption("h", false, "display help");
		options.addOption("e", false, "display examples");
		options.addOption("c", false, "display DB configuration");
		options.addOption("w", true, "write output to file");
		options.addOption("d", false, "detailed output");
		options.addOption("x", false, "write XML files named after UID");
		options.addOption("xf", true,
				"write XML file with speicified file name");
		options.addOption("i", false, "case-insensitive value matching");

		CommandLineParser parser = new GnuParser();
		//try {
		CommandLine cmd = parser.parse(options, args);
//		} catch (UnrecognizedOptionException e) {
//			System.out.println("ERROR: unrecognized option");
//		}

		if (args.length == 0 || cmd.hasOption("h")) {
			displayHelp();
			return;
		}
		if (cmd.hasOption("e")) {
			displayExamples();
			return;
		}

		if (cmd.hasOption("x")&&cmd.hasOption("xf")) {
			System.out.println("ERROR: Options x and xf must not be used at the same time.");
			return;
		}

		// read dbConfig
		try {
			dbConfig = readDbConfig();
		} catch (DatabaseException e) {
			System.out
					.println("Problem in reading database configuration file:");
			System.out.println(e.toString());
			System.out.print("Problem in reading database configuration file.");
			if (dbConfig != null && dbConfig.fileLocation != null) {
				System.out.println("Check " + dbConfig.fileLocation);
			} else {
				System.out
						.println("Check ./dbConfig.properties or $ACSDATA/config/dbConfig.properties.");
			}
			return;
		}
		if (cmd.hasOption("c")) {
			displayConfig();
			return;
		}

		if (cmd.hasOption("w")) {
			try {
				writeFile = new PrintStream(cmd.getOptionValue("w"));
			} catch (FileNotFoundException e) {
				System.out.println("Problem in creating file:");
				System.out.println(e.toString());
				System.out.println("Problem in creating file "
						+ cmd.getOptionValue("w"));
				return;
			}
		}

		if (cmd.hasOption("x") || cmd.hasOption("xf")) {
			// XML ouptut
			xmlOut = true;
		}
		if (cmd.hasOption("xf")) {
			// filename specified
			xmlFile = cmd.getOptionValue("xf");
			xmlFileList.add(xmlFile);
			try {
				xmlStream = new PrintStream(xmlFile);
			} catch (FileNotFoundException e) {
				System.out.println("Problem in creating file:");
				System.out.println(e.toString());
				System.out.println("Problem in creating file " + xmlFile);
				return;
			}
		}

		if (cmd.hasOption('d')) {
			// detailed ouptut
			detailedOut = true;
		}

		if (cmd.hasOption('i')) {
			// case insensitive (values only!)
			caseIns = true;
		}

		args = cmd.getArgs();
		int argCount = 0; // where to continue reading the command line args
		HashMap<String, String> parameterMap = new HashMap<String, String>(); // stores
		// param/value
		// pairs
		HashSet<String> genericSearches = new HashSet<String>(); // stores
		// generic

		// query constructed:
		String xpath = basicXpath;

		String queryInfo = "Queried for "; // output, what kind of query is
		// submitted

		// read parameter/value pairs:
		for (int i = argCount; i < args.length; i++) {
			String parVal = args[i];
			int sep = parVal.indexOf('=');
			// single value without parameter
			if (sep < 0) {
				genericSearches.add(parVal);
			} else {
				// test whether the parameter name contains special symbols, throw exception then:
				if (!parVal.substring(0, sep).matches("\\w+")) {
					System.out.println("ERROR: parameter name contains special symbols: "+parVal.substring(0,sep));
					return;
				}
				parameterMap.put(parVal.substring(0, sep).toLowerCase(), parVal
						.substring(sep + 1));
			}
		}
		// read parameter/value pairs and
		// construct XPath query:
		for (int i = argCount; i < args.length; i++) {
			String parVal = args[i];
			int sep = parVal.indexOf('=');
			// the following two strings are used for case-insensitive matches:
			String preCaseInsString = caseIns ? "translate(" : "";
			String postCaseInsString = caseIns ? ",\"ABCDEFGHIJKLMNOPQRSTUVWXYZ\",\"abcdefghijklmnopqrstuvwxyz\")"
					: "";

			String xpathExpr = null;
			if (sep < 0) {
				// single value without parameter
				if (caseIns) {
					parVal = parVal.toLowerCase();
				}
				xpathExpr = "[contains(" + preCaseInsString + "."
						+ postCaseInsString + ", \"" + parVal + "\")]";
				queryInfo += parVal + " contained in doc, ";
				if (caseIns) {
					System.out
							.println("ERROR: Case-insensitive full-text search not supported in the moment, due to bug in Oracle.");
					// The following happens:
					// SQL> SELECT value(xmlres) xml, archive_uid FROM
					// xml_obsproject_entities, table(xmlsequence(extract(xml,
					// '/prj:ObsProject[contains(translate(string(.),"ABCDEFGHIJKLMNOPQRSTUVWXYZ","abcdefghijklmnopqrstuvwxyz"),
					// "maur")]','xmlns:prj="Alma/ObsPrep/ObsProject'))) xmlres
					// WHERE deleted=0 AND hidden=0 AND dirty=0;
					// ERROR:
					// ORA-03113: end-of-file on communication channel
					return;
				}
			} else {
				// parameter/value pair separated by =:
				String param = parVal.substring(0, sep).toLowerCase();
				String value = parVal.substring(sep + 1);
				if (caseIns) {
					value = value.toLowerCase();
				}
				// find the xpath determined by parameter:
				try {
					xpathExpr = xPathList.get(paramsList.indexOf(param));
					if (value.charAt(0) == '=') {
						// exact match, not containment:
						xpathExpr = "[" + preCaseInsString + xpathExpr
								+ postCaseInsString + " =\""
								+ value.substring(1) + "\"]";
						queryInfo += "pre-defined parameter " + param
								+ " with exact content " + value.substring(1)
								+ ", ";
					} else {
						// containment match:
						xpathExpr = "[" + xpathExpr + "[contains("
								+ preCaseInsString + "." + postCaseInsString
								+ ", \"" + value + "\")]]";
						queryInfo += "pre-defined parameter " + param
								+ " containing " + value + ", ";
					}
				} catch (Exception e) {
					// we have only one parameter containing the =
					if (value.equals("")) {
						// no value specified, we just search for all docs
						// containing the parameter:
						try {
							xpathExpr = xPathList
									.get(paramsList.indexOf(param));
							xpathExpr = "[" + xpathExpr + "]";
							queryInfo += "pre-defined parameter " + param
									+ " occurring in doc, ";
						} catch (Exception e1) {
							// Parameter not in parameter list. Use original
							// parameter in element search:
							if (parVal.substring(0, sep).equals("ObsProject")) {
								// special case: just query for ObsProjects, then we cannot use the standard XPath construction
								xpathExpr = "";
							} else {
								xpathExpr = "[.//prj:"
										+ parVal.substring(0, sep) + "]";
							}
							queryInfo += "XML element "
									+ parVal.substring(0, sep)
									+ " occurring in doc, ";
							elementSearch.add(parVal.substring(0, sep));
						}
					} else {
						// nothing to worry about: The parameter is not in the
						// parameter
						// list. Instead of param, we use the original (not
						// lowercased) parameter.
						// Now we try to use it as a normal element name
						if (value.charAt(0) == '=') {
							// exact match, not containment:
							xpathExpr = "[" + preCaseInsString + ".//prj:"
									+ parVal.substring(0, sep)
									+ postCaseInsString + "=\""
									+ value.substring(1) + "\"]";
							queryInfo += "XML element "
									+ parVal.substring(0, sep)
									+ " with exact content "
									+ value.substring(1) + ", ";
							elementSearch.add(parVal.substring(0, sep));
						} else {
							// containment match:
							xpathExpr = "[" + ".//prj:"
									+ parVal.substring(0, sep) + "[contains("
									+ preCaseInsString + "."
									+ postCaseInsString + ", \"" + value
									+ "\")]]";
							queryInfo += "XML element "
									+ parVal.substring(0, sep) + " containing "
									+ value + ", ";
							elementSearch.add(parVal.substring(0, sep));
						}
					}
				}
			}
			xpath = xpath + xpathExpr;
		}

		// replace ' with '':
		xpath = xpath.replaceAll("'", "''");

		// submit query
		String sql;// = "select
		// extract(obsEnts.xml,'"+xpath+"','xmlns:prj=\"Alma/ObsPrep/ObsProject\"')
		// from xml_obsproject_entities obsEnts;";

		sql = "SELECT value(xmlres) xml, archive_uid FROM "
				+ obsProjTab
				+ ", table(xmlsequence(extract(xml, '"
				+ xpath
				+ "','xmlns:prj=\"Alma/ObsPrep/ObsProject\"'))) xmlres WHERE deleted=0 AND hidden=0 AND dirty=0";

		// System.out.println("submitting query: " + sql);

		Namespace ns = Namespace.getNamespace("prj", "Alma/ObsPrep/ObsProject");

		try {
			ocpds = DatabaseConnectionPool.instance(m_logger);
		} catch (Exception e) {
			System.out.println("Problem in creating database connection: ");
			System.out.println(e.toString());
			System.out
					.println("Problem in creating database connection. Check Oracle system and "
							+ dbConfig.fileLocation + ".");
			return;
		}
		try {
			readConn = ocpds.getConnection();
		} catch (Exception e) {
			System.out.println("Problem in creating database connection: ");
			System.out.println(e.toString());
			System.out
					.println("Problem in creating database connection. Check Oracle system and "
							+ dbConfig.fileLocation + ".");
			return;
		}
		Statement stmt = null;
		try {
			stmt = readConn.createStatement();
		} catch (Exception e) {
			System.out.println("Problem in creating database connection: ");
			System.out.println(e.toString());
			System.out
					.println("Problem in creating database connection. Check Oracle system and "
							+ dbConfig.fileLocation + ".");
			try {
				stmt.close();
				ocpds.close(readConn);
			} catch (Exception e1) {
				// nothing to do here
			}
			return;
		}
		ResultSet curs = null;
		try {
			curs = stmt.executeQuery(sql);
		} catch (Exception e) {
			System.out.println("Problem in executing database query: ");
			System.out.println(e.toString());
			System.out.println("Executed query: " + sql);
			System.out.println("Problem in executing database query.");
			try {
				curs.close();
				stmt.close();
				ocpds.close(readConn);
			} catch (Exception e1) {
				// nothing to do here
			}
			return;
		}
		int i = 0;
		System.out.println();
		try {
			while (curs.next()) {
				org.jdom.Document doc = builder.build(new StringReader(
						((XMLType) curs.getObject(1)).getStringVal()));
				Element root = doc.getRootElement();
				// System.out.println("+++"+root.getName());
				String projName = root.getChildTextNormalize("projectName", ns);
				String pi = root.getChildTextNormalize("pI", ns);
				String version = root.getChildTextNormalize("version", ns);
				String uid = curs.getString(2);
				// get associated ASDMS:
				List<String> asdms = getASDMs(uid);

				// in case of detailed and/or file output, we need more infos:
				String projCode = root.getChildTextNormalize("code");
				if (projCode == null) {
					projCode = "";
				}
				Iterator<Element> it = root.getDescendants(new ElementFilter(
						"transitionName"));
				String lineSpecies = it.hasNext() ? it.next().getText() : "";
				it = root.getDescendants(new ElementFilter("sourceName"));
				String source = "";
				while (it.hasNext()) {
					source += "  " + it.next().getText();
				}
				it = root.getDescendants(new ElementFilter("polarisation"));
				String polar = it.hasNext() ? it.next().getText() : "";
				it = root.getDescendants(new ElementFilter("filename"));
				String projFilename = it.hasNext() ? it.next().getText() : "";

				// System.out.println(((XMLType)
				// curs.getObject(1)).getStringVal());
				i++;
				System.out.println(i + ": Proj: " + projName);
				System.out.println("PI: " + pi);
				System.out.println("version: " + version);
				System.out.println(uid); // uid
				if (asdms.isEmpty()) {
					System.out.println("ASDMs: None");
				} else {
					System.out.print("ASDMs:");
					for (Iterator<String> it1 = asdms.iterator(); it1.hasNext();) {
						System.out.print(" " + it1.next());
					}
					System.out.println();
				}

				if (detailedOut) {
					System.out.println("Project Code: " + projCode);
					System.out.println(lineSpecies);
					System.out.println("Source:" + source);
					System.out.println("Polarization: " + polar);
					System.out.println("Project filename: " + projFilename);
				}
				System.out.println("");
				if (xmlOut) {
					if (xmlFile == null) {
						// we have to open a new file (and close it afterwards)
						// for each XML document
						String fn = uid.substring(6).replace('/', ':') + ".xml";
						try {
							xmlStream = new PrintStream(fn);
							xmlFileList.add(fn);
						} catch (FileNotFoundException e) {
							System.out.println("Problem in creating file:");
							System.out.println(e.toString());
							System.out
									.println("Problem in creating file " + fn);
							return;
						}
					}
					xmlStream
							.println(new XMLOutputter(Format.getPrettyFormat())
									.outputString(root));
					if (xmlFile == null) {
						// we have to close the file for each XML document
						xmlStream.close();
					}
				}

				// file output:
				if (writeFile != null) {
					writeFile.println(i + ": Proj: " + projName);
					writeFile.println("PI: " + pi);
					writeFile.println("version: " + version);
					writeFile.println(uid); // uid
					if (asdms.isEmpty()) {
						writeFile.println("ASDMs: None");
					} else {
						writeFile.print("ASDMs:");
						for (Iterator<String> it1 = asdms.iterator(); it1
								.hasNext();) {
							writeFile.print(" " + it1.next());
						}
						writeFile.println();
					}
					writeFile.println("Project Code: " + projCode);
					writeFile.println(lineSpecies);
					writeFile.println("Source:" + source);
					writeFile.println("Polarization: " + polar);
					writeFile.println("Project filename: " + projFilename);
					writeFile.println("");

				}
			}
		} catch (SQLException e) {
			System.out.println("Problem in reading database result: ");
			System.out.println(e.toString());
			System.out.println("Executed query was: " + sql);
			System.out.println("Problem in reading database result.");
			try {
				curs.close();
				stmt.close();
				ocpds.close(readConn);
			} catch (Exception e1) {
				// nothing to do here
			}
			return;
		} catch (JDOMException e) {
			System.out.println("Problem in parsing database result: ");
			System.out.println(e.toString());
			System.out.println("Executed query was: " + sql);
			System.out.println("Problem in parsing database result.");
			try {
				curs.close();
				stmt.close();
				ocpds.close(readConn);
			} catch (Exception e1) {
				// nothing to do here
			}
			return;
		} catch (IOException e) {
			System.out.println("Problem in writing to file:");
			System.out.println(e.toString());
			System.out.println("Problem in writing to file "
					+ cmd.getOptionValue("w"));
			try {
				curs.close();
				stmt.close();
				ocpds.close(readConn);
			} catch (Exception e1) {
				// nothing to do here
			}
			return;
		}

		if (i == 0 && !elementSearch.isEmpty()) {
			// no results returned in element query: check whether the element
			// name really appears.
			xpath = "/prj:ObsProject";
			for (Iterator<String> it = elementSearch.iterator(); it.hasNext();) {
				xpath += "[.//prj:" + it.next() + "]";
			}
			sql = "SELECT value(xmlres) xml, archive_uid FROM "
					+ obsProjTab
					+ ", table(xmlsequence(extract(xml, '"
					+ xpath
					+ "','xmlns:prj=\"Alma/ObsPrep/ObsProject\"'))) xmlres WHERE deleted=0 AND hidden=0 AND dirty=0";
			try {
				curs = stmt.executeQuery(sql);
				if (!curs.next()) {
					// The element search required an invalid element. This has
					// to be printed out:
					System.out
							.print("ERROR: Search returned 0 results. One of the following XML elements does not appear in ObsProjects (mind case-sensitivity): ");
					for (Iterator<String> it = elementSearch.iterator(); it
							.hasNext();) {
						System.out.print(it.next() + " ");
					}
					System.out.println();
				}
			} catch (Exception e) {
				// something wrong here, but no idea how to proceed...
				System.out.println("Internal problem: " + e.toString());
			}
		}

		System.out.println(queryInfo
				+ (caseIns ? "using case-insensitive text search."
						: "using case-sensitive text search."));
		System.out.println("Found " + i + " result(s). ");
		if (writeFile != null) {
			System.out.println("Output written to " + cmd.getOptionValue("w")
					+ ".");
		}

		if (!xmlFileList.isEmpty()) {
			System.out.print("XML output written to: ");
			for (Iterator<String> it1 = xmlFileList.iterator(); it1.hasNext();) {
				System.out.print(" " + it1.next());
			}
			System.out.println();
		}
		try {
			curs.close();
			stmt.close();
			ocpds.close(readConn);
		} catch (Exception e1) {
			// nothing to do here
		}
	}

	private static void displayConfig() {
		System.out.println("Database configuration read from "
				+ dbConfig.fileLocation + ":");
		System.out.println(dbConfig.toString());
	}

	private static List<String> getASDMs(String uid) throws SQLException,
			JDOMException, IOException {
		ArrayList<String> out = new ArrayList<String>();

		
		// Example: /ExecBlockTable[.//projectId/EntityRef/attribute::entityId="uid://B203/X6643/X1"]//ContainerEntity
		Statement stmt = readConn.createStatement();
		String xpath = "/ExecBlockTable[.//projectId/EntityRef/attribute::entityId=\""
				+ uid + "\"]/ContainerEntity"; // xpath to reach correct
		// pointers to ASMDs in
		// execblocks
		// In order to extract the entityId attribute from the xmlsequence table an additional
		// XPath is required on the result set.
		String sql = "SELECT extractvalue(value(xmlres),'*:ContainerEntity/@entityId') xml FROM"
				+ execblockTab
				+ ", table(xmlsequence(extract(xml, '"
				+ xpath
				+ "','xmlns=\"Alma/XASDM/ExecBlockTable\"'))) xmlres WHERE deleted=0 AND hidden=0 AND dirty=0";
		// System.out.println(sql);
		ResultSet curs = stmt.executeQuery(sql);
		while (curs.next()) {
			// NOTE: Following is not required anymore, kept here for reference
			//
			// unfortunately we have to evaluate the attribute here (no idea why
			// it doesn't work in XPath, always returns 0 results)
			//org.jdom.Document doc = builder.build(new StringReader(
			//		((XMLType) curs.getObject(1)).getStringVal()));
			// /attribute::entityId"
			//out.add(doc.getRootElement().getAttributeValue("entityId"));
			out.add((String) curs.getObject(1));
		}
		return out;
	}

	private static DBConfiguration readDbConfig() throws DatabaseException {
		return DBConfiguration.instance(m_logger);
	}

	private static void displayExamples() {
		System.out.println("EXAMPLES for projectQuery");
		System.out.println();
		System.out.println("- Project name:");

		System.out.println("  projectQuery ProjName=Protoplaneta");

		System.out.println("- PI:");

		System.out.println("  projectQuery -i pi=\"g. galilei\"");

		System.out
				.println("  (Case insensitive, blanks have to be enclosed in \".");

		System.out.println("- Project Code");

		System.out.println("  projectQuery projCode==LW001");

		System.out.println("  (Exact match, not containment)");

		System.out.println("- Continuum/Spectral line:");

		System.out.println("  projectQuery obsType=Cont");

		System.out.println("- Source name:");

		System.out.println("  projectQuery sourceName=L1527 sourceName=L1551");

		System.out.println("  (Both sources must be in project)");

		System.out.println("- Polarization: ");

		System.out.println("  projectQuery pol=4");

		System.out.println("  (Polarization value can be 1,2, or 4)");

		System.out.println("- Line Species:");

		System.out.println("  projectQuery lineSpecies=\"CO(3-2)\"");

		System.out.println("- Rest frequency:");

		System.out.println("  projectQuery restFreq=90");

		System.out.println("  (No intervals allowed yet, unit is GHz)");

		System.out.println("- Generic query:");

		System.out.println("  projectQuery searchterm");

		System.out
				.println("  (Returns projects that contain \"searchterm\" somewhere)");

		System.out.println("- Other APDM fields:");

		System.out.println("  projectQuery minIntegrationTime==2.0");

		System.out
				.println("  (minIntegrationTime is name of XML element in project, always case-sensitive!)");

		System.out.println("- File output:");

		System.out
				.println("  projectQuery -w fileOut.txt ProjName=Protoplaneta");

		System.out
				.println("  (writes (detailed) project summary to fileOut.txt)");

		System.out
				.println("  projectQuery -xf fileOut.xml ProjName=Protoplaneta");

		System.out.println("  (writes XML file to fileOut.xml)");

		System.out.println("  projectQuery -x ProjName=Protoplaneta");

		System.out
				.println("  (writes XML file to file named after UIDs, eg. X1b:Xa44:X1.xml)");

		System.out.println("- Conjunctive queries:");

		System.out.println("  projectQuery ProjName=Protoplaneta pi=galilei");
	}

	private static void displayHelp() {
		System.out
				.println("projectQuery [-h] [-e] [-c] [-w <filename>] [-d] [-x|-xf <filename>] [-i] <parameter>[=[=]<value>] [<parameter>[=[=]<value>]]");
		System.out.println();

		System.out.println("-h              show help page");
		System.out.println("-e              show examples");
		System.out
				.println("-c              show configuration (which database is used, with which user)");
		System.out
				.println("-w              write the summary output to file with name <filename> (works also with -d)");
		System.out.println("-d              detailed output");
		System.out
				.println("-xf             write ObsProject XML entities to file with name <filename>");
		System.out
				.println("-x              create one XML file for each matching entity with the UIDs as filenames.");
		System.out.println("-i              ignore case for value searches. (Does not work for full text search, only with parameter=value searches.)");
		System.out.println();
		System.out
				.println("The command line switches -w, -d, -x, -xf, -i must be specofoed in the above order, if multiple of them are used. -x and -xf cannot be used at the same time.");
		System.out.println();
		System.out
				.println("<parameter> is either one of the real element names of the ObsProject schema or one of \"projname\", \"pi\", \"projcode\", \"obstype\", \"sourcename\", \"linespecies\",\"pol\", \"restfreq\" where the element names are case sensitive while the aliases in the list are not.");
		System.out.println();
		System.out
				.println("<value>     if specified should contain a valid ASCII string, if -i is specified as well the search will be case in-sensitive. ");
		System.out
				.println("            If no value is specified, but still the '=' sign then the program will just check for the existence of the element and return the entities containing the element.");
		System.out.println("            For example ");
		System.out.println("                          projectQuery projName=");
		System.out
				.println("            will return all projects in the archive.");
		System.out
				.println("            If two consecutive equal signs ('==') are given between the parameter name and the value the program will search for the value string exactly, with one equal sign ('=') it will execute a containment query, i.e. it will match all entities where the paramter value contains the given string.");
		System.out
				.println("            *NOTE*: If reserved characters of the unix shell are used in the value string they have to be properly escaped. If the search value contains a space the whole value has to be enclosed in double quotes, like 'parameter=\"My value\"'");
		System.out
				.println("            If no '=' sign is specified than the string is matched against all elements, i.e. if that string is contained in any element value (not element name) then the entity matches the query. ");
		System.out.println();
		System.out
				.println("            *NOTE*: There is no space allowed on either side of the '=' sign, i.e. 'parameter=value' works correctly but 'parameter = value' does not.");
		System.out.println();
		System.out
				.println("If more than one parameter=value pair is given they all have to match, i.e. it is effectively an AND query.");

	}
}
