
/* This file is part of xtmodcoll-3.0, (c) xtmod.com
 * 
 * xtmodcoll-3.0 is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * xtmodcoll-3.0 is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser Public License
 * along with xtmodcoll-3.0.  If not, see <http://www.gnu.org/licenses/>.
 */

package com.xtmod.util.collections;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;


/**
 * This is a utility class to compare two lists (ignoring order of elements). This is
 * typically used to compare an existing collection of objects against a newer version of
 * that collection, which is why this class refers to its lists as "Existing" and
 * "Incoming". After specifying the two lists, and triggering the computation, the results
 * can be retrieved through the get-methods.
 * <p>
 * This takes three type parameters: E is the element type of the first list ("existing"),
 * I is the element type of the second list ("incoming"), and C is the type they have in
 * common (the type used to correspond elements from both lists, often String). To use this
 * class, you need to provide implementations of the identify-methods which convert elements
 * from the lists into the common type C.
 * <p>
 * The common type is used to identify which elements correspond to each other. After two
 * corresponding elements have been found, the algorithm checks if the incoming element
 * denotes an "update" of the existing element, or if they are considered equal. This gets
 * decided by the isUpdate() method that you need to implement, too.
 */
@SuppressWarnings("unchecked")
public abstract class ListDiff<E, C extends Comparable, I> {


	protected SortableList<E> iexist;
	protected SortableList<I> iincom;

	protected SortableList<E> removed;
	protected SortableList<E> unchanged;
	protected SortableList<E> updated;
	protected SortableList<I> duplicate;
	protected SortableList<I> updates;
	protected SortableList<I> added;



	// Input
	// =====================================================================


	/**
	 * Sets both lists and diffs them.
	 * 
	 * @param exist - the first list
	 * @param incom - the second list
	 */
	public void diff (List<E> exist, List<I> incom) {
		iexist = makeExistList(exist);
		iincom = makeIncomList(incom);

		iexist.sortByIdent();
		iincom.sortByIdent();

		diffLists();
		
		removed.sortByIndex();
		unchanged.sortByIndex();
		updated.sortByIndex();
		
		duplicate.sortByIndex();
		updates.sortByIndex();
		added.sortByIndex();
	}



	// Computations
	// =====================================================================


	/**
	 * Convert given {@code E} object into a Comparable of type {@code C}
	 */
	abstract protected C identifyExisting (E x);

	/**
	 * Convert given {@code I} object into a Comparable of type {@code C}
	 */
	abstract protected C identifyIncoming (I x);

	/**
	 * Compare an existing object with its incoming version.
	 * <p>
	 * Once an incoming element has been identified as corresponding to a certain existing
	 * element, this method decides whether the two elements are considered equal or if
	 * the incoming represents an "update" (an interior modification, e.g. a different
	 * userobject in a treenode).
	 * 
	 * It may suffice to implement this as <code>!e.equals(i)</code>.
	 */
	abstract protected boolean isUpdate (E e, I i);



	
	/*
	 * These two methods are almost identical but not fully: they use different identify()
	 * methods.
	 */

	protected SortableList<E> makeExistList(List<E> origlist) {
		int n = origlist.size();
		SortableList<E> list = new SortableList<E>(n);
		for (int index=0; index<n; index++) {
			E elem = origlist.get(index);
			list.add(new SortableElem<E>(index, elem, identifyExisting(elem)));
		}
		return list;
	}

	protected SortableList<I> makeIncomList(List<I> origlist) {
		int n = origlist.size();
		SortableList<I> list = new SortableList<I>(n);
		for (int index=0; index<n; index++) {
			I elem = origlist.get(index);
			list.add(new SortableElem<I>(index, elem, identifyIncoming(elem)));
		}
		return list;
	}

	

	protected void diffLists () {

		removed = new SortableList<E>();
		added = new SortableList<I>();
		
		unchanged = new SortableList<E>();
		duplicate = new SortableList<I>();

		updated = new SortableList<E>();
		updates = new SortableList<I>();

		int ib = 0;
		int ia = 0;

		for (;;) {

			if (ia == iexist.size()) {
				added.addAll(iincom.subList(ib, iincom.size()));
				break;
			}
			if (ib == iincom.size()) {
				removed.addAll(iexist.subList(ia, iexist.size()));
				break;
			}

			SortableElem<E> a = iexist.get(ia);
			SortableElem<I> b = iincom.get(ib);

			int eq = (a.ident).compareTo(b.ident);

			if (eq < 0) {
				/* a<b : no such b */
				removed.add(a);
				ia += 1;
			}
			else
			if (eq == 0) {
				/* a=b : */
				if (isUpdate(a.elem, b.elem)) {
					updated.add(a);
					updates.add(b);
				} else {
					unchanged.add(a);
					duplicate.add(b);
				}
				ia += 1;
				ib += 1;
			}
			else
			if (eq > 0) {
				/* a>b : */
				added.add(b);
				ib += 1;
			}
		}

	}


	/**
	 * Elem to be stored in SortableList. Holds certain fields that can be sorted by.
	 */
	protected class SortableElem<OBJ> {
		protected OBJ elem;      // object provided by client, either <E> or <I>
		protected int origIndex; // the index (used for sorting)
		protected C ident;       // the id (used for sorting)
		protected SortableElem (int oi, OBJ e, C id) { origIndex=oi; elem=e; ident=id;}
	}

	/**
	 * List consisting of SortableElems. Can be sorted back and forth, which we
	 * need for our diff list algorithm.
	 */
	protected class SortableList<OBJ> extends ArrayList<SortableElem<OBJ>> {

		protected SortableList() {
			super();
		}

		protected SortableList(int size) {
			super(size);
		}

		protected void sortByIdent () {
			Comparator<SortableElem<OBJ>> comp;
			comp = new Comparator<SortableElem<OBJ>>() {
				public int compare (SortableElem<OBJ> o1, SortableElem<OBJ> o2) {
					return (o1.ident.compareTo(o2.ident));
				}
			};
			Collections.sort(this, comp);
		}

		protected void sortByIndex () {
			Comparator<SortableElem<OBJ>> comp;
			comp= new Comparator<SortableElem<OBJ>>() {
				public int compare (SortableElem<OBJ> o1, SortableElem<OBJ> o2) {
					return (o1.origIndex - o2.origIndex);
				}
			};
			Collections.sort(this, comp);
		}

		protected List<OBJ> toElems () {
			ArrayList<OBJ> ret = new ArrayList<OBJ>(this.size());
			for (SortableElem<OBJ> x : this)
				ret.add(x.elem);
			return ret;
		}

		protected List<Integer> toIndices () {
			ArrayList<Integer> ret = new ArrayList<Integer>(this.size());
			for (SortableElem<OBJ> x : this)
				ret.add(x.origIndex);
			return ret;
		}

	}
	

	
	// Output
	// =====================================================================

	/**
	 * The existing objects that correspond to incoming "Duplicates".
	 */
	public List<E> getUnchanged () {
		return unchanged.toElems();
	}

	/**
	 * Same, expressed as indices in the existing list.
	 */
	public List<Integer> getUnchangedIndices () {
		return unchanged.toIndices();
	}

	/**
	 * The existing objects that correspond to incoming "Updates".
	 */
	public List<E> getUpdated () {
		return updated.toElems();
	}

	/**
	 * Same, expressed as indices in the existing list.
	 */
	public List<Integer> getUpdatedIndices () {
		return updated.toIndices();
	}
	
	/**
	 * The existing objects that don't correspond to anything incoming.
	 */
	public List<E> getRemoved () {
		return removed.toElems();
	}

	/**
	 * Same, expressed as indices in the existing list.
	 */
	public List<Integer> getRemovedIndices () {
		return removed.toIndices();
	}
	
	/**
	 * The incoming objects that correspond to "Unchanged" existing.
	 */
	public List<I> getDuplicates () {
		return duplicate.toElems();
	}

	/**
	 * Same, expressed as indices in the incoming list.
	 */
	public List<Integer> getDuplicatesIndices () {
		return duplicate.toIndices();
	}
	
	/**
	 * The incoming objects that correspond to "Updated" existing.
	 */
	public List<I> getUpdates () {
		return updates.toElems();
	}

	/**
	 * Same, expressed as indices in the incoming list.
	 */
	public List<Integer> getUpdatesIndices () {
		return updates.toIndices();
	}
	
	/**
	 * The incoming objects that don't correspond to anything existing.
	 */
	public List<I> getAdded () {
		return added.toElems();
	}

	/**
	 * Same, expressed as indices in the incoming list.
	 */
	public List<Integer> getAddedIndices () {
		return added.toIndices();
	}

	/**
	 * Returns if both lists have equal contents (order ignored).
	 */
	public boolean areEqual () {
		return (removed.isEmpty() && added.isEmpty() && updated.isEmpty());
	}


	@Override
	public String toString() {
		StringBuilder s = new StringBuilder(); 
		for (int i=0; i<updated.size(); i++)
			s.append(" Upd  ["+updated.get(i).origIndex+"]" + updated.get(i).elem)
			.append(" < ["+updates.get(i).origIndex+"]"+updates.get(i).elem+"\n");
		for (int i=0; i<removed.size(); i++)
			s.append(" Rem  ["+removed.get(i).origIndex+"]"+removed.get(i).elem+"\n");
		for (int i=0; i<added.size(); i++)
			s.append(" Add  < ["+added.get(i).origIndex+"]"+added.get(i).elem+"\n");
		return s.toString();
	}

}






