/** MonitorDAOImpl is the implementation of MonitorDAO interface.
 * This class is used by blobbers to persist monitoring informacion through
 * store method. Besides that, is on charge of Control Device
 * properties autoconfiguration.
 *
 * @author Pablo Burgos
 * @since ACS-8_0_0-B Jun2009
 * @version "@(#) $Id: MonitorDAOImpl.java,v 1.1.2.6 2010/05/21 19:46:00 vgonzale Exp $
 */
package alma.archive.tmcdb.DAO;

import java.sql.Timestamp;
import java.util.HashMap;
import java.util.List;
import java.util.logging.Logger;

import javax.persistence.EntityManager;
import javax.persistence.EntityTransaction;
import javax.persistence.NoResultException;
import javax.persistence.NonUniqueResultException;
import javax.persistence.PersistenceException;
import javax.persistence.Query;

import alma.DAOErrType.wrappers.AcsJDBConnectionFailureEx;
import alma.DAOErrType.wrappers.AcsJDynConfigFailureEx;
import alma.DAOErrType.wrappers.AcsJGettingMonitorCharacteristicsEx;
import alma.DAOErrType.wrappers.AcsJStoreFailureEx;
import alma.acsErrTypeAlarmSourceFactory.ACSASFactoryNotInitedEx;
import alma.acsErrTypeAlarmSourceFactory.FaultStateCreationErrorEx;
import alma.acsErrTypeAlarmSourceFactory.SourceCreationErrorEx;
import alma.alarmsystem.source.ACSAlarmSystemInterface;
import alma.alarmsystem.source.ACSAlarmSystemInterfaceFactory;
import alma.alarmsystem.source.ACSFaultState;
import alma.archive.tmcdb.Persistence.Util.ComponentNameHelper;
import alma.archive.tmcdb.Persistence.Util.TMCDBConfig;
import alma.archive.tmcdb.Persistence.Util.TMCDBPersistence;
import alma.archive.tmcdb.Persistence.pojo.Assembly;
import alma.archive.tmcdb.Persistence.pojo.AssemblyType;
import alma.archive.tmcdb.Persistence.pojo.BACIProperty;
import alma.archive.tmcdb.Persistence.pojo.Component;
import alma.archive.tmcdb.Persistence.pojo.ComponentType;
import alma.archive.tmcdb.Persistence.pojo.Configuration;
import alma.archive.tmcdb.Persistence.pojo.DefaultBACIProperty;
import alma.archive.tmcdb.Persistence.pojo.DefaultComponent;
import alma.archive.tmcdb.Persistence.pojo.DefaultMonitorPoint;
import alma.archive.tmcdb.Persistence.pojo.HWConfiguration;
import alma.archive.tmcdb.Persistence.pojo.MonitorData;
import alma.archive.tmcdb.Persistence.pojo.MonitorPoint;

@SuppressWarnings("deprecation")
public class MonitorDAOImpl implements MonitorDAO {

	private Logger log;
	private HashMap<String, Object> myConfiguredComponentsMap = new HashMap<String, Object>();
	private TMCDBPersistence myPersistenceLayer = new TMCDBPersistence();
	private EntityManager entityManagerStore;
	private EntityTransaction transactionStore;
	private boolean dbConnectionEnabled = false;
	private boolean monitoringOnlyEnabled = false;
	private static final int MaxAttempts = 10;
	private int countAttempts = 0;

	public MonitorDAOImpl(Logger inLogger) {
		this.log = inLogger;
		dbConnectionEnabled = TMCDBConfig.getInstance().isDBConnectionEnabled();
		monitoringOnlyEnabled = TMCDBConfig.getInstance()
				.isMonitoringOnlyEnabled();

	}

	public void close() {
		this.myPersistenceLayer.close();
	}

	public void openTransactionStore() throws Exception {
		if (dbConnectionEnabled) {
			try {
				entityManagerStore = this.myPersistenceLayer.getEntityManager();
				transactionStore = entityManagerStore.getTransaction();
				transactionStore.begin();
			} catch (PersistenceException ex) {
				/*
				 * Here a Persistence exception has been caught. After
				 * MaxAttempts, connections to the Database will not be
				 * attempted anymore. Refer to COMP-4240
				 */
				countAttempts += 1;
				if (countAttempts > MaxAttempts) {
					dbConnectionEnabled = false;
					send_alarm("Monitoring", "DAO", 1, true);
				}
				throw new AcsJDBConnectionFailureEx(
						"DAO couldn't get connected to Database. Attempt "
								+ countAttempts + "out of " + MaxAttempts, ex);
			}
		}
	}

	public void closeTransactionStore() throws Exception {
		if (dbConnectionEnabled) {
			try {
				transactionStore.commit();
				entityManagerStore.close();
			} catch (PersistenceException ex) {
				/*
				 * Here a Persistence exception has been caught. After
				 * MaxAttempts, connections to the Database will not be
				 * attempted anymore. Refer to COMP-4240
				 */
				countAttempts += 1;
				if (countAttempts > MaxAttempts) {
					dbConnectionEnabled = false;
					send_alarm("Monitoring", "DAO", 1, true);
				}
				throw new AcsJDBConnectionFailureEx(
						"DAO couldn't get connected to Database. Attempt "
								+ countAttempts + "out of " + MaxAttempts, ex);
			}
		}
	}

	private boolean hasFailedToConfigured(ComponentData inData) {
		/*
		 * We must be sure to just try auto-configuration once for a given
		 * component name. This is to avoid new attempts each time data with
		 * that component information is retrieved from a monitordatablock.
		 */
		String hashEntry = inData.componentName + ":" + inData.propertyName
				+ ":" + inData.index;

		if (this.myConfiguredComponentsMap.containsKey(hashEntry)) {
			return true;
		}
		return false;
	}

	private void failedToConfigured(ComponentData inData) {
		String hashEntry = inData.componentName + ":" + inData.propertyName
				+ ":" + inData.index;
		this.myConfiguredComponentsMap.put(hashEntry, null);
		log.fine("Dynamic configuration failed for blob " + hashEntry);
	}

	private Long getConfigurationId(EntityManager entityManagerStore,
			String configurationName) throws NonUniqueResultException,
			NoResultException {
		Query query = entityManagerStore
				.createNamedQuery("findConfigurationByName");
		query.setParameter("configurationName", configurationName);
		Configuration conf = (Configuration) query.getSingleResult();
		Long configurationId = conf.getConfigurationId();
		log.fine("ConfigurationId = " + configurationId);
		return configurationId;
	}

	private Long getHwConfigurationId(EntityManager entityManagerStore,
			Long swConfigurationId) throws NonUniqueResultException,
			NoResultException {
		Query query = entityManagerStore
				.createNamedQuery("findHwConfBySwConfigId");
		query.setParameter("swConfigurationId", swConfigurationId);
		HWConfiguration conf = (HWConfiguration) query.getSingleResult();
		Long hwConfigurationId = conf.getConfigurationId();
		log.fine("HwConfigurationId = " + hwConfigurationId);
		return hwConfigurationId;
	}

	private Long getAssemblyId(EntityManager entityManagerStore,
			Long configurationId, String serialNumber)
			throws NonUniqueResultException, NoResultException {
		Query query = entityManagerStore
				.createNamedQuery("findAssemblyBySerialNumberAndConfigurationId");
		query.setParameter("serialNumber", serialNumber);
		query.setParameter("configurationId", configurationId);
		Assembly assembly = (Assembly) query.getSingleResult();
		Long assemblyId = assembly.getAssemblyId();
		log.fine("assemblyId = " + assemblyId);
		return assemblyId;
	}

	private Long getComponentId(EntityManager entityManagerStore,
			Long configurationId, String componentName)
			throws NonUniqueResultException, NoResultException {
		Query query = entityManagerStore
				.createNamedQuery("findComponentByComponentName");
		String tokens[] = ComponentNameHelper.getPathAndName(componentName);
		query.setParameter("path", tokens[0]);
		query.setParameter("componentName", tokens[1]);
		query.setParameter("configurationId", configurationId);
		Component comp = (Component) query.getSingleResult();
		Long componentId = comp.getComponentId();
		log.fine("componentId = " + componentId);
		return componentId;
	}

	private Long getBaciPropertyId(EntityManager entityManagerStore,
			Long componentId, String propertyName)
			throws NonUniqueResultException, NoResultException {
		Query query = entityManagerStore
				.createNamedQuery("findBACIPropertyIdByPropertyNameANDComponentId");
		query.setParameter("componentId", componentId);
		query.setParameter("propertyName", propertyName);
		BACIProperty baciProp = (BACIProperty) query.getSingleResult();
		Long baciPropertyId = baciProp.getBACIPropertyId();
		log.fine("baciPropertyId = " + baciPropertyId);
		return baciPropertyId;
	}

	private Long getMonitorPointId(EntityManager entityManagerStore,
			Long assemblyId, Long baciPropertyId, int index)
			throws NonUniqueResultException, NoResultException {
		Query query = entityManagerStore
				.createNamedQuery("findMonitorPointIdByAssemblyIdANDBACIPropertyIdANDIndex");
		query.setParameter("assemblyId", assemblyId);
		query.setParameter("baciPropertyId", baciPropertyId);
		query.setParameter("index", index);
		MonitorPoint mp = (MonitorPoint) query.getSingleResult();
		Long monitorPointId = mp.getMonitorPointId();
		log.fine("index = " + index);
		log.fine("monitorPointId = " + monitorPointId);
		return monitorPointId;
	}

	public MonitorCharacteristicIDs getMonitorCharacteristicIDs(
			EntityManager entityManagerStore, String configurationName,
			ComponentData inData) throws AcsJGettingMonitorCharacteristicsEx,
			AcsJDynConfigFailureEx {

		// constructor of MonitorCharacteristicIDs sets all to -1L and false
		MonitorCharacteristicIDs monitorCharacteristicIDs = new MonitorCharacteristicIDs();
		try {
			Long configurationId = getConfigurationId(entityManagerStore,
					configurationName);
			monitorCharacteristicIDs.setConfigurationId(configurationId);
		} catch (NoResultException e) {
			// this means problems
			throw new AcsJGettingMonitorCharacteristicsEx(
					"Could not find configuration '" + configurationName + "'",
					e);
		} catch (NonUniqueResultException e) {
			// this means problems too
			throw new AcsJGettingMonitorCharacteristicsEx(
					"Found multiple configurations with name '"
							+ configurationName + "'", e);
		}

		try {
			Long hwCconfigurationId = getHwConfigurationId(entityManagerStore,
					monitorCharacteristicIDs.getConfigurationId());
			monitorCharacteristicIDs.setHwConfigurationId(hwCconfigurationId);
		} catch (NoResultException e) {
			// this means problems
			throw new AcsJGettingMonitorCharacteristicsEx(
					"Could not find HW configuration for configuration '"
							+ configurationName + "'", e);
		} catch (NonUniqueResultException e) {
			// this means problems too
			throw new AcsJGettingMonitorCharacteristicsEx(
					"Found multiple HW configurations for configuration '"
							+ configurationName + "'", e);
		}

		try {
			Long assemblyId = getAssemblyId(entityManagerStore,
					monitorCharacteristicIDs.getConfigurationId(),
					inData.serialNumber.toUpperCase());
			monitorCharacteristicIDs.setAssemblyId(assemblyId);
		} catch (NoResultException e) {
			/*
			 * at this point we know that autoconfiguration is needed there is
			 * no assembly within this configuration with the given serialnumber
			 */
			Long assemblyId = configureNewAssembly(monitorCharacteristicIDs,
					inData);
			monitorCharacteristicIDs.setAssemblyId(assemblyId);
		} catch (NonUniqueResultException e) {
			// this means problems
			throw new AcsJGettingMonitorCharacteristicsEx(
					"Found multiple assemblyIds with serialnumber="
							+ inData.serialNumber.toUpperCase(), e);
		}

		try {
			Long componentId = getComponentId(entityManagerStore,
					monitorCharacteristicIDs.getConfigurationId(),
					inData.componentName);
			monitorCharacteristicIDs.setComponentId(componentId);
		} catch (NoResultException ex) {
			/*
			 * at this point we know that autoconfiguration is needed there is
			 * no component named as componentName within this configuration
			 * since the assembly is known (getAssemblyId returned a valid
			 * value) it means that that an existing assembly has been moved to
			 * a new component
			 */
			if (monitoringOnlyEnabled) {
				Long componentId = configureNewComponent(
						monitorCharacteristicIDs, inData);
				monitorCharacteristicIDs.setComponentId(componentId);
			} else {
				throw new AcsJGettingMonitorCharacteristicsEx(
						"Component Not Found. This must be configured by TMCDB procedure",
						ex);
			}
		} catch (NonUniqueResultException e) {
			// this means problems
			throw new AcsJGettingMonitorCharacteristicsEx(
					"Found multiple componentId matching the given componentName.",
					e);
		}

		try {
			Long baciPropertyId = getBaciPropertyId(entityManagerStore,
					monitorCharacteristicIDs.getComponentId(),
					inData.propertyName);
			monitorCharacteristicIDs.setBACIPropertyId(baciPropertyId);
		} catch (NoResultException e) {
			/*
			 * at this point we know that autoconfiguration is needed there is
			 * no baciproperty named as propertyName for the given componentId
			 * within this configuration since the assembly and the component is
			 * known (getAssemblyId and getComponentId returned a valid value)
			 * it means that that a new property is being monitored (change to
			 * the devices spreadsheet) static tables (defaultBaciProperty) must
			 * have been updated to be able to store this data
			 */
			if (monitoringOnlyEnabled) {
				Long baciPropertyId = configureNewProperty(
						monitorCharacteristicIDs, inData);
				monitorCharacteristicIDs.setBACIPropertyId(baciPropertyId);
			} else {
				throw new AcsJGettingMonitorCharacteristicsEx(
						"BACI property  Not Found. This must be configure bt TMCDB procedure",
						e);
			}
		} catch (NonUniqueResultException e) {
			// this means problems
			throw new AcsJGettingMonitorCharacteristicsEx(
					"Found multiple BACI properties matching the given property name and componentId.",
					e);
		}

		try {
			Long monitorPointId = getMonitorPointId(entityManagerStore,
					monitorCharacteristicIDs.getAssemblyId(),
					monitorCharacteristicIDs.getBACIPropertyId(), inData.index);
			monitorCharacteristicIDs.setIndex(inData.index);
			monitorCharacteristicIDs.setMonitorPointId(monitorPointId);
			monitorCharacteristicIDs.setIsOnDB(true);
		} catch (NoResultException e) {
			/*
			 * at this point we know that autoconfiguration is needed there is
			 * no monitorpoint matching the given assemblyId, baciPropertyId and
			 * index within this configuration since the baciProperty is known
			 * (getBaciPropertyId returned a valid value) it means that that a
			 * new monitorpoint is being monitored (change to the devices
			 * spreadsheet) static tables (defaultMonitorPoint) must have been
			 * updated to be able to store this data
			 */
			Long monitorPointId = configureNewMonitorPoint(
					monitorCharacteristicIDs, inData);
			monitorCharacteristicIDs.setIndex(inData.index);
			monitorCharacteristicIDs.setMonitorPointId(monitorPointId);
			// we leave setIsOnDB as false to signal that this blob was just
			// configured
		} catch (NonUniqueResultException e) {
			// this means problems
			throw new AcsJGettingMonitorCharacteristicsEx(
					"Found multiple monitor points matching the given assemblyId, baciPropertyId and index.",
					e);
		}

		return monitorCharacteristicIDs;

	}

	/**
	 * Store method is the main method inside MonitorDAOImpl class. This method
	 * attempts to persists the ComponentData object into the database. This
	 * method will attempt to get the Monitor Characteristics IDs. These ID's
	 * are the primary keys of Dynamic Tables on TMC schema definition:
	 * Assembly, Component, BACIProperty and MonitorPoint. If one of these ID's
	 * can not be found on Database, for the running Configuration ID,
	 * autoconfiguration will be attempt. Autoconfiguration is a Dynamic
	 * Algorithm, that infers Dynamic Monitoring Information from the data
	 * inside static tables DefaultComponent, DefaultBACIProperty,
	 * DefaultMonitorPoint. If the inference process is successful, Dynamic
	 * Tables will be filled in with correct monitoring information.
	 * 
	 * @param inData
	 * @throws Exception
	 */
	public void store(ComponentData inData) throws Exception {
		try {

			log.fine("Handling component: " + inData);

			/**
			 * Here comes a rule to prevent monitoring information coming from
			 * LA antennas to be persisted on database.
			 */
			if (!inData.componentName.contains("CONTROL/LA")
					&& dbConnectionEnabled) {
				if (!hasFailedToConfigured(inData)) {
					MonitorCharacteristicIDs monitorCharacteristicIDs;
					try {
						String confName = TMCDBConfig.getInstance()
								.getConfigurationName();
						monitorCharacteristicIDs = getMonitorCharacteristicIDs(
								entityManagerStore, confName, inData);
					} catch (AcsJGettingMonitorCharacteristicsEx e) {
						/*
						 * This exception typically is thrown when a
						 * NonUniqueResultException has been received
						 */
						throw new AcsJStoreFailureEx(
								"Failure when getting monitor characteristics",
								e);
					} catch (AcsJDynConfigFailureEx e) {
						/*
						 * This exception is thrown when an attempt to auto
						 * configure has been made and it failed TODO: No
						 * further attempts to auto configure this monitor point
						 * should be made
						 */
						failedToConfigured(inData);

						log
								.fine("Monitor point could not be autoconfigured for component = "
										+ inData.componentName
										+ ", serialNumber = "
										+ inData.serialNumber
										+ ", propertyName = "
										+ inData.propertyName
										+ ", index = "
										+ inData.index + " .");

						throw new AcsJStoreFailureEx(
								"Failure when configuring DB for: "
										+ inData.componentName + ":"
										+ inData.propertyName + ":"
										+ inData.index, e);
					}
					persistNewMonitorData(entityManagerStore, inData,
							monitorCharacteristicIDs);
				} else {
					throw new AcsJStoreFailureEx(
							"Dropping this blob since monitor point could not be configured previously");
				}
			}
		} catch (PersistenceException ex) {
			/*
			 * Here a Persistence exception has been caught. After MaxAttempts,
			 * connections to the Database will not be attempted anymore. Refer
			 * to COMP-4240
			 */
			countAttempts += 1;
			if (countAttempts > MaxAttempts) {
				dbConnectionEnabled = false;
				send_alarm("Monitoring", "DAO", 1, true);
			}
			throw new AcsJDBConnectionFailureEx(
					"DAO couldn't get connected to Database. Attempt "
							+ countAttempts + "out of " + MaxAttempts, ex);
		}
	}

	/*
	 * This methods gets the IDL URI based upong the component name
	 */
	public String getComponentIDL(String componentName) throws Exception {
		if ((componentName.substring(componentName.lastIndexOf("/") + 1))
				.equalsIgnoreCase("ACME")) {
			return "IDL:alma/Control/ACME:1.0";
		} else {
			ComponentResolver cmpRes = ComponentResolver.getResolver();
			String componentIDL = cmpRes.resolve(componentName);
			return componentIDL;
		}
	}

	/*
	 * This methods asumes that the AssemblyTypeName corresponds to the string
	 * after the last '/' of the Corresponding component's IDL URI example:
	 * "IDL:alma/Control/MountVertex:1.0" -> "MountVertex"
	 */
	private String getAssemblyTypeName(String componentIDL) {
		String assemblyTypeName = componentIDL.split(":")[1];
		String[] aux = assemblyTypeName.split("/");
		assemblyTypeName = aux[aux.length - 1];
		return assemblyTypeName;
	}

	public Long configureNewAssembly(
			MonitorCharacteristicIDs monitorCharacteristicIDs,
			ComponentData inData) throws AcsJDynConfigFailureEx {
		log.fine("Going to add configuration for assembly with SN = "
				+ inData.serialNumber.toUpperCase());

		/**
		 * Now let's look the assembly type associated with inData
		 */
		String assemblyIDL;
		try {
			assemblyIDL = getComponentIDL(inData.componentName);
		} catch (Exception e) {
			throw new AcsJDynConfigFailureEx(
					"Could not get IDL based on component name. Aborting dynamic configuration",
					e);
		}

		String assemblyTypeName = getAssemblyTypeName(assemblyIDL);

		/*
		 * The next queries and updates must be handled as a transaction since
		 * we can have several blobbers modifying the tables involved in
		 * autoconfiguration (dirty reading can happens). In some sense this is
		 * a nested transaction inside store transaction.
		 */
		EntityManager entityManager = this.myPersistenceLayer
				.getEntityManager();
		EntityTransaction transaction = entityManager.getTransaction();
		transaction.begin();
		/*
		 * Transacction must be enclosed in a try catch statement, to be able to
		 * rollback the transaccion in case
		 */
		try {
			AssemblyType assemblyType = getAssemblyTypeByLikeAssemblyCode(
					assemblyTypeName, entityManager);
			persistNewAssembly(entityManager, assemblyType,
					monitorCharacteristicIDs.getHwConfigurationId(), inData);
			transaction.commit();
			Long newAssemblyId = getAssemblyId(entityManager,
					monitorCharacteristicIDs.getHwConfigurationId(),
					inData.serialNumber.toUpperCase());
			return newAssemblyId;
		} catch (NonUniqueResultException e) {
			throw new AcsJDynConfigFailureEx(
					"Found multiple assembly types matching the assembly code "
							+ assemblyTypeName + ".", e);
		} catch (NoResultException e) {
			throw new AcsJDynConfigFailureEx(
					"Found no assembly type matching the assembly code "
							+ assemblyTypeName + ".", e);
		} catch (Exception e) {
			if (transaction != null) {
				try {
					log.fine("Exception detected, rollback.");
					transaction.rollback();
				} catch (RuntimeException rbEx) {
					log.fine("Couldn't roll back transaction: "
							+ rbEx.toString());
				}
			}
			throw new AcsJDynConfigFailureEx(
					"Failure while persisting new assembly: "
							+ assemblyTypeName + ".", e);
		} finally {
			if (entityManager != null) {
				entityManager.close();
			}
		}
	}

	private Long configureNewComponent(
			MonitorCharacteristicIDs monitorCharacteristicIDs,
			ComponentData inData) throws AcsJDynConfigFailureEx {
		log.fine("Going to add configuration for component = "
				+ inData.componentName);
		Long configurationId = monitorCharacteristicIDs.getConfigurationId();

		String assemblyIDL;
		try {
			assemblyIDL = getComponentIDL(inData.componentName);
		} catch (Exception e) {
			throw new AcsJDynConfigFailureEx(
					"Could not get IDL based on component name. Aborting dynamic configuration",
					e);
		}

		String assemblyTypeName = getAssemblyTypeName(assemblyIDL);

		/*
		 * The next queries and updates must be handled as a transaction since
		 * we can have several blobbers modifying the tables involved in
		 * autoconfiguration (dirty reading can happens). In some sense this is
		 * a nested transaction inside store transaction.
		 */
		EntityManager entityManager = this.myPersistenceLayer
				.getEntityManager();
		EntityTransaction transaction = entityManager.getTransaction();
		transaction.begin();
		/*
		 * Transacction must be enclosed in a try catch statement, to be able to
		 * rollback the transaccion in case
		 */
		try {

			/* Now let's look for its defaultComponentType */
			ComponentType componentType = getComponentTypeByLikeIDL(
					entityManager, assemblyIDL);
			/* Let's find the default component associated with inData */
			DefaultComponent defaultComponent = getDefaultComponentByLikeAssemblyTypeName(
					entityManager, assemblyTypeName);

			/*
			 * At this point we should have all information needed to persist
			 * the new objects into the database.
			 */
			persistNewComponent(entityManager, configurationId, componentType,
					inData, defaultComponent);
			transaction.commit();
			Long newComponentId = getComponentId(entityManager,
					monitorCharacteristicIDs.getConfigurationId(),
					inData.componentName);
			return newComponentId;
		} catch (NonUniqueResultException e) {
			throw new AcsJDynConfigFailureEx(
					"Found multiple component types matching the given IDL"
							+ assemblyIDL + ".", e);
		} catch (NoResultException e) {
			throw new AcsJDynConfigFailureEx(
					"Found no component types matching the given IDL"
							+ assemblyIDL + ".", e);
		} catch (Exception e) {
			if (transaction != null) {
				try {
					log.fine("Exception detected, rollback.");
					transaction.rollback();
				} catch (RuntimeException rbEx) {
					log.fine("Couldn't roll back transaction: "
							+ rbEx.toString());
				}
			}
			throw new AcsJDynConfigFailureEx(
					"Failure while persisting new component: "
							+ inData.componentName + ".", e);
		} finally {
			if (entityManager != null) {
				entityManager.close();
			}
		}
	}

	private Long configureNewProperty(
			MonitorCharacteristicIDs monitorCharacteristicIDs,
			ComponentData inData) throws AcsJDynConfigFailureEx {
		log.fine("Going to add configuration for property = "
				+ inData.propertyName);
		Long configurationId = monitorCharacteristicIDs.getConfigurationId();
		Long componentId = monitorCharacteristicIDs.getComponentId();
		String assemblyIDL;
		try {
			assemblyIDL = getComponentIDL(inData.componentName);
		} catch (Exception e) {
			throw new AcsJDynConfigFailureEx(
					"Could not get IDL based on component name. Aborting dynamic configuration",
					e);
		}

		String assemblyTypeName = getAssemblyTypeName(assemblyIDL);

		/*
		 * The next queries and updates must be handled as a transaction since
		 * we can have several blobbers modifying the tables involved in
		 * autoconfiguration (dirty reading can happens). In some sense this is
		 * a nested transaction inside store transaction.
		 */
		EntityManager entityManager = this.myPersistenceLayer
				.getEntityManager();
		EntityTransaction transaction = entityManager.getTransaction();
		transaction.begin();
		/*
		 * Transacction must be enclosed in a try catch statement, to be able to
		 * rollback the transaccion in case
		 */
		try {
			DefaultComponent defaultComponent = getDefaultComponentByLikeAssemblyTypeName(
					entityManager, assemblyTypeName);
			DefaultBACIProperty defaultBACIProperty = getDefaultBACIPropertyByDefaultComponentIdAndPropertyName(
					entityManager, defaultComponent.getDefaultComponentId(),
					inData.propertyName);

			persistNewBACIProperty(entityManager, defaultBACIProperty,
					componentId);
			transaction.commit();
			Long newBaciPropertyId = getBaciPropertyId(entityManager,
					componentId, inData.propertyName);
			return newBaciPropertyId;
		} catch (NonUniqueResultException e) {
			throw new AcsJDynConfigFailureEx(
					"Found multiple default baci properties matching the given property name"
							+ inData.propertyName + ".", e);
		} catch (NoResultException e) {
			throw new AcsJDynConfigFailureEx(
					"Found no default baci properties matching the given property name"
							+ inData.propertyName + ".", e);
		} catch (Exception e) {
			if (transaction != null) {
				try {
					log.fine("Exception detected, rollback.");
					transaction.rollback();
				} catch (RuntimeException rbEx) {
					log.fine("Couldn't roll back transaction: "
							+ rbEx.toString());
				}
			}
			throw new AcsJDynConfigFailureEx(
					"Failure while persisting new baci property: "
							+ inData.propertyName + ".", e);
		} finally {
			if (entityManager != null) {
				entityManager.close();
			}
		}
	}

	private Long configureNewMonitorPoint(
			MonitorCharacteristicIDs monitorCharacteristicIDs,
			ComponentData inData) throws AcsJDynConfigFailureEx {
		log.fine("Going to add configuration for monitor point index = "
				+ inData.index + ", of property = " + inData.propertyName);
		Long configurationId = monitorCharacteristicIDs.getConfigurationId();
		Long assemblyId = monitorCharacteristicIDs.getAssemblyId();
		Long componentId = monitorCharacteristicIDs.getComponentId();
		Long propertyId = monitorCharacteristicIDs.getBACIPropertyId();
		String assemblyIDL;
		try {
			assemblyIDL = getComponentIDL(inData.componentName);
		} catch (Exception e) {
			throw new AcsJDynConfigFailureEx(
					"Could not get IDL based on component name. Aborting dynamic configuration",
					e);
		}

		String assemblyTypeName = getAssemblyTypeName(assemblyIDL);

		/*
		 * The next queries and updates must be handled as a transaction since
		 * we can have several blobbers modifying the tables involved in
		 * autoconfiguration (dirty reading can happens). In some sense this is
		 * a nested transaction inside store transaction.
		 */
		EntityManager entityManager = this.myPersistenceLayer
				.getEntityManager();
		EntityTransaction transaction = entityManager.getTransaction();
		transaction.begin();
		/*
		 * Transacction must be enclosed in a try catch statement, to be able to
		 * rollback the transaccion in case
		 */
		try {
			DefaultComponent defaultComponent = getDefaultComponentByLikeAssemblyTypeName(
					entityManager, assemblyTypeName);
			DefaultBACIProperty defaultBACIProperty = getDefaultBACIPropertyByDefaultComponentIdAndPropertyName(
					entityManager, defaultComponent.getDefaultComponentId(),
					inData.propertyName);
			DefaultMonitorPoint defaultMonitorPoint = getDefaultMonitorPointByDefaultBACIPropId(
					entityManager, defaultBACIProperty
							.getDefaultBACIPropertyId(), inData.index);
			persistNewMonitorPoint(entityManager, defaultMonitorPoint,
					propertyId, assemblyId);
			transaction.commit();
			Long newMonitorPointId = getMonitorPointId(entityManager,
					assemblyId, propertyId, inData.index);
			return newMonitorPointId;
		} catch (NonUniqueResultException e) {
			throw new AcsJDynConfigFailureEx(
					"Found multiple default monitor point matching the index"
							+ inData.index + ".", e);
		} catch (NoResultException e) {
			throw new AcsJDynConfigFailureEx(
					"Found no default monitor point matching the given index"
							+ inData.index + ".", e);
		} catch (Exception e) {
			if (transaction != null) {
				try {
					log.fine("Exception detected, rollback.");
					transaction.rollback();
				} catch (RuntimeException rbEx) {
					log.fine("Couldn't roll back transaction: "
							+ rbEx.toString());
				}
			}
			throw new AcsJDynConfigFailureEx(
					"Failure while persisting new monitorpoint: "
							+ inData.propertyName + ".", e);
		} finally {
			if (entityManager != null) {
				entityManager.close();
			}
		}
	}

	private AssemblyType getAssemblyTypeByLikeAssemblyCode(
			String assemblyTypeName, EntityManager entityManager)
			throws NonUniqueResultException, NoResultException {
		Query query = entityManager
				.createNamedQuery("findAssemblyTypeByLikeAssemblyCode");
		query.setParameter("assemblyTypeName", assemblyTypeName);
		AssemblyType assemblyType = (AssemblyType) query.getSingleResult();

		log.fine("Assembly Type name to be associated is "
				+ assemblyType.getAssemblyTypeName());

		return assemblyType;
	}

	private Assembly getAssemblyBySerialNumberAndConfigurationId(
			EntityManager entityManager, Long configurationId,
			String serialNumber) throws NonUniqueResultException,
			NoResultException {
		Query query = entityManager
				.createNamedQuery("findAssemblyBySerialNumberAndConfigurationId");
		query.setParameter("serialNumber", serialNumber.toUpperCase());
		query.setParameter("configurationId", configurationId);
		Assembly assembly = (Assembly) query.getSingleResult();

		return assembly;
	}

	private DefaultComponent getDefaultComponentByLikeAssemblyTypeName(
			EntityManager entityManager, String assemblyTypeName)
			throws NonUniqueResultException, NoResultException {
		Query query = entityManager
				.createNamedQuery("findDefaultComponentByLikeAssemblyTypeName");
		query.setParameter("assemblyTypeName", assemblyTypeName);
		DefaultComponent defaultComponent = (DefaultComponent) query
				.getSingleResult();

		log.fine("Default component: " + defaultComponent.getAssemblyName());

		return defaultComponent;
	}

	private ComponentType getComponentTypeByLikeIDL(
			EntityManager entityManager, String assemblyIDL)
			throws NonUniqueResultException, NoResultException {
		Query query = entityManager
				.createNamedQuery("findComponentTypeBylikeIDL");
		query.setParameter("idl", assemblyIDL);
		ComponentType componentType = (ComponentType) query.getSingleResult();

		log.fine("Component Type with idl " + componentType.getIdl()
				+ " associated with component.");

		return componentType;
	}

	private DefaultBACIProperty getDefaultBACIPropertyByDefaultComponentIdAndPropertyName(
			EntityManager entityManager, Long defaultComponentId,
			String propertyName) throws NonUniqueResultException,
			NoResultException {
		Query baciQuery = entityManager
				.createNamedQuery("findDefaultBACIPropertyByDefaultComponentId");
		baciQuery.setParameter("defaultComponentId", defaultComponentId);
		baciQuery.setParameter("propertyName", propertyName);
		DefaultBACIProperty defaultBACIProperty = (DefaultBACIProperty) baciQuery
				.getSingleResult();

		return defaultBACIProperty;
	}

	private Component getComponentByComponentNameAndConfigurationId(
			EntityManager entityManager, Long configurationId,
			String componentName) throws NonUniqueResultException,
			NoResultException {
		Query query = entityManager
				.createNamedQuery("findComponentByComponentName");
		query.setParameter("componentName", componentName);
		query.setParameter("configurationId", configurationId);
		Component component = (Component) query.getSingleResult();

		return component;
	}

	private DefaultMonitorPoint getDefaultMonitorPointByDefaultBACIPropId(
			EntityManager entityManager, Long defaultBACIPropId, int index)
			throws NoResultException, NonUniqueResultException {
		Query monitorQuery = entityManager
				.createNamedQuery("findDefaultMonitorPointListByDefaultBACIPropId");
		monitorQuery.setParameter("defaultBACIPropertyId", defaultBACIPropId);
		monitorQuery.setParameter("index", index);
		DefaultMonitorPoint defaultMonitorPoint = (DefaultMonitorPoint) monitorQuery
				.getSingleResult();

		return defaultMonitorPoint;
	}

	private void persistNewAssembly(EntityManager entityManager,
			AssemblyType assemblyType, Long configurationId,
			ComponentData inData) {
		Assembly assembly = new Assembly();
		// assembly.setAssemblyId(0L);
		assembly.setAssemblyTypeName(assemblyType.getAssemblyTypeName());
		assembly.setConfigurationId(configurationId);
		assembly.setSerialNumber(inData.serialNumber);
                assembly.setData(null);

		entityManager.persist(assembly);

		log.fine("Assembly " + assemblyType.getAssemblyTypeName()
				+ "with serial number " + inData.serialNumber
				+ " was added to configuration id " + configurationId);
	}

	private void persistNewComponent(EntityManager entityManager,
			Long configurationId, ComponentType componentType,
			ComponentData inData, DefaultComponent defaultComponent) {
		Component component = new Component();
                String tokens[] = ComponentNameHelper.getPathAndName(inData.componentName);
		// component.setComponentId(0L);
		component.setComponentTypeId(componentType.getComponentTypeId());
		component.setComponentName(tokens[1]);
		component.setConfigurationId(configurationId);
		component.setContainerId(null);
		component.setImplLang(defaultComponent.getImplLang());
		component.setRealTime(defaultComponent.getRealTime());
		component.setCode(defaultComponent.getCode());
		component.setPath(tokens[0]);
		component.setIsAutoStart(defaultComponent.getIsAutoStart());
		component.setIsDefault(defaultComponent.getIsDefault());
		component.setIsStandaloneDefined(defaultComponent
				.getIsStandaloneDefined());
                component.setIsControl(1);
		component.setKeepAliveTime(defaultComponent.getKeepAliveTime());
		component.setMinLogLevel(defaultComponent.getMinLogLevel());
		component.setMinLogLevelLocal(defaultComponent.getMinLogLevelLocal());

		entityManager.persist(component);

		log.fine("Component " + inData.componentName
				+ " was added to configuration id " + configurationId);
	}

	private void persistNewBACIProperty(EntityManager entityManager,
			DefaultBACIProperty defaultBACIProperty, Long componentId) {
		BACIProperty baciProperty = new BACIProperty();
		// baciProperty.setBACIPropertyId(0L);
		baciProperty.setComponentId(componentId);
		baciProperty.setPropertyName(defaultBACIProperty.getPropertyName());
		baciProperty.setDescription(defaultBACIProperty.getDescription());
		baciProperty.setIsSequence(1);
		baciProperty.setFormat(defaultBACIProperty.getFormat());
		baciProperty.setUnits(defaultBACIProperty.getUnits());
		baciProperty.setResolution(defaultBACIProperty.getResolution());
		baciProperty.setArchiveMinInt(defaultBACIProperty.getArchiveMinInt());
		baciProperty.setArchiveMaxInt(defaultBACIProperty.getArchiveMaxInt());
		baciProperty.setDefaultTimerTrig(defaultBACIProperty
				.getDefaultTimerTrig());
		baciProperty.setMinTimerTrig(defaultBACIProperty.getMinTimerTrig());
		baciProperty.setArchiveDelta(defaultBACIProperty.getArchiveDelta());
		baciProperty.setInitializeDEVIO(defaultBACIProperty
				.getInitializeDEVIO());
		baciProperty.setMinDeltaTrig(defaultBACIProperty.getMinDeltaTrig());
		baciProperty.setDefaultValue(defaultBACIProperty.getDefaultValue());
		baciProperty.setArchivePriority(defaultBACIProperty
				.getArchivePriority());

		entityManager.persist(baciProperty);

		log.fine("BACI Property " + defaultBACIProperty.getPropertyName()
				+ " was added to configuration .");
	}

	private void persistNewMonitorPoint(EntityManager entityManager,
			DefaultMonitorPoint defaultMonitorPoint, Long baciPropertyId,
			Long assemblyId) {
		MonitorPoint monitorPoint = new MonitorPoint();
		monitorPoint.setBaciPropertyId(baciPropertyId);
		monitorPoint.setMonitorPointName(defaultMonitorPoint
				.getMonitorPointName());
		monitorPoint.setAssemblyId(assemblyId);
		monitorPoint.setIndex(defaultMonitorPoint.getIndex());
		monitorPoint.setDatatype(defaultMonitorPoint.getDatatype());
		monitorPoint.setRCA(defaultMonitorPoint.getRCA());
		monitorPoint.setTERelated(defaultMonitorPoint.getTERelated());
		monitorPoint.setRawDatatype(defaultMonitorPoint.getRawDatatype());
		monitorPoint.setWorldDatatype(defaultMonitorPoint.getWorldDatatype());
		monitorPoint.setUnits(defaultMonitorPoint.getUnits());
		monitorPoint.setScale(defaultMonitorPoint.getScale());
		monitorPoint.setOffset(defaultMonitorPoint.getOffset());
		monitorPoint.setMinRange(defaultMonitorPoint.getMinRange());
		monitorPoint.setMaxRange(defaultMonitorPoint.getMinRange());
		monitorPoint.setDescription(defaultMonitorPoint.getDescription());

		entityManager.persist(monitorPoint);

		log.fine("Monitor Point " + defaultMonitorPoint.getMonitorPointName()
				+ " has been configured.");
	}

	private void persistNewMonitorData(EntityManager entityManager,
			ComponentData inData,
			MonitorCharacteristicIDs monitorCharacteristicIDs) {
		MonitorData monitorData = new MonitorData();

		monitorData.setMonitorPointId(monitorCharacteristicIDs
				.getMonitorPointId());
		monitorData.setStartTime(inData.startTime);
		monitorData.setEndTime(inData.stopTime);
		monitorData.setMonitorTs(new Timestamp(System.currentTimeMillis()));
		monitorData.setSampleSize(inData.sampleSize);
		monitorData.setMonitorClob(inData.clob);

		if (inData.statistics != null) {
			monitorData.setMinStat(inData.statistics.min.doubleValue());
			monitorData.setMaxStat(inData.statistics.max.doubleValue());
			monitorData.setMeanStat(inData.statistics.mean.doubleValue());
			monitorData.setStdDevStat(inData.statistics.stdDev.doubleValue());
		}

		entityManagerStore.persist(monitorData);
	}

	public List getMonitorData(long monitorPointId, Timestamp startTimestamp,
			Timestamp stopTimestamp) {
		EntityManager entityManager = this.myPersistenceLayer
				.getEntityManager();

		Query query = entityManager
				.createNamedQuery("findMonitorDataByMonitorPointIdAndTimestampRange");
		query.setParameter("monitorPointId", monitorPointId);
		query.setParameter("startTimestamp", startTimestamp);
		query.setParameter("stopTimestamp", stopTimestamp);

		return query.getResultList();
	}

	private void send_alarm(String faultFamily, String faultMember,
			int faultCode, boolean active) {
		try {
			ACSAlarmSystemInterface alarmSource = ACSAlarmSystemInterfaceFactory
					.createSource("ALARM_SYSTEM_SOURCES");
			ACSFaultState fs = ACSAlarmSystemInterfaceFactory.createFaultState(
					faultFamily, faultMember, faultCode);
			if (active) {
				fs.setDescriptor(ACSFaultState.ACTIVE);
			} else {
				fs.setDescriptor(ACSFaultState.TERMINATE);
			}

			fs.setUserTimestamp(new Timestamp(System.currentTimeMillis()));

			alarmSource.push(fs);

		} catch (ACSASFactoryNotInitedEx ex) {
			log.severe("Alarm with FF=" + faultFamily + " FM=" + faultMember
					+ " FC=" + faultCode + " could not be thrown. Message="
					+ ex.getMessage());
		} catch (SourceCreationErrorEx ex) {
			log.severe("Alarm with FF=" + faultFamily + " FM=" + faultMember
					+ " FC=" + faultCode + " could not be thrown. Message="
					+ ex.getMessage());
		} catch (FaultStateCreationErrorEx ex) {
			log.severe("Alarm with FF=" + faultFamily + " FM=" + faultMember
					+ " FC=" + faultCode + " could not be thrown. Message="
					+ ex.getMessage());
		}
	}
}
