/*
* Copyright (C) The Community OpenORB Project. All rights reserved.
*
* This software is published under the terms of The OpenORB Community Software
* License version 1.0, a copy of which has been included with this distribution
* in the LICENSE.txt file.
*/
package org.openorb.util.urlhandler.resource;

import java.io.FileNotFoundException;
import java.io.IOException;

import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLConnection;
import java.net.URLStreamHandler;

import org.openorb.util.ExceptionTool;

/**
 * This class is an extension of the URLStreamHandler class. It overrides
 * the method openConnection and adds a level of indirection to the way
 * resources are loaded from a URL.
 *
 * @author  Chris Wood
 * @version $Revision: 1.10 $ $Date: 2002/07/14 19:12:41 $
 */
public class Handler
    extends URLStreamHandler
{
    /**
     * Opens a connection to the specified URL.
     *
     * @param url A URL to open a connection to.
     * @return The established connection.
     * @throws IOException When the file specified by the
     * URL could not be found.
     */
    protected URLConnection openConnection( final URL url )
        throws IOException
    {
        String cln = url.getHost();

        String resrce = url.getFile().substring( 1 );

        URL realURL;

        if ( cln != null && cln.length() != 0 )
        {
            Class clz;

            try
            {
                clz = Thread.currentThread().getContextClassLoader().
                        loadClass( cln );
            }
            catch ( final ClassNotFoundException ex )
            {
                throw ( IOException ) ExceptionTool.initCause(
                      new MalformedURLException( "Class " + cln
                      + " cannot be found (" + ex + ")" ), ex );
            }

            realURL = clz.getResource( resrce );

            if ( realURL == null )
                throw new FileNotFoundException(
                      "Class resource " + resrce + " of class "
                      + cln + " cannot be found" );

        }
        else
        {
            realURL = ClassLoader.getSystemResource( resrce );

            if ( realURL == null )
                throw new FileNotFoundException( "System resource "
                      + resrce + " cannot be found" );
        }
        return realURL.openConnection();
    }
}

