/*
 *    ALMA - Atacama Large Millimiter Array
 *    (c) European Southern Observatory, 2002
 *    Copyright by ESO (in the framework of the ALMA collaboration),
 *    All rights reserved
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License as published by the Free Software Foundation; either
 *    version 2.1 of the License, or (at your option) any later version.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
 *    MA 02111-1307  USA
 *
 *    Created on Sep 14, 2005
 *
 */
 
  
// $Author: hmeuss $
// $Date: 2006/10/26 16:57:10 $
// $Log: ArchiveMasterClient.java,v $
// Revision 1.7  2006/10/26 16:57:10  hmeuss
// Heikos change: workaround for possible problem in state machine listener or whereever
//
// Revision 1.6  2006/10/10 14:59:11  hsommer
// ContainerException now called AcsJContainerServicesEx
//
// Revision 1.5  2006/09/19 15:50:53  swilliam
// Create new archive SysV-style script, retire old archiveStart/archiveStop scripts
// Remove archive dependency startup from ACS component
// Change test dbConfig to port 8180
//
// Revision 1.4  2005/11/21 16:36:23  hmeuss
// improved archiveStart command
//
// Revision 1.3  2005/09/26 15:46:52  hmeuss
// improved shutdown procedure
//
// Revision 1.2  2005/09/15 11:30:24  hmeuss
// manager reference is now fetched in Java client instead of script
//
// Revision 1.1  2005/09/15 09:21:28  hmeuss
// Created new script archiveStart which initializes the master component
// 
 
package alma.archive.manager;

import java.util.logging.Logger;

import org.apache.commons.lang.ArrayUtils;

import alma.ACS.MasterComponent;
import alma.ACS.MasterComponentHelper;
import alma.ACS.ROstringSeq;
import alma.ACS.SUBSYSSTATE_AVAILABLE;
import alma.ACS.SUBSYSSTATE_ERROR;
import alma.ACS.SUBSYSSTATE_INITIALIZING_PASS1;
import alma.ACS.SUBSYSSTATE_OFFLINE;
import alma.ACS.SUBSYSSTATE_ONLINE;
import alma.ACS.SUBSYSSTATE_PREINITIALIZED;
import alma.ACS.SUBSYSSTATE_SHUTDOWN;
import alma.ACS.MasterComponentImpl.StateChangeListener;
import alma.ACS.MasterComponentImpl.StateChangeSemaphore;
import alma.ACS.MasterComponentPackage.SubsystemStateEvent;
import alma.JavaContainerError.wrappers.AcsJContainerServicesEx;
import alma.acs.component.client.ComponentClient;
import alma.acs.genfw.runtime.sm.AcsStateUtil;

/**
 * @author hmeuss
 *
 */
public class ArchiveMasterClient extends ComponentClient {

	private static String masterName = "ARCHIVE_MASTER_COMP";

	public static final String[] shutdownHierarchy = new String[] {SUBSYSSTATE_AVAILABLE.value, SUBSYSSTATE_OFFLINE.value, SUBSYSSTATE_SHUTDOWN.value};
	public static final String[] initializingPass1Hierarchy = new String[] {SUBSYSSTATE_AVAILABLE.value, SUBSYSSTATE_OFFLINE.value, SUBSYSSTATE_INITIALIZING_PASS1.value};
	public static final String[] preInitHierarchy = new String[] {SUBSYSSTATE_AVAILABLE.value, SUBSYSSTATE_OFFLINE.value, SUBSYSSTATE_PREINITIALIZED.value};
	public static final String[] onlineHierarchy = new String[] {SUBSYSSTATE_AVAILABLE.value, SUBSYSSTATE_ONLINE.value};
	public static final String[] errorHierarchy = new String[] {SUBSYSSTATE_AVAILABLE.value, SUBSYSSTATE_ERROR.value};

	/**
	 * @param logger
	 * @param managerLoc
	 * @param clientName
	 * @throws Exception
	 */
	public ArchiveMasterClient(Logger logger, String managerLoc, String clientName) throws Exception {
		super(logger, managerLoc, clientName);
	}

	/**
	 * The main method calls INITPASS1 and INITPASS2 on the Archive master component by default.
	 * If arguments are given, these arguments specify, which transition is performed.
	 * Example: java ArchiveSubsystemMasterImpl INITPASS2
	 * 
	 * @param args
	 * @throws Exception
	 */
		public static void main(String[] args) throws Exception {
			
			if (args.length>1) {
				masterName=args[1];
			}

			System.out.println("Connecting to "+masterName);
			
			// get location of manager
			String managerLoc = null;
			if (System.getProperty("ACS.manager") != null)
			{
				managerLoc = System.getProperty("ACS.manager").trim();
			}
			else
			{
				System.out.println("ERROR: You must specify MANAGER_REFERENCE");
			}
					
			// construct client object:
			ArchiveMasterClient client = new ArchiveMasterClient(null, managerLoc, "ArchiveMasterClient");
			
			
			// get reference to Archive subsystem master:
			//ArchiveSubsystemMasterIF master=null;
			MasterComponent master=null; 
			if (client.getContainerServices().findComponents(masterName, null).length > 0) {
				try {
					master = MasterComponentHelper.narrow(client.getContainerServices().getComponent(masterName));
				} catch (AcsJContainerServicesEx e) {
					client.m_logger.severe(e.getContextInfo());
					System.out.println("ERROR: Could not connect to "+masterName+":\n"+e.toString());
				}
			} else {

				System.out.println("ERROR: Did not find component"+masterName);
			}
						
			if (args.length>0) {
				if  (args[0].equalsIgnoreCase("INITPASS1")) {
					master.doTransition(SubsystemStateEvent.SUBSYSEVENT_INITPASS1);
				} else if (args[0].equalsIgnoreCase("INITPASS2")) {
					master.doTransition(SubsystemStateEvent.SUBSYSEVENT_INITPASS2);
				} else if (args[0].equalsIgnoreCase("REINIT")) {
					master.doTransition(SubsystemStateEvent.SUBSYSEVENT_REINIT);
				} else if (args[0].equalsIgnoreCase("START")) {
					master.doTransition(SubsystemStateEvent.SUBSYSEVENT_START);
				} else if (args[0].equalsIgnoreCase("STOP")) {
					master.doTransition(SubsystemStateEvent.SUBSYSEVENT_STOP);
				} else if (args[0].equalsIgnoreCase("SHUTDOWNPASS1")) {
					master.doTransition(SubsystemStateEvent.SUBSYSEVENT_SHUTDOWNPASS1);
				} else if (args[0].equalsIgnoreCase("SHUTDOWNPASS2")) {
					master.doTransition(SubsystemStateEvent.SUBSYSEVENT_SHUTDOWNPASS2);
				} else if (args[0].equalsIgnoreCase("ERROR")) {
					master.doTransition(SubsystemStateEvent.SUBSYSEVENT_ERROR);
				} else {
					System.out.println("Unknown event " + args[0] + " will be ignored!");
				}
			} 
			else {

				ROstringSeq statesProperty = master.currentStateHierarchy();

				StateChangeListener listener = new StateChangeListener(client.m_logger);		
				listener.createMonitor(statesProperty, client.getContainerServices());

				StateChangeSemaphore sync = listener.getStateChangeSemaphore();


				if (!listener.verifyCurrentState(shutdownHierarchy)) {
					throw new IllegalStateException("The ARCHIVE subsystem was not in the expected SHUTDOWN state.");
				}				
				
				sync.reset();
				master.doTransition(SubsystemStateEvent.SUBSYSEVENT_INITPASS1);
				sync.waitForStateChanges(2);
				
				// Now we ought to be in SUBSYSSTATE_PREINITIALIZED if there had actually been two valid notifications.
				// Due to some unknown problem (2006-10), sometimes we receive the state change notification too early (or too many of them).
				// Therefore as a workaround, we wait till the state is right.
				while (true) {
					String[] actualState = listener.getCurrentState();
					if (ArrayUtils.isEquals(shutdownHierarchy, actualState) ||
						ArrayUtils.isEquals(initializingPass1Hierarchy, actualState)) {
						// this is the dreaded case where we got notified too early
						try {
							System.out.println("Mastercomponent is still in state " + AcsStateUtil.stateHierarchyNamesToString(actualState) + "....");
							Thread.sleep(1000);
						} catch (InterruptedException ex) {
							; // nothing
						}
					}
					// checking for ERROR state must always be done, regardless of the above workaround
					else if (ArrayUtils.isEquals(errorHierarchy, actualState)) {
						throw new Exception("Found ARCHIVE master component in ERROR state. You may try to send it a REINIT.");
					}
					else if (!ArrayUtils.isEquals(preInitHierarchy, actualState)) {
						throw new Exception("Found ARCHIVE master component in state " + AcsStateUtil.stateHierarchyNamesToString(actualState) +
								" when " + AcsStateUtil.stateHierarchyNamesToString(preInitHierarchy) + " was expected.");
					}
					else {
						break;
					}
				}

				// now we are in SUBSYSSTATE_PREINITIALIZED as it should be.
				// TODO: add similar checks as for the above transition
				
				sync.reset();
				master.doTransition(SubsystemStateEvent.SUBSYSEVENT_INITPASS2);
				sync.waitForStateChanges(2);
				
				// now we should be ONLINE
				
				System.out.println("Initialized Archive subsystem.");
			}
		}
		
}
