/*
* Copyright (C) The Community OpenORB Project. All rights reserved.
*
* This software is published under the terms of The OpenORB Community Software
* License version 1.0, a copy of which has been included with this distribution
* in the LICENSE.txt file.
*/
package org.openorb.iiop;

import java.util.List;
import java.util.ArrayList;

import org.apache.avalon.framework.logger.LogEnabled;
import org.apache.avalon.framework.logger.AbstractLogEnabled;

import org.openorb.net.ServerManager;
import org.openorb.net.ClientManager;
import org.openorb.net.TransportClientInitializer;

import org.openorb.PI.FeatureInitializer;
import org.openorb.PI.FeatureInitInfo;
import org.openorb.PI.CodecFactoryManager;
import org.openorb.PI.SimpleIORInterceptor;
import org.openorb.util.Trace;

import org.openorb.util.ExceptionTool;

import org.omg.PortableInterceptor.ORBInitInfo;
import org.omg.PortableInterceptor.ServerRequestInterceptor;
import org.omg.PortableInterceptor.ServerRequestInfo;
import org.omg.PortableInterceptor.ForwardRequest;
import org.omg.IOP.TaggedComponent;
import org.omg.IOP.TAG_CODE_SETS;
import org.omg.IOP.TAG_INTERNET_IOP;
import org.omg.CONV_FRAME.CodeSetComponentInfo;
import org.omg.CONV_FRAME.CodeSetComponentInfoHelper;
import org.omg.IIOP.ListenPoint;
import org.omg.IIOP.ListenPointHelper;

/**
 *
 * @author Chris Wood
 * @version $Revision: 1.21 $ $Date: 2002/07/14 19:05:40 $ 
 */
public class IIOPProtocolInitializer 
    extends AbstractLogEnabled 
    implements FeatureInitializer
{
    public IIOPProtocolInitializer()
    {
    }

    public void init( ORBInitInfo orbinfo, FeatureInitInfo featureinfo )
    {

        if ( getLogger().isErrorEnabled() && Trace.isLow() ) 
            getLogger().debug( "init" );

        // initialize the codeset database
        CodeSetDatabaseInitializer csdb = new CodeSetDatabaseInitializer();
        csdb.enableLogging( getLogger().getChildLogger( "csdb" ) );
        csdb.initialize();
        featureinfo.setFeature( "CodeSetDatabase", csdb );

        org.omg.CORBA.ORB orb = featureinfo.orb();

        // initialize codec factory.
        CDRCodecFactory codec_factory = new CDRCodecFactory( orb );

        org.omg.IOP.Codec codec = null;

        CodecFactoryManager cfm = ( CodecFactoryManager )
                featureinfo.getFeature( "CodecFactoryManager" );

        if ( cfm != null )
        {
            cfm.register_codec_factory( new org.omg.IOP.Encoding( 
              org.omg.IOP.ENCODING_CDR_ENCAPS.value, ( byte ) 1, ( byte ) 0 ), codec_factory );
            cfm.register_codec_factory( new org.omg.IOP.Encoding( 
              org.omg.IOP.ENCODING_CDR_ENCAPS.value, ( byte ) 1, ( byte ) 1 ), codec_factory );
            org.omg.IOP.Encoding enc = new org.omg.IOP.Encoding( 
              org.omg.IOP.ENCODING_CDR_ENCAPS.value, ( byte ) 1, ( byte ) 2 );
            cfm.register_codec_factory( enc, codec_factory );

            try
            {
                codec = codec_factory.create_codec( enc );
            }
            catch (final Exception ex)
            {
                getLogger().error("Error during create_codec().", ex);
                return ;
            }
        }

        ServerManager svrmgr = ( ServerManager ) featureinfo.getFeature( "ServerCPCManager" );
        ClientManager cltmgr = ( ClientManager ) featureinfo.getFeature( "ClientCPCManager" );

        ListenPoint [] biDirListenPoints = null;

        org.openorb.CORBA.kernel.ORBLoader loader = featureinfo.getLoader();

        IIOPServerProtocol svrproto = null;

        if ( svrmgr != null )
        {
            // create codeset component IOR interceptor
            org.omg.CORBA.Any any = orb.create_any();

            CodeSetComponentInfo codesetInfo = csdb.getServerCodeSets();
            CodeSetComponentInfoHelper.insert( any, codesetInfo );

            List iorComponents = new ArrayList();

            try
            {
                byte buf[] = codec.encode_value( any );

                if ( buf != null )
                    iorComponents.add( new TaggedComponent( TAG_CODE_SETS.value, buf ) );
            }
            catch (final org.omg.IOP.CodecPackage.InvalidTypeForEncoding ex)
            {
                final String error = "Unable to encode code set component.";
                getLogger().error(error, ex);
                throw ExceptionTool.initCause(new org.omg.CORBA.INITIALIZE(error), ex);
            }

            // get alternate IIOP endpoints to publish in IORs
            java.util.Iterator itt = loader.properties( "iiop.alternateAddr" );

            while ( itt.hasNext() )
            {
                String hostport = ( ( org.openorb.CORBA.kernel.Property ) itt.next() ).getValue();
                int idx = hostport.indexOf( ':' );

                if ( idx < 0 )
                    continue;

                String host = hostport.substring( 0, idx );

                int port;

                try
                {
                    port = Integer.parseInt( hostport.substring( idx + 1 ) );
                }
                catch (final NumberFormatException ex)
                {
                    continue;
                }

                if ( port > 0xFFFF )
                    continue;

                ListenPoint lp = new ListenPoint( host, ( short ) port );

                ListenPointHelper.insert( any, lp );

                try
                {
                    byte buf[] = codec.encode_value( any );

                    if ( buf != null )
                        iorComponents.add( new TaggedComponent(
                                org.omg.IOP.TAG_ALTERNATE_IIOP_ADDRESS.value, buf ) );
                }
                catch (final org.omg.IOP.CodecPackage.InvalidTypeForEncoding ex)
                {
                    final String error = "Unable to encode alternate endpoint component.";
                    getLogger().error(error, ex);
                    throw ExceptionTool.initCause(new org.omg.CORBA.INITIALIZE(error), ex);
                }
            }

            // create IOR interceptor for assigning components
            if ( !iorComponents.isEmpty() )
            {
                TaggedComponent [] altAddrComponents = new TaggedComponent[ iorComponents.size() ];
                iorComponents.toArray( altAddrComponents );

                try
                {
                    orbinfo.add_ior_interceptor( new SimpleIORInterceptor( "codesets and alts",
                                                 TAG_INTERNET_IOP.value, altAddrComponents ) );
                }
                catch ( org.omg.PortableInterceptor.ORBInitInfoPackage.DuplicateName ex )
                {
                    // ignore.
                }

            }

            boolean listen = true;

            // setup server interceptor for activating bidir.
            if ( loader.getBooleanProperty( "iiop.allowBiDir", true ) )
            {
                try
                {
                    orbinfo.add_server_request_interceptor( BIDIR_INTERCEPTOR );
                }
                catch ( org.omg.PortableInterceptor.ORBInitInfoPackage.DuplicateName ex )
                {
                    // ignore.
                }

                listen = !loader.getBooleanProperty( "iiop.biDirOnlyServer", false );
            }

            IIOPTransportServerInitializer svrInit;

            try
            {
                svrInit = ( IIOPTransportServerInitializer ) loader.constructClass( 
                  "iiop.TransportServerInitializerClass", 
                  "org.openorb.iiop.IIOPTransportServerInitializer",
                  new Object [] { orbinfo, featureinfo }, 
                  new Class[] { ORBInitInfo.class, FeatureInitInfo.class } );

                if ( svrInit instanceof LogEnabled )
                {
                    ((LogEnabled) svrInit).enableLogging(getLogger());
                }
            }
            catch (final java.lang.reflect.InvocationTargetException ex )
            {
                final String error = "Unable to initialize TransportServerInitializer.";

                final Throwable ex1 = ex.getTargetException();

                getLogger().error(error, ex1);

                if (ex1 instanceof Error)
                {
                    throw (Error) ex1;
                }
                
                if (ex1 instanceof RuntimeException)
                {
                    throw (RuntimeException) ex1;
                }

                throw ExceptionTool.initCause(new org.omg.CORBA.INITIALIZE( 
                      "Unable to initialize TransportServerInitializer ("
                      + ex1 + ")", 0, org.omg.CORBA.CompletionStatus.COMPLETED_NO), ex1);
            }

            if ( listen )
                svrInit.open();

            svrproto = new IIOPServerProtocol( svrmgr, codec_factory, svrInit );

            svrmgr.register_protocol( org.omg.IOP.TAG_INTERNET_IOP.value, svrproto );

            if ( listen )
                svrproto.open();

            // get the bidir listen points so the client side can sent them in
            // the bidir service context.
            biDirListenPoints = svrInit.getBiDirEndpoints();
        }

        IIOPClientProtocol cltproto = null;

        if ( cltmgr != null )
        {
            // initialize client side.

            TransportClientInitializer cltInit;

            try
            {
                cltInit = ( TransportClientInitializer ) loader.constructClass(
                        "iiop.TransportClientInitializerClass",
                        "org.openorb.iiop.IIOPTransportClientInitializer",
                        new Object [] { orbinfo, featureinfo },
                        new Class[] { ORBInitInfo.class, FeatureInitInfo.class } );

                if ( cltInit instanceof LogEnabled )
                {
                    ((LogEnabled) cltInit).enableLogging(getLogger());
                }
            }
            catch ( java.lang.reflect.InvocationTargetException ex )
            {
                final String error = "Unable to initialize TransportClientInitializer.";

                getLogger().error(error, ex.getTargetException());

                throw ExceptionTool.initCause(new org.omg.CORBA.INITIALIZE( 
                        error + " (" + ex.getTargetException() + ")", 0, 
                        org.omg.CORBA.CompletionStatus.COMPLETED_NO), ex);
            }

            cltproto = new IIOPClientProtocol( cltmgr, codec_factory,
                    csdb.getClientCodeSets(), cltInit, biDirListenPoints );
            cltmgr.register_protocol( org.omg.IOP.TAG_INTERNET_IOP.value, cltproto );

            // setup bidirectional interceptor / access.
            if ( svrproto != null )
            {
                cltproto.setServerProtocol( svrproto );
                svrproto.setClientProtocol( cltproto );
            }
        }
    }


    private static final ServerRequestInterceptor BIDIR_INTERCEPTOR =
            new BiDirServerInterceptor();

    private static final class BiDirServerInterceptor
        extends org.omg.CORBA.LocalObject
        implements ServerRequestInterceptor
    {
        public String name()
        {
            return "IIOPBiDirServerInterceptor";
        }

        public void receive_request_service_contexts( ServerRequestInfo ri )
            throws ForwardRequest
        {
        }

        public void receive_request( ServerRequestInfo ri )
            throws ForwardRequest
        {
            if ( ri instanceof IIOPServerRequest )
            {
                IIOPServerRequest request = ( IIOPServerRequest ) ri;
                IIOPServerChannel channel = ( IIOPServerChannel ) request.channel();

                channel.checkBiDirActivation( request );
            }
        }

        public void send_other( org.omg.PortableInterceptor.ServerRequestInfo ri )
            throws ForwardRequest
        {
        }

        public void send_reply( org.omg.PortableInterceptor.ServerRequestInfo ri )
        {
        }

        public void send_exception( org.omg.PortableInterceptor.ServerRequestInfo ri )
            throws ForwardRequest
        {
        }

        public void destroy()
        {
        }
    }
}

