/*
 *    ALMA - Atacama Large Millimiter Array
 *    (c) European Southern Observatory, 2002
 *    Copyright by ESO (in the framework of the ALMA collaboration),
 *    All rights reserved
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License as published by the Free Software Foundation; either
 *    version 2.1 of the License, or (at your option) any later version.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
 *    MA 02111-1307  USA
 *
 */
package alma.archive.idregistry.server;

import alma.archive.idregistry.exceptions.DatabaseException;
import alma.archive.idregistry.exceptions.DocumentDoesNotExistException;
import alma.archive.idregistry.exceptions.PermissionDeniedException;
import alma.archive.idregistry.shared.ArchiveDescriptor;
import alma.userrepository.errors.UserRepositoryException;

public interface ArchiveIdRegistry {
    /**
     * Returns an ArchiveID for the given descriptor. The descriptor holds
     * information about a particular instance of the archive. The returned
     * value is unique between archives. The descriptor is stored in the archive
     * once the ID has been associated with it.
     *
     * @param archiveDescriptor
     *            the ArchiveDescriptor of the archive applying for an ID
     * @param username
     *            The user who is performing this operation
     * @param password
     *            The password for the given username
     * @return an Archive ID of the format 000
     * @throws UserRepositoryException if a problem occurred communicating with the user repository
     * @throws DatabaseException if a problem occurred storing the ArchiveDescriptor
     * @throws PermissioDeniedException if the user is not authenticated or does not have sufficient privileges to store an ArchiveDescriptor
     */
    public String getArchiveId(ArchiveDescriptor archiveDescriptor,
            String username, String password) throws UserRepositoryException, DatabaseException,
            PermissionDeniedException;
    
    /**
     * Return the Describing XML for a particular Archive Identifier. Given a
     * particular archive id queries the Archive for that descriptor. The
     * username will be used to check that the user has permissions to access
     * the descriptor.
     *
     * @param archiveID
     *            The archive ID used to perform the query
     * @param username
     *            The username for permissions
     * @return Returns the archive descriptor for the given archive ID
     * @throws UserRepositoryException if a problem occurred communicating with the user repository
     * @throws DatabaseException if a problem occurred retrieving the ArchiveDescriptor
     * @throws PermissionDeniedException if the user is not authenticated or does not have sufficient privileges to retrieve an ArchiveDescriptor
     */
    public ArchiveDescriptor getArchiveDescriptor(String archiveId, String username, String password)
    throws PermissionDeniedException, UserRepositoryException, DatabaseException, DocumentDoesNotExistException;
}
