/*
 * 	  Created on 09-Sep-2003
 * 
 *    ALMA - Atacama Large Millimiter Array
 *    (c) European Southern Observatory, 2002
 *    Copyright by ESO (in the framework of the ALMA collaboration),
 *    All rights reserved
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License as published by the Free Software Foundation; either
 *    version 2.1 of the License, or (at your option) any later version.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
 *    MA 02111-1307  USA
 */
package alma.archive.database.xmldb;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.StringReader;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.ListIterator;
import java.util.logging.Logger;

import org.jdom.Attribute;
import org.jdom.CDATA;
import org.jdom.Comment;
import org.jdom.Document;
import org.jdom.Element;
import org.jdom.JDOMException;
import org.jdom.ProcessingInstruction;
import org.jdom.Text;
import org.jdom.input.SAXBuilder;
import org.jdom.output.Format;
import org.jdom.output.XMLOutputter;
import org.jdom.xpath.XPath;
import org.xmldb.api.base.XMLDBException;

import alma.archive.database.helpers.DBConfiguration;
import alma.archive.database.helpers.DatabaseHelper;
import alma.archive.database.interfaces.DBCursor;
import alma.archive.database.interfaces.InternalIF;
import alma.archive.database.interfaces.InternalIFFactory;
import alma.archive.database.interfaces.SchemaManager;
import alma.archive.database.interfaces.UserManager;
import alma.archive.exceptions.ArchiveException;
import alma.archive.exceptions.ModuleCriticalException;
import alma.archive.exceptions.access.EntityDirtyException;
import alma.archive.exceptions.access.PermissionDeniedException;
import alma.archive.exceptions.general.DatabaseException;
import alma.archive.exceptions.general.EntityAlreadyDeletedException;
import alma.archive.exceptions.general.EntityDoesNotExistException;
import alma.archive.exceptions.general.EntityExistsException;
import alma.archive.exceptions.general.EntityUndeletedException;
import alma.archive.exceptions.general.HistoryInconsistencyException;
import alma.archive.exceptions.general.IllegalTimestampException;
import alma.archive.exceptions.general.UndefinedNamespaceException;
import alma.archive.exceptions.general.UnknownSchemaException;
import alma.archive.exceptions.syntax.IllegalHistoryNumberException;
import alma.archive.exceptions.syntax.MalformedPermissionsException;
import alma.archive.exceptions.syntax.MalformedQueryException;
import alma.archive.exceptions.syntax.MalformedURIException;
import alma.archive.exceptions.syntax.MalformedXMLException;
import alma.archive.exceptions.syntax.MalformedXPointerException;
import alma.archive.exceptions.syntax.UnderspecifiedQueryException;
import alma.archive.exceptions.syntax.UnknownFlagException;
import alma.archive.exceptions.user.UserDoesNotExistException;
import alma.archive.wrappers.ArchiveTimeStamp;
import alma.archive.wrappers.DocumentData;
import alma.archive.wrappers.Permissions;

import alma.acs.container.archive.Range;
import alma.ArchiveIdentifierError.wrappers.AcsJArchiveIdentifierErrorEx;
import alma.ArchiveIdentifierError.wrappers.AcsJIdentifierUnexpectedEx;
import alma.ArchiveIdentifierError.wrappers.AcsJRangeExhaustedEx;

/**
 * @author simon XML:DB implementation of the internal interface, more detailed
 *         documentation of the interface cna be found in the parent class.
 */
public class XmldbInternalIF implements InternalIF {
	private Logger logger = null;

	private XmldbDatabase database = null;

	private static XmldbInternalIF _instance = null;

	private static XmldbSchemaManager smanager = null;

	private static int access_count = 0;

	/**
	 * Return an instance of the InternalIF, create the class if necessary
	 * 
	 * @param logger
	 * @return
	 * @throws DatabaseException
	 */
	public static InternalIF instance(Logger logger) throws DatabaseException {
		if (_instance == null) {
			XmldbDatabase db;
			try {
				db = XmldbDatabase.instance(logger);
			} catch (XMLDBException e) {
				logger.severe(e.getLocalizedMessage());
				throw new DatabaseException(e);
			}
			_instance = new XmldbInternalIF(db, logger);
		}
		access_count++;
		return _instance;
	}

	/**
	 * @see alma.archive.database.interfaces.InternalIF#init()
	 */
	public void init() throws DatabaseException {
		smanager.refresh();
	}

	/**
	 * Closes the class and frees resources
	 */
	public void close() throws DatabaseException {
		access_count--;
		if (access_count == 0) {
			smanager.close();
			database.close();
			_instance = null;
		}
	}

	/**
	 * Create an instance of the class, connects to the database and the test
	 * manager.
	 * 
	 * @param logger
	 * @throws DatabaseException
	 */
	protected XmldbInternalIF(XmldbDatabase db, Logger logger)
			throws DatabaseException {
		this.logger = logger;

		database = db;
		smanager = XmldbSchemaManager.instance(logger);
	}

	/**
	 * @see alma.archive.database.interfaces.InternalIF
	 */
	public void store(URI uri, String xml, URI schema, String schemaName,
			String owner, Permissions permissions, String user, boolean emitLogs)
			throws DatabaseException, ArchiveException, MalformedURIException,
			MalformedXMLException, EntityExistsException,
			UnknownSchemaException, UserDoesNotExistException,
			MalformedPermissionsException, ModuleCriticalException {

		if (!DatabaseHelper.checkArchiveIdStoragePermission(uri,
				InternalIFFactory.getIdentifierManager(logger))) {
			// TODO define new exception for this case
			throw new ArchiveException("Cannot store document with UID "
					+ uri.toString()
					+ ": does not match with ArchiveId: "
					+ InternalIFFactory.getIdentifierManager(logger)
							.getArchiveId());
		}

		// TODO: check permissions and owner/user
		if (smanager.verifySchema(schema, schemaName)) {
			DocumentDataXml dd = new DocumentDataXml(uri, schema, schemaName,
					owner, permissions);
			database.putDescriptor(uri, dd.toXmlString());
			database.putLatest(uri, xml);
		}
	}

	/**
	 * @see alma.archive.database.interfaces.InternalIF
	 */
	public void update(URI uri, ArchiveTimeStamp timeStamp, String xml,
			URI schema, boolean force, String user)
			throws PermissionDeniedException, DatabaseException,
			ArchiveException, MalformedURIException,
			HistoryInconsistencyException, MalformedXMLException,
			EntityDoesNotExistException, UserDoesNotExistException,
			UnknownSchemaException {
		// TODO: Check user permissions
		String schemaName = smanager.getSchemaName(schema);
		DocumentDataXml dd = database.getDescriptor(uri);

		if (!force) {
			ArchiveTimeStamp latestTimeStamp = dd.getLatestTimeStamp();
			if (!timeStamp.equals(latestTimeStamp))
				throw new HistoryInconsistencyException("Presented timestamp: "
						+ timeStamp.toISOString()
						+ " does not equal latest timestamp: "
						+ latestTimeStamp.toISOString());
		}

		// get the old document out, and into the hisory
		String oldLatest = database.getlatest(uri);
		database.putDocuments(uri, dd.getLatestTimeStamp(), oldLatest);
		database.delLatest(uri);

		dd.addLatest();
		dd.setSchema(schema, schemaName);
		database.putLatest(uri, xml);

		database.delDescriptor(uri);
		database.putDescriptor(uri, dd.toXmlString());
	}

	/**
	 * @see alma.archive.database.interfaces.InternalIF
	 */
	public String get(URI uri, String user) throws PermissionDeniedException,
			EntityDirtyException, EntityDoesNotExistException,
			DatabaseException, ArchiveException, UserDoesNotExistException,
			MalformedURIException {
		if (!database.exists(uri))
			throw new EntityDoesNotExistException("The URI: "
					+ uri.toASCIIString() + " was not found in the database");
		else {
			DocumentDataXml dd = database.getDescriptor(uri);
			if (dd.isDeleted())
				throw new EntityDoesNotExistException("The entity: "
						+ uri.toASCIIString() + " is flagged as deleted");

			if (dd.isDirty())
				throw new EntityDirtyException("The entity: "
						+ uri.toASCIIString() + " is flagged as dirty");

			// need to get from user to role, here or inside
			if (!dd.checkReadPermision(user))
				throw new PermissionDeniedException(user
						+ " does not have permission to access: "
						+ uri.toASCIIString());
			return database.getlatest(uri);
		}
	}

	/**
	 * @see alma.archive.database.interfaces.InternalIF
	 */
	public String[] get(URI uri, String xpath, HashMap namespaces, String user)
			throws PermissionDeniedException, EntityDirtyException,
			EntityDoesNotExistException, DatabaseException, ArchiveException,
			MalformedURIException, IllegalHistoryNumberException,
			UserDoesNotExistException, MalformedXPointerException,
			UndefinedNamespaceException {
		if (!database.exists(uri))
			throw new EntityDoesNotExistException("The URI: "
					+ uri.toASCIIString() + " was not found in the database");
		else {
			DocumentDataXml dd = database.getDescriptor(uri);
			if (dd.isDeleted())
				throw new EntityDoesNotExistException("The entity: "
						+ uri.toASCIIString() + " is flagged as deleted");

			if (dd.isDirty())
				throw new EntityDirtyException("The entity: "
						+ uri.toASCIIString() + " is flagged as dirty");

			if (!dd.checkReadPermision(user))
				throw new PermissionDeniedException(user
						+ " does not have permission to access: "
						+ uri.toASCIIString());

			SAXBuilder builder = new SAXBuilder();
			builder.setIgnoringElementContentWhitespace(true);
			try {
				Document doc = builder.build(new StringReader(database
						.getlatest(uri)));
				Element root = doc.getRootElement();

				XPath path = XPath.newInstance(xpath);
				List results = path.selectNodes(root);
				if (results.size() == 0)
					return new String[0];
				ListIterator iter = results.listIterator();
				String[] string_array = new String[results.size()];

				while (iter.hasNext()) {
					// there has go to be a better way of doing this

					XMLOutputter out = new XMLOutputter(Format.getRawFormat());
					// XMLOutputter out = new XMLOutputter("",false,"UTF-8");
					int index = iter.nextIndex();
					Object n = iter.next();
					String result_string = "";
					if (n instanceof Element) {
						result_string = out.outputString((Element) n);
					}
					if (n instanceof Text) {
						result_string = out.outputString((Text) n);
					}
					if (n instanceof CDATA) {
						result_string = out.outputString((CDATA) n);
					}
					if (n instanceof Comment) {
						result_string = out.outputString((Comment) n);
					}
					if (n instanceof ProcessingInstruction) {
						result_string = out
								.outputString((ProcessingInstruction) n);
					}
					if (n instanceof Attribute) {
						Attribute tmp = (Attribute) n;
						result_string = tmp.getValue();
					}
					string_array[index] = result_string;
				}
				return string_array;
			} catch (JDOMException e) {
				logger.severe(e.getLocalizedMessage());
				throw new DatabaseException(e);
			} catch (IOException e) {
				logger.severe(e.getLocalizedMessage());
				throw new DatabaseException(e);
			}
		}
	}

	/**
	 * @see alma.archive.database.interfaces.InternalIF
	 */
	public String get(URI uri, ArchiveTimeStamp timestamp, String user)
			throws PermissionDeniedException, EntityDirtyException,
			EntityDoesNotExistException, DatabaseException, ArchiveException,
			MalformedURIException, UserDoesNotExistException,
			IllegalTimestampException {
		if (!database.exists(uri))
			throw new EntityDoesNotExistException("The URI: "
					+ uri.toASCIIString() + " was not found in the database");
		else {
			DocumentDataXml dd = database.getDescriptor(uri);
			if (dd.isDeleted())
				throw new EntityDoesNotExistException("The entity: "
						+ uri.toASCIIString() + " is flagged as deleted");

			if (dd.isDirty())
				throw new EntityDirtyException("The entity: "
						+ uri.toASCIIString() + " is flagged as dirty");

			ArchiveTimeStamp latestts = dd.getLatestTimeStamp();
			if (latestts.equals(timestamp)) {
				return database.getlatest(uri);
			} else {
				return database.getDocuments(uri, timestamp);
			}
		}
	}

	/**
	 * @see alma.archive.database.interfaces.InternalIF
	 */
	public String getDirty(URI uri, String user)
			throws PermissionDeniedException, EntityDoesNotExistException,
			DatabaseException, ArchiveException, UserDoesNotExistException,
			MalformedURIException {
		if (!database.exists(uri))
			throw new EntityDoesNotExistException("The URI: "
					+ uri.toASCIIString() + " was not found in the database");
		else {
			DocumentDataXml dd = database.getDescriptor(uri);
			if (dd.isDeleted())
				throw new EntityDoesNotExistException("The entity: "
						+ uri.toASCIIString() + " is flagged as deleted");
			// need to get from user to role, here or inside
			if (!dd.checkReadPermision(user))
				throw new PermissionDeniedException(user
						+ " does not have permission to access: "
						+ uri.toASCIIString());
			return database.getlatest(uri);
		}
	}

	/**
	 * @see alma.archive.database.interfaces.InternalIF
	 */
	public void delete(URI uri, String user) throws PermissionDeniedException,
			EntityDoesNotExistException, DatabaseException, ArchiveException,
			MalformedURIException, UserDoesNotExistException,
			EntityAlreadyDeletedException {
		if (!database.exists(uri))
			throw new EntityDoesNotExistException("The URI: "
					+ uri.toASCIIString() + " was not found in the database");
		else {
			DocumentDataXml dd = database.getDescriptor(uri);
			if (dd.isDeleted())
				throw new EntityAlreadyDeletedException("The entity: "
						+ uri.toASCIIString() + " is flagged as deleted");
			else {
				dd.setDelete(true);
				database.delDescriptor(uri);
				database.putDescriptor(uri, dd.toXmlString());
			}
		}
	}

	/**
	 * @see alma.archive.database.interfaces.InternalIF
	 */
	public void undelete(URI uri, String user)
			throws PermissionDeniedException, DatabaseException,
			ArchiveException, MalformedURIException,
			EntityDoesNotExistException, UserDoesNotExistException,
			EntityUndeletedException {
		if (!database.exists(uri))
			throw new EntityDoesNotExistException("The URI: "
					+ uri.toASCIIString() + " was not found in the database");
		else {
			DocumentDataXml dd = database.getDescriptor(uri);
			if (!dd.isDeleted())
				throw new EntityUndeletedException("The entity: "
						+ uri.toASCIIString() + " is not flagged as deleted");
			else {
				dd.setDelete(false);
				database.delDescriptor(uri);
				database.putDescriptor(uri, dd.toXmlString());
			}
		}
	}

	/**
	 * @see alma.archive.database.interfaces.InternalIF
	 */
	public void dirty(URI uri, String user) throws PermissionDeniedException,
			EntityDoesNotExistException, DatabaseException, ArchiveException,
			MalformedURIException, UserDoesNotExistException {
		if (!database.exists(uri))
			throw new EntityDoesNotExistException("The URI: "
					+ uri.toASCIIString() + " was not found in the database");
		else {
			DocumentDataXml dd = database.getDescriptor(uri);

			dd.setDirty(true);
			database.delDescriptor(uri);
			database.putDescriptor(uri, dd.toXmlString());
		}
	}

	/**
	 * @see alma.archive.database.interfaces.InternalIF
	 */
	public void clean(URI uri, String user) throws PermissionDeniedException,
			DatabaseException, ArchiveException, MalformedURIException,
			EntityDoesNotExistException, UserDoesNotExistException,
			EntityUndeletedException {
		if (!database.exists(uri))
			throw new EntityDoesNotExistException("The URI: "
					+ uri.toASCIIString() + " was not found in the database");
		else {
			DocumentDataXml dd = database.getDescriptor(uri);

			dd.setDirty(false);
			database.delDescriptor(uri);
			database.putDescriptor(uri, dd.toXmlString());
		}
	}

	/**
	 * @see alma.archive.database.interfaces.InternalIF
	 */
	public void hidden(URI uri, String user) throws PermissionDeniedException,
			EntityDoesNotExistException, DatabaseException, ArchiveException,
			MalformedURIException, UserDoesNotExistException {
		if (!database.exists(uri))
			throw new EntityDoesNotExistException("The URI: "
					+ uri.toASCIIString() + " was not found in the database");
		else {
			DocumentDataXml dd = database.getDescriptor(uri);

			dd.setHidden(true);
			database.delDescriptor(uri);
			database.putDescriptor(uri, dd.toXmlString());
		}
	}

	/**
	 * @see alma.archive.database.interfaces.InternalIF
	 */
	public void visible(URI uri, String user) throws PermissionDeniedException,
			DatabaseException, ArchiveException, MalformedURIException,
			EntityDoesNotExistException, UserDoesNotExistException,
			EntityUndeletedException {
		if (!database.exists(uri))
			throw new EntityDoesNotExistException("The URI: "
					+ uri.toASCIIString() + " was not found in the database");
		else {
			DocumentDataXml dd = database.getDescriptor(uri);

			dd.setHidden(false);
			database.delDescriptor(uri);
			database.putDescriptor(uri, dd.toXmlString());
		}
	}

	/**
	 * @see alma.archive.database.interfaces.InternalIF
	 */
	public void setVirtual(URI uri, String user, boolean virtual)
			throws PermissionDeniedException, DatabaseException,
			ArchiveException, MalformedURIException,
			EntityDoesNotExistException, UserDoesNotExistException {
		if (!database.exists(uri))
			throw new EntityDoesNotExistException("The URI: "
					+ uri.toASCIIString() + " was not found in the database");
		else {
			DocumentDataXml dd = database.getDescriptor(uri);

			dd.setVirtual(virtual);
			database.delDescriptor(uri);
			database.putDescriptor(uri, dd.toXmlString());
		}
	}

	/**
	 * @see alma.archive.database.interfaces.InternalIF
	 */
	public DocumentData status(URI uri, String user)
			throws PermissionDeniedException, EntityDoesNotExistException,
			DatabaseException, ArchiveException, UserDoesNotExistException,
			IllegalHistoryNumberException, MalformedURIException {
		if (!database.exists(uri))
			throw new EntityDoesNotExistException("The URI: "
					+ uri.toASCIIString() + " was not found in the database");
		else {
			DocumentDataXml dd = database.getDescriptor(uri);
			return dd.getDocumentData();
		}
	}

	/**
	 * @see alma.archive.database.interfaces.InternalIF
	 */
	public DocumentData[] allStatus(URI uri, String user)
			throws PermissionDeniedException, EntityDoesNotExistException,
			DatabaseException, ArchiveException, UserDoesNotExistException,
			MalformedURIException {
		if (!database.exists(uri))
			throw new EntityDoesNotExistException("The URI: "
					+ uri.toASCIIString() + " was not found in the database");
		else {
			DocumentDataXml dd = database.getDescriptor(uri);
			return dd.getAllDocumentData();
		}
	}

	/**
	 * @see alma.archive.database.interfaces.InternalIF
	 */
	public DocumentData status(URI uri, ArchiveTimeStamp timestamp, String user)
			throws PermissionDeniedException, EntityDoesNotExistException,
			DatabaseException, ArchiveException, MalformedURIException,
			UserDoesNotExistException, IllegalTimestampException {
		if (!database.exists(uri))
			throw new EntityDoesNotExistException("The URI: "
					+ uri.toASCIIString() + " was not found in the database");
		else {
			DocumentDataXml dd = database.getDescriptor(uri);
			return dd.getDocumentData(timestamp);
		}
	}

	/**
	 * @see alma.archive.database.interfaces.InternalIF
	 */
	protected DocumentData status(URI uri) throws PermissionDeniedException,
			EntityDoesNotExistException, DatabaseException, ArchiveException,
			MalformedURIException, UserDoesNotExistException,
			IllegalTimestampException {
		if (!database.exists(uri))
			throw new EntityDoesNotExistException("The URI: "
					+ uri.toASCIIString() + " was not found in the database");
		else {
			DocumentDataXml dd = database.getDescriptor(uri);
			return dd.getDocumentData();
		}
	}

	/**
	 * @see alma.archive.database.interfaces.InternalIF
	 */
	public DBCursor query(String query, String schema, HashMap namespaces,
			boolean dirtyRead, String user) throws PermissionDeniedException,
			DatabaseException, ArchiveException, MalformedQueryException,
			UnderspecifiedQueryException, UnknownSchemaException,
			UserDoesNotExistException, UndefinedNamespaceException {
		return database.queryLatest(query, namespaces, schema, dirtyRead,
				false, user);
	}

	public Collection<String> getLog(String timeFrom, String timeTo,
			short minType, short maxType, String routine, String source,
			String process, int maxRow) throws ArchiveException {
		logger
				.warning("queryLog only supported for Oracle database. Check your dbConfig.properties file!");
		throw new ArchiveException(
				"queryLog only supported for Oracle database.");
	}

	/**
	 * @see alma.archive.database.interfaces.InternalIF
	 */
	public DBCursor queryContent(String query, String schema,
			HashMap namespaces, boolean dirtyRead, String user)
			throws PermissionDeniedException, DatabaseException,
			ArchiveException, MalformedQueryException,
			UnderspecifiedQueryException, UnknownSchemaException,
			UserDoesNotExistException, UndefinedNamespaceException {
		// for XMLDB we can safely use normal query evaluation
		return database.queryLatest(query, namespaces, schema, dirtyRead,
				false, user);
	}

	public URI[] queryRecent(ArchiveTimeStamp timestamp, String schema,
			String user) throws PermissionDeniedException, DatabaseException,
			ArchiveException, MalformedQueryException,
			UnderspecifiedQueryException, UnknownSchemaException,
			UserDoesNotExistException, UndefinedNamespaceException {
		return database.queryRecent(timestamp, schema, user);
	}

	public URI[] queryInterval(ArchiveTimeStamp timeFrom,
			ArchiveTimeStamp timeTo, String schema, String XPathQuery,
			String user) throws DatabaseException, ArchiveException {
		return database.queryInterval(timeFrom, timeTo, schema, XPathQuery,
				user);
	}

	/**
	 * @see alma.archive.database.interfaces.InternalIF
	 */
	public URI[] queryIDs(String query, String schema, HashMap namespaces,
			boolean dirtyRead, String user) throws PermissionDeniedException,
			DatabaseException, ArchiveException, MalformedQueryException,
			UnderspecifiedQueryException, UnknownSchemaException,
			UserDoesNotExistException, UndefinedNamespaceException {
		XmldbCursor cursor = database.queryLatest(query, namespaces, schema,
				dirtyRead, false, user);
		return cursor.uriList();
	}

	/**
	 * @see alma.archive.database.interfaces.InternalIF
	 */
	public URI[] queryIDsAll(String query, String schema, HashMap namespaces,
			String user) throws PermissionDeniedException, DatabaseException,
			ArchiveException, MalformedQueryException,
			UnderspecifiedQueryException, UnknownSchemaException,
			UserDoesNotExistException, UndefinedNamespaceException {
		XmldbCursor cursor = database.queryLatest(query, namespaces, schema,
				true, true, user);
		return cursor.uriList();
	}

	/**
	 * @see alma.archive.database.interfaces.InternalIF
	 */
	public void setFlag(URI uid, String flagName, boolean flagValue, String user)
			throws PermissionDeniedException, DatabaseException,
			ArchiveException, MalformedURIException,
			EntityDoesNotExistException, UserDoesNotExistException,
			UnknownFlagException {
		// TODO: not sure what this does
	}

	/**
	 * @see alma.archive.database.interfaces.InternalIF
	 */
	public void changeOwner(URI uid, String newOwner, String user)
			throws PermissionDeniedException, DatabaseException,
			ArchiveException, MalformedURIException,
			EntityDoesNotExistException, UserDoesNotExistException {
		// TODO: Chenge the owner of the Entity
	}

	/**
	 * @see alma.archive.database.interfaces.InternalIF
	 */
	public SchemaManager getSchemaManager(String user) throws ArchiveException,
			PermissionDeniedException, UserDoesNotExistException {
		// TODO: Some kind of permissions check on the user
		return XmldbSchemaManager.instance(logger);
	}

	/**
	 * @see alma.archive.database.interfaces.InternalIF
	 */
	public UserManager getUserManager(String user) throws ArchiveException,
			PermissionDeniedException, UserDoesNotExistException {
		// TODO: Some kind of permissions check on the user
		return XmldbUserManager.getInstance(logger);
	}

	/**
	 * @see alma.archive.database.interfaces.InternalIF
	 */
	public void remove(URI uri, boolean keepHead, String user)
			throws PermissionDeniedException, DatabaseException,
			ArchiveException, MalformedURIException, UserDoesNotExistException,
			EntityDoesNotExistException {
		if (!database.exists(uri))
			throw new EntityDoesNotExistException("The URI: "
					+ uri.toASCIIString() + " was not found in the database");
		else {
			DocumentDataXml dd = database.getDescriptor(uri);

			List stamps = dd.getOtherTimeStamp();
			ListIterator iter = stamps.listIterator();
			while (iter.hasNext()) {
				ArchiveTimeStamp ts = (ArchiveTimeStamp) iter.next();
				database.delDocuments(uri, ts);
			}
			database.delDescriptor(uri);

			if (keepHead) {
				dd.removeHistory();
				database.delDescriptor(uri);
				database.putDescriptor(uri, dd.toXmlString());
			} else {
				database.delLatest(uri);
			}
		}
	}

	/**
	 * @see alma.archive.database.interfaces.InternalIF
	 */
	public void setPermission(URI uri, Permissions permissions, String user)
			throws PermissionDeniedException, DatabaseException,
			ArchiveException, MalformedURIException, UserDoesNotExistException,
			MalformedPermissionsException {
		if (!database.exists(uri))
			throw new EntityDoesNotExistException("The URI: "
					+ uri.toASCIIString() + " was not found in the database");
		else {
			DocumentDataXml dd = database.getDescriptor(uri);
			dd.setPermissions(permissions);
			database.delDescriptor(uri);
			database.putDescriptor(uri, dd.toXmlString());
		}
	}

	/**
	 * @see alma.archive.database.interfaces.InternalIF
	 */
	public void restoreState(ArchiveTimeStamp timestamp, String user)
			throws PermissionDeniedException, DatabaseException,
			ArchiveException, UserDoesNotExistException,
			IllegalTimestampException {
		// GNDN
	}

	/**
	 * @see alma.archive.database.interfaces.InternalIF
	 */
	public void notifyConfigChange() {
		// GNDN
	}

	/**
	 * @see alma.archive.database.interfaces.InternalIF
	 */
	public void cleanTestArea(String user) throws PermissionDeniedException,
			DatabaseException, ArchiveException, UserDoesNotExistException {
		// TODO: Need to figure out some way of getting separate access to the
		// db
		try {
			XmldbConnector connector = XmldbConnector.instance(logger, true);
			connector.cleanTestCollections();
		} catch (XMLDBException e) {
			logger.severe(e.getLocalizedMessage());
			throw new DatabaseException(e);
		}

		XmldbSchemaManager.instance(logger).removeAll();
	}

	/**
	 * Loads IdentifierRange.xsd into DB. Necessary after removing all schemas
	 * (eg. archiveCleanTest)
	 * 
	 * @param user
	 *            The user doing the operation
	 * @throws DatabaseException
	 * @throws UserDoesNotExistException
	 * @throws MalformedURIException
	 * @throws UnknownSchemaException
	 * @throws PermissionDeniedException
	 * @throws MalformedXMLException
	 */
	protected void loadIdentifierRangeSchema(String user)
			throws DatabaseException, UserDoesNotExistException,
			MalformedURIException, UnknownSchemaException,
			PermissionDeniedException, MalformedXMLException {
		// now add the IdentfierRange schema:

		// // read IdentifierRange.xsd from jar file:
		ClassLoader loader = DBConfiguration.class.getClassLoader();
		StringBuffer UIDschema = new StringBuffer();
		BufferedReader UIDschemaReader;
		URL UIDschemaFile = loader.getResource("IdentifierRange.xsd");
		if (UIDschemaFile == null) {
			logger
					.severe("Could not find IdentifierRange.xsd. Assumed location: IdentifierRange.xsd in archive_xmlstore_if.jar");
			throw new DatabaseException(
					"Could not find IdentifierRange.xsd. Assumed location: IdentifierRange.xsd in archive_xmlstore_if.jar");
		}
		logger.info("-------- Loading " + UIDschemaFile);
		try {
			UIDschemaReader = new BufferedReader(new InputStreamReader(
					UIDschemaFile.openStream()));
		} catch (IOException e1) {
			logger
					.severe("Problems when opening schema file for UID range. Location: IndentifierRange.xsd in archive_xmlstore_if.jar");
			throw new DatabaseException(
					"Problems when opening schema file for UID range. Location: IndentifierRange.xsd in archive_xmlstore_if.jar");
		}
		try {
			while (UIDschemaReader.ready()) {
				UIDschema.append(UIDschemaReader.readLine());
			}
		} catch (IOException e) {
			logger
					.severe("Problems when opening schema file for UID range. Location: IndentifierRange.xsd in archive_xmlstore_if.jar");
			throw new DatabaseException(
					"Problems when opening schema file for UID range. Location: IndentifierRange.xsd in archive_xmlstore_if.jar");
		}

		URI idRangeURI;
		try {
			idRangeURI = new Range(InternalIFFactory.getIdentifierManager(
					logger).getNewRange()).next();
			// TODO fetch the UID somehow else
		} catch (ArchiveException e) {
			throw new DatabaseException(e);
		} catch (ModuleCriticalException e) {
			throw new DatabaseException(e.getCause());
		} catch (AcsJArchiveIdentifierErrorEx e) {
			throw new DatabaseException(e);
		}
		logger.info("Storing IdentifierRange.xsd under UID: " + idRangeURI);
		XmldbSchemaManager.instance(logger).addSchema("IdentifierRange",
				UIDschema.toString(), "", idRangeURI, user, new Permissions());
	}

	/**
	 * Try to write an the read a little document to the database, just to check
	 * that it is awake and cooperating.
	 * 
	 * @return
	 */
	public boolean ping() {
		// TODO: Implement ping
		/*
		 * try { String document = "<test></test>"; URI uri = new
		 * URI("archive://ping"); database.putLatest(uri,document); String val =
		 * database.getlatest(uri); if (document.equalsIgnoreCase(val)){ return
		 * true; } else{ return false; } } catch (URISyntaxException e){ return
		 * false; } catch (DatabaseException e){ return false; }
		 */
		return true;
	}

	public void updateXML(URI uid, String schema, String newChild)
			throws DatabaseException, EntityDoesNotExistException,
			MalformedXMLException {
		// get old document
		String xml;
		try {
			xml = (String) database.getlatest(uid);
		} catch (DatabaseException e) {
			// this means document is not there, throw other exception:
			throw new EntityDoesNotExistException(uid.toString());
		}
		xml = DatabaseHelper.addAsLastChild(xml, newChild);
		database.putLatest(uid, xml);
	}
	/**
	 * Adds xmlElement as last child of the (first) element found by xPath in
	 * document uid (belonging to schema schema)
	 */
	public void addElement(URI uid, String schemaName, String xPath,
			String xmlElement, String user) throws EntityDoesNotExistException,
			DatabaseException, PermissionDeniedException,
			ModuleCriticalException {
		throw new DatabaseException("Operation addElement only supported for Oracle.");
	}
		
	/**
	 * The (first) element pointed at by xPath in document uid (belonging to
	 * schema schema) will be replaced by element xmlElement.
	 */
	public void updateElement(URI uid, String schemaName, String xPath,
			String xmlElement, String user) throws EntityDoesNotExistException,
			DatabaseException, PermissionDeniedException,
			ModuleCriticalException {
		throw new DatabaseException("Operation updateElement only supported for Oracle.");
			}

	/**
	 * The (first) element pointed at by xPath in document uid (belonging to
	 * schema schema) will be deleted.
	 */
	public void deleteElement(URI uid, String schemaName, String xPath,
			String user) throws EntityDoesNotExistException, DatabaseException,
			PermissionDeniedException, ModuleCriticalException {
		throw new DatabaseException("Operation deleteElement only supported for Oracle.");
			}

	public void storeLog(String message, int level, String attributeValue,
			String attributeValue2, String attributeValue3,
			String attributeValue4, String attributeValue5,
			String attributeValue6, String attributeValue7,
			String attributeValue8, String attributeValue9,
			String attributeValue10, String attributeValue11,
			String attributeValue12, String attributeValue13,
			String attributeValue14, String attributeValue15,
			String[] dataStrings, String xml, boolean commit, String steEnv) {
		// do nothing here, only used in Oracle

	}

}
