/*
 * 	  Created on 09-Sep-2003
 * 
 *    ALMA - Atacama Large Millimiter Array
 *    (c) European Southern Observatory, 2002
 *    Copyright by ESO (in the framework of the ALMA collaboration),
 *    All rights reserved
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License as published by the Free Software Foundation; either
 *    version 2.1 of the License, or (at your option) any later version.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
 *    MA 02111-1307  USA
 */
package alma.archive.database.xmldb;

import java.net.URI;
import java.net.URISyntaxException;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Set;
import java.util.Vector;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.xmldb.api.base.Collection;
import org.xmldb.api.base.Resource;
import org.xmldb.api.base.ResourceSet;
import org.xmldb.api.base.XMLDBException;
import org.xmldb.api.modules.XMLResource;
import org.xmldb.api.modules.XPathQueryService;

import alma.archive.database.helpers.DBConfiguration;
import alma.archive.exceptions.ArchiveException;
import alma.archive.exceptions.cursor.CursorClosedException;
import alma.archive.exceptions.general.DatabaseException;
import alma.archive.exceptions.general.UnknownSchemaException;
import alma.archive.wrappers.ArchiveTimeStamp;

/**
 * @author simon
 * 
 * This class controls the XML:DB Database resource, it is a singleton
 * implementation. There are three collections that the database acessed:
 * descriptor, latest and documents.
 * 
 * The descriptor collection contains all of the meta data relating to the
 * documents stored in the archive. This meta data relates to all versions of a
 * document, it also contains the uri of the schema.
 * 
 * The documents collection contains all of the older versions of documents,
 * when the documents are stored the timestamp is appended to the uri so it is
 * possible to differntiate between them.
 * 
 * The latest collection stores the latest version of documents, this done so
 * that it possible to query one the most recent versions.
 * 
 * This class must be closed when it is finshed with, not closing will cause
 * threading problems.
 */
public class XmldbDatabase {
	private Logger logger;

	private XmldbConnector connector = null;

	protected DBConfiguration dbConfig = null;

	private String _testns = "uid://test";

	private URI testns;

	private Collection descriptor = null;

	private Collection latest = null;

	private Collection documents = null;

	private static XmldbDatabase _instance = null;

	private static int access_count = 0;

	/**
	 * Returns a pointer to the single class instance, creates the class if it
	 * is neccessary. When the pointer is returned a counter is incremented, it
	 * is only decremneted when close is called.
	 * 
	 * @param logger
	 * @return
	 * @throws DatabaseException
	 * @throws XMLDBException
	 */
	public static XmldbDatabase instance(Logger logger)
			throws DatabaseException, XMLDBException {
		if (_instance == null) {
			DBConfiguration config = DBConfiguration.instance(logger);
			boolean testMode = config.testMode;
			XmldbConnector conn = XmldbConnector.instance(logger, testMode);
			_instance = new XmldbDatabase(conn, logger);
		}
		access_count++;
		return _instance;
	}

	/**
	 * Attempts to free all resources and close the class. When this is called
	 * the instance counter is decremented, when it is zero the collections are
	 * closed.
	 * 
	 * @throws XMLDBException
	 */
	public void close() throws DatabaseException {
		access_count--;
		if (access_count == 0) {
			if (descriptor != null) {
				connector.shutdownCollection(descriptor);
				descriptor = null;
			}
			if (latest != null) {
				connector.shutdownCollection(latest);
				latest = null;
			}
			if (documents != null) {
				connector.shutdownCollection(documents);
				documents = null;
			}
			_instance = null;
		}
	}

	/**
	 * Creates the connections to the XML:DB database, also configures the
	 * cache. All of the collections are created with the connector, then the
	 * descriptor and latest cache are setup.
	 * 
	 * @param logger
	 * @throws DatabaseException
	 * @throws XMLDBException
	 */
	protected XmldbDatabase(XmldbConnector conn, Logger logger)
			throws DatabaseException, XMLDBException {
		this.logger = logger;
		dbConfig = DBConfiguration.instance(logger);
		// Get a connector
		if (connector == null) {
			connector = conn;
		}

		try {
			testns = new URI(_testns);
		} catch (URISyntaxException e) {
			e.printStackTrace();
		}

		// Check that there is configuration information
		if (dbConfig == null) {
			dbConfig = DBConfiguration.instance(logger);
		}
	}

	/**
	 * @deprecated Used to check that none of the collections have accidentaly
	 *             become set to null, this will be removed in the future.
	 * @throws XMLDBException
	 */
	private void connect() throws DatabaseException {
		try {
			if (descriptor == null) {
				logger.info("ARCHIVE: Fetching descriptor Collection");
				descriptor = connector.getDescriptor();
			}
			if (latest == null) {
				logger.info("ARCHIVE: Fetching latest Collection");
				latest = connector.getLatest();
			}
			if (documents == null) {
				logger.info("ARCHIVE: Fetching documents Collection");
				documents = connector.getDocuments();
			}
		} catch (XMLDBException e) {
			throw new DatabaseException(
					"Problems fetching collections from the Database", e);
		}
	}

	/**
	 * refreshs database information after removal of collections
	 * 
	 * @throws DatabaseException
	 * 
	 * @throws DatabaseException
	 */
	protected void refresh() throws DatabaseException {
		logger.info("Refreshing database info.");
		descriptor = null;
		latest = null;
		documents = null;
		connect();
	}

	/**
	 * Checks if a particular document exists in either the descriptor or latest
	 * collections.
	 * 
	 * @param uri
	 * @return
	 */
	public boolean exists(URI uri) throws DatabaseException {
		if (uri.equals(testns))
			return true;
		connect();
		if ((inCollection(descriptor, uri)) || (inCollection(latest, uri)))
			return true;

		return false;
	}

	/**
	 * Used to check if a descriptor exists, for a particular URI.
	 * 
	 * @param uri
	 * @return
	 */
	public boolean descriptorExists(URI uri) throws DatabaseException {
		connect();
		return inCollection(descriptor, uri);
	}

	/**
	 * Find out whether or not a particular document exists in a collection,
	 * takes into account the differnt libraries. May be moved to the connector
	 * in the future as there are some database dependancies.
	 * 
	 * @param col
	 * @param uri
	 * @return
	 * @throws DatabaseException
	 */
	private boolean inCollection(Collection col, URI uri)
			throws DatabaseException {
		synchronized (col) {
			try {
				String xmldbUri = URItoString(uri);
				Resource res = col.getResource(xmldbUri);
				if (res == null) {
					return false;
				} else {
					if (res.getContent() == null) {
						return false;
					} else
						return true;
				}
			} catch (XMLDBException e) {
				throw new DatabaseException(
						"Problem when checking existance of: "
								+ uri.toASCIIString() + " ", e);
			}
		}
	}

	/**
	 * Stores XML at a particular location in a collection
	 * 
	 * @param col
	 *            the collection to store in
	 * @param uri
	 *            the uri to store the xml at
	 * @param xml
	 *            the content
	 */
	public void put(Collection col, URI uri, String xml)
			throws DatabaseException {
		try {
			String xmldbUri = URItoString(uri);
			XMLResource newDocument = (XMLResource) col.createResource(
					xmldbUri, "XMLResource");
			newDocument.setContent(xml);
			col.storeResource(newDocument);
		} catch (XMLDBException e) {
			throw new DatabaseException("Problem when puting: "
					+ uri.toASCIIString() + " ", e);
		}
	}

	/**
	 * Put xml into the descriptor collection
	 * 
	 * @param uri
	 * @param xml
	 * @throws DatabaseException
	 */
	public void putDescriptor(URI uri, String xml) throws DatabaseException {
		connect();
		synchronized (descriptor) {
			put(descriptor, uri, xml);
		}
	}

	/**
	 * Put xml into the latest collection
	 * 
	 * @param uri
	 * @param xml
	 * @throws DatabaseException
	 */
	public void putLatest(URI uri, String xml) throws DatabaseException {
		connect();
		synchronized (latest) {
			put(latest, uri, xml);
		}
	}

	/**
	 * Put xml into the documents collection
	 * 
	 * @param uri
	 * @param ts
	 * @param xml
	 * @throws DatabaseException
	 */
	public void putDocuments(URI uri, ArchiveTimeStamp ts, String xml)
			throws DatabaseException {
		connect();
		synchronized (documents) {
			put(documents, URIAddTime(uri, ts), xml);
		}
	}

	/**
	 * Get xml from a collection
	 * 
	 * @param col
	 *            the collection to retrive from
	 * @param uri
	 *            the location to retrive from
	 * @return the xml
	 * @throws DatabaseException
	 */
	private String get(Collection col, URI uri) throws DatabaseException {
		try {
			String xmldbUri = URItoString(uri);
			XMLResource document = (XMLResource) col.getResource(xmldbUri);
			if (document == null) {
				throw new DatabaseException(
						"No content (Null XMLResource) was returned by the database URI: "
								+ uri.toASCIIString()
								+ " probably nothign there");
			}
			String xml = (String) document.getContent();
			if (xml == null) {
				throw new DatabaseException(
						"No content (Null XML) was returned by the database URI: "
								+ uri.toASCIIString()
								+ " probably nothign there");
			} else
				return xml;
		} catch (XMLDBException e) {
			throw new DatabaseException("Problem when geting: "
					+ uri.toASCIIString() + " ", e);
		}
	}

	/**
	 * Get xml from the descriptor collection
	 * 
	 * @param uri
	 * @return
	 * @throws DatabaseException
	 */
	public DocumentDataXml getDescriptor(URI uri) throws DatabaseException {
		connect();
		synchronized (descriptor) {
			String xml = null;
			xml = get(descriptor, uri);

			if (xml == null) {
				throw new DatabaseException(
						"No content was returned by the database URI: "
								+ uri.toASCIIString()
								+ " from collection Descriptor "
								+ "probably nothign there");
			} else
				return new DocumentDataXml(xml);
		}
	}

	/**
	 * Get xml from the latest collection
	 * 
	 * @param uri
	 * @return
	 * @throws DatabaseException
	 */
	public String getlatest(URI uri) throws DatabaseException {
		connect();
		synchronized (latest) {
			String xml = null;
			xml = get(latest, uri);

			if (xml == null) {
				throw new DatabaseException(
						"No content was returned by the database URI: "
								+ uri.toASCIIString()
								+ " from collection Latest "
								+ "probably nothign there");
			} else
				return xml;
		}
	}

	/**
	 * Does nothign at present, database always in synch.
	 * 
	 * @param uri
	 */
	public void flushLatest(URI uri) {

	}

	/**
	 * Get xml from the documents colection
	 * 
	 * @param uri
	 * @param ts
	 * @return
	 * @throws DatabaseException
	 */
	public String getDocuments(URI uri, ArchiveTimeStamp ts)
			throws DatabaseException {
		connect();
		synchronized (documents) {
			String xml = get(documents, URIAddTime(uri, ts));
			if (xml == null) {
				throw new DatabaseException(
						"No content was returned by the database URI: "
								+ uri.toASCIIString()
								+ " from collection Documents "
								+ "probably nothign there");
			} else
				return xml;
		}
	}

	/**
	 * Removes the xml from a given collection and location
	 * 
	 * @param col
	 * @param uri
	 * @throws DatabaseException
	 */
	private void del(Collection col, URI uri) throws DatabaseException {
		try {
			String xmldbUri = URItoString(uri);
			XMLResource document = (XMLResource) col.getResource(xmldbUri);
			if (document == null) {
				throw new DatabaseException("Unable to delete: "
						+ uri.toString()
						+ " The XMLResource returned by the database was Null");
			}
			col.removeResource(document);
		} catch (XMLDBException e) {
			throw new DatabaseException("Problem when geting: "
					+ uri.toASCIIString() + " ", e);
		}
	}

	/**
	 * Remove xml from the decriptor collection
	 * 
	 * @param uri
	 * @throws DatabaseException
	 */
	public void delDescriptor(URI uri) throws DatabaseException {
		connect();
		synchronized (descriptor) {
			del(descriptor, uri);
		}
	}

	/**
	 * Remove xml from the latest collection
	 * 
	 * @param uri
	 * @throws DatabaseException
	 */
	public void delLatest(URI uri) throws DatabaseException {
		connect();
		synchronized (latest) {
			del(latest, uri);
		}
	}

	/**
	 * Remove xml from the documents collection
	 * 
	 * @param uri
	 * @param ts
	 * @throws DatabaseException
	 */
	public void delDocuments(URI uri, ArchiveTimeStamp ts)
			throws DatabaseException {
		connect();
		synchronized (documents) {
			del(documents, URIAddTime(uri, ts));
		}
	}

	/**
	 * Attempts to register a set fo namespaces for a query
	 * 
	 * @param service
	 * @param namespaces
	 * @throws XMLDBException
	 */
	private void registerNamespaces(XPathQueryService service,
			HashMap namespaces) throws XMLDBException {
		if (namespaces != null) {
			Set set = namespaces.keySet();
			Iterator iter = set.iterator();
			while (iter.hasNext()) {
				String tag = (String) iter.next();
				String namespace = (String) namespaces.get(tag);
				service.setNamespace(tag, namespace);
			}
		} else {
			logger.log(Level.INFO, "ARCHIVE: Namespaces not set for query");
		}
	}

	/**
	 * Querys the latest collection of the database.
	 * 
	 * @param query
	 *            the XPath query
	 * @param namespaces
	 *            a HashMap of namespaces
	 * @param schema
	 *            the name of the schema for the document
	 * @param dirtyRead
	 *            whether documents flagged as dirty shoudl be included
	 * @param allRead
	 *            whether all flags should be ingnored
	 * @param user
	 *            the user
	 * @return
	 * @throws DatabaseException
	 */
	public XmldbCursor queryLatest(String query, HashMap namespaces,
			String schema, boolean dirtyRead, boolean allRead, String user)
			throws DatabaseException {
		connect();
		synchronized (latest) {
			try {
				XPathQueryService service = (XPathQueryService) latest
						.getService("XPathQueryService", "1.0");

				registerNamespaces(service, namespaces);

				ResourceSet results = service.query(query);

				XmldbCursor cursor = new XmldbCursor(this, results, schema,
						dirtyRead, allRead, user, logger);

				return cursor;
			} catch (XMLDBException e) {
				throw new DatabaseException(
						"XPath query failed " + query + " ", e);
			}
		}
	}

	/**
	 * Querys recent incoming documents, later than timestamp
	 * 
	 * @param timestamp
	 *            Only documents later than timestamp are returned
	 * @param schema
	 *            the name of the schema for the document
	 * @param user
	 *            the user
	 * @return
	 * @throws DatabaseException
	 */
	public URI[] queryRecent(ArchiveTimeStamp timestamp, String schema,
			String user) throws DatabaseException {
		try {
			return queryInterval(timestamp, null, schema, null, user);
		} catch (UnknownSchemaException e) {
			throw new DatabaseException(e);
		}
	}

	/**
	 * Queries documents stored between timeFrom and timeTo
	 * 
	 * @param timeFrom
	 *            Only documents later than timeFrom are returned
	 * @param timeTo
	 *            Only documents later than timeFrom are returned, maybe null,
	 *            then it's ignored
	 * @param schema
	 *            the name of the schema for the document
	 * @param query
	 *            XPath query to be executed, may be null, then it's ignored
	 * @param user
	 *            the user
	 * @return
	 * @throws DatabaseException
	 * @throws UnknownSchemaException 
	 */
	public URI[] queryInterval(ArchiveTimeStamp timeFrom,
			ArchiveTimeStamp timeTo, String schema, String XPathQuery,
			String user) throws DatabaseException, UnknownSchemaException {
		// due to inconvenient storage architecture, we have to proceed as
		// follows:
		// 1) query docs with correct timestamps
		// 2) query docs matching submitted query
		// 3) Return intersection of the two result sets

		logger.finest("Interval query: "+XPathQuery+"   from: "+timeFrom.toISOString()+"   to: "+((timeTo==null)?"now":timeTo.toISOString()));
		
		connect();

		// find docs matching timestamp constraints
		XmldbCursor uidTime;
		synchronized (descriptor) {
			String query = "/descriptor[@schemaname=\"" + schema
					+ "\"][history/latest/attribute::timestamp>\""
					+ timeFrom.toISOString() + "\"]";
			if (timeTo != null) {
				query = query + "[history/latest/attribute::timestamp<\""
						+ timeTo.toISOString() + "\"]";
			}

			//logger.info("Query: "+query);
			try {
				XPathQueryService service = (XPathQueryService) descriptor
						.getService("XPathQueryService", "1.0");

				ResourceSet results = service.query(query);
				uidTime = new XmldbCursor(this, results, schema, false, false,
						user, logger);
			} catch (XMLDBException e) {
				throw new DatabaseException(
						"XPath query failed " + query + " ", e);
			}
		}

		if (XPathQuery != null) {
			HashMap namespaces = XmldbSchemaManager.instance(logger)
					.getSchemaNamespaces(
							XmldbSchemaManager.instance(logger).getSchemaURI(
									schema));
			// find docs matching the query
			XmldbCursor uidQuery;
			synchronized (latest) {
				try {
					XPathQueryService service = (XPathQueryService) latest
							.getService("XPathQueryService", "1.0");

					registerNamespaces(service, namespaces);

					ResourceSet results = service.query(XPathQuery);

					//System.out.println(results.getResource(0).getContent());

					uidQuery = new XmldbCursor(this, results, schema,
							false, false, user, logger);
				} catch (XMLDBException e) {
					throw new DatabaseException("XPath query failed " + XPathQuery
							+ " ", e);
				}
			}

			// now form intersection
			
			// put uidTime in a set, then construct vector out of correct element
			Set<URI> uids = new HashSet();
			try {
				while (uidTime.hasNext()) {
					uids.add(uidTime.next().getUri());
				}
			} catch (ArchiveException e) {
				throw new DatabaseException("Problems in handling cursor. "+e);
			}
			//logger.info("TimeUIDs: "+uids);
			Vector<URI> out = new Vector();
			try {
				while (uidQuery.hasNext()) {
					URI next = uidQuery.next().getUri();
					//logger.info("query uid: "+next);
					if (uids.contains(next)) {
						out.add(next);
						uids.remove(next);
					}
				}
				uidTime.close();
				uidQuery.close();
			} catch (ArchiveException e) {
				throw new DatabaseException("Problems in handling cursor. "+e);
			}
			return out.toArray(new URI[0]);
			
		} else {
			// construct UID list out of uidTime only
			URI[] ret = uidTime.uriList();
			try {
				uidTime.close();
			} catch (ArchiveException e) {
				throw new DatabaseException("Problems in handling cursor. "+e);
			}
			return ret;
		}
	}

	private String URItoString(URI uri) {
		String tmp = uri.toASCIIString();
		tmp = tmp.replaceAll("://", "schemeSeperator");
		tmp = tmp.replaceAll("/", "slash");
		tmp = tmp.replaceAll("#", "hash");
		return tmp;
	}

	private URI URIAddTime(URI uri, ArchiveTimeStamp ts) {
		try {
			String suri = uri.toASCIIString();
			suri = suri + "#" + ts.toISOString();
			return new URI(suri);
		} catch (URISyntaxException e) {
			logger.severe(e.getLocalizedMessage());
			return null;
		}
	}
}