/*
 * 	  Created on 09-Sep-2003
 * 
 *    ALMA - Atacama Large Millimiter Array
 *    (c) European Southern Observatory, 2002
 *    Copyright by ESO (in the framework of the ALMA collaboration),
 *    All rights reserved
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License as published by the Free Software Foundation; either
 *    version 2.1 of the License, or (at your option) any later version.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
 *    MA 02111-1307  USA
 */
package alma.archive.database.xmldb;

import java.io.IOException;
import java.io.StringReader;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.ListIterator;

import org.jdom.Document;
import org.jdom.Element;
import org.jdom.JDOMException;
import org.jdom.input.SAXBuilder;
import org.jdom.output.Format;
import org.jdom.output.XMLOutputter;
import org.jdom.xpath.XPath;

import alma.archive.exceptions.general.DatabaseException;
import alma.archive.wrappers.ArchiveTimeStamp;
import alma.archive.wrappers.DocumentData;
import alma.archive.wrappers.Permissions;

/**
 * 
 * @author simon
 *
 * This class is used to wrap the DocumentData class and deliver XML to the
 * XML:DB Database. The structure of the XML is given below:
 * 
 * <pre>
 * \<descriptor uri="" schemauri="" owner="" deleted="" virtual=""\>
 * 	\<permissions read="" write=""/\>
 * 	\<administrative hidden="" dirty=""/\>
 * 	\<history\>
 * 		\<version number="" timestamp=""/\>
 * 		....
 *		\<latest number="" timestamp=""/\>
 *	\</history\>
 * \</descriptor\>
 * </pre>
 *  
 * The class can be created either by parsing a string or from a set of
 * initial values.
 */
public class DocumentDataXml
{	 
	private Document doc = null;
	 
	/*
	 * These elements allow quick access without
	 * resorting to XPaths. 
	 */ 
	private Element root = null;
	private Element perm = null;
	private Element admin = null;
	private Element history = null;
	private Element latest = null;
	
	
	/**
	 * Constructor for an empty structure, adds essential attributes
	 * and creats the structure. The is only in memeory at this point
	 * so it is viatal that it is output to a string and written to 
	 * the database at some point.
	 * 
	 * @param uri uri of the docuement that this relates to.
	 * @param schema uri of the schema for this document.
	 * @param schemaName the name of the schema pointed to by the schema uri
	 * @param owner the owner of the document
	 * @param permissions a permissions calss for the document
	 */
	public DocumentDataXml(
			URI uri, 
			URI schema, 
			String schemaName, 
			String owner, 
			Permissions permissions)
	{
		root = new Element("descriptor");
		
		root.setAttribute("uri",uri.toASCIIString());
		root.setAttribute("schemauri",schema.toASCIIString());
		root.setAttribute("schemaname",schemaName);
		root.setAttribute("owner",owner);
		//Reasonable default
		root.setAttribute("deleted","false");
		root.setAttribute("virtual","false");
		
		//Permissions element
		perm = new Element("permissions");
		perm.setAttribute("read",permissions.getRead());
		perm.setAttribute("write",permissions.getWrite());
		
		//Administration element
		admin = new Element("administration");
		admin.setAttribute("hidden","false");
		admin.setAttribute("dirty","false");
		
		//History element, populated later
		history = new Element("history");
		
		//Latest element must always exist
		latest = new Element("latest");
		//latest.setAttribute("number","1");
		latest.setAttribute("timestamp",getTimeStamp());
		history.addContent(latest);
	
		//Buld the document
		root.addContent(perm);
		root.addContent(admin);
		root.addContent(history);
		
		doc = new Document(root);
	}
	
	/**
	 * Creates the structure from an XML string
	 * @param xml the string to create the structure
	 */
	public DocumentDataXml(String xml)
	{
		SAXBuilder builder = new SAXBuilder();
		builder.setIgnoringElementContentWhitespace(true);
		try
		{
			doc = builder.build(new StringReader(xml));
			root = doc.getRootElement();
			
			perm = root.getChild("permissions");
			admin = root.getChild("administration");
			history = root.getChild("history");
			latest = history.getChild("latest");
		}
		catch (JDOMException e)
		{
			e.printStackTrace();
			//MPA: no logging, no action. is it ok?
		}
		catch (IOException e)
		{
			e.printStackTrace();
        //MPA: no logging, no action. is it ok?
		}	
	}
	
	/**
	 * Removes the entire hitory contained in the structure,
	 * does not remove the latest version.
	 */
	public void removeHistory()
	{
		history.removeChildren("version");
	}
	
	/**
	 * This is used to shift the most recent entry into the history,
	 * from latest to a version. In the process it creates a new
	 * latest with a current timestamp.
	 */
	public void addLatest()
	{	
		if (history != null)
		{
			//Create a new version and copy across the timestamp
			Element version = new Element("version");
			version.setAttribute("timestamp",latest.getAttributeValue("timestamp"));
			history.addContent(version);
			//Set the latest timestamp
			latest.setAttribute("timestamp",getTimeStamp());
		}
	}
	
	/**
	 * Used to return a DocumentData structure for a particular version.
	 * @param version the element contaning a particular timestamp 
	 * @return returns the DocumentData for this version
	 */
	private DocumentData getDocumentData(Element version)
	{
		try 
		{
			//short vers = Short.parseShort(version.getAttributeValue("number"));
			ArchiveTimeStamp timestamp = new ArchiveTimeStamp(version.getAttributeValue("timestamp"));
			URI schema = new URI(root.getAttributeValue("schemauri"));
		
			String owner = root.getAttributeValue("owner");
			Permissions permissions = new Permissions(perm.getAttributeValue("read"),perm.getAttributeValue("write"));
			boolean hidden = this.isHidden();
			boolean dirty = this.isDirty();
			boolean deleted = this.isDeleted();
			boolean virtual = this.isVirtual();
			
			// Might be used at some point
			String admin ="";
			return new DocumentData(timestamp,schema,owner,permissions,hidden,dirty,deleted,virtual,admin);
		}
		catch (URISyntaxException e) {e.printStackTrace();return null;}
		//MPA: no logging, no action. is it ok?
 catch (DatabaseException e) {
	 
	 return null;
		}
	}
	
	/**
	 * Creates and returns the DocumentData structure for the latest
	 * version of the document.
	 */
	public DocumentData getDocumentData()
	{
		return getDocumentData(latest);
	}
	
	/**
	 * Create and return the document data for a particular timestamp.
	 * This function uses an XPath so a performace hit should be expected.
	 * @param timestamp the time in question
	 */
	public DocumentData getDocumentData(ArchiveTimeStamp timestamp)
	{	
		try
		{
			String _path = "//version[@timestamp=\"" + timestamp.toISOString() + "\"]";
			XPath path = XPath.newInstance(_path);
			Element result = (Element)path.selectSingleNode(history);
			if (result == null)
			{
				return getDocumentData(latest);
			}
			else
			{
				return getDocumentData(result);
			}
		}
		catch (JDOMException e)
		{
			e.printStackTrace();
			return null;
			//MPA: no action, no logging. Is it ok?
		}
	}
	
	/**
	 * Creates an array of DocumentData
	 * @return
	 */
	public DocumentData[] getAllDocumentData()
	{
		List children = history.getChildren();
		DocumentData[] result = new DocumentData[children.size()];
		ListIterator iter = children.listIterator();
		while (iter.hasNext())
		{
			result[iter.nextIndex()] = getDocumentData((Element)iter.next());
		}
		return result;
	}
	
	/**
	 * Returns the URI for the document
	 */
	public URI getURI()
	{
		try
		{
			return new URI(root.getAttributeValue("uri"));
		}
		catch (URISyntaxException e)
		{
			e.printStackTrace();
			return null;
			//MPA: no action, no logging. is it ok?
		}
	}
	
	/**
	 * Returns the URI for the document scehma 
	 */
	public URI getSchemaURI()
	{
		try
		{
			return new URI(root.getAttributeValue("schemauri"));
		}
		catch (URISyntaxException e)
		{
			e.printStackTrace();
			return null;
            //MPA: no action, no logging. is it ok?
		}
	}
	
	/**
	 * Returns the name of the document schema
	 */
	public String getSchemaName()
	{
		String schemaName = root.getAttributeValue("schemaname"); 
		return schemaName;
	}
	
	/**
	 * Allows the scehma and it's URI to be changed
	 * @param schemaUri the URI of the schema
	 * @param schemaName the name of the schema
	 */
	public void setSchema(URI schemaUri, String schemaName)
	{
		root.setAttribute("schemauri",schemaUri.toASCIIString());
		root.setAttribute("schemaname",schemaName);
	}
	
	/**
	 * returns the most recent time stamp
	 * @return
	 */
	public ArchiveTimeStamp getLatestTimeStamp()
	{
		try {
			return new ArchiveTimeStamp(latest.getAttributeValue("timestamp"));
		} catch (DatabaseException e) {
			return null;
		}
	}
	
	/**
	 * Returns a list of all of the timestamps
	 * @return a List of ArchiveTimeStamp
	 */
	public List getOtherTimeStamp()
	{
		List children = history.getChildren("version");
		Iterator iter = children.iterator();
		List stamps = new ArrayList();
		while (iter.hasNext())
		{
			Element e = (Element)iter.next();
			ArchiveTimeStamp s;
			try {
				s = new ArchiveTimeStamp(e.getAttributeValue("timestamp"));
				stamps.add(s);
			} catch (DatabaseException e1) {
				// don't know what to do here, since there is no logger availbale...
			}
		}
		return stamps;
	}
	
	/**
	 * Returns the dirty flag
	 */
	public boolean isDirty()
	{
		String value = admin.getAttributeValue("dirty");
		if (value.equalsIgnoreCase("true")) {return true;}
		else {return false;}
	}
	
	/**
	 * Sets the dirty flag
	 * @param dirty value for the flag
	 */
	public void setDirty(boolean dirty)
	{
		admin.setAttribute("dirty",Boolean.toString(dirty));
	}
	
	/**
	 * Returns the hidden flag
	 */
	public boolean isHidden()
	{
		String value = admin.getAttributeValue("hidden");
		if (value.equalsIgnoreCase("true")) {return true;}
		else {return false;}
	}
	
	/**
	 * Sets the hidden flag
	 * @param hidden value for the flag
	 */
	public void setHidden(boolean hidden)
	{
		admin.setAttribute("hidden",Boolean.toString(hidden));
	}
	
	/**
	 * Returns the deleted flag
	 */
	public boolean isDeleted()
	{
		String value = root.getAttributeValue("deleted");
		if (value.equalsIgnoreCase("true")) {return true;}
		else {return false;}
	}
	
	/**
	 * Sets the deleted flag
	 * @param del vlaue for the flag
	 */
	public void setDelete(boolean del)
	{
		root.setAttribute("deleted",Boolean.toString(del));
	}
	
	/**
	 * Returns the virtual flag, if there is no virtual flag
	 * returns false for backwards compatability.
	 */
	public boolean isVirtual()
	{
		String value = root.getAttributeValue("virtual");
		if (value == null) return false;
		if (value.equalsIgnoreCase("true")) {return true;}
		else {return false;}
	}
	
	/**
	 * Sets the virtual flag
	 * @param vir the value for the flag
	 */
	public void setVirtual (boolean vir)
	{
		root.setAttribute("virtual",Boolean.toString(vir));
	}
	
//	public ArchiveTimeStamp closestTime(ArchiveTimeStamp timeStamp)
//	{
//		if (timeStamp.after(new ArchiveTimeStamp(latest.getAttributeValue("timestamp"))))
//		{
//			return null;
//		}
//		else
//		{
//			ArchiveTimeStamp t = null;
//			
//			List list = history.getChildren();
//			Iterator iter = list.iterator();
//			while(iter.hasNext())
//			{
//				Element ver = (Element)iter.next();
//				String ts = ver.getAttributeValue("timestamp");
//				ArchiveTimeStamp time = new ArchiveTimeStamp(ts);
//				
//				if (t == null)
//				{
//					t = time;
//				} 
//				else if ((timeStamp.before(time)) || (timeStamp.equals(time)))
//				{
//					t = time;
//				}
//			}
//			return t;
//		}
//	}

	/**
	 * Should return the read permission given a particular role,
	 * at present returns true.
	 */
	public boolean checkReadPermision(String role)
	{
		// TODO work out some way of checking permisson
		return true;
	}
	
	/**
	 * Should return the write permission given a particular role,
	 * at presnet returns true.
	 */
	public boolean checkWritePermision(String role)
	{
		// TODO work out some way of checking permisson
		return true;
	}
	
	/**
	 * Returns a very simple Permissions structure, it is imagined that this
	 * will bacome far more complex.
	 */
	public Permissions getPermissions()
	{
		return new Permissions(perm.getAttributeValue("read"),perm.getAttributeValue("write"));
	}
	
	/**
	 * Sets the permission to the contents of the Permissions class
	 * @param permissions the new Permissions
	 */
	public void setPermissions(Permissions permissions)
	{
		perm.setAttribute("read",permissions.getRead());
		perm.setAttribute("write",permissions.getWrite());	
	}
	
	/**
	 * Returns a String representation of the XML structure
	 */
	public String toXmlString()
	{
		XMLOutputter out = new XMLOutputter(Format.getRawFormat());
		// XMLOutputter out = new XMLOutputter("",false,"UTF-8");
		return out.outputString(doc);
	}
	
	/**
	 * Returns an ISO compliant timestamp string for now
	 * @return
	 */
	private String getTimeStamp()
	{
		ArchiveTimeStamp ts = new ArchiveTimeStamp(); 
		return ts.toISOString();
	}
	
	public static void main(String[] args) throws URISyntaxException
	{
		URI uri = new URI("uid://X0000/X00");
		URI schemauri = new URI("uid://X0000/X01");
		Permissions perm = new Permissions();
		
		DocumentDataXml test = new DocumentDataXml(uri,schemauri,"nonsense","bob",perm); 
		test.addLatest();
		test.addLatest();
		System.out.println(test.toXmlString());
		
		DocumentDataXml rTrip = new DocumentDataXml(test.toXmlString());
		System.out.println(rTrip.toXmlString());
		rTrip.addLatest();
		System.out.println(rTrip.toXmlString());
	}
}
