/*
 * 	  Created on 21-Sep-2004
 * 
 *    ALMA - Atacama Large Millimiter Array
 *    (c) European Southern Observatory, 2002
 *    Copyright by ESO (in the framework of the ALMA collaboration),
 *    All rights reserved
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License as published by the Free Software Foundation; either
 *    version 2.1 of the License, or (at your option) any later version.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
 *    MA 02111-1307  USA
 */
package alma.archive.database.vdoc;

import java.io.File;
import java.lang.reflect.Field;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Vector;
import java.util.logging.Logger;

import alma.archive.database.vdoc.*;

import org.jdom.Element;

import sun.misc.Launcher;
import alma.archive.database.interfaces.InternalIF;
import alma.archive.exceptions.general.VDocException;

/**
 * @author simon
 * 
 * This class holds the central functionality and all of the abstract methods
 * for the ContextSource classes. All children of this class must implement 
 */
public abstract class Definition
{	
	protected static Logger logger = null;
	private URI contextid = null;
	
	/**
	 * Create and instance of the correct type of subclass based on the 
	 * XML Definition. All of the parameters will be checked as well.
	 * @param definition
	 * @return
	 * @throws VDocException
	 */
	public static Definition instance(Element definition) 
		throws 
			VDocException
	{
		String type = definition.getAttributeValue("type");
		String contextid = definition.getAttributeValue("contextid");
		Definition _instance;
		try{
			_instance = instance(type,new URI(contextid));
		}
		catch (URISyntaxException e){
			throw new VDocException(e);
//			MPA: no logging
		}
		List parameters = definition.getChildren("param");
		Iterator iter = parameters.iterator();
		while (iter.hasNext())
		{
			Element parameter = (Element)iter.next();
			String paramName = parameter.getAttributeValue("name");
			String paramValue = parameter.getAttributeValue("val");
			_instance.setParameter(paramName,paramValue);
		}
		return _instance;
	}
	
	/**
	 * Create an empty instance of the correct type of of Defenition.
	 * @param type
	 * @return
	 * @throws VDocException
	 */
	public static Definition instance(String type, URI contextid)
		throws
			VDocException
	{
		//Create the name of the class
		String className = "alma.archive.database.vdoc." + 
							type +
							"Definition";
		
		try
		{
			Definition _instance = 
				(Definition)Class.forName(className).newInstance();
			_instance.setLogger();
			_instance.setContextid(contextid);
			
			return _instance;
		}
		catch (ClassNotFoundException e){
			throw new VDocException (
				"Unable to find the required class: " + 
				e.getMessage());
			//MPA: would be better to also pass parameter e
		} 
		catch (IllegalAccessException e) {
			throw new VDocException (e.getMessage());
			//MPA: no logging and would be better to also pass parameter e
		}
		catch (InstantiationException e){
			throw new VDocException (e.getMessage());
//			MPA: no logging and would be better to also pass parameter e
		}
	}
	
	public static boolean isAvailable(String type)
	{
		try
		{
			String className = "alma.archive.database.vdoc." + 
				type + "Definition";
			Class test = Class.forName(className);
		}
		catch (ClassNotFoundException e){
			return false;
			//MPA: logging missing
		}
		return true;
	}
	
	/**
	 * Sets up a default logger for all of the subclasses.
	 *
	 */
	public void setLogger()
	{
		if (logger == null)
		{
			logger = Logger.getAnonymousLogger();
		}
	}
	
	/**
	 * Set a particular logger for all of the subclasses, will change all of
	 * them as it is a static field.
	 * @param logr
	 */
	public void setLogger(Logger logr)
	{
		logger = logr;
	}

	
	/**
	 * Returns an XML description of the of the Source types available.
	 * @return
	 * @throws VDocException
	 */
	/*
	public static Element getSourceTypes() throws VDocException
	{
		ArrayList classList = locateChildren
			("Definition","alma.archive.database.vdoc");
		Element types = new Element("types");
		Iterator iter = classList.iterator();
		while (iter.hasNext())
		{
			Element type = new Element("type");
			type.setAttribute("name",(String)iter.next());
			types.addContent(type);
		}
		return types;
	}
	*/
	
	/**
	 * This takes advantage of a gratuitous hack to list all of the loded
	 * classes.
	 * @param cl
	 * @return
	 * @throws VDocException
	 */
	private static Iterator listClasses(ClassLoader cl) 
		throws 
			VDocException
	{
		try
		{
			Class cl_class = cl.getClass();
			while (cl_class != java.lang.ClassLoader.class)
			{
				cl_class = cl_class.getSuperclass();
			}
			Field classes_field = cl_class.getDeclaredField("classes");
			classes_field.setAccessible(true);
			Vector classes = (Vector)classes_field.get(cl);
			return classes.iterator();
		}
		catch (SecurityException e){
			throw new VDocException(e);
		}
		catch (NoSuchFieldException e){
			throw new VDocException(e);
		}
		catch (IllegalArgumentException e){
			throw new VDocException(e);
		}
		catch (IllegalAccessException e){
			throw new VDocException(e);
		}
		//MPA: 4 log missing
	}
	
	/**
	 * Returns a list of the child classes.
	 * @param parentClass
	 * @param packageName
	 * @return
	 * @throws VDocException
	 */
	private static ArrayList locateChildren(
			String parentClass, 
			String packageName) 
		throws 
			VDocException
	{
		ClassLoader cl = ClassLoader.getSystemClassLoader();
		Iterator iter = listClasses(cl);
		ArrayList results = new ArrayList();
		while(iter.hasNext())
		{
			Class cls = (Class)iter.next();
			System.out.println(cls.getName());
			Class sclass = cls.getSuperclass();
			if (sclass != null)
			{
				if (sclass.getName().equalsIgnoreCase(
					packageName + "." + parentClass))
				{
					results.add(cls.getName());
				}
			}
		}
		
		/*
		//Translate the package name into an absolute path
        String name = new String(packageName);
        name = name.replace('.','/');
        //Go and find the file
        URL url = ClassLoader.getSystemResource(name);
        
        
        
        System.out.println(url.toString());
        File directory = new File(url.getFile());
        
        
        if (directory.exists())
        {        	
        	String [] files = directory.list();
            //Run through all of the files
        	for (int i=0;i<files.length;i++) 
            {
        		System.out.println(files[i].toString());
                if (files[i].endsWith(".class"))
                {
                    String classname = 
                    	files[i].substring(0,files[i].length()-6);
                    try
					{
                        //Get the class definition
                    	Class classDef = 
                    		Class.forName(packageName + "." + classname);
                    	//Get the parent class
                    	Class superClass = classDef.getSuperclass();
                    	if (superClass.getName().equalsIgnoreCase
                    			(packageName + "." + parentClass))
                    	{
                    		results.add(classname);
                    	}
                    } catch (ClassNotFoundException e) {
                        logger.severe(e.getLocalizedMessage());
                    }
                }
            }
        }
        */
        return results;
	}
	
	private String getName()
	{
		String name = this.getClass().getName();
		int lastdot = name.lastIndexOf(".");
		name = name.substring(lastdot + 1);
		name = name.replaceAll("Definition","");
		return name;
	}
	
	/**
	 * Returns an XML document describing the parameters for a particular type
	 * of Definition. The parameters themselves must begin with "param" e.g.
	 * paramXPath.
	 * @param type
	 * @return
	 * @throws VDocException
	 */
	public Element getElement() throws VDocException
	{
		Element definition = new Element("definition");
		definition.setAttribute("type",getName());
		definition.setAttribute("contextid",contextid.toASCIIString());
		
		Field[] fields = this.getClass().getDeclaredFields();
		for (int x = 0; x < fields.length; x++)
		{
			Field field = fields[x];
			String name = field.getName();
			if (name.indexOf("param") != -1)
			{
				try
				{
					Element parameter = new Element("parameter");
					name = name.replaceAll("param","");
					parameter.setAttribute("name",name);
					//type name might need to be shortened
					parameter.setAttribute("type",field.getType().getName());
					parameter.setAttribute("val",field.get(this).toString());
					definition.addContent(parameter);
				}
				catch (IllegalAccessException e)
				{
					throw new VDocException(e);
					//MPA: logging missing
				}
			}
		}
		
		return definition;
	}
	
	/**
	 * Extracts all of the parameter information from the XML and uses
	 * it to set variables. Unfortunatly this means that the variables 
	 * must be public. 
	 * @param source
	 * @throws VDocException
	 */
	public void setParameter (
			String paramName,
			String paramValue)
		throws 
			VDocException
	{
		Class sourceDef = this.getClass();
		try
		{
			Field param = sourceDef.getField("param" + paramName);
			String type = param.getType().getName();			
			if (type.equalsIgnoreCase("int"))
			{
				param.setInt(this,Integer.parseInt(paramValue));
			}
			else if (type.equalsIgnoreCase("double"))
			{
				param.setDouble(this,Double.parseDouble(paramValue));
			}
			else if (type.equalsIgnoreCase("long"))
			{
				param.setLong(this,Long.parseLong(paramValue));
			}
			else if (type.equalsIgnoreCase("boolean"))
			{
				param.setBoolean(
					this,Boolean.valueOf(paramValue).booleanValue());
			}
			else if (type.equalsIgnoreCase("java.lang.String"))
			{
				param.set(this,paramValue);
			}
			else
			{
				throw new VDocException(
					type + " Is not a supported type");
			}
		}
		catch (SecurityException e){
			throw new VDocException(e);
		}
		catch (NoSuchFieldException e){
			throw new VDocException(e);
		}
		catch (IllegalAccessException e){
			throw new VDocException(e);
		}
//		MPA: logging missing, 3 times
	}
	
	public void setContextid(URI contextid){
		this.contextid = contextid;
	}
	
	public URI getContextid(){
		return contextid;
	}
	
	
	public abstract Element createList(InternalIF internal) 
		throws
			VDocException;
}
