/*
 *    ALMA - Atacama Large Millimiter Array
 *    (c) European Southern Observatory, 2002
 *    Copyright by ESO (in the framework of the ALMA collaboration),
 *    All rights reserved
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License as published by the Free Software Foundation; either
 *    version 2.1 of the License, or (at your option) any later version.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
 *    MA 02111-1307  USA
 *
 *    Created on Oct 23, 2003
 *
 */

// $Author: hmeuss $
// $Date: 2007/04/05 08:57:34 $
// $Log: SchemaManagerImpl.java,v $
// Revision 1.17  2007/04/05 08:57:34  hmeuss
// history table now gets also deleted when all schemas are removed. This fixes the bug of archiveLoadSchema -c corrupting the Database.
//
// Revision 1.16  2006/10/11 09:33:42  hmeuss
// IdenitiferRange.xsd is loaded into DB after removing all schemas.
//
// Revision 1.14  2006/10/10 12:43:28  hmeuss
// IdentifierRange.xsd is loaded into DB after removing all schemas (eg. calling archiveLoadSchema -c).
// Not tested yet, but should work without problems.
//
// Revision 1.13  2006/09/21 13:38:27  hsommer
// corrected for exception changes due to new queryContent method
//
// Revision 1.12  2006/09/21 08:53:43  hmeuss
// Implemented queryContent method, that correctly queries XPath expressions returning a non-node type (eg. ending with an attribute step)
//
// Revision 1.11  2006/09/06 10:14:28  hsommer
// generics
//
// Revision 1.10  2006/07/07 09:03:31  hmeuss
// removed bug, that prevented old documents belonging to old schemas from being retrieved
//
// Revision 1.9  2006/07/07 08:53:16  hmeuss
// removed bug, that prevented old documents belonging to old schemas from being retrieved
//
// Revision 1.8  2006/06/08 15:21:29  hmeuss
// fixed bug for the  -c option of the schema loader
//
// Revision 1.7  2006/02/16 12:49:02  mpasquat
// added comments about exception handling problems
//
// Revision 1.6  2005/07/21 13:48:12  hmeuss
// replaced DB2 in log messages by Oracle
//
// Revision 1.5  2004/10/28 09:03:32  hmeuss
// - Implementation of refresh in oracle.SchemaManager
// - Added ModuleCriticalException to refresh
//
// Revision 1.4  2004/09/23 11:59:14  hmeuss
// Oracle DatabaseReader now creates a new connection object in every method call.
//
// Added ModuleCriticalException to more methods of the internal IF
//
// Revision 1.3  2004/06/01 15:53:12  hmeuss
// Added treatment of virtual documents
//
// Revision 1.2  2004/05/05 08:45:55  hmeuss
// *** empty log message ***
//
// Revision 1.1  2004/04/05 13:59:29  hmeuss
// Internal IF implementation adapted to Oracle
//
// Revision 1.11  2004/02/17 14:37:32  hmeuss
// Added some schema treatment
//
// Revision 1.10  2004/02/11 16:20:09  hmeuss
// *** empty log message ***
//
// Revision 1.9  2004/01/29 14:17:05  hmeuss
// Adapted to the new interface
//
// Revision 1.8  2004/01/23 15:11:45  hmeuss
// Added permission check in DB2CursorImpl
//
// Revision 1.7  2004/01/19 16:29:46  hmeuss
// Splitted functionality of Db2Database into Db2DatabaseReader, Db2DatabaseWriter, Db2DatabaseCache
//
// Revision 1.6  2004/01/13 12:56:32  hmeuss
// implemented removeAll and listSchema in DB2SchemaManager
//
// Revision 1.5  2004/01/13 10:43:13  hmeuss
// *** empty log message ***
//
// Revision 1.4  2003/12/18 13:20:05  hmeuss
// *** empty log message ***
//
// Revision 1.3  2003/12/01 12:35:47  sfarrow
// *** empty log message ***
//
// Revision 1.2  2003/11/20 12:47:31  hmeuss
// *** empty log message ***
//
// Revision 1.1  2003/10/30 08:36:26  hmeuss
// *** empty log message ***
// 

package alma.archive.database.oracle;

import java.net.URI;
import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.logging.Logger;

import alma.archive.database.interfaces.SchemaManager;
import alma.archive.exceptions.ArchiveException;
import alma.archive.exceptions.ModuleCriticalException;
import alma.archive.exceptions.access.PermissionDeniedException;
import alma.archive.exceptions.general.DatabaseException;
import alma.archive.exceptions.general.DocumentDoesNotExistException;
import alma.archive.exceptions.general.EntityDoesNotExistException;
import alma.archive.exceptions.general.EntityExistsException;
import alma.archive.exceptions.general.HistoryInconsistencyException;
import alma.archive.exceptions.general.NamespaceDefinedException;
import alma.archive.exceptions.general.UnknownSchemaException;
import alma.archive.exceptions.syntax.MalformedURIException;
import alma.archive.exceptions.syntax.MalformedXMLException;
import alma.archive.wrappers.Permissions;

/**
 * @author hmeuss
 *
 */
public class SchemaManagerImpl implements SchemaManager {

	protected Logger m_logger = Logger.getLogger("ARCHIVE");
	// user using the schema manager
	String m_user;
	Database m_database;
	DatabaseReader m_databaseReader;
	DatabaseWriter m_databaseWriter;
	DatabaseCache m_databaseCache;

	// (virtual) uid for the schema of schemas (unused)
	String schemaSchemaURIstring = "http://www.w3.org/2001/XMLSchema";
	URI schemaSchemaURI; // set by constructor  (unused)
	

	/**
	 * 
	 */
	public SchemaManagerImpl(String user, Database database)
		throws DatabaseException {
		super();

		m_database = database;
		m_databaseReader = database.dbReader;
		m_databaseWriter = database.dbWriter;
		m_databaseCache = database.dbCache;
		m_user = user;
		try {
			schemaSchemaURI = new URI(schemaSchemaURIstring);
		} catch (URISyntaxException e) {
			// this should not happen, since URI is hardcoded
			throw new DatabaseException(
				"URI used for schema of schmeas (hardcoded variable schemaSchemaURIstring) "
					+ schemaSchemaURIstring
					+ " is no valid URI");
		}
	}

	/**
	 * 
	 */
	public SchemaManagerImpl(String user) throws DatabaseException, ModuleCriticalException {
		super();

		m_database = Database.instance();
		m_databaseReader = m_database.dbReader;
		m_databaseWriter = m_database.dbWriter;
		m_databaseCache = m_database.dbCache;
		m_user = user;
		try {
			schemaSchemaURI = new URI(schemaSchemaURIstring);
		} catch (URISyntaxException e) {
			throw new DatabaseException(
				"URI used for schema of schmeas (hardcoded variable schemaSchemaURIstring) "
					+ schemaSchemaURIstring
					+ " is no valid URI");
		}
	}

	public void removeAll() throws EntityDoesNotExistException, DatabaseException, PermissionDeniedException, ModuleCriticalException
	{
		for (Iterator schemaIt=((HashMap) m_databaseCache.uri2schemaName.clone()).values().iterator(); schemaIt.hasNext(); ) {
			String schema=(String) schemaIt.next();
			m_databaseWriter.removeSchema(schema);
		}
		m_databaseWriter.removeHistoryTab();
		m_databaseWriter.removeNamespaces();
		try {
		m_databaseWriter.loadIdentifierRangeSchema();
		} catch (Exception e) {
		throw new DatabaseException(e);
		}
	}
	/**
	 * @see alma.archive.database.interfaces.SchemaManager#close()
	 */
	public void close() throws DatabaseException {
		// do nothing
	}

	/**
	 * @see alma.archive.database.interfaces.SchemaManager#getSchemaURI(java.lang.String)
	 */
	public URI getSchemaURI(String schema)
		throws DatabaseException, ArchiveException, UnknownSchemaException {
		// have a look in the cached schema names:
		URI ret = m_databaseCache.schemaName2uri.get(schema);
		if (ret == null)
			throw new UnknownSchemaException(schema);
		return ret;
	}

	/**
	 * @see alma.archive.database.interfaces.SchemaManager#getSchemaURI(java.lang.String, int)
	 */
	public URI getSchemaURI(String schema, int version)
		throws ArchiveException, UnknownSchemaException, DatabaseException, ModuleCriticalException {

		return m_databaseReader.getSchemaURI(schema, version);
	}

	/**
	 * @throws UnknownSchemaException
	 * @throws DatabaseException
	 * @throws ModuleCriticalException
	 * @see alma.archive.database.interfaces.SchemaManager#getSchemaName(java.net.URI)
	 */
	public String getSchemaName(URI schemaURI)
		throws
			ArchiveException,
			MalformedURIException, UnknownSchemaException, DatabaseException, ModuleCriticalException {
				return m_databaseReader.getSchemaName(schemaURI);
	}

	/**
	 * @see alma.archive.database.interfaces.SchemaManager#getSchemaVersion(java.net.URI)
	 */
	public int getSchemaVersion(URI schemaUri)
		throws
			ArchiveException,
			UnknownSchemaException,
			MalformedURIException, DatabaseException, ModuleCriticalException {
				// TODO use a cache here, too! This method is called for *every* retrieve!
				return m_databaseReader.getSchemaVersion(schemaUri);
	}

	/**
	 * @see alma.archive.database.interfaces.SchemaManager#getSchemaNamespaces(java.net.URI)
	 */
	public HashMap<String, String> getSchemaNamespaces(URI schemaUri)
		throws ArchiveException, UnknownSchemaException, DatabaseException, ModuleCriticalException {
			return m_databaseReader.getSchemaNamespaces(schemaUri);
	}

	/**
	 * @see alma.archive.database.interfaces.SchemaManager#registerNamespace(java.lang.String, java.net.URI)
	 */
	public void registerNamespace(String prefix, URI namespace)
		throws NamespaceDefinedException, DatabaseException, ArchiveException, MalformedURIException {
			m_databaseWriter.registerNamespace(prefix, namespace);
	}

	/**
	 * @see alma.archive.database.interfaces.SchemaManager#removeNamespace(java.lang.String)
	 */
	public void removeNamespace(String prefix)
		throws DatabaseException, ArchiveException, MalformedURIException {
			m_databaseWriter.removeNamespace(prefix);
	}

	/**
	 * @see alma.archive.database.interfaces.SchemaManager#namespaceExists(java.net.URI)
	 */
	public boolean namespaceExists(URI namespace) throws ArchiveException, DatabaseException, ModuleCriticalException {
		return m_databaseReader.namespaceExists(namespace); 
	}

	/**
	 * @see alma.archive.database.interfaces.SchemaManager#assignNamespace(java.lang.String, java.net.URI)
	 */
	public void assignNamespace(String prefix, URI schemaUri)
		throws DatabaseException, ArchiveException, UnknownSchemaException {
			m_databaseWriter.assignNamespace(prefix, schemaUri);
	}

	/**
	 * @see alma.archive.database.interfaces.SchemaManager#withdrawNamespace(java.lang.String, java.net.URI)
	 */
	public void withdrawNamespace(String prefix, URI schemaUri)
		throws DatabaseException, ArchiveException, UnknownSchemaException {
			m_databaseWriter.withdrawNamespace(prefix, schemaUri); 
	}

	/**
	 * @see alma.archive.database.interfaces.SchemaManager#namespaces()
	 */
	public HashMap<String, String> namespaces() throws ArchiveException, DatabaseException, ModuleCriticalException {
		return m_databaseReader.getSchemaNamespaces(null);
	}

	/**
	 * @see alma.archive.database.interfaces.SchemaManager#addSchema(java.lang.String, java.lang.String, java.net.URI, java.lang.String, alma.archive.wrappers.Permissions, java.lang.String).
	 * The indexConfig argument contains the DAD file for Oracle as a string.
	 */
	public void addSchema(
		String schemaName,
	String xml,
	String indexConfig,
		URI schemaURInew,
		String owner,
		Permissions permissions)
		throws
			DatabaseException,
			ArchiveException,
			MalformedURIException,
			UnknownSchemaException,
			PermissionDeniedException,
			MalformedXMLException {
		// check schema existence
		// existence check can be done by catching an SQL exception in DatabaseWriter 
	    
	    //MPA: last 4 exceptio never thrown
		if (m_databaseReader.existsSchema(schemaName)) {
			throw new EntityExistsException(schemaName);
		}		

		// add schema to schemas table
		m_databaseWriter.addSchemaAndTables(schemaURInew, schemaName, xml, indexConfig);

		// TODO possibly fill tables for xpath-SQL translation 
		// TODO check permissions!!!!!!!!!
	}

	/**
	 * @see alma.archive.database.interfaces.SchemaManager#updateSchema(java.lang.String, java.lang.String, java.net.URI, java.net.URI, java.lang.String, alma.archive.wrappers.Permissions, java.lang.String)
	 */
	public void updateSchema(
		String schemaName,
	String xml,
	String indexConfig,
		URI schemaURIold,
		URI schemaURInew,
		String owner,
		Permissions permissions)
		throws
			ArchiveException,
			MalformedURIException,
			UnknownSchemaException,
			PermissionDeniedException,
			MalformedXMLException, DatabaseException, DocumentDoesNotExistException, HistoryInconsistencyException, ModuleCriticalException {
		
// TODO check permissions

		// check schema existence + version consistency
		if (!m_databaseReader.checkSchemaConsistency(schemaName, schemaURIold)) {
			throw new HistoryInconsistencyException(
				"The schema document "
					+ schemaURInew
					+ " is not the latest schema document of schema "
					+ schemaName);
		}

int version = m_databaseReader.getSchemaVersion(schemaURIold);
		// insert schema into database
		m_databaseWriter.addSchema(
			schemaURInew,
			schemaName,
			version+1,   // new version increments by 1
			xml,
			indexConfig);

		// TODO attach new DAD file to old side tables
		// TODO possibly fill tables for xpath-SQL translation 
	}

	/**
	 * @see alma.archive.database.interfaces.SchemaManager#removeSchema(java.lang.String, java.lang.String)
	 */
	public void removeSchema(String schemaName)
		throws DatabaseException, ArchiveException, UnknownSchemaException, PermissionDeniedException {
		// TODO check permissions
		m_databaseWriter.removeSchema(schemaName);
	}

	/**
	 * @see alma.archive.database.interfaces.SchemaManager#listSchema()
	 */
	public List<String> listSchema() throws DatabaseException, ArchiveException {
		return new ArrayList<String>(m_databaseCache.schemaName2uri.keySet());
	}

	/**
	 * @see alma.archive.database.interfaces.SchemaManager#refresh()
	 */
	public void refresh() throws DatabaseException, ModuleCriticalException {
		m_databaseReader.initSchemaMaps(); 
	}

}
