/*
 *    ALMA - Atacama Large Millimiter Array
 *    (c) European Southern Observatory, 2002
 *    Copyright by ESO (in the framework of the ALMA collaboration),
 *    All rights reserved
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License as published by the Free Software Foundation; either
 *    version 2.1 of the License, or (at your option) any later version.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
 *    MA 02111-1307  USA
 *
 *    Created on Jul 3, 2003
 *
 */

// $Author: hmeuss $
// $Date: 2006/12/15 21:57:52 $
// $Log: CursorImpl.java,v $
// Revision 1.9  2006/12/15 21:57:52  hmeuss
// fixed bug, that some connections were not closed
//
// Revision 1.8  2006/10/09 11:59:20  hmeuss
// fixed bug in query construction for content queries
//
// Revision 1.7  2006/09/21 13:38:06  hsommer
// corrected for exception changes due to new queryContent method
//
// Revision 1.6  2006/09/21 08:53:43  hmeuss
// Implemented queryContent method, that correctly queries XPath expressions returning a non-node type (eg. ending with an attribute step)
//
// Revision 1.5  2006/02/16 12:47:51  mpasquat
// added comments about exception handling problems
//
// Revision 1.4  2005/07/21 12:29:48  hmeuss
// Changed design of test area
//
// Revision 1.3  2004/09/23 11:59:14  hmeuss
// Oracle DatabaseReader now creates a new connection object in every method call.
//
// Added ModuleCriticalException to more methods of the internal IF
//
// Revision 1.2  2004/07/15 09:46:14  hmeuss
// Removedf superfluous jar files
//
// Revision 1.1  2004/04/05 13:59:29  hmeuss
// Internal IF implementation adapted to Oracle
//
// Revision 1.10  2004/01/23 15:11:45  hmeuss
// Added permission check in DB2CursorImpl
//
// Revision 1.9  2003/12/18 13:20:05  hmeuss
// *** empty log message ***
//
// Revision 1.8  2003/11/20 12:47:31  hmeuss
// *** empty log message ***
//
// Revision 1.7  2003/10/30 09:23:27  hmeuss
// *** empty log message ***
//
// Revision 1.5  2003/08/29 09:02:04  hmeuss
// Java internal interface updated
//
// Revision 1.4  2003/08/06 12:31:59  hmeuss
// Query and Cursor are now somehow functioning
//
// Revision 1.3  2003/07/24 15:12:14  hmeuss
// Beginning the implementation for real DB2, version 8 functionality
//
// Revision 1.2  2003/07/14 12:24:33  hmeuss
// First DB2 implementation
// 
package alma.archive.database.oracle;

import java.io.IOException;
import java.io.StringReader;
import java.net.URI;
import java.net.URISyntaxException;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.Iterator;
import java.util.List;
import java.util.logging.Logger;

import oracle.xdb.XMLType;

import org.jaxen.BaseXPath;
import org.jaxen.JaxenException;
import org.jaxen.expr.Step;
import org.jaxen.jdom.JDOMXPath;
import org.jdom.JDOMException;
import org.jdom.input.SAXBuilder;

import alma.archive.database.helpers.DatabaseHelper;
import alma.archive.database.interfaces.DBCursor;
import alma.archive.exceptions.ModuleCriticalException;
import alma.archive.exceptions.cursor.CursorClosedException;
import alma.archive.exceptions.general.DatabaseException;
import alma.archive.exceptions.syntax.MalformedQueryException;
import alma.archive.wrappers.ResultStruct;
import java.sql.Connection;


/**
 * 
 * The implementation of a Cursor uses an underlying JDBC ResultSet in order to
 * implement the functionality.
 * 
 * The implementation also checks the read permissions
 * 
 * Therefore the query underlying the resultset must contain readPermissions and
 * owner of the respective entities
 * 
 * @author hmeuss
 *  
 */
public class CursorImpl implements DBCursor {

	/* Underlying JDBC ResultSet: */
	private ResultSet m_result;

	/* user having submitted the query, needed for checking permissions */
	private String m_user;

	/*
	 * underlying connection that has to be closed when all results have been
	 * fetched.
	 */
	private Connection m_conn;

	/*
	 * flag, whether m_conn is still accessible. (m_conn is closed after the
	 * last result is fetched from db.)
	 */
	private boolean m_closed = false;

	/* next xml fragment in resultSet to be retrieved */
	private ResultStruct nextResult = null;

	/*
	 * set to null for normal evaluation. If different from null, this last
	 * XPath step has to evaluated on the results in the cursor.
	 */
	private BaseXPath lastStep = null;

	/*
	 * for caching multiple results that have to be cached in the case of
	 * lastStep!=null (ie. content queries) and that an intermediate fragment
	 * (from m_result) matches in more than place to the XPath query
	 */
	private Iterator resultStack = null;

	private URI resultStackUri = null;

	static private SAXBuilder builder;

	private Logger logger;

	/**
	 * Constructor for Cursor rs is the underlying resultset (living in Oracle),
	 * conn is the underlying connection (to be closed when cursor is closed),
	 * user the user requesting the data
	 * 
	 * rs must contain uid, xml fragment, and readPermissions
	 * 
	 *  
	 */

	public CursorImpl(
	// TODO shall we synchronize?
			ResultSet rs, Connection conn, String user)
			throws MalformedQueryException, DatabaseException,
			ModuleCriticalException {

		logger = InternalIfImpl.instance().logger;

		m_conn = conn;
		m_user = user;
		m_result = rs;
		builder = new SAXBuilder();
	}

	/**
	 * @see alma.archiveinternal.InternalCursorOperations#hasNext()
	 */
	public boolean hasNext() throws DatabaseException {
		if (nextResult == null) {
			nextResult = fetchNextResult();
		}
		return nextResult != null;
	}

	public void close() {
		try {
			if (!m_closed) {
				m_result.close();
				DatabaseConnectionPool.instance(logger).close(m_conn);
			}
		} catch (Exception e) {
			// ignore
			logger.info("Ignoring exception: "
					+ e.toString());
		} finally {
			m_closed = true;
		}
	}

	/**
	 * Tells the Cursor, that on every result, the last step has still to
	 * evaluated before returning the cursor to the user.
	 * @throws DatabaseException
	 * 
	 * @author hmeuss
	 */
	protected void reevaluateLastStep(Step last) throws DatabaseException {
		try {
		lastStep = new JDOMXPath("string(*/" + last.getText()+")");
		// lastStep = new JDOMXPath("/*/" + last.getText());
		} catch (JaxenException e) {
			logger.warning("Could not create XPath out of "+last.toString());
			throw new DatabaseException("Could not create XPath out of "+last.toString());
		}
	}

	/**
	 * @see alma.archiveinternal.InternalCursorOperations#next()
	 */
	public ResultStruct next() throws DatabaseException, CursorClosedException {
		if (nextResult == null) {
			nextResult = fetchNextResult();
		}

		if (m_closed) {
			throw new CursorClosedException();
		}

		ResultStruct out = nextResult;
		nextResult = null;
		return out;
	}

	public ResultStruct[] nextBlock(short size) throws DatabaseException {
		// TODO try m_result.setFetchSize(size)
		short count = 0;
		ResultStruct[] ret = new ResultStruct[size];

		while (count < size && hasNext()) {
			try {
				ret[count] = next();
			} catch (CursorClosedException e) {
				// no more results available
				//MPA: logging missing
				return ret;
			}
			count++;
		}

		return ret;
	}

	//////////////////////////////////////////////////////////////////////
	//                private helpers //
	//////////////////////////////////////////////////////////////////////

	/**
	 * 
	 * get the next row from result set and construct ResultStruct. This method
	 * does also check permissions.
	 *  
	 */
	private ResultStruct fetchNextResult() throws DatabaseException {
		// Should this method use a semaphore?

		// first check whether still some old results are left in resultStacK:
		if (resultStack != null && resultStack.hasNext()) {
			ResultStruct out = new ResultStruct(resultStackUri, resultStack
					.next().toString());
			return out;
		} else {
			resultStack = null;
		}

		String uid = null;
		try {
			if (m_closed || !m_result.next()) {
				// no more results in underlying resultSet: close statement
				if (!m_closed) {
					m_closed = true;
					m_result.close();
					DatabaseConnectionPool.instance(logger).close(m_conn);
				}
				return null;
			}

			/* get permissions and owner */
			String readPermissions = m_result
					.getString(DBConfig.colName_readPermissions);
			String owner = m_result.getString(DBConfig.colName_owner);
			/* if permissions are not ok: get next element from resultSet */
			if (!DatabaseHelper.checkAccessPermissions(m_user, readPermissions,
					owner)) {
				return fetchNextResult();
			}

			uid = m_result.getString(DBConfig.colName_uid);
			String xml = ((XMLType) m_result.getObject(DBConfig.colName_xml)).getStringVal();

			if (lastStep != null) {
				Object result=null;
				try {
				// now reevaluate the last step and use the modified result
					org.jdom.Document doc = builder.build(new StringReader(xml));
					result = lastStep.evaluate(doc);
				} catch (JaxenException e) {
					logger.warning("Could not evaluate XPath "+lastStep.toString()+" on document.");
					throw new DatabaseException("Could not evaluate XPath "+lastStep.toString()+" on document.");
				} catch (JDOMException e) {
					logger.warning("Could not parse document retrieved from Oracle. ");
					throw new DatabaseException("Could not parse document retrieved from Oracle. ");
				} catch (IOException e) {
					throw new DatabaseException("Caught IOException when reading String. This should NEVER happen!");
				}
				if (result == null) {
					// this should not happen due to the construction of the
					// queries!!!
					return null;
				}
				if (result instanceof List) {
					// probably multiple results (ie. Nodes) in this fragment!
					resultStack = ((List) result).iterator();
					if (!resultStack.hasNext()) {
						// this should not occur
						throw new DatabaseException(
								"Unexpected failure of XPath handling in CursorImpl for content queries.");
					}
					resultStackUri = new URI(uid);
					xml = resultStack.next().toString();
				} else {
					// only one result, with String, Double or Boolean type
					xml = result.toString();
				}
			}

			ResultStruct out = new ResultStruct(new URI(uid), xml);
			return out;
		} catch (SQLException e) {
			logger.warning("SQL Exception: " + e);
			throw new DatabaseException(
					"Could not retrieve next result from cursor.");
			//MPA: retrown without e
		} catch (URISyntaxException e) {
			logger.warning("Found invalid UID in database: " + uid
					+ ". Will not be part of query result.");
			// continue with next result:
			return fetchNextResult();
		}
	}

}
