/*
 * 	  Created on 01-Feb-2005
 * 
 *    ALMA - Atacama Large Millimiter Array
 *    (c) European Southern Observatory, 2002
 *    Copyright by ESO (in the framework of the ALMA collaboration),
 *    All rights reserved
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License as published by the Free Software Foundation; either
 *    version 2.1 of the License, or (at your option) any later version.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
 *    MA 02111-1307  USA
 */
package alma.archive.database.lite;

import java.io.IOException;
import java.io.StringReader;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;

import org.jdom.Document;
import org.jdom.Element;
import org.jdom.JDOMException;
import org.jdom.input.SAXBuilder;
import org.jdom.xpath.XPath;

import alma.archive.database.helpers.DatabaseHelper;
import alma.archive.database.interfaces.DBCursor;
import alma.archive.database.interfaces.InternalIF;
import alma.archive.database.interfaces.SchemaManager;
import alma.archive.database.interfaces.UserManager;
import alma.archive.exceptions.ArchiveException;
import alma.archive.exceptions.ModuleCriticalException;
import alma.archive.exceptions.access.EntityDirtyException;
import alma.archive.exceptions.access.PermissionDeniedException;
import alma.archive.exceptions.general.DatabaseException;
import alma.archive.exceptions.general.EntityAlreadyDeletedException;
import alma.archive.exceptions.general.EntityDoesNotExistException;
import alma.archive.exceptions.general.EntityExistsException;
import alma.archive.exceptions.general.EntityUndeletedException;
import alma.archive.exceptions.general.HistoryInconsistencyException;
import alma.archive.exceptions.general.IllegalTimestampException;
import alma.archive.exceptions.general.UndefinedNamespaceException;
import alma.archive.exceptions.general.UnknownSchemaException;
import alma.archive.exceptions.syntax.IllegalHistoryNumberException;
import alma.archive.exceptions.syntax.MalformedPermissionsException;
import alma.archive.exceptions.syntax.MalformedQueryException;
import alma.archive.exceptions.syntax.MalformedURIException;
import alma.archive.exceptions.syntax.MalformedXMLException;
import alma.archive.exceptions.syntax.MalformedXPointerException;
import alma.archive.exceptions.syntax.UnderspecifiedQueryException;
import alma.archive.exceptions.syntax.UnknownFlagException;
import alma.archive.exceptions.user.UserDoesNotExistException;
import alma.archive.wrappers.ArchiveTimeStamp;
import alma.archive.wrappers.DocumentData;
import alma.archive.wrappers.Permissions;

/**
 * @author simon
 * 
 * A lite implementation of the InternalIF interface, this is designed to fill a
 * gap in testing intermediate layers of software and eliminate the need for a
 * database.
 */
public class LiteInternalIF implements InternalIF {
	private String _schemas = "archive://schemas";

	private String _namespaces = "archive://namespaces";

	private String _testns = "uid://test";

	private String _schemans = "http://www.w3.org/2001/XMLSchema";

	private URI schemasUri, namespacesUri, testns, schemans;

	HashMap documents = null;

	HashMap documentsData = null;

	HashMap deletedDocuments = null;

	private LiteSchemaManager smanager = null;

	/**
	 * @throws DatabaseException
	 * 
	 */
	public LiteInternalIF() throws DatabaseException {
		documents = new HashMap();
		documentsData = new HashMap();
		deletedDocuments = new HashMap();
		try {
			schemasUri = new URI(_schemas);
			namespacesUri = new URI(_namespaces);
			testns = new URI(_testns);
			schemans = new URI(_schemans);
		} catch (URISyntaxException e) {
			throw new DatabaseException(e);
		}
	}

	public void close() throws DatabaseException, ModuleCriticalException {
	}

	public void init() throws DatabaseException, ModuleCriticalException {
	}

	/**
	 * @see alma.archive.database.interfaces.InternalIF#store(java.net.URI,
	 *      java.lang.String, java.net.URI, java.lang.String, java.lang.String,
	 *      alma.archive.wrappers.Permissions, java.lang.String)
	 */
	public void store(URI uid, String xml, URI schema, String schemaName,
			String owner, Permissions permissions, String user, boolean emitLogs)
			throws ModuleCriticalException, DatabaseException,
			ArchiveException, MalformedURIException, MalformedXMLException,
			EntityExistsException, UnknownSchemaException,
			UserDoesNotExistException, MalformedPermissionsException {
		// System.out.println("Storing:" + xml + " at: " + uid.toASCIIString());
		if (documents.containsKey(uid)) {
			throw new EntityExistsException();
		}
		documents.put(uid, xml);
		DocumentData dd = new DocumentData(new ArchiveTimeStamp(), schema,
				owner, permissions, false, false, false, false, "");
		documentsData.put(uid, dd);
	}

	/**
	 * @see alma.archive.database.interfaces.InternalIF#update(java.net.URI,
	 *      alma.archive.wrappers.ArchiveTimeStamp, java.lang.String,
	 *      java.net.URI, boolean, java.lang.String)
	 */
	public void update(URI uid, ArchiveTimeStamp timeStamp, String xml,
			URI schema, boolean force, String user)
			throws ModuleCriticalException, PermissionDeniedException,
			DatabaseException, ArchiveException, MalformedURIException,
			HistoryInconsistencyException, MalformedXMLException,
			EntityDoesNotExistException, UserDoesNotExistException,
			UnknownSchemaException {
		if (documents.containsKey(uid)) {
			documents.put(uid, xml);
		} else {
			throw new EntityDoesNotExistException();
		}
	}

	/**
	 * @see alma.archive.database.interfaces.InternalIF#get(java.net.URI,
	 *      java.lang.String)
	 */
	public String get(URI uid, String user) throws ModuleCriticalException,
			PermissionDeniedException, EntityDirtyException,
			EntityDoesNotExistException, DatabaseException, ArchiveException,
			UserDoesNotExistException, MalformedURIException {
		if (documents.containsKey(uid)) {
			return (String) documents.get(uid);
		} else {
			throw new EntityDoesNotExistException();
		}
	}

	/**
	 * @see alma.archive.database.interfaces.InternalIF#get(java.net.URI,
	 *      java.lang.String, java.util.HashMap, java.lang.String)
	 */
	public String[] get(URI uid, String xpath, HashMap namespaces, String user)
			throws ModuleCriticalException, PermissionDeniedException,
			EntityDirtyException, EntityDoesNotExistException,
			DatabaseException, ArchiveException, MalformedURIException,
			IllegalHistoryNumberException, UserDoesNotExistException,
			MalformedXPointerException, UndefinedNamespaceException {
		String[] result = null;
		result = (String[]) documents.values().toArray(result);
		return null;
	}

	/**
	 * @see alma.archive.database.interfaces.InternalIF#get(java.net.URI,
	 *      alma.archive.wrappers.ArchiveTimeStamp, java.lang.String)
	 */
	public String get(URI uid, ArchiveTimeStamp timestamp, String user)
			throws ModuleCriticalException, PermissionDeniedException,
			EntityDirtyException, EntityDoesNotExistException,
			DatabaseException, ArchiveException, MalformedURIException,
			UserDoesNotExistException, IllegalTimestampException {
		if (documents.containsKey(uid)) {
			return (String) documents.get(uid);
		} else {
			throw new EntityDoesNotExistException();
		}
	}

	/**
	 * @see alma.archive.database.interfaces.InternalIF#getDirty(java.net.URI,
	 *      java.lang.String)
	 */
	public String getDirty(URI uid, String user)
			throws ModuleCriticalException, PermissionDeniedException,
			EntityDoesNotExistException, DatabaseException, ArchiveException,
			UserDoesNotExistException, MalformedURIException {
		if (documents.containsKey(uid)) {
			return (String) documents.get(uid);
		} else {
			throw new EntityDoesNotExistException();
		}
	}

	/**
	 * @see alma.archive.database.interfaces.InternalIF#delete(java.net.URI,
	 *      java.lang.String)
	 */
	public void delete(URI uid, String user) throws ModuleCriticalException,
			PermissionDeniedException, EntityDoesNotExistException,
			DatabaseException, ArchiveException, MalformedURIException,
			UserDoesNotExistException, EntityAlreadyDeletedException {
		if (documents.containsKey(uid)) {
			deletedDocuments.put(uid, documents.remove(uid));
		} else {
			throw new EntityDoesNotExistException();
		}
	}

	/**
	 * @see alma.archive.database.interfaces.InternalIF#undelete(java.net.URI,
	 *      java.lang.String)
	 */
	public void undelete(URI uid, String user)
			throws PermissionDeniedException, DatabaseException,
			ArchiveException, MalformedURIException,
			EntityDoesNotExistException, UserDoesNotExistException,
			EntityUndeletedException {
		if (deletedDocuments.containsKey(uid)) {
			documents.put(uid, deletedDocuments.remove(uid));
		} else {
			throw new EntityDoesNotExistException();
		}
	}

	/**
	 * @see alma.archive.database.interfaces.InternalIF#dirty(java.net.URI,
	 *      java.lang.String)
	 */
	public void dirty(URI uid, String user) throws ModuleCriticalException,
			PermissionDeniedException, EntityDoesNotExistException,
			DatabaseException, ArchiveException, MalformedURIException,
			UserDoesNotExistException {

	}

	/**
	 * @see alma.archive.database.interfaces.InternalIF#clean(java.net.URI,
	 *      java.lang.String)
	 */
	public void clean(URI uid, String user) throws ModuleCriticalException,
			PermissionDeniedException, DatabaseException, ArchiveException,
			MalformedURIException, EntityDoesNotExistException,
			UserDoesNotExistException, EntityUndeletedException {

	}

	/**
	 * @see alma.archive.database.interfaces.InternalIF#hidden(java.net.URI,
	 *      java.lang.String)
	 */
	public void hidden(URI uid, String user) throws PermissionDeniedException,
			EntityDoesNotExistException, DatabaseException, ArchiveException,
			MalformedURIException, UserDoesNotExistException {

	}

	/**
	 * @see alma.archive.database.interfaces.InternalIF#visible(java.net.URI,
	 *      java.lang.String)
	 */
	public void visible(URI uid, String user) throws ModuleCriticalException,
			PermissionDeniedException, DatabaseException, ArchiveException,
			MalformedURIException, EntityDoesNotExistException,
			UserDoesNotExistException, EntityUndeletedException {

	}

	/**
	 * @see alma.archive.database.interfaces.InternalIF#setVirtual(java.net.URI,
	 *      java.lang.String, boolean)
	 */
	public void setVirtual(URI uid, String user, boolean virtual)
			throws ModuleCriticalException, PermissionDeniedException,
			DatabaseException, ArchiveException, MalformedURIException,
			EntityDoesNotExistException, UserDoesNotExistException {
		// TODO Auto-generated method stub

	}

	/**
	 * @see alma.archive.database.interfaces.InternalIF#status(java.net.URI,
	 *      java.lang.String)
	 */
	public DocumentData status(URI uid, String user)
			throws ModuleCriticalException, PermissionDeniedException,
			EntityDoesNotExistException, DatabaseException, ArchiveException,
			UserDoesNotExistException, IllegalHistoryNumberException,
			MalformedURIException {
		if (documentsData.containsKey(uid)) {
			return (DocumentData) documentsData.get(uid);
		} else {
			throw new EntityDoesNotExistException();
		}
	}

	/**
	 * @see alma.archive.database.interfaces.InternalIF#allStatus(java.net.URI,
	 *      java.lang.String)
	 */
	public DocumentData[] allStatus(URI uid, String user)
			throws ModuleCriticalException, PermissionDeniedException,
			EntityDoesNotExistException, DatabaseException, ArchiveException,
			UserDoesNotExistException, MalformedURIException {
		DocumentData[] results = new DocumentData[1];
		results[0] = new DocumentData(new ArchiveTimeStamp(), testns, "owner",
				new Permissions(), false, false, false, false, "admin");
		return results;
	}

	/**
	 * @see alma.archive.database.interfaces.InternalIF#status(java.net.URI,
	 *      alma.archive.wrappers.ArchiveTimeStamp, java.lang.String)
	 */
	public DocumentData status(URI uid, ArchiveTimeStamp timestamp, String user)
			throws ModuleCriticalException, PermissionDeniedException,
			EntityDoesNotExistException, DatabaseException, ArchiveException,
			MalformedURIException, UserDoesNotExistException,
			IllegalTimestampException {
		if (documentsData.containsValue(uid)) {
			return (DocumentData) documentsData.get(uid);
		} else {
			throw new EntityDoesNotExistException();
		}
	}

	public URI[] queryRecent(ArchiveTimeStamp timestamp, String schema,
			String user) throws PermissionDeniedException, DatabaseException,
			ArchiveException, MalformedQueryException,
			UnderspecifiedQueryException, UnknownSchemaException,
			UserDoesNotExistException, UndefinedNamespaceException {
		// TODO implement this!!
		System.out
				.println("SEVERE: queryRecent() is not implemented for this database type. Please contact the Archive team.");
		return null;
	}

	public URI[] queryInterval(ArchiveTimeStamp timeFrom,
			ArchiveTimeStamp timeTo, String schema, String XPathQuery,
			String user) {
		// TODO implement this!!
		System.out
				.println("SEVERE: queryInterval() is not implemented for this database type. Please contact the Archive team.");
		return null;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see alma.archive.database.interfaces.InternalIF#query(java.lang.String,
	 *      java.lang.String, java.util.HashMap, boolean, java.lang.String)
	 */
	public DBCursor query(String query, String schema, HashMap namespaces,
			boolean dirtyRead, String user) throws ModuleCriticalException,
			PermissionDeniedException, DatabaseException, ArchiveException,
			MalformedQueryException, UnderspecifiedQueryException,
			UnknownSchemaException, UserDoesNotExistException,
			UndefinedNamespaceException {
		HashMap subset = filter(query);
		return new LiteCursor(subset);
	}

	public Collection<String> getLog(String timeFrom, String timeTo,
			short minType, short maxType, String routine, String source,
			String process, int maxRow) throws ArchiveException {
		throw new ArchiveException(
				"queryLog only supported for Oracle database.");
	}

	public DBCursor queryContent(String query, String schema,
			HashMap namespaces, boolean dirtyRead, String user)
			throws ModuleCriticalException, PermissionDeniedException,
			DatabaseException, ArchiveException, MalformedQueryException,
			UnderspecifiedQueryException, UnknownSchemaException,
			UserDoesNotExistException, UndefinedNamespaceException {
		return query(query, schema, namespaces, dirtyRead, user);
	}

	/**
	 * @see alma.archive.database.interfaces.InternalIF#queryIDs(java.lang.String,
	 *      java.lang.String, java.util.HashMap, boolean, java.lang.String)
	 */
	public URI[] queryIDs(String query, String schema, HashMap namespaces,
			boolean dirtyRead, String user) throws ModuleCriticalException,
			PermissionDeniedException, DatabaseException, ArchiveException,
			MalformedQueryException, UnderspecifiedQueryException,
			UnknownSchemaException, UserDoesNotExistException,
			UndefinedNamespaceException {
		HashMap subset = filter(query);
		URI[] results = new URI[0];
		results = (URI[]) subset.keySet().toArray(results);
		return results;
	}

	/**
	 * @see alma.archive.database.interfaces.InternalIF#queryIDsAll(java.lang.String,
	 *      java.lang.String, java.util.HashMap, java.lang.String)
	 */
	public URI[] queryIDsAll(String query, String schema, HashMap namespaces,
			String user) throws ModuleCriticalException,
			PermissionDeniedException, DatabaseException, ArchiveException,
			MalformedQueryException, UnderspecifiedQueryException,
			UnknownSchemaException, UserDoesNotExistException,
			UndefinedNamespaceException {
		HashMap subset = filter(query);
		URI[] results = new URI[0];
		results = (URI[]) subset.keySet().toArray(results);
		return results;
	}

	private HashMap filter(String query) throws MalformedQueryException,
			DatabaseException {
		try {
			HashMap results = new HashMap();

			XPath path = XPath.newInstance(query);
			Iterator iter = documents.keySet().iterator();
			while (iter.hasNext()) {
				URI uri = (URI) iter.next();
				// System.out.println(uri.toASCIIString());
				String content = (String) documents.get(uri);
				SAXBuilder builder = new SAXBuilder();
				try {
					Document doc = builder.build(new StringReader(content));
					Element root = doc.getRootElement();
					if (path.selectSingleNode(root) != null) {
						results.put(uri, content);
					}
				} catch (JDOMException e) {
					throw new DatabaseException(e);
				} catch (IOException e) {
					throw new DatabaseException(e);
				}
			}
			return results;
		} catch (JDOMException e) {
			throw new MalformedQueryException();
		}
	}

	/**
	 * @see alma.archive.database.interfaces.InternalIF#setFlag(java.net.URI,
	 *      java.lang.String, boolean, java.lang.String)
	 */
	public void setFlag(URI uid, String flagName, boolean flagValue, String user)
			throws ModuleCriticalException, PermissionDeniedException,
			DatabaseException, ArchiveException, MalformedURIException,
			EntityDoesNotExistException, UserDoesNotExistException,
			UnknownFlagException {

	}

	/**
	 * @see alma.archive.database.interfaces.InternalIF#changeOwner(java.net.URI,
	 *      java.lang.String, java.lang.String)
	 */
	public void changeOwner(URI uid, String newOwner, String user)
			throws ModuleCriticalException, PermissionDeniedException,
			DatabaseException, ArchiveException, MalformedURIException,
			EntityDoesNotExistException, UserDoesNotExistException {

	}

	/**
	 * @see alma.archive.database.interfaces.InternalIF#getSchemaManager(java.lang.String)
	 */
	public SchemaManager getSchemaManager(String user)
			throws ModuleCriticalException, ArchiveException,
			PermissionDeniedException, UserDoesNotExistException {
		if (smanager == null) {
			smanager = new LiteSchemaManager();
		}
		return smanager;
	}

	/**
	 * @see alma.archive.database.interfaces.InternalIF#getUserManager(java.lang.String)
	 */
	public UserManager getUserManager(String user)
			throws ModuleCriticalException, ArchiveException,
			PermissionDeniedException, UserDoesNotExistException {
		// TODO Auto-generated method stub
		return null;
	}

	/**
	 * @see alma.archive.database.interfaces.InternalIF#remove(java.net.URI,
	 *      boolean, java.lang.String)
	 */
	public void remove(URI uid, boolean keepHead, String user)
			throws ModuleCriticalException, PermissionDeniedException,
			DatabaseException, ArchiveException, MalformedURIException,
			UserDoesNotExistException, EntityDoesNotExistException {
		if (!keepHead) {
			documents.remove(uid);
		}
	}

	/**
	 * @see alma.archive.database.interfaces.InternalIF#setPermission(java.net.URI,
	 *      alma.archive.wrappers.Permissions, java.lang.String)
	 */
	public void setPermission(URI uid, Permissions permissions, String user)
			throws ModuleCriticalException, PermissionDeniedException,
			DatabaseException, ArchiveException, MalformedURIException,
			UserDoesNotExistException, MalformedPermissionsException {

	}

	/**
	 * @see alma.archive.database.interfaces.InternalIF#restoreState(alma.archive.wrappers.ArchiveTimeStamp,
	 *      java.lang.String)
	 */
	public void restoreState(ArchiveTimeStamp timestamp, String user)
			throws ModuleCriticalException, PermissionDeniedException,
			DatabaseException, ArchiveException, UserDoesNotExistException,
			IllegalTimestampException {

	}

	/**
	 * @see alma.archive.database.interfaces.InternalIF#cleanTestArea(java.lang.String)
	 */
	public void cleanTestArea(String user) throws ModuleCriticalException,
			PermissionDeniedException, DatabaseException, ArchiveException,
			UserDoesNotExistException {
		documents = new HashMap();
		deletedDocuments = new HashMap();
	}

	/**
	 * @see alma.archive.database.interfaces.InternalIF#notifyConfigChange()
	 */
	public void notifyConfigChange() throws ModuleCriticalException,
			DatabaseException {

	}

	public boolean ping() {
		return true;
	}

	public void updateXML(URI uid, String schema, String newChild)
			throws DatabaseException, EntityDoesNotExistException,
			MalformedXMLException {
		if (documents.containsKey(uid)) {
			// get old document
			String xml = (String) documents.get(uid);
			xml = DatabaseHelper.addAsLastChild(xml, newChild);
			documents.put(uid, xml);
		} else {
			throw new EntityDoesNotExistException((uid.toString()));
		}
	}

	/**
	 * Adds xmlElement as last child of the (first) element found by xPath in
	 * document uid (belonging to schema schema)
	 */
	public void addElement(URI uid, String schemaName, String xPath,
			String xmlElement, String user) throws EntityDoesNotExistException,
			DatabaseException, PermissionDeniedException,
			ModuleCriticalException {
		throw new DatabaseException("Operation addElement only supported for Oracle.");
	}
		
	/**
	 * The (first) element pointed at by xPath in document uid (belonging to
	 * schema schema) will be replaced by element xmlElement.
	 */
	public void updateElement(URI uid, String schemaName, String xPath,
			String xmlElement, String user) throws EntityDoesNotExistException,
			DatabaseException, PermissionDeniedException,
			ModuleCriticalException {
		throw new DatabaseException("Operation updateElement only supported for Oracle.");
			}

	/**
	 * The (first) element pointed at by xPath in document uid (belonging to
	 * schema schema) will be deleted.
	 */
	public void deleteElement(URI uid, String schemaName, String xPath,
			String user) throws EntityDoesNotExistException, DatabaseException,
			PermissionDeniedException, ModuleCriticalException {
		throw new DatabaseException("Operation deleteElement only supported for Oracle.");
			}
		
	public void storeLog(String message, int level, String attributeValue,
			String attributeValue2, String attributeValue3,
			String attributeValue4, String attributeValue5,
			String attributeValue6, String attributeValue7,
			String attributeValue8, String attributeValue9,
			String attributeValue10, String attributeValue11,
			String attributeValue12, String attributeValue13,
			String attributeValue14, String attributeValue15,
			String[] dataStrings, String xml, boolean commit, String steEnv) {
		// do nothing here, only used in Oracle

	}

}
