/*
 * 	  Created on 16-Nov-2005
 * 
 *    ALMA - Atacama Large Millimiter Array
 *    (c) European Southern Observatory, 2002
 *    Copyright by ESO (in the framework of the ALMA collaboration),
 *    All rights reserved
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License as published by the Free Software Foundation; either
 *    version 2.1 of the License, or (at your option) any later version.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
 *    MA 02111-1307  USA
 */
package alma.archive.client;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.FilenameFilter;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Hashtable;
import java.util.Iterator;

import org.jdom.Document;
import org.jdom.Element;
import org.jdom.JDOMException;
import org.jdom.input.SAXBuilder;
import org.jdom.output.Format;
import org.jdom.output.XMLOutputter;
import org.jdom.xpath.XPath;

import alma.archive.wrappers.TextFileWindow;

/**
 * @author simon
 */
public class LogAggregator
{

	private class LogEntry
	{
		private String tag = "";
		private String id = "";
		private String message = "";
		private String line = "";
		private String comments = "";
		private String path = "";
		private String fileName = "";
		private String packageName = "";
		private String codeSnippet = "";
		
		// ARCHIVE:<id>0001</id>A little message
		// ARCHIVE: A little message with a : in the middle
		
		/**
		 * This constructor will parse the string version of the log message,
		 * works using the colons so must only use them for this purpose.
		 */
		public LogEntry(
				String code, 
				int lineNumber, 
				File file, 
				String packageName,
				String codeSnippet)
		{
			//The line number is always set from the code as it will change
			line = Integer.toString(lineNumber);
			this.path = file.getAbsolutePath();
			this.fileName = file.getName();
			this.packageName = packageName;
			this.codeSnippet = codeSnippet;
			String [] parts = code.split(":");
			tag = parts[0];
			message = code.substring(tag.length() + 1);
			
			int start = message.indexOf("<id>");
			int end = message.indexOf("</id>");
			if ((start >= 0) && (end > 0))
			{
				id = message.substring(start + 4,end);
				message = message.substring(end + 5);
			}
			comments = "Insert comment here";
		}
		
		/*
		public LogEntry(Element log)
		{
			tag = log.getAttributeValue("tag");
			id = log.getAttributeValue("id");
			message = log.getAttributeValue("message");
			line = log.getAttributeValue("line");
			comments = log.getText();
		}
		*/
		
		public Element toXml()
		{
			Element log = new Element("log");
			log.setAttribute("tag",tag);
			log.setAttribute("id",id);
			log.setAttribute("message",message);
			log.setAttribute("line",line);
			log.setAttribute("fileName",fileName);
			log.setAttribute("packageName",packageName);
			
			Element comment = new Element("comments");
			comment.setText(comments);
			log.addContent(comment);
			
			Element code = new Element("code");
			code.setText(codeSnippet);
			log.addContent(code);
			return log;
		}
		
		public void update(Element log)
		{
			log.setAttribute("message",message);
			log.setAttribute("line",line);
			
			Element code = log.getChild("code");
			code.setText(codeSnippet);
		}
		
		public boolean hasId()
		{
			if (id == "") return false;
			else return true;
		}
		public String getLine(){return line;}
		public String getPath(){return path;}
		public String getId(){return id;}
		public void setId(String id){this.id = id;}
		
		public void setComments(String comments){this.comments = comments;}
		
		public String toString()
		{
			if (hasId()){
				return tag + ":<id>" + id + "</id>" + message;
			}
			else{
				return tag + ":" + message;
			}
		}
		
		public String toStringShort()
		{
			return tag + ":<id>" + id + "</id>";	
		}
		
		public String oldTag()
		{
			return tag + ":";
		}
		
		public String newTag()
		{
			return tag + ":<id>" + id + "</id>";
		}
	}

	private String targetDirectory;
	private String tag;
	private String outputFile;
	/**
	 * 
	 */
	public LogAggregator(String targetDirectory, String tag, String outputFile)
	{
		this.targetDirectory = targetDirectory;
		this.tag = tag;
		this.outputFile = outputFile;
	}
	
	public void run()
	{
		ArrayList logs = extractLogEntrys(targetDirectory,tag);
		ArrayList changed = assignIds(logs);
		updateChanged(changed);
		updateXml(outputFile,tag,logs);
	}
	
	public void printLogs()
	{
		ArrayList logs = extractLogEntrys(targetDirectory,tag);
		Iterator iter = logs.iterator();
		while (iter.hasNext())
		{
			LogEntry log = (LogEntry)iter.next();
			String msg = log.toString();
			System.out.println(log.getLine() + "::" + msg);
		}
	}
	
	private ArrayList assignIds(ArrayList logs)
	{	
		//run through and find the used id's
		//fetch new ones to assign
		Iterator iter = logs.iterator();
		ArrayList noid = new ArrayList();
		int maxid = 0;
		while (iter.hasNext())
		{
			LogEntry log = (LogEntry)iter.next();
			if (log.hasId())
			{
				String strInt = log.getId();
				int intid = Integer.parseInt(strInt);
				if (intid > maxid) maxid = intid;
			}
			else
			{
				noid.add(log);
			}
		}
		
		iter = noid.iterator();
		while (iter.hasNext())
		{
			LogEntry log = (LogEntry)iter.next();
			maxid++;
			log.setId(Integer.toString(maxid));
		}
		return noid;
	}
	
	private void updateChanged(ArrayList changed)
	{
		Hashtable files = new Hashtable();
		ArrayList fileNames = new ArrayList();
		
		Iterator iter = changed.iterator();
		while (iter.hasNext())
		{
			LogEntry log = (LogEntry)iter.next();		
			if (files.containsKey(log.getPath()))
			{
				Hashtable lines = (Hashtable)files.get(log.getPath());
				lines.put(log.getLine(),log);
			}
			else
			{
				Hashtable lines = new Hashtable();
				lines.put(log.getLine(),log);
				files.put(log.getPath(),lines);
				
				if (!fileNames.contains(log.getPath()))
					fileNames.add(log.getPath());
			}
		}
		
		iter = fileNames.iterator();
		while (iter.hasNext())
		{
			String fileName = (String)iter.next();
			Hashtable lines = (Hashtable)files.get(fileName);
			try{
				File file = new File(fileName);
				File newFile = new File(fileName + ".orig");
				file.renameTo(newFile);
				
				BufferedReader in = new BufferedReader(new FileReader(newFile));
				BufferedWriter out = new BufferedWriter(new FileWriter(file));
				int lineNumber = 0;
				String str;
				while ((str = in.readLine()) != null) {
		        	lineNumber++;
		        	if (lines.containsKey(Integer.toString(lineNumber)))
		        	{
		        		LogEntry log = 
		        			(LogEntry)lines.get(Integer.toString(lineNumber));
		        		str = str.replaceFirst(log.oldTag(),log.newTag());
		        	}
		        	out.write(str + "\n");
		        }
		        in.close();
		        out.close();
			}
			catch (IOException e){
				e.printStackTrace();
			}
		}
	}
	
	private void updateXml(String fileName, String tag, ArrayList logs)
	{
		File xmlFile = new File(fileName);
		String outFileName = fileName;
		Element aggregation = null;
		Element module = null;
		
		if (xmlFile.exists()){
			//outFileName = outFileName + ".new";
			SAXBuilder builder = new SAXBuilder();
			builder.setIgnoringElementContentWhitespace(true);
			try{
				Document doc = builder.build(
					new FileReader(xmlFile));
				aggregation = doc.getRootElement();
				String _path = "module[@name='" + tag + "']";
				XPath path = XPath.newInstance(_path);
				module = (Element)path.selectSingleNode(aggregation);
			}
			catch (JDOMException e) {
				e.printStackTrace();
			}
			catch (IOException e) {
				e.printStackTrace();
			}
			xmlFile.delete();
		}
		else{
			aggregation = new Element("aggregation");
		}
		
		if (module == null){
			module = new Element("module");
			module.setAttribute("name",tag);
			aggregation.addContent(module);
		}
		
		Iterator iter = logs.iterator();
		while (iter.hasNext()){
			LogEntry log = (LogEntry)iter.next();
			String id = log.getId();
			
			Element logElement = null;
			try{
				XPath path = XPath.newInstance("log[@id='" + id +"'");
				logElement = (Element)path.selectSingleNode(module);
				if (logElement == null){
					logElement = log.toXml();
					module.addContent(logElement);
				}
				else{
					log.update(logElement);
				}
			}
			catch (JDOMException e){
				e.printStackTrace();
			}
		}
		
		try{
			FileWriter writer = new FileWriter(outFileName);
			XMLOutputter out = new XMLOutputter(Format.getPrettyFormat());
			out.output(aggregation,writer);
		}
		catch (IOException e){
			e.printStackTrace();
		}
	}
	
	private ArrayList extractLogEntrys(String targetDirectory, String tag)
	{
		ArrayList results = new ArrayList();
		//run through all of the files looking getting the logs
		File dir = new File(targetDirectory);
		if (dir.exists())
		{
			FilenameFilter filter = new FilenameFilter() {
		        public boolean accept(File dir, String name) {
		            if (name.startsWith(".")) return false;
		            if (name.endsWith(".orig")) return false;
		            return true;
		        }
		    };
			String[] children = dir.list(filter);
			
			for (int x = 0; x < children.length; x++)
			{
				String fileName = dir.getAbsolutePath() + "/" + children[x];
				File file = new File(fileName);
				if (!file.isDirectory())
				{
					ArrayList fileResults = extractLogsFromFile(
						file,tag);
					results.addAll(fileResults);
				}
				else
				{
					//Recursion
					ArrayList subResults = extractLogEntrys(file.getPath(),tag);
					results.addAll(subResults);
				}
			}
		}	
		return results;
	}
	
	private ArrayList extractLogsFromFile(File file, String tag)
	{
		ArrayList result = new ArrayList();
		
		try
		{
	        //BufferedReader in = new BufferedReader(new FileReader(file));
			TextFileWindow in = new TextFileWindow(5,4,file);
	        String str;
	        String packageName = "";
	        int lineNumber = 0;
	        while ((str = in.readLine()) != null) {
	        	if (str.startsWith("package"))
	        	{
	        		packageName = str.substring(8,str.length()-1);
	        	}
	        	
	        	lineNumber++;
	        	
	            int start = str.indexOf(tag + ":");
	            if (start > 0)
	            {
	            	int offset = 0;
	            	int end = str.indexOf(");",start);
	            	while ((end < 0) && ((str += in.readLine()) != null))
	            	{
	            		offset++;
	            		//fetch the next line and add it
	            		end = str.indexOf(");",start);
	            	}
	            	String msg = str.substring(start,end);
	            	msg = msg.replaceAll("\t","");
	            	if (msg.endsWith("\""))
	            	{
	            		msg = msg.substring(0,msg.length() - 1);
	            	}
	            	LogEntry log = new LogEntry(
	            		msg,lineNumber,file,packageName,in.readWindow());
	            	result.add(log);
	            	lineNumber += offset;
	            }
	        }
	        in.close();
	    } catch (IOException e) {
	    	e.printStackTrace();
	    }
		
		return result;
	}
	
	private ArrayList usedIds(String targetDirectory, String tag)
	{
		return null;
	}

	public static void main(String[] args)
	{
		//arguents LogAggregator directory ModuleTag outputFile
		if (args.length != 3)
		{
			System.out.println(
			"Arguments: [working directory] [tag e.g. ARCHIVE] [xml file name]");
		}
		else
		{
			String directory = args[0];
			String tag = args[1];
			String outputFile = args[2];
			LogAggregator la = new LogAggregator(directory,tag,outputFile);
			la.run();
		}
	}
}
