/*
 *    ALMA - Atacama Large Millimiter Array
 *    (c) European Southern Observatory, 2002
 *    Copyright by ESO (in the framework of the ALMA collaboration),
 *    All rights reserved
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License as published by the Free Software Foundation; either
 *    version 2.1 of the License, or (at your option) any later version.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
 *    MA 02111-1307  USA
 *
 */
package alma.userrepository.roledirectory.acs;

import java.util.ArrayList;
import java.util.List;
import java.util.logging.Level;

import alma.UserRepositoryError.DuplicateObjectErrorEx;
import alma.UserRepositoryError.InvalidAttributeErrorEx;
import alma.UserRepositoryError.ObjectNotFoundErrorEx;
import alma.UserRepositoryError.PermissionErrorEx;
import alma.UserRepositoryError.UserRepositoryErrorEx;
import alma.UserRepositoryError.wrappers.AcsJDuplicateObjectErrorEx;
import alma.UserRepositoryError.wrappers.AcsJInvalidAttributeErrorEx;
import alma.UserRepositoryError.wrappers.AcsJObjectNotFoundErrorEx;
import alma.UserRepositoryError.wrappers.AcsJPermissionErrorEx;
import alma.UserRepositoryError.wrappers.AcsJUnknownCauseEx;
import alma.acs.component.ComponentImplBase;
import alma.userrepository.RoleDirectoryOperations;
import alma.userrepository.UserRole;
import alma.userrepository.UserRoleStruct;
import alma.userrepository.errors.DuplicateObjectException;
import alma.userrepository.errors.InvalidAttributeException;
import alma.userrepository.errors.ObjectNotFoundException;
import alma.userrepository.errors.PermissionException;
import alma.userrepository.errors.UserRepositoryException;
import alma.userrepository.roledirectory.Role;
import alma.userrepository.roledirectory.RoleDirectory;
import alma.userrepository.roledirectory.RoleDirectorySession;
import alma.userrepository.shared.DirectorySessionFactory;

/**
 * @author stewart
 * 
 */
public class RoleDirectoryImpl extends ComponentImplBase implements
        RoleDirectoryOperations {

    private ConnectionManager myConManager = ConnectionManager.getInstance();

    // ///////////////////////////////////////////////////////////
    // Implementation of ComponentLifecycle
    // ///////////////////////////////////////////////////////////

    public RoleDirectoryImpl() {
    }

    /**
     * (non-Javadoc)
     * 
     * @see alma.acs.component.ComponentLifecycle#cleanUp()
     */
    public void cleanUp() {
        m_logger.info("cleanUp() called...");
        try {
            myConManager.releaseAll();
        } catch (UserRepositoryException e) {
            m_logger
                    .log(
                            Level.SEVERE,
                            "Exceptions thrown while cleaning up UserRepository connections",
                            e);
        }
        super.cleanUp();
    }

    // ///////////////////////////////////////////////////////////
    // Implementation of ACSComponent
    // ///////////////////////////////////////////////////////////

    /**
     * (non-Javadoc)
     * 
     * @see alma.userrepository.RoleDirectory#assignRole(java.lang.String,
     *      java.lang.String, java.lang.String)
     */
    public void assignRole(String inConnectionId, String inUID, UserRole inRole)
            throws DuplicateObjectErrorEx, InvalidAttributeErrorEx,
            ObjectNotFoundErrorEx, PermissionErrorEx, UserRepositoryErrorEx {
        RoleDirectorySession session = myConManager.retrieve(inConnectionId);

        try {
            session.assignRole(inUID, idlRoleToJavaRole(inRole));
        } catch (UserRepositoryException e) {
            if (e instanceof DuplicateObjectException) {
                AcsJDuplicateObjectErrorEx acsE = new AcsJDuplicateObjectErrorEx(
                        e);
                throw acsE.toDuplicateObjectErrorEx();
            }
            if (e instanceof InvalidAttributeException) {
                AcsJInvalidAttributeErrorEx acsE = new AcsJInvalidAttributeErrorEx(
                        e);
                throw acsE.toInvalidAttributeErrorEx();
            }
            if (e instanceof ObjectNotFoundException) {
                AcsJObjectNotFoundErrorEx acsE = new AcsJObjectNotFoundErrorEx(
                        e);
                throw acsE.toObjectNotFoundErrorEx();
            }
            if (e instanceof PermissionException) {
                AcsJPermissionErrorEx acsE = new AcsJPermissionErrorEx(e);
                throw acsE.toPermissionErrorEx();
            }
            if (e instanceof UserRepositoryException) {
                AcsJUnknownCauseEx acsE = new AcsJUnknownCauseEx(e);
                throw acsE.toUserRepositoryErrorEx();
            }
        }
    }

    /**
     * (non-Javadoc)
     * 
     * @see alma.userrepository.RoleDirectory#getAllRoles(java.lang.String)
     */
    public UserRoleStruct[] getAllRoles(String inConnectionId)
            throws UserRepositoryErrorEx {

        RoleDirectorySession session = myConManager.retrieve(inConnectionId);
        try {
            return stringArrayToUserRoleArray(session.getAllRoles());
        } catch (UserRepositoryException e) {
            AcsJUnknownCauseEx acsE = new AcsJUnknownCauseEx(e);
            throw acsE.toUserRepositoryErrorEx();
        }
    }

    /**
     * (non-Javadoc)
     * 
     * @see alma.userrepository.RoleDirectory#getAllUserRoles(java.lang.String,
     *      java.lang.String)
     */
    public UserRoleStruct[] getAllUserRoles(String inConnectionId, String inUID)
            throws UserRepositoryErrorEx {

        RoleDirectorySession session = myConManager.retrieve(inConnectionId);
        try {
            return stringArrayToUserRoleArray(session.getAllUserRoles(inUID));
        } catch (UserRepositoryException e) {
            AcsJUnknownCauseEx acsE = new AcsJUnknownCauseEx(e);
            throw acsE.toUserRepositoryErrorEx();
        }
    }

    /**
     * (non-Javadoc)
     * 
     * @see alma.userrepository.RoleDirectory#getExplicitUserRoles(java.lang
     *      .String, java.lang.String)
     */
    public UserRoleStruct[] getExplicitUserRoles(String inConnectionId,
            String inUID) throws UserRepositoryErrorEx {

        RoleDirectorySession session = myConManager.retrieve(inConnectionId);
        try {
            return stringArrayToUserRoleArray(session
                    .getExplicitUserRoles(inUID));
        } catch (UserRepositoryException e) {
            AcsJUnknownCauseEx acsE = new AcsJUnknownCauseEx(e);
            throw acsE.toUserRepositoryErrorEx();
        }
    }

    /**
     * (non-Javadoc)
     * 
     * @see alma.userrepository.RoleDirectory#revokeRole(java.lang.String,
     *      java.lang.String, UserRole)
     */
    public void revokeRole(String inConnectionId, String inUID, UserRole inRole)
            throws UserRepositoryErrorEx, InvalidAttributeErrorEx,
            ObjectNotFoundErrorEx, PermissionErrorEx {
        RoleDirectorySession session = myConManager.retrieve(inConnectionId);

        try {
            session.revokeRole(inUID, idlRoleToJavaRole(inRole));
        } catch (UserRepositoryException e) {
            if (e instanceof InvalidAttributeException) {
                m_logger.log(Level.WARNING, "Unexpected error", e);
                AcsJInvalidAttributeErrorEx acsE = new AcsJInvalidAttributeErrorEx(
                        e);
                throw acsE.toInvalidAttributeErrorEx();
            }
            if (e instanceof ObjectNotFoundException) {
                AcsJObjectNotFoundErrorEx acsE = new AcsJObjectNotFoundErrorEx(
                        e);
                throw acsE.toObjectNotFoundErrorEx();
            }
            if (e instanceof PermissionException) {
                AcsJPermissionErrorEx acsE = new AcsJPermissionErrorEx(e);
                throw acsE.toPermissionErrorEx();
            }
            if (e instanceof UserRepositoryException) {
                AcsJUnknownCauseEx acsE = new AcsJUnknownCauseEx(e);
                throw acsE.toUserRepositoryErrorEx();
            }
        }
    }

    /**
     * (non-Javadoc)
     * 
     * @see alma.userrepository.RoleDirectory#createConnection(java.lang.String,
     *      java.lang.String)
     */
    public String createConnection(String inUID, String inPassword)
            throws UserRepositoryErrorEx {
        String connectionId = null;
        try {
            connectionId = myConManager.connect(inUID, inPassword);
        } catch (UserRepositoryException e) {
            AcsJUnknownCauseEx acsE = new AcsJUnknownCauseEx(e);
            throw acsE.toUserRepositoryErrorEx();
        }

        return connectionId;
    }

    /**
     * (non-Javadoc)
     * 
     * @see alma.userrepository.RoleDirectory#releaseConnection(java.lang.String)
     */
    public void releaseConnection(String inConnectionId)
            throws UserRepositoryErrorEx {
        try {
            myConManager.release(inConnectionId);
        } catch (UserRepositoryException e) {
            AcsJUnknownCauseEx acsE = new AcsJUnknownCauseEx(e);
            throw acsE.toUserRepositoryErrorEx();
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see alma.userrepository.RoleDirectory#isAlive()
     */
    public boolean isAlive() {
        return DirectorySessionFactory.getInstance().isAlive();
    }

    private UserRoleStruct[] stringArrayToUserRoleArray(
            RoleDirectory inMap) {
        List<UserRoleStruct> outList = new ArrayList<UserRoleStruct>();
        for (String applicationName : inMap.getApplicationNames()) {
            List<UserRole> userRoleList = new ArrayList<UserRole>();
            for (Role role : inMap.getRoles(applicationName)) {
                userRoleList.add(javaRoleToIDLRole(role));
            }
            UserRoleStruct struct = new UserRoleStruct(applicationName,
                    userRoleList.toArray(new UserRole[0]));
            outList.add(struct);
        }
        return outList.toArray(new UserRoleStruct[0]);
    }

    private UserRole javaRoleToIDLRole(Role inRole) {
        UserRole outRole = new UserRole();
        outRole.applicationName = inRole.getApplicationName();
        outRole.roleName = inRole.getRoleName();
        return outRole;
    }

    private Role idlRoleToJavaRole(UserRole inRole) {
        Role outRole = new Role(inRole.applicationName, inRole.roleName);
        return outRole;
    }
}
