/*
 *    ALMA - Atacama Large Millimiter Array
 *    (c) European Southern Observatory, 2002
 *    Copyright by ESO (in the framework of the ALMA collaboration),
 *    All rights reserved
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License as published by the Free Software Foundation; either
 *    version 2.1 of the License, or (at your option) any later version.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
 *    MA 02111-1307  USA
 *
 */
package alma.userrepository.projectdirectory.acs;

import java.util.ArrayList;
import java.util.List;
import java.util.logging.Logger;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.omg.CORBA.StringHolder;

import alma.ACS.ComponentStates;
import alma.ProjectCodeModule.ProjectCodeStruct;
import alma.ProjectCodeModule.ProjectTypeCodeEnum;
import alma.UserRepositoryError.DuplicateObjectErrorEx;
import alma.UserRepositoryError.InvalidAttributeErrorEx;
import alma.UserRepositoryError.InvalidOperationErrorEx;
import alma.UserRepositoryError.ObjectNotFoundErrorEx;
import alma.UserRepositoryError.PermissionErrorEx;
import alma.UserRepositoryError.UserRepositoryErrorEx;
import alma.UserRepositoryError.wrappers.AcsJDuplicateObjectErrorEx;
import alma.UserRepositoryError.wrappers.AcsJInvalidAttributeErrorEx;
import alma.UserRepositoryError.wrappers.AcsJInvalidOperationErrorEx;
import alma.UserRepositoryError.wrappers.AcsJObjectNotFoundErrorEx;
import alma.UserRepositoryError.wrappers.AcsJPermissionErrorEx;
import alma.UserRepositoryError.wrappers.AcsJUnknownCauseEx;
import alma.acs.component.ComponentLifecycle;
import alma.acs.container.ContainerServices;
import alma.archive.projectcode.ProjectCode;
import alma.archive.projectcode.TypeCode;
import alma.userrepository.ProjectDirectoryJ;
import alma.userrepository.ProjectRole;
import alma.userrepository.errors.DuplicateObjectException;
import alma.userrepository.errors.InvalidAttributeException;
import alma.userrepository.errors.InvalidOperationException;
import alma.userrepository.errors.ObjectNotFoundException;
import alma.userrepository.errors.PermissionException;
import alma.userrepository.errors.UserRepositoryException;
import alma.userrepository.projectdirectory.ProjectDirectorySession;
import alma.userrepository.shared.DirectorySessionFactory;
import alma.xmljbind.userrepository.user.User;

public class ProjectDirectoryImpl implements ComponentLifecycle,
        ProjectDirectoryJ {
    private ContainerServices m_containerServices;

    private Logger m_logger;

    protected Log log = LogFactory.getLog(this.getClass());

    private ConnectionManager MGR = ConnectionManager.getInstance();

    // ///////////////////////////////////////////////////////////
    // Implementation of ComponentLifecycle
    // ///////////////////////////////////////////////////////////

    /*
     * (non-Javadoc)
     * 
     * @seealma.acs.component.ComponentLifecycle#initialize(alma.acs.container.
     * ContainerServices)
     */
    public void initialize(ContainerServices containerServices) {
        m_containerServices = containerServices;
        m_logger = m_containerServices.getLogger();
        m_logger.info("initialize() called...");
    }

    /*
     * (non-Javadoc)
     * 
     * @see alma.acs.component.ComponentLifecycle#execute()
     */
    public void execute() {
        m_logger.info("execute() called...");
    }

    /*
     * (non-Javadoc)
     * 
     * @see alma.acs.component.ComponentLifecycle#cleanUp()
     */
    public void cleanUp() {
        m_logger.info("cleanUp() called...");
        try {
            MGR.releaseAll();
        } catch (UserRepositoryException e) {
            m_logger
                    .severe("Exceptions thrown while cleaning up UserRepository connections");
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see alma.acs.component.ComponentLifecycle#aboutToAbort()
     */
    public void aboutToAbort() {
        cleanUp();
        m_logger.info("managed to abort...");
    }

    // ///////////////////////////////////////////////////////////
    // Implementation of ACSComponent
    // ///////////////////////////////////////////////////////////

    /*
     * (non-Javadoc)
     * 
     * @see alma.ACS.ACSComponentOperations#componentState()
     */
    public ComponentStates componentState() {
        return m_containerServices.getComponentStateManager().getCurrentState();
    }

    /*
     * (non-Javadoc)
     * 
     * @see alma.ACS.ACSComponentOperations#name()
     */
    public String name() {
        return m_containerServices.getName();
    }

    // ///////////////////////////////////////////////////////////
    // Implementation of UserRepositoryComponentJ
    // ///////////////////////////////////////////////////////////

    /*
     * (non-Javadoc)
     * 
     * @see
     * alma.userrepository.ProjectDirectoryJ#assignProjectRole(java.lang.String,
     * java.lang.String, alma.userrepository.ProjectRole, java.lang.String)
     */
    public void assignProjectRole(String connectionId, String member,
            ProjectRole role, ProjectCodeStruct projectId)
            throws DuplicateObjectErrorEx, InvalidAttributeErrorEx,
            ObjectNotFoundErrorEx, PermissionErrorEx, UserRepositoryErrorEx,
            InvalidOperationErrorEx {
        ProjectDirectorySession session = MGR.retrieve(connectionId);
        alma.userrepository.projectdirectory.ProjectRole javaRole = convertFromCastor(role);

        try {
            session.assignProjectRole(member, idlCodeToJavaCode(projectId),
                    javaRole);
        } catch (UserRepositoryException e) {
            if (e instanceof DuplicateObjectException) {
                AcsJDuplicateObjectErrorEx acsE = new AcsJDuplicateObjectErrorEx(
                        e);
                throw acsE.toDuplicateObjectErrorEx();
            }
            if (e instanceof InvalidAttributeException) {
                AcsJInvalidAttributeErrorEx acsE = new AcsJInvalidAttributeErrorEx(
                        e);
                throw acsE.toInvalidAttributeErrorEx();
            }
            if (e instanceof ObjectNotFoundException) {
                AcsJObjectNotFoundErrorEx acsE = new AcsJObjectNotFoundErrorEx(
                        e);
                throw acsE.toObjectNotFoundErrorEx();
            }
            if (e instanceof PermissionException) {
                AcsJPermissionErrorEx acsE = new AcsJPermissionErrorEx(e);
                throw acsE.toPermissionErrorEx();
            }
            if (e instanceof InvalidOperationException) {
                AcsJInvalidOperationErrorEx acsE = new AcsJInvalidOperationErrorEx(
                        e);
                throw acsE.toInvalidOperationErrorEx();
            }
            if (e instanceof UserRepositoryException) {
                AcsJUnknownCauseEx acsE = new AcsJUnknownCauseEx(e);
                throw acsE.toUserRepositoryErrorEx();
            }
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * alma.userrepository.ProjectDirectoryJ#createNewProject(java.lang.String,
     * java.lang.String)
     */
    public void createNewProject(String connectionId,
            ProjectCodeStruct projectId) throws UserRepositoryErrorEx,
            DuplicateObjectErrorEx, InvalidAttributeErrorEx, PermissionErrorEx {
        ProjectDirectorySession session = MGR.retrieve(connectionId);

        try {
            session.createNewProject(idlCodeToJavaCode(projectId));
        } catch (UserRepositoryException e) {
            if (e instanceof DuplicateObjectException) {
                AcsJDuplicateObjectErrorEx acsE = new AcsJDuplicateObjectErrorEx(
                        e);
                throw acsE.toDuplicateObjectErrorEx();
            }
            if (e instanceof InvalidAttributeException) {
                AcsJInvalidAttributeErrorEx acsE = new AcsJInvalidAttributeErrorEx(
                        e);
                throw acsE.toInvalidAttributeErrorEx();
            }
            if (e instanceof PermissionException) {
                AcsJPermissionErrorEx acsE = new AcsJPermissionErrorEx(e);
                throw acsE.toPermissionErrorEx();
            }
            if (e instanceof UserRepositoryException) {
                AcsJUnknownCauseEx acsE = new AcsJUnknownCauseEx(e);
                throw acsE.toUserRepositoryErrorEx();
            }
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see alma.userrepository.ProjectDirectoryJ#exists(java.lang.String,
     * java.lang.String)
     */
    public boolean exists(String connectionId, ProjectCodeStruct projectId)
            throws UserRepositoryErrorEx {
        ProjectDirectorySession session = MGR.retrieve(connectionId);

        boolean result = false;

        try {
            result = session.exists(idlCodeToJavaCode(projectId));
        } catch (UserRepositoryException e) {
            AcsJUnknownCauseEx acsE = new AcsJUnknownCauseEx(e);
            throw acsE.toUserRepositoryErrorEx();
        }

        return result;
    }

    /*
     * (non-Javadoc)
     * 
     * @see alma.userrepository.ProjectDirectoryJ#getMembers(java.lang.String,
     * java.lang.String, alma.userrepository.ProjectRole)
     */
    public String[] getMembers(String connectionId,
            ProjectCodeStruct projectId, ProjectRole role)
            throws UserRepositoryErrorEx {
        ProjectDirectorySession session = MGR.retrieve(connectionId);

        alma.userrepository.projectdirectory.ProjectRole javaRole = convertFromCastor(role);
        String[] membersArray = {};

        try {
            List<String> members = session.listMembers(
                    idlCodeToJavaCode(projectId), javaRole);
            membersArray = members.toArray(new String[members.size()]);
        } catch (UserRepositoryException e) {
            AcsJUnknownCauseEx acsE = new AcsJUnknownCauseEx(e);
            throw acsE.toUserRepositoryErrorEx();
        }

        return membersArray;
    }

    /*
     * (non-Javadoc)
     * 
     * @see alma.userrepository.ProjectDirectoryJ#isMember(java.lang.String,
     * java.lang.String, java.lang.String, alma.userrepository.ProjectRole)
     */
    public boolean isMember(String connectionId, String member,
            ProjectCodeStruct projectId, ProjectRole role)
            throws UserRepositoryErrorEx {
        ProjectDirectorySession session = MGR.retrieve(connectionId);
        alma.userrepository.projectdirectory.ProjectRole javaRole = convertFromCastor(role);

        boolean result = false;
        try {
            result = session.isMember(member, idlCodeToJavaCode(projectId),
                    javaRole);
        } catch (UserRepositoryException e) {
            AcsJUnknownCauseEx acsE = new AcsJUnknownCauseEx(e);
            throw acsE.toUserRepositoryErrorEx();
        }

        return result;
    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * alma.userrepository.ProjectDirectoryJ#junkMethodSoWeGenerateBindingClass
     * (alma.xmljbind.userrepository.user.User, org.omg.CORBA.StringHolder)
     */
    public void junkMethodSoWeGenerateBindingClass(User junkIn,
            StringHolder junkOut) {
    }

    /*
     * (non-Javadoc)
     * 
     * @see alma.userrepository.ProjectDirectoryJ#listProjects(java.lang.String)
     */
    public ProjectCodeStruct[] listProjects(String connectionId)
            throws UserRepositoryErrorEx {
        ProjectDirectorySession session = MGR.retrieve(connectionId);

        List<ProjectCodeStruct> structList = new ArrayList<ProjectCodeStruct>();
        try {
            List<ProjectCode> codeList = session.listProjects();
            for (ProjectCode code : codeList) {
                structList.add(javaCodeToIDLCode(code));
            }
        } catch (UserRepositoryException e) {
            AcsJUnknownCauseEx acsE = new AcsJUnknownCauseEx(e);
            throw acsE.toUserRepositoryErrorEx();
        }

        return structList.toArray(new ProjectCodeStruct[0]);
    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * alma.userrepository.ProjectDirectoryJ#revokeProjectRole(java.lang.String,
     * java.lang.String, alma.userrepository.ProjectRole, java.lang.String)
     */
    public void revokeProjectRole(String connectionId, String member,
            ProjectRole role, ProjectCodeStruct projectId)
            throws UserRepositoryErrorEx, InvalidAttributeErrorEx,
            InvalidOperationErrorEx, ObjectNotFoundErrorEx, PermissionErrorEx {
        ProjectDirectorySession session = MGR.retrieve(connectionId);
        alma.userrepository.projectdirectory.ProjectRole javaRole = convertFromCastor(role);

        try {
            session.revokeProjectRole(member, idlCodeToJavaCode(projectId),
                    javaRole);
        } catch (UserRepositoryException e) {
            if (e instanceof InvalidAttributeException) {
                AcsJInvalidAttributeErrorEx acsE = new AcsJInvalidAttributeErrorEx(
                        e);
                throw acsE.toInvalidAttributeErrorEx();
            }
            if (e instanceof ObjectNotFoundException) {
                AcsJObjectNotFoundErrorEx acsE = new AcsJObjectNotFoundErrorEx(
                        e);
                throw acsE.toObjectNotFoundErrorEx();
            }
            if (e instanceof PermissionException) {
                AcsJPermissionErrorEx acsE = new AcsJPermissionErrorEx(e);
                throw acsE.toPermissionErrorEx();
            }
            if (e instanceof InvalidOperationException) {
                AcsJInvalidOperationErrorEx acsE = new AcsJInvalidOperationErrorEx(
                        e);
                throw acsE.toInvalidOperationErrorEx();
            }
            if (e instanceof UserRepositoryException) {
                AcsJUnknownCauseEx acsE = new AcsJUnknownCauseEx(e);
                throw acsE.toUserRepositoryErrorEx();
            }
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * alma.userrepository.ProjectDirectoryJ#createConnection(java.lang.String,
     * java.lang.String)
     */
    public String createConnection(String uid, String password)
            throws UserRepositoryErrorEx {
        String connectionId = null;
        try {
            connectionId = MGR.connect(uid, password);
        } catch (UserRepositoryException e) {
            AcsJUnknownCauseEx acsE = new AcsJUnknownCauseEx(e);
            throw acsE.toUserRepositoryErrorEx();
        }

        return connectionId;
    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * alma.userrepository.ProjectDirectoryJ#releaseConnection(java.lang.String)
     */
    public void releaseConnection(String connectionId)
            throws UserRepositoryErrorEx {
        try {
            MGR.release(connectionId);
        } catch (UserRepositoryException e) {
            AcsJUnknownCauseEx acsE = new AcsJUnknownCauseEx(e);
            throw acsE.toUserRepositoryErrorEx();
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see alma.userrepository.ProjectDirectoryJ#isAlive()
     */
    public boolean isAlive() {
        return DirectorySessionFactory.getInstance().isAlive();
    }

    public alma.userrepository.projectdirectory.ProjectRole convertFromCastor(
            alma.userrepository.ProjectRole castorRole) {
        if (castorRole.equals(alma.userrepository.ProjectRole.PI))
            return alma.userrepository.projectdirectory.ProjectRole.PI;
        if (castorRole.equals(alma.userrepository.ProjectRole.COI))
            return alma.userrepository.projectdirectory.ProjectRole.COI;
        if (castorRole.equals(alma.userrepository.ProjectRole.SYSTEM))
            return alma.userrepository.projectdirectory.ProjectRole.SYSTEM;
        if (castorRole.equals(alma.userrepository.ProjectRole.ANYROLE))
            return alma.userrepository.projectdirectory.ProjectRole.ANY;
        return null;
    }

    public ProjectCodeStruct javaCodeToIDLCode(ProjectCode inCode) {
        ProjectCodeStruct outStruct = new ProjectCodeStruct();
        outStruct.year = inCode.getYear();
        outStruct.semester = inCode.getSemester();
        outStruct.sequenceNumber = inCode.getSequenceNumber();
        outStruct.typeCode = ProjectTypeCodeEnum.from_int(inCode.getTypeCode()
                .ordinal());
        return outStruct;
    }

    public ProjectCode idlCodeToJavaCode(ProjectCodeStruct inStruct) {
        return new ProjectCode(inStruct.year, inStruct.semester,
                inStruct.sequenceNumber, Enum.valueOf(TypeCode.class,
                        inStruct.typeCode.toString()));
    }
}
