/*
 *    ALMA - Atacama Large Millimiter Array
 *    (c) European Southern Observatory, 2002
 *    Copyright by ESO (in the framework of the ALMA collaboration),
 *    All rights reserved
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License as published by the Free Software Foundation; either
 *    version 2.1 of the License, or (at your option) any later version.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
 *    MA 02111-1307  USA
 *
 */
package alma.userrepository.addressbook.acs;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.logging.Logger;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.omg.CORBA.StringHolder;

import alma.ACS.ComponentStates;
import alma.JavaContainerError.wrappers.AcsJContainerServicesEx;
import alma.UserRepositoryError.DataExpiredErrorEx;
import alma.UserRepositoryError.InvalidSearchFilterErrorEx;
import alma.UserRepositoryError.ObjectNotFoundErrorEx;
import alma.UserRepositoryError.PermissionErrorEx;
import alma.UserRepositoryError.UserRepositoryErrorEx;
import alma.UserRepositoryError.wrappers.AcsJDataExpiredErrorEx;
import alma.UserRepositoryError.wrappers.AcsJInvalidSearchFilterErrorEx;
import alma.UserRepositoryError.wrappers.AcsJObjectNotFoundErrorEx;
import alma.UserRepositoryError.wrappers.AcsJPermissionErrorEx;
import alma.UserRepositoryError.wrappers.AcsJUnknownCauseEx;
import alma.acs.component.ComponentLifecycle;
import alma.acs.container.ContainerServices;
import alma.userrepository.AddressBookEntryJ;
import alma.userrepository.AddressBookJ;
import alma.userrepository.addressbook.AddressBookSession;
import alma.userrepository.domainmodel.AddressBookEntry;
import alma.userrepository.errors.DataExpiredException;
import alma.userrepository.errors.InvalidSearchFilterException;
import alma.userrepository.errors.ObjectNotFoundException;
import alma.userrepository.errors.PermissionException;
import alma.userrepository.errors.UserRepositoryException;
import alma.userrepository.shared.DirectorySessionFactory;
import alma.xmljbind.userrepository.address.Address;
import alma.xmljbind.userrepository.address.AddressEntityT;
import alma.xmljbind.userrepository.preferences.Preferences;
import alma.xmljbind.userrepository.preferences.PreferencesEntityT;
import alma.xmljbind.userrepository.user.User;
import alma.xmljbind.userrepository.user.UserEntityT;

/**
 * @author stewart
 * 
 */
public class AddressBookImpl implements ComponentLifecycle, AddressBookJ {
    private ContainerServices m_containerServices;

    private Logger m_logger;

    protected Log log = LogFactory.getLog(this.getClass());

    private ConnectionManager MGR = ConnectionManager.getInstance();

    // ///////////////////////////////////////////////////////////
    // Implementation of ComponentLifecycle
    // ///////////////////////////////////////////////////////////

    /*
     * (non-Javadoc)
     * 
     * @seealma.acs.component.ComponentLifecycle#initialize(alma.acs.container.
     * ContainerServices)
     */
    public void initialize(ContainerServices containerServices) {
        m_containerServices = containerServices;
        m_logger = m_containerServices.getLogger();
        m_logger.info("initialize() called...");
    }

    /*
     * (non-Javadoc)
     * 
     * @see alma.acs.component.ComponentLifecycle#execute()
     */
    public void execute() {
        m_logger.info("execute() called...");
    }

    /*
     * (non-Javadoc)
     * 
     * @see alma.acs.component.ComponentLifecycle#cleanUp()
     */
    public void cleanUp() {
        m_logger.info("cleanUp() called...");
        try {
            MGR.releaseAll();
        } catch (UserRepositoryException e) {
            m_logger
                    .severe("Exceptions thrown while cleaning up UserRepository connections");
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see alma.acs.component.ComponentLifecycle#aboutToAbort()
     */
    public void aboutToAbort() {
        cleanUp();
        m_logger.info("managed to abort...");
    }

    // ///////////////////////////////////////////////////////////
    // Implementation of ACSComponent
    // ///////////////////////////////////////////////////////////

    /*
     * (non-Javadoc)
     * 
     * @see alma.ACS.ACSComponentOperations#componentState()
     */
    public ComponentStates componentState() {
        return m_containerServices.getComponentStateManager().getCurrentState();
    }

    /*
     * (non-Javadoc)
     * 
     * @see alma.ACS.ACSComponentOperations#name()
     */
    public String name() {
        return m_containerServices.getName();
    }

    // ///////////////////////////////////////////////////////////
    // Implementation of UserRepositoryComponentJ
    // ///////////////////////////////////////////////////////////

    public String createConnection(String uid, String password)
            throws UserRepositoryErrorEx {
        String connectionId = null;
        try {
            connectionId = MGR.connect(uid, password);
        } catch (UserRepositoryException e) {
            AcsJUnknownCauseEx acsE = new AcsJUnknownCauseEx(e);
            throw acsE.toUserRepositoryErrorEx();
        }

        return connectionId;
    }

    public void releaseConnection(String connectionId)
            throws UserRepositoryErrorEx {
        try {
            MGR.release(connectionId);
        } catch (UserRepositoryException e) {
            AcsJUnknownCauseEx acsE = new AcsJUnknownCauseEx(e);
            throw acsE.toUserRepositoryErrorEx();
        }
    }

    public boolean isAlive() {
        return DirectorySessionFactory.getInstance().isAlive();
    }

    // - user branch ----------------------------------------------------------

    /*
     * (non-Javadoc)
     * 
     * @see alma.userrepository.AddressBookJ#authenticate(java.lang.String,
     * java.lang.String)
     */
    public boolean authenticate(String uid, String password)
            throws UserRepositoryErrorEx {
        boolean outResult = false;
        try {
            outResult = DirectorySessionFactory.authenticate(uid, password);
        } catch (PermissionException e) {
            // authentication failed.
        } catch (UserRepositoryException e) {
            AcsJUnknownCauseEx acsE = new AcsJUnknownCauseEx(e);
            throw acsE.toUserRepositoryErrorEx();
        }
        return outResult;
    }

    /*
     * (non-Javadoc)
     * 
     * @see alma.userrepository.AddressBookJ#commitEntry(java.lang.String,
     * alma.userrepository.AddressBookEntryJ)
     */
    public void commitEntry(String connectionId, AddressBookEntryJ entryJ)
            throws PermissionErrorEx, DataExpiredErrorEx, UserRepositoryErrorEx {
        AddressBookSession session = MGR.retrieve(connectionId);

        try {
            AcsAddressBookEntry acsEntry = new AcsAddressBookEntry(entryJ);
            // External sync watchdog operates on LDAP user beans, so as we're
            // using an LDAP backend we convert to the correct type before
            // running
            // commit()
            AddressBookEntry entryWrapper = acsEntry.getLdapAddressBookEntry();

            synchronized (ConnectionManager.writeLock) {
                session.commitEntry(entryWrapper);
            }
        } catch (UserRepositoryException e) {
            if (e instanceof PermissionException) {
                AcsJPermissionErrorEx acsE = new AcsJPermissionErrorEx(e);
                throw acsE.toPermissionErrorEx();
            }
            if (e instanceof DataExpiredException) {
                AcsJDataExpiredErrorEx acsE = new AcsJDataExpiredErrorEx(e);
                throw acsE.toDataExpiredErrorEx();
            }
            if (e instanceof UserRepositoryException) {
                AcsJUnknownCauseEx acsE = new AcsJUnknownCauseEx(e);
                throw acsE.toUserRepositoryErrorEx();
            }
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see alma.userrepository.AddressBookJ#deleteEntry(java.lang.String,
     * alma.userrepository.AddressBookEntryJ)
     */
    public void deleteEntry(String connectionId, AddressBookEntryJ entry)
            throws PermissionErrorEx, ObjectNotFoundErrorEx,
            UserRepositoryErrorEx {
        AddressBookSession session = MGR.retrieve(connectionId);

        try {
            AddressBookEntry entryWrapper = new AcsAddressBookEntry(entry);
            synchronized (ConnectionManager.writeLock) {
                session.deleteEntry(entryWrapper);
            }
        } catch (UserRepositoryException e) {
            if (e instanceof PermissionException) {
                AcsJPermissionErrorEx acsE = new AcsJPermissionErrorEx(e);
                throw acsE.toPermissionErrorEx();
            }
            if (e instanceof ObjectNotFoundException) {
                AcsJObjectNotFoundErrorEx acsE = new AcsJObjectNotFoundErrorEx(
                        e);
                throw acsE.toObjectNotFoundErrorEx();
            }
            if (e instanceof UserRepositoryException) {
                AcsJUnknownCauseEx acsE = new AcsJUnknownCauseEx(e);
                throw acsE.toUserRepositoryErrorEx();
            }
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see alma.userrepository.AddressBookJ#deleteEntryByUid(java.lang.String,
     * java.lang.String)
     */
    public void deleteEntryByUid(String connectionId, String uid)
            throws PermissionErrorEx, ObjectNotFoundErrorEx,
            UserRepositoryErrorEx {
        AddressBookSession session = MGR.retrieve(connectionId);

        try {
            synchronized (ConnectionManager.writeLock) {
                session.deleteEntryByUid(uid);
            }
        } catch (UserRepositoryException e) {
            if (e instanceof PermissionException) {
                AcsJPermissionErrorEx acsE = new AcsJPermissionErrorEx(e);
                throw acsE.toPermissionErrorEx();
            }
            if (e instanceof ObjectNotFoundException) {
                AcsJObjectNotFoundErrorEx acsE = new AcsJObjectNotFoundErrorEx(
                        e);
                throw acsE.toObjectNotFoundErrorEx();
            }
            if (e instanceof UserRepositoryException) {
                AcsJUnknownCauseEx acsE = new AcsJUnknownCauseEx(e);
                throw acsE.toUserRepositoryErrorEx();
            }
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see alma.userrepository.AddressBookJ#exists(java.lang.String,
     * java.lang.String)
     */
    public boolean exists(String connectionId, String uid)
            throws UserRepositoryErrorEx {
        AddressBookSession session = MGR.retrieve(connectionId);
        try {
            return session.exists(uid);
        } catch (UserRepositoryException e) {
            AcsJUnknownCauseEx acsE = new AcsJUnknownCauseEx(e);
            throw acsE.toUserRepositoryErrorEx();
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see alma.userrepository.AddressBookJ#getUser(java.lang.String,
     * java.lang.String)
     */
    public User getUser(String connectionId, String uid)
            throws PermissionErrorEx, ObjectNotFoundErrorEx,
            UserRepositoryErrorEx {
        AddressBookSession session = MGR.retrieve(connectionId);
        User user = null;

        try {
            m_logger.finest("Calling getUser(" + uid
                    + ") on internal interface");
            user = new AcsUser(session.getUser(uid));
        } catch (UserRepositoryException e) {
            if (e instanceof ObjectNotFoundException) {
                AcsJObjectNotFoundErrorEx acsE = new AcsJObjectNotFoundErrorEx(
                        e);
                throw acsE.toObjectNotFoundErrorEx();
            }
            if (e instanceof PermissionException) {
                AcsJPermissionErrorEx acsE = new AcsJPermissionErrorEx(e);
                throw acsE.toPermissionErrorEx();
            }
            if (e instanceof UserRepositoryException) {
                System.out.println(e.toString());
                AcsJUnknownCauseEx acsE = new AcsJUnknownCauseEx(e);
                throw acsE.toUserRepositoryErrorEx();
            }
        }

        try {
            AcsUser userWrapper = new AcsUser(user);
            User acsUser = userWrapper.getCastor();

            UserEntityT usrEntity = new UserEntityT();
            m_containerServices.assignUniqueEntityId(usrEntity);
            m_logger.finest("Assigning entityId '" + usrEntity.getEntityId()
                    + "' to user");
            acsUser.setUserEntity(usrEntity);

            return acsUser;
        } catch (AcsJContainerServicesEx e) {
            AcsJUnknownCauseEx acsE = new AcsJUnknownCauseEx(e);
            throw acsE.toUserRepositoryErrorEx();
        } catch (Exception e) {
            System.out.println(e.toString());
            AcsJUnknownCauseEx acsE = new AcsJUnknownCauseEx(e);
            throw acsE.toUserRepositoryErrorEx();
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see alma.userrepository.AddressBookJ#getPreferences(java.lang.String,
     * java.lang.String)
     */
    public Preferences getPreferences(String connectionId, String uid)
            throws PermissionErrorEx, ObjectNotFoundErrorEx,
            UserRepositoryErrorEx {
        AddressBookSession session = MGR.retrieve(connectionId);
        Preferences preferences = null;

        try {
            m_logger.finest("Calling getPreferences(" + uid
                    + ") on internal interface");
            preferences = new AcsPreferences(session.getPreferences(uid));
        } catch (UserRepositoryException e) {
            if (e instanceof ObjectNotFoundException) {
                AcsJObjectNotFoundErrorEx acsE = new AcsJObjectNotFoundErrorEx(
                        e);
                throw acsE.toObjectNotFoundErrorEx();
            }
            if (e instanceof PermissionException) {
                AcsJPermissionErrorEx acsE = new AcsJPermissionErrorEx(e);
                throw acsE.toPermissionErrorEx();
            }
            if (e instanceof UserRepositoryException) {
                System.out.println(e.toString());
                AcsJUnknownCauseEx acsE = new AcsJUnknownCauseEx(e);
                throw acsE.toUserRepositoryErrorEx();
            }
        }

        try {
            AcsPreferences preferencesWrapper = new AcsPreferences(preferences);
            Preferences acsPreferences = preferencesWrapper.getCastor();

            PreferencesEntityT preferencesEntity = new PreferencesEntityT();
            m_containerServices.assignUniqueEntityId(preferencesEntity);
            m_logger.finest("Assigning entityId '"
                    + preferencesEntity.getEntityId() + "' to preferences");
            acsPreferences.setPreferencesEntity(preferencesEntity);

            return acsPreferences;
        } catch (AcsJContainerServicesEx e) {
            AcsJUnknownCauseEx acsE = new AcsJUnknownCauseEx(e);
            throw acsE.toUserRepositoryErrorEx();
        } catch (Exception e) {
            System.out.println(e.toString());
            AcsJUnknownCauseEx acsE = new AcsJUnknownCauseEx(e);
            throw acsE.toUserRepositoryErrorEx();
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see alma.userrepository.AddressBookJ#getEntry(java.lang.String,
     * java.lang.String)
     */
    public AddressBookEntryJ getEntry(String connectionId, String uid)
            throws PermissionErrorEx, ObjectNotFoundErrorEx,
            UserRepositoryErrorEx {
        AddressBookSession session = MGR.retrieve(connectionId);
        AddressBookEntry entry = null;

        try {
            entry = session.getEntry(uid);
        } catch (UserRepositoryException e) {
            if (e instanceof ObjectNotFoundException) {
                AcsJObjectNotFoundErrorEx acsE = new AcsJObjectNotFoundErrorEx(
                        e);
                throw acsE.toObjectNotFoundErrorEx();
            }
            if (e instanceof PermissionException) {
                AcsJPermissionErrorEx acsE = new AcsJPermissionErrorEx(e);
                throw acsE.toPermissionErrorEx();
            }
            if (e instanceof UserRepositoryException) {
                AcsJUnknownCauseEx acsE = new AcsJUnknownCauseEx(e);
                throw acsE.toUserRepositoryErrorEx();
            }
        }

        try {
            AcsAddressBookEntry entryWrapper = new AcsAddressBookEntry(entry);
            AddressBookEntryJ acsEntry = entryWrapper.getAddressBookEntryJ();

            UserEntityT usrEntity = new UserEntityT();
            m_containerServices.assignUniqueEntityId(usrEntity);
            acsEntry.thisUser.setUserEntity(usrEntity);

            PreferencesEntityT prefsEntity = new PreferencesEntityT();
            m_containerServices.assignUniqueEntityId(prefsEntity);
            acsEntry.thisPreferences.setPreferencesEntity(prefsEntity);

            for (Address address : acsEntry.addressArray) {
                AddressEntityT addressEntity = new AddressEntityT();
                m_containerServices.assignUniqueEntityId(addressEntity);
                address.setAddressEntity(addressEntity);
            }

            return acsEntry;
        } catch (AcsJContainerServicesEx e) {
            AcsJUnknownCauseEx acsE = new AcsJUnknownCauseEx(e);
            throw acsE.toUserRepositoryErrorEx();
        } catch (Exception e) {
            AcsJUnknownCauseEx acsE = new AcsJUnknownCauseEx(e);
            throw acsE.toUserRepositoryErrorEx();
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * alma.userrepository.AddressBookJ#junkMethodSoWeGenerateBindingClass(alma
     * .xmljbind.userrepository.user.User, org.omg.CORBA.StringHolder)
     */
    public void junkMethodSoWeGenerateBindingClass(User junkIn,
            StringHolder junkOut) {
        // TODO Auto-generated method stub
    }

    /*
     * (non-Javadoc)
     * 
     * @see alma.userrepository.AddressBookJ#listUsers(java.lang.String)
     */
    public String[] listUsers(String connectionId) throws UserRepositoryErrorEx {
        AddressBookSession session = MGR.retrieve(connectionId);

        String[] results = {};
        try {
            List<String> l = session.listUsers();
            results = l.toArray(new String[l.size()]);
        } catch (UserRepositoryException e) {
            if (e instanceof UserRepositoryException) {
                AcsJUnknownCauseEx acsE = new AcsJUnknownCauseEx(e);
                throw acsE.toUserRepositoryErrorEx();
            }
        }

        return results;
    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * alma.userrepository.AddressBookJ#searchGetAttributes(java.lang.String,
     * java.lang.String, java.lang.String[])
     */
    public String[] searchGetAttribute(String connectionId,
            String searchFilter, String attribute)
            throws InvalidSearchFilterErrorEx, UserRepositoryErrorEx {
        AddressBookSession session = MGR.retrieve(connectionId);
        List<String> attributesList = new ArrayList<String>();
        attributesList.add(attribute);

        String[] results = {};
        try {
            List<Map<String, String>> maps = session.searchGetAttributes(
                    searchFilter, attributesList);
            List<String> l = getValuesFromMaps(maps);
            results = l.toArray(new String[l.size()]);
        } catch (UserRepositoryException e) {
            if (e instanceof InvalidSearchFilterException) {
                AcsJInvalidSearchFilterErrorEx acsE = new AcsJInvalidSearchFilterErrorEx(
                        e);
                throw acsE.toInvalidSearchFilterErrorEx();
            }
            if (e instanceof UserRepositoryException) {
                AcsJUnknownCauseEx acsE = new AcsJUnknownCauseEx(e);
                throw acsE.toUserRepositoryErrorEx();
            }
        }

        return results;
    }

    private List<String> getValuesFromMaps(List<Map<String, String>> maps) {
        List<String> vals = new ArrayList<String>();
        for (Map<String, String> map : maps) {
            vals.addAll(map.values());
        }
        return vals;
    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * alma.userrepository.AddressBookJ#searchGetLocationAttributes(java.lang
     * .String, java.lang.String, java.lang.String, java.lang.String[])
     */
    public String[] searchLocationGetAttribute(String connectionId,
            String searchBase, String searchFilter, String attribute)
            throws InvalidSearchFilterErrorEx, ObjectNotFoundErrorEx,
            UserRepositoryErrorEx {
        AddressBookSession session = MGR.retrieve(connectionId);
        List<String> attributesList = new ArrayList<String>();
        attributesList.add(attribute);

        String[] results = {};
        try {
            List<Map<String, String>> maps = session
                    .searchLocationGetAttributes(searchBase, searchFilter,
                            attributesList);
            List<String> l = getValuesFromMaps(maps);
            results = l.toArray(new String[l.size()]);
        } catch (UserRepositoryException e) {
            if (e instanceof InvalidSearchFilterException) {
                AcsJInvalidSearchFilterErrorEx acsE = new AcsJInvalidSearchFilterErrorEx(
                        e);
                throw acsE.toInvalidSearchFilterErrorEx();
            }
            if (e instanceof ObjectNotFoundException) {
                AcsJObjectNotFoundErrorEx acsE = new AcsJObjectNotFoundErrorEx(
                        e);
                throw acsE.toObjectNotFoundErrorEx();
            }
            if (e instanceof UserRepositoryException) {
                AcsJUnknownCauseEx acsE = new AcsJUnknownCauseEx(e);
                throw acsE.toUserRepositoryErrorEx();
            }
        }

        return results;
    }

    /*
     * (non-Javadoc)
     * 
     * @see alma.userrepository.AddressBookJ#searchGetLocation(java.lang.String,
     * java.lang.String)
     */
    public String[] searchGetLocation(String connectionId, String searchFilter)
            throws InvalidSearchFilterErrorEx, UserRepositoryErrorEx {
        AddressBookSession session = MGR.retrieve(connectionId);

        String[] results = {};
        try {
            List<String> l = session.searchGetLocation(searchFilter);
            results = l.toArray(new String[l.size()]);
        } catch (UserRepositoryException e) {
            if (e instanceof InvalidSearchFilterException) {
                AcsJInvalidSearchFilterErrorEx acsE = new AcsJInvalidSearchFilterErrorEx(
                        e);
                throw acsE.toInvalidSearchFilterErrorEx();
            }
            if (e instanceof UserRepositoryException) {
                AcsJUnknownCauseEx acsE = new AcsJUnknownCauseEx(e);
                throw acsE.toUserRepositoryErrorEx();
            }
        }

        return results;
    }

    /*
     * (non-Javadoc)
     * 
     * @see alma.userrepository.AddressBookJ#setPassword(java.lang.String,
     * java.lang.String, java.lang.String)
     */
    public void setPassword(String connectionId, String uid, String password)
            throws PermissionErrorEx, ObjectNotFoundErrorEx,
            UserRepositoryErrorEx {
        AddressBookSession session = MGR.retrieve(connectionId);

        try {
            session.setPassword(uid, password);
        } catch (UserRepositoryException e) {
            if (e instanceof ObjectNotFoundException) {
                AcsJObjectNotFoundErrorEx acsE = new AcsJObjectNotFoundErrorEx(
                        e);
                throw acsE.toObjectNotFoundErrorEx();
            }
            if (e instanceof PermissionException) {
                AcsJPermissionErrorEx acsE = new AcsJPermissionErrorEx(e);
                throw acsE.toPermissionErrorEx();
            }
            if (e instanceof UserRepositoryException) {
                AcsJUnknownCauseEx acsE = new AcsJUnknownCauseEx(e);
                throw acsE.toUserRepositoryErrorEx();
            }
        }
    }

}
