/*******************************************************************************
*
* "@(#) $Id: rtAlarm_k.c,v 1.3 2007/10/15 07:25:59 bjeram Exp $"
*
* ALMA - Atacama Large Millimiter Array
* (c) Associated Universities Inc., 2003 
*
*This library is free software; you can redistribute it and/or
*modify it under the terms of the GNU Lesser General Public
*License as published by the Free Software Foundation; either
*version 2.1 of the License, or (at your option) any later version.
*
*This library is distributed in the hope that it will be useful,
*but WITHOUT ANY WARRANTY; without even the implied warranty of
*MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
*Lesser General Public License for more details.
*
*You should have received a copy of the GNU Lesser General Public
*License along with this library; if not, write to the Free Software
*Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
*
* who       when        what
* --------  --------    ----------------------------------------------
* bjeram  2007-06-29  created
*/

/* 
 * System Headers
 */
#ifdef  __KERNEL__
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/delay.h>
#include <linux/time.h>
#endif

/*
 * RTAI specific
 */
#ifndef  __KERNEL__
#include <stdlib.h>
#include <rtai_lxrt.h>
#endif
#include <rtai_malloc.h>
#include <rtai_registry.h>


#include <rtLog.h>

/* 
 * Local Headers
 */
#include "rtAlarm.h"

#ifdef __KERNEL__
MODULE_AUTHOR("Bogdan Jeram");
MODULE_DESCRIPTION("rtAlarm");
MODULE_LICENSE("GPL");
#endif

#define modName	"rtAlarm"

/***********************************************************************************/
rtAlarmSource_t *rtAlarmSourceInit(char *alarmSourceName, int alreadyCreated)
{
    rtlogRecord_t logRecord;
    rtAlarmSource_t *rtas;
    MBX *mbx;

    RTLOG_INFO(modName, "rtAlarmSourceInit %s, %d", alarmSourceName, alreadyCreated); 

    if (alreadyCreated)
	{
	mbx = (MBX*)rt_get_adr(nam2num(alarmSourceName)); /** here we just check if we have MBX 
							   in principal we can use rt_get_adr_cnt, but it is not supported in LXRT*/
	if (mbx == 0) 
	    {
	    rtlogRecord_t logRecord;
	    RTLOG_ERROR(modName, "failed to find already created RT Alarm source(MBX): %s", alarmSourceName);

	    return 0;
	    }/*if*/
	}
    
    mbx = rt_named_mbx_init(alarmSourceName, RTALARM_MBX_SIZE*sizeof(rtAlarm_t));
    if (mbx == 0)
	{
	rtlogRecord_t logRecord;
	RTLOG_ERROR(modName, "failed to create MBX: %s", alarmSourceName);

	return 0;
	}
    
#ifdef __KERNEL__
    rtas = (rtAlarmSource_t*)rt_malloc(sizeof(rtAlarmSource_t));
#else
    rtas = (rtAlarmSource_t*)malloc(sizeof(rtAlarmSource_t));
#endif
    rtas->mbx = mbx;
    strcpy(rtas->name, alarmSourceName);

    return rtas;
}/*rtAlarmSourceInit*/

/***********************************************************************************/
void rtAlarmSourceDelete(rtAlarmSource_t *alarmSource, int flush)
{
    rtAlarm_t alarm;
    rtlogRecord_t logRecord;
    int ret;

    RTLOG_INFO(modName, "rtAlarmSourceDelete %s", alarmSource->name);

    if(alarmSource==0 || alarmSource->mbx==0)
	{
	return;
	}/*if*/
    if(flush)
	{
	/* send to the logging system  all alarms that remained in mbx */
	while(!rt_mbx_receive_timed(alarmSource->mbx, &alarm, sizeof(rtAlarm_t), 1000000))
	    {
	    RTLOG_WARNING(modName,
			  "Alarm: Code=%d, Descriptor=%d, Property=%s has not been read from RT Alarm Source: %s", 
			  alarm.alarmCode,
			  alarm.descriptor,
			  alarm.property,
			  alarmSource->name 
		);
	    }/*while */
	}/*if */

    ret = rt_named_mbx_delete(alarmSource->mbx);
    if (ret<0)
	{
	RTLOG_WARNING(modName, "Problem deleting named MBX: %s. Reason: %d", alarmSource->name, ret);
	}
#ifdef __KERNEL__
    rt_free(alarmSource); 
#else
    free(alarmSource);
#endif
}/*rtAlarmSourceDelete*/

/***********************************************************************************/
int rtAlarmPush(rtAlarmSource_t *as, rtAlarmCode_t alarmCode, rtAlarmState_t state)
{
    rtAlarm_t alarm;

    alarm.alarmCode = alarmCode;
    alarm.timeStamp = rt_get_time();
    alarm.property[0] = 0;
    alarm.descriptor = state;

    return _rtAlarmPush(as, &alarm);
}/*rtAlarmPush*/

/***********************************************************************************/
int rtAlarmWithPropertyPush(rtAlarmSource_t *as, rtAlarmCode_t alarmCode, rtAlarmState_t state, char *property)
{
    rtAlarm_t alarm;

    alarm.alarmCode = alarmCode;
    alarm.timeStamp = rt_get_time();
    strncpy(alarm.property, property, 255);
    alarm.descriptor = state;

    return _rtAlarmPush(as, &alarm);
}/*rtAlarmWithPropertyPuhs*/

int _rtAlarmPush(rtAlarmSource_t *as, rtAlarm_t *alarm)
{
    rtlogRecord_t logRecord;
    int ret;
    
    if (as!=0 && as->mbx!=0)
	{
	ret = rt_mbx_send(as->mbx, alarm, sizeof(rtAlarm_t));
	if (ret==0)
	    {
	    return ret; /*=0*/
	    }
	else
	    {
	    RTLOG_WARNING(modName,
			  "Following alarm could not be sent: source %s, code %d, descriptor %d, property %s. Reason: %d", 
			  as->name, 
			  alarm->alarmCode,
			  alarm->descriptor,
			  alarm->property,
			  ret
		);
	    return ret;
	    }/*if-else*/
	}
    else
	{
	RTLOG_WARNING(modName,
		      "Following alarm could not be sent: source %s, code %d, descriptor %d, property %s. Reason: rtAlarmSource is NULL or MBX has not been initialized.", 
		      (as->name!=0) ? as->name : "NULL", 
		      alarm->alarmCode,
		      alarm->descriptor,
		      alarm->property
	    );
	}/*if-else*/
    
    return -1;
}/* _rtAlarmPush*/


#ifdef __KERNEL__

static int __init rtAlarm_init(void)
{ 
    rtlogRecord_t logRecord; 
    RTLOG_INFO(modName, "module initialized/loaded successfully");
    return 0;
}/*rtAlarm_init*/

static void __exit rtAlarm_exit(void)
{
    rtlogRecord_t logRecord;
    RTLOG_INFO(modName, "module exited successfully");
}/*rtAlarm_exit*/


module_init(rtAlarm_init);
module_exit(rtAlarm_exit);


#ifdef CONFIG_KBUILD
EXPORT_SYMBOL(rtAlarmSourceInit);
EXPORT_SYMBOL(rtAlarmSourceDelete);
EXPORT_SYMBOL(rtAlarmPush);
EXPORT_SYMBOL(rtAlarmWithPropertyPush);
EXPORT_SYMBOL(_rtAlarmPush);
#endif /*CONFIG_KBUILD*/

#endif /*__KERNEL__*/

/*___oOo___*/
