#ifndef _rtAlarmImpl_H_
#define _rtAlarmImpl_H_
/*******************************************************************************
* ALMA - Atacama Large Millimiter Array
* (c) European Southern Observatory, 2007 
* 
* This library is free software; you can redistribute it and/or
* modify it under the terms of the GNU Lesser General Public
* License as published by the Free Software Foundation; either
* version 2.1 of the License, or (at your option) any later version.
* 
* This library is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
* Lesser General Public License for more details.
* 
* You should have received a copy of the GNU Lesser General Public
* License along with this library; if not, write to the Free Software
* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
*
* "@(#) $Id: rtAlarmImpl.h,v 1.6 2009/10/13 11:02:50 bjeram Exp $"
*
* who       when      what
* --------  --------  ----------------------------------------------
* bjeram  2007-06-26  created
*/

/************************************************************************
 *
 *----------------------------------------------------------------------
 */

#ifndef __cplusplus
#error This is a C++ include file and cannot be used from plain C
#endif


#include <acsThread.h>
#include <baciCharacteristicComponentImpl.h>
#include <AlarmSystemInterface.h>
#include <expat.h>
#include <rtlogTimeDataThread.h>

#include <rtai_mbx.h>

#include "rtAlarmData.h"
#include "rtAlarmS.h"


namespace rtAlarm
{

    /**
     * structure that contains data to create RT alarm source:
     * name of the source, faultFamily and faultMember
     */
    typedef struct _rtAlarmSourceData
    {
	ACE_CString name; 	///< name of the source. It is used for creating MBX and RT task so first 6(5) leters should be different!!
	ACE_CString faultFamily;
	ACE_CString faultMember;

	TimeDataThread *timeDataSource_mp; ///< Source(Thread) that reads time data structure from RT and it is used to synch time
    } rtAlarmSourceData;

/**
 * rtAlarmThread thread which reads alarms from the kernel space. There is one thread per RT alarm source 
 */
class rtAlarmThread : public ACS::Thread
{
  public:
    rtAlarmThread(const ACE_CString &_name, rtAlarmSourceData &_asd);

    virtual ~rtAlarmThread();

    void onStart();

    void run();

  private:
    acsalarm::AlarmSystemInterface* alarmSource_mp;  ///< The alarm system source
    auto_ptr<acsalarm::FaultState> faultState_map; ///< Alarm Fault state;
   
    MBX* alarmMBX_p;   ///< MBX where alarms comes from kernel space
    RT_TASK* readingAlarmsTask;  ///< task that is used to read from MBX
   
    rtAlarmSourceData &asd;  ///< information about RT alarm source
};//class rtAlarmThread


/**
 * Implementation of the rtAlarmImpl component
 */
class rtAlarmImpl :
    public virtual baci::CharacteristicComponentImpl,
    public POA_ACS_RT::rtAlarm
{
  public:
    rtAlarmImpl(const ACE_CString &name, maci::ContainerServices* containerServices);

    virtual ~rtAlarmImpl();

    /**
     * @throw ACSErr::ACSbaseExImpl
     */
    virtual void initialize();

    virtual void cleanUp();
  protected:
    TimeDataThread *timeDataThread_mp; ///< Thread that reads time data structure from RT

    /// vector for threads which retrieve alarms from kernel space
    std::vector<rtAlarmThread*> rtAlarmThreads;
    /// vector that holds information about rt alarm sources
    std::vector<rtAlarmSourceData> rtAlarmSources;
    
    /**
       class to parse entry from CDB. It is used inside rtAlarmImpl
     */
    class rtAlarmCDBParser
    {
      public:
	/** parse xml and result put to the vector
	 * @parm xml XML that has to be parsed
         * @parm asdl vector where parsed data will be stored
	 * @return 0 if OK
	 */
	int parse(char *xml, std::vector<rtAlarmSourceData>* asdl);
	
      private:
	// expat handlers for elements
	static void startElm(void *data, const XML_Char *el, const XML_Char **attr);
	static void endElm(void *data, const XML_Char *el);

	// temporary storage for retreived data ( hope that parsing is not done in paralel)
	static rtAlarmSourceData asd;
    };//class rtAlarmSourceCDB
}; //class rtAlarmImpl

};//namespace rtAlarm
#endif /*!_rtAlarmImpl_H_*/
