/*
 *    ALMA - Atacama Large Millimiter Array
 *    (c) European Southern Observatory, 2002
 *    Copyright by ESO (in the framework of the ALMA collaboration),
 *    All rights reserved
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License as published by the Free Software Foundation; either
 *    version 2.1 of the License, or (at your option) any later version.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
 *    MA 02111-1307  USA
 *
 */
package alma.userrepository.shared.ldap;

import java.util.Properties;

import alma.userrepository.shared.PropertiesLoader;

public class ConnectionProperties extends PropertiesLoader {
    private static final String RESOURCE_NAME = "ldap.connection";

    // JNDI property keys
    static final String INITIAL_FACTORY_KEY = "java.naming.factory.initial";
    static final String STATE_FACTORY_KEY = "java.naming.factory.state";
    static final String OBJECT_FACTORY_KEY = "java.naming.factory.object";
    static final String AUTHENTICATION_KEY = "java.naming.security.authentication";
    static final String PROVIDER_URL_KEY = "java.naming.provider.url";

    // User and password to use for 'anonymous' access.
    static final String READ_ONLY_USER_DN_KEY = "alma.userrepository.ldap.readonly.user.dn";
    static final String READ_ONLY_USER_PASSWORD_KEY = "alma.userrepository.ldap.readonly.user.password";

    // OS account default properties
    private String initialFactory = "com.sun.jndi.ldap.LdapCtxFactory";
    private String stateFactory = "alma.userrepository.addressbook.ldap.UserRepositoryStateFactory:com.sun.jndi.ldap.obj.LdapGroupFactory";
    private String objectFactory = "alma.userrepository.addressbook.ldap.UserRepositoryObjectFactory:com.sun.jndi.ldap.obj.LdapGroupFactory";
    private String authentication = "simple";
    private String providerUrl = "ldap://127.0.0.1:1389/";
    private String userName = "uid=readUser,ou=master,dc=alma,dc=info";
    private String userPassword = "oS!iMLDAP!";

    private volatile static ConnectionProperties instance = null;

    public static ConnectionProperties getInstance() {
        if (instance == null) {
            synchronized (ConnectionProperties.class) {
                if (instance == null) {
                    instance = new ConnectionProperties();
                }
            }
        }
        return instance;
    }

    private ConnectionProperties() {
        super(RESOURCE_NAME);
        getProperties();
    }

    @Override
    protected Properties getDefaultProperties() {
        Properties outProperties = new Properties();
        outProperties.put(INITIAL_FACTORY_KEY, getInitialFactory());
        outProperties.put(STATE_FACTORY_KEY, getStateFactory());
        outProperties.put(OBJECT_FACTORY_KEY, getObjectFactory());
        outProperties.put(AUTHENTICATION_KEY, getAuthentication());
        outProperties.put(PROVIDER_URL_KEY, getProviderUrl());
        outProperties.put(READ_ONLY_USER_DN_KEY, getReadOnlyUserName());
        outProperties.put(READ_ONLY_USER_PASSWORD_KEY,
                getReadOnlyUserPassword());
        return outProperties;
    }

    @Override
    protected void updateSettingsFromProperties() {
        try {
            setInitialFactory(myProperties.getProperty(INITIAL_FACTORY_KEY));
            setStateFactory(myProperties.getProperty(STATE_FACTORY_KEY));
            setObjectFactory(myProperties.getProperty(OBJECT_FACTORY_KEY));
            setAuthentication(myProperties.getProperty(AUTHENTICATION_KEY));
            setProviderUrl(myProperties.getProperty(PROVIDER_URL_KEY));
            setReadOnlyUserName(myProperties.getProperty(READ_ONLY_USER_DN_KEY));
            setReadOnlyUserPassword(myProperties
                    .getProperty(READ_ONLY_USER_PASSWORD_KEY));
        } catch (Exception e) {
            if (log.isErrorEnabled())
                log.error("Error updating properties: " + e);
        }
    }

    @Override
    protected void updatePropertiesFromSettings() {
        myProperties.put(INITIAL_FACTORY_KEY, getInitialFactory());
        myProperties.put(STATE_FACTORY_KEY, getStateFactory());
        myProperties.put(OBJECT_FACTORY_KEY, getObjectFactory());
        myProperties.put(AUTHENTICATION_KEY, getAuthentication());
        myProperties.put(PROVIDER_URL_KEY, getProviderUrl());
        myProperties.put(READ_ONLY_USER_DN_KEY, getReadOnlyUserName());
        myProperties
                .put(READ_ONLY_USER_PASSWORD_KEY, getReadOnlyUserPassword());
    }

    // -------------------------------------------------------------- Properties

    public String getInitialFactory() {
        return initialFactory;
    }

    public void setInitialFactory(String factoryInitial) {
        this.initialFactory = factoryInitial;
    }

    public String getObjectFactory() {
        return objectFactory;
    }

    public void setObjectFactory(String factoryObject) {
        this.objectFactory = factoryObject;
    }

    public String getStateFactory() {
        return stateFactory;
    }

    public void setStateFactory(String factoryState) {
        this.stateFactory = factoryState;
    }

    public String getProviderUrl() {
        return providerUrl;
    }

    public void setProviderUrl(String providerUrl) {
        this.providerUrl = providerUrl;
    }

    public String getAuthentication() {
        return authentication;
    }

    public void setAuthentication(String security) {
        this.authentication = security;
    }

    public String getReadOnlyUserName() {
        return this.userName;
    }

    public void setReadOnlyUserName(String inName) {
        this.userName = inName;
    }

    public String getReadOnlyUserPassword() {
        return this.userPassword;
    }

    public void setReadOnlyUserPassword(String inPassword) {
        this.userPassword = inPassword;
    }

}
