/*
 *    ALMA - Atacama Large Millimiter Array
 *    (c) European Southern Observatory, 2002
 *    Copyright by ESO (in the framework of the ALMA collaboration),
 *    All rights reserved
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License as published by the Free Software Foundation; either
 *    version 2.1 of the License, or (at your option) any later version.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
 *    MA 02111-1307  USA
 *
 */
package alma.userrepository.roledirectory.ldap;

import java.util.ArrayList;
import java.util.List;

import javax.naming.InvalidNameException;
import javax.naming.NameNotFoundException;
import javax.naming.NamingEnumeration;
import javax.naming.NamingException;
import javax.naming.NoPermissionException;
import javax.naming.directory.AttributeInUseException;
import javax.naming.directory.DirContext;
import javax.naming.directory.InvalidAttributeValueException;
import javax.naming.directory.NoSuchAttributeException;
import javax.naming.directory.SearchControls;
import javax.naming.directory.SearchResult;
import javax.naming.ldap.LdapName;
import javax.naming.ldap.Rdn;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import alma.userrepository.errors.DuplicateObjectException;
import alma.userrepository.errors.InvalidAttributeException;
import alma.userrepository.errors.ObjectNotFoundException;
import alma.userrepository.errors.PermissionException;
import alma.userrepository.errors.UserRepositoryException;
import alma.userrepository.roledirectory.Role;
import alma.userrepository.roledirectory.RoleDirectory;
import alma.userrepository.roledirectory.RoleDirectorySession;
import alma.userrepository.shared.ldap.DnDirectory;

import com.sun.jndi.ldap.obj.GroupOfNames;

public class LdapRoleDirectorySession implements RoleDirectorySession {
    protected Log log = LogFactory.getLog(this.getClass());

    private DirContext ctx = null;

    @SuppressWarnings("unused")
    private LdapRoleDirectorySession() {
    }

    public LdapRoleDirectorySession(DirContext ctx)
            throws UserRepositoryException {
        this.ctx = ctx;
    }

    /**
     * @see alma.userrepository.roledirectory.RoleDirectorySession#close()
     */
    public void close() throws UserRepositoryException {
        try {
            ctx.close();
        } catch (NamingException e) {
            if (log.isErrorEnabled()) {
                log.error("Couldn't close session: ", e);
            }
            throw new UserRepositoryException("Couldn't close session: ", e);
        }
    }

    /**
     * @throws UserRepositoryException
     * @see alma.userrepository.roledirectory.RoleDirectorySession#getAllRoles
     */
    public RoleDirectory getAllRoles() throws UserRepositoryException {
        return translateList("ALL ROLES", listRoles());
    }

    /**
     * @see alma.userrepository.roledirectory.RoleDirectorySession#getAllUserRoles
     */
    public RoleDirectory getAllUserRoles(String inUserId)
            throws UserRepositoryException {
        return translateList(inUserId, listRoles(inUserId));
    }

    /**
     * @see alma.userrepository.roledirectory.RoleDirectorySession#getExplicitUserRoles
     */
    public RoleDirectory getExplicitUserRoles(String inUserId)
            throws UserRepositoryException {
        return translateList(inUserId, listExplicitRoles(inUserId));
    }

    /**
	 */
    private List<String> listRoles() throws UserRepositoryException {
        List<String> outRoles = new ArrayList<String>();

        SearchControls controls = new SearchControls();
        controls.setSearchScope(SearchControls.SUBTREE_SCOPE);
        controls.setReturningAttributes(new String[0]);
        String filter = "(cn=*)";

        if (log.isTraceEnabled()) {
            log.trace("Searching " + DnDirectory.ROLE_BASE + " with filter '"
                    + filter + "'.");
        }

        NamingEnumeration<SearchResult> ne;
        try {
            ne = ctx.search(DnDirectory.ROLE_BASE, filter, controls);
        } catch (NamingException e) {
            throw new UserRepositoryException(
                    "Unexpected error occurred while searching contents of 'roles' branch.",
                    e);
        }

        try {
            while (ne.hasMoreElements()) {
                SearchResult result = ne.next();
                outRoles.add(result.getNameInNamespace());
            }
        } catch (NamingException e) {
            throw new UserRepositoryException(
                    "Unexpected error occurred while fetching roles.", e);
        }

        return outRoles;
    }

    /**
	 */
    private List<String> listRoles(String uid) throws UserRepositoryException {
        String userDn = DnDirectory.getUserDn(uid);
        List<String> l;
        try {
            l = recursiveFindRolesWithMember(userDn);
        } catch (NamingException e) {
            throw new UserRepositoryException(
                    "Unexpected error occurred while listing roles for user with uid '"
                            + uid + "'.", e);
        }
        return l;
    }

    /**
     */
    private List<String> listExplicitRoles(String uid)
            throws UserRepositoryException {
        List<String> outRoles = new ArrayList<String>();

        SearchControls controls = new SearchControls();
        controls.setSearchScope(SearchControls.SUBTREE_SCOPE);
        controls.setReturningAttributes(new String[0]);
        controls.setDerefLinkFlag(true);
        String filter = "(member=" + DnDirectory.getUserDn(uid) + ")";

        NamingEnumeration<SearchResult> ne;
        try {
            ne = ctx.search(DnDirectory.ROLE_BASE, filter, controls);
        } catch (NamingException e) {
            throw new UserRepositoryException(
                    "Unexpected error occurred while searching for user '"
                            + uid + "' in roles.", e);
        }

        while (ne.hasMoreElements()) {
            try {
                SearchResult result = ne.next();
                outRoles.add(result.getNameInNamespace());
            } catch (NamingException e) {
                throw new UserRepositoryException(
                        "Unexpected error occurred while iterating results for 'match uid="
                                + uid + "' in roles'.", e);
            }
        }

        return outRoles;
    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * alma.userrepository.roledirectory.RoleDirectorySession#addUserToRole(
     * java.lang.String, java.lang.String)
     */
    public void assignRole(String uid, Role role)
            throws UserRepositoryException {
        // get the fqdn of the user and role
        String roleDn = DnDirectory.getRoleDn(role);
        String userDn = DnDirectory.getUserDn(uid);

        // we do not need to use a fresh connection, as the security
        // is calculated dynamically by the LDAP server on each invocation.

        try {
            GroupOfNames group = (GroupOfNames) ctx.lookup(roleDn);
            group.addMember(userDn);
            group.close();
        } catch (AttributeInUseException e1) {
            throw new DuplicateObjectException("The user with uid '" + uid
                    + "' already belongs to the '" + role + "' role.", e1);
        } catch (NameNotFoundException e2) {
            throw new ObjectNotFoundException("The role '" + role
                    + "' was not found.", e2);
        } catch (NoPermissionException e3) {
            throw new PermissionException(
                    "Insufficient permission to add user with uid '" + uid
                            + "' to the '" + role + "' role.", e3);
        } catch (InvalidAttributeValueException e4) {
            throw new InvalidAttributeException("The fully qualified user dn '"
                    + userDn + "' is invalid.", e4);
        } catch (NamingException e5) {
            throw new UserRepositoryException(
                    "Unexpected error occurred while adding user with uid '"
                            + uid + "' to the '" + role + "' role.", e5);
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * alma.userrepository.roledirectory.RoleDirectorySession#removeUserFromRole
     * (java.lang.String, java.lang.String)
     */
    public void revokeRole(String uid, Role role)
            throws UserRepositoryException {
        // get the fqdn of the user and role
        String userDn = DnDirectory.getUserDn(uid);
        String roleDn = DnDirectory.getRoleDn(role);

        // we do not need to use a fresh connection, as the security
        // is calculated dynamically by the LDAP server on each invocation.

        try {
            GroupOfNames group = (GroupOfNames) ctx.lookup(roleDn);
            group.removeMember(userDn);
            group.close();
        } catch (NoSuchAttributeException e1) {
            throw new InvalidAttributeException("The user with uid '" + uid
                    + "' is not a member of the '" + role + "' role.", e1);
        } catch (NameNotFoundException e2) {
            throw new ObjectNotFoundException("The role '" + role
                    + "' was not found.", e2);
        } catch (NoPermissionException e3) {
            throw new PermissionException(
                    "Insufficient permission to add user with uid '" + uid
                            + "' to the '" + role + "' role.", e3);
        } catch (NamingException e4) {
            throw new UserRepositoryException(
                    "Unexpected error occurred while adding user with uid '"
                            + uid + "' to the '" + role + "' role.", e4);
        }
    }

    private List<String> recursiveFindRolesWithMember(String memberDn)
            throws NamingException {
        List<String> outRoles = new ArrayList<String>();

        SearchControls controls = new SearchControls();
        controls.setSearchScope(SearchControls.SUBTREE_SCOPE);
        controls.setReturningAttributes(new String[0]);
        String filter = "(member=" + memberDn + ")";

        for (NamingEnumeration<SearchResult> ne = ctx.search(
                DnDirectory.ROLE_BASE, filter, controls); ne.hasMoreElements();) {
            SearchResult result = ne.next();
            String roleDN = result.getNameInNamespace();
            outRoles.add(roleDN);
            outRoles.addAll(recursiveFindRolesWithMember(roleDN));
        }

        return outRoles;
    }

    /**
     * Converts a List of DNs into a List of Role objects.
     * 
     * @param inList
     *            a List containing Strings being the role DN.
     * 
     * @return a RoleDirectory containing the corresponding Role objects.
     */
    private RoleDirectory translateList(String inUserName, List<String> inList) {
        RoleDirectory outDirectory = new RoleDirectory(inUserName);
        for (String roleDN : inList) {
            try {
                LdapName dn = new LdapName(roleDN);
                Rdn nameRdn = dn.getRdn(dn.size() - 1);
                String roleName = (String) nameRdn.getValue();
                Rdn groupRdn = dn.getRdn(dn.size() - 2);
                String roleApplication = (String) groupRdn.getValue();
                Role role = new Role(roleName, roleApplication);
                outDirectory.add(role);
            } catch (InvalidNameException e) {
                // TODO Auto-generated catch block
                e.printStackTrace();
            }
        }
        return outDirectory;
    }
}
