/*
 *    ALMA - Atacama Large Millimiter Array
 *    (c) European Southern Observatory, 2002
 *    Copyright by ESO (in the framework of the ALMA collaboration),
 *    All rights reserved
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License as published by the Free Software Foundation; either
 *    version 2.1 of the License, or (at your option) any later version.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
 *    MA 02111-1307  USA
 *
 */
package alma.userrepository.roledirectory;

import java.util.List;
import java.util.Map;

import alma.userrepository.errors.UserRepositoryException;
import alma.userrepository.shared.DirectorySession;

/**
 * @author Stewart
 * 
 */
public interface RoleDirectorySession extends DirectorySession {
    /**
     * Assign the specified role to the user with the given uid.
     * 
     * For an LDAP backend, a number of specific UserRepositoryExceptions may be
     * expected: a DuplicateObjectException will be thrown if the user already
     * has the role; an InvalidAttributeException will be thrown if the member
     * is invalid; an ObjectNotFoundException will be thrown if the role does
     * not exist; a PermissionException will be thrown if the session privileges
     * are insufficient to perform this operation
     * 
     * @param uid
     *            the user uid
     * 
     * @param role
     *            the role to assign
     * 
     * @throws UserRepositoryException
     *             if some other error occurred during the operation
     */
    public void assignRole(String uid, Role role)
            throws UserRepositoryException;

    /**
     * Gets all roles available in the system.
     * 
     * @return a RoleDirectory containing all available roles in the system.
     * 
     * @throws UserRepositoryException
     */
    public RoleDirectory getAllRoles() throws UserRepositoryException;

    /**
     * Gets all roles explicitly assigned to a user. This method does not return
     * any additional roles gained through the role hierarchy.
     * 
     * @param inUserId
     *            a String being the user id for whom we should retrieve roles
     * 
     * @return a RoleDirectory containing all roles explicitly assigned to the user.
     * 
     * @throws UserRepositoryException
     *             in case anything goes wring while retrieving the information.
     */
    public RoleDirectory getExplicitUserRoles(String inUserId)
            throws UserRepositoryException;

    /**
     * Gets all roles attributed to a user, whether assigned explicitly or
     * gained through role inheritance.
     * 
     * @param inUserId
     *            a String being the user id for whom we should retrieve roles
     * 
     * @return a RoleDirectory containing all roles assigned to the user.
     * 
     * @throws UserRepositoryException
     *             in case anything goes wring while retrieving the information.
     */
    public RoleDirectory getAllUserRoles(String inUserId)
            throws UserRepositoryException;

    /**
     * Remove a user from a role.
     * 
     * For an LDAP backend, a number of specific UserRepositoryExceptions may be
     * expected: a DuplicateObjectException will be thrown if the user already
     * has the role; an InvalidAttributeException will be thrown if the member
     * is invalid; an ObjectNotFoundException will be thrown if the role does
     * not exist; a PermissionException will be thrown if the session privileges
     * are insufficient to perform this operation
     * 
     * @param uid
     *            the uid of the user to remove
     * 
     * @param role
     *            the role from which should remove the user
     * 
     * @throws UserRepositoryException
     *             if some other error occurred during the operation
     */
    public void revokeRole(String uid, Role role)
            throws UserRepositoryException;

}
