package alma.userrepository.roledirectory;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * This class contains roles assigned to a specific user.
 * 
 * @author apersson
 */
public class RoleDirectory {

    /**
     * The user name this role directory is related to.
     */
    private String myUserName;

    /**
     * Map containing the roles for the different applications.
     */
    private Map<String, List<Role>> myRoleMap = new HashMap<String, List<Role>>();

    /**
     * Constructor taking the user name of the user this role directory is valid
     * for.
     * 
     * @param inUserName
     *            a String being the user name.
     * 
     * @throws IllegalArgumentException
     *             in case the specified user name is <code>null</code> or empty
     *             string.
     */
    public RoleDirectory(String inUserName) {
        if (inUserName == null || "".equals(inUserName)) {
            throw new IllegalArgumentException(
                    "inUserName cannot be null or empty string.");
        }
        this.myUserName = inUserName;
    }

    /**
     * Adds the given role to the directory. It is ensured that the role will
     * only exist once.
     * 
     * @param inRole
     *            the Role to add to the directory.
     * 
     * @return a boolean indicating if the role was added; <code>true</code>
     *         means the role was added, <code>false</code> means that the role
     *         already existed.
     * 
     * @throws IllegalArgumentException
     *             in case the specified role is <code>null</code>.
     */
    public boolean add(Role inRole) {
        if (inRole == null) {
            throw new IllegalArgumentException("inRole cannot be null.");
        }
        boolean outValue = false;
        List<Role> roleList = this.myRoleMap.get(inRole.getApplicationName());
        if (roleList == null) {
            roleList = new ArrayList<Role>();
            this.myRoleMap.put(inRole.getApplicationName(), roleList);
        }
        if (!roleList.contains(inRole)) {
            roleList.add(inRole);
            outValue = true;
        }
        return outValue;
    }

    /**
     * Adds all roles in the given collection. The roles may be for different
     * applications. It is ensured that a role will only exist once.
     * 
     * @param inRoleList
     *            a List containing Role objects
     * 
     * @return boolean indicating if the directory was changed as a result of
     *         this operation; <code>true</code> means that at least one role
     *         was added, <code>false</code> means that all roles already
     *         existed.
     * 
     * @throws IllegalArgumentException
     *             in case the specified list is <code>null</code> or if the
     *             list contains a <code>null</code> value.
     */
    public boolean addAll(List<Role> inRoleList) {
        if (inRoleList == null) {
            throw new IllegalArgumentException("inRoleList cannot be null.");
        }
        boolean outValue = false;
        for (Role role : inRoleList) {
            if (add(role)) {
                outValue = true;
            }
        }
        return outValue;
    }

    /**
     * Clears the directory of all roles.
     */
    public void clear() {
        this.myRoleMap = new HashMap<String, List<Role>>();
    }

    /**
     * Returns all application names that this directory holds roles for. The
     * returned list will be sorted alphabetically.
     * 
     * @return a List if String objects being the application names.
     */
    public List<String> getApplicationNames() {
        List<String> outList = new ArrayList<String>(this.myRoleMap.keySet());
        Collections.sort(outList);
        return outList;
    }

    /**
     * Gets all roles related to the given application. The returned list will
     * be sorted alphabetically.
     * 
     * @param inApplicationName
     *            a String being the application name for which to return the
     *            roles.
     * 
     * @return a List of Role objects related to the given application or
     *         <code>null</code> if the application is unknown to this
     *         directory.
     * 
     * @throws IllegalArgumentException
     *             in case inApplicationName is <code>null</code> or empty
     *             string.
     */
    public List<Role> getRoles(String inApplicationName) {
        if (inApplicationName == null || "".equals(inApplicationName)) {
            throw new IllegalArgumentException(
                    "inApplicationName cannot be null or empty string.");
        }
        List<Role> outList = this.myRoleMap.get(inApplicationName);
        if (outList != null) {
            /* Return a copy of the list. */
            outList = new ArrayList<Role>(outList);
            Collections.sort(outList);
        }
        return outList;
    }

    /**
     * Gets the name of the user this directory relates to.
     * 
     * @return a String being the user name.
     */
    public String getUserName() {
        return this.myUserName;
    }

    /**
     * Checks if the given Role is assigned to the user/directory.
     * 
     * @param inRole
     *            the Role to look for.
     * 
     * @return a boolean being <code>true</code> if the Role is assigned,
     *         <code>false</code> otherwise.
     * 
     * @throws IllegalArgumentException
     *             in case inRole is <code>null</code>.
     */
    public boolean hasRole(Role inRole) {
        if (inRole == null) {
            throw new IllegalArgumentException("inRole cannot be null.");
        }
        boolean outValue = false;
        List<Role> roleList = this.myRoleMap.get(inRole.getApplicationName());
        if (roleList != null) {
            if (roleList.contains(inRole)) {
                outValue = true;
            }
        }
        return outValue;
    }

    /**
     * Checks if all the given Roles are assigned to the user/directory. The
     * roles may be for different applications.
     * 
     * @param inRoleList
     *            a List containing Role objects.
     * 
     * @return a boolean being <code>true</code> if all roles are assigned,
     *         <code>false</code> otherwise.
     * 
     * @throws IllegalArgumentException
     *             in case the specified list is <code>null</code> or if the
     *             list contains a <code>null</code> value.
     */
    public boolean hasAllRoles(List<Role> inRoleList) {
        if (inRoleList == null) {
            throw new IllegalArgumentException("inRoleList cannot be null.");
        }
        boolean outValue = true;
        for (Role role : inRoleList) {
            if (!hasRole(role)) {
                outValue = false;
                break;
            }
        }
        return outValue;
    }

    /**
     * Checks if any of the given Roles are assigned to the user/directory. The
     * roles may be for different applications.
     * 
     * @param inRoleList
     *            a List containing Role objects.
     * 
     * @return a boolean being <code>true</code> if any of the roles are
     *         assigned, <code>false</code> otherwise.
     * 
     * @throws IllegalArgumentException
     *             in case the specified list is <code>null</code> or if the
     *             list contains a <code>null</code> value.
     */
    public boolean hasAnyRole(List<Role> inRoleList) {
        if (inRoleList == null) {
            throw new IllegalArgumentException("inRoleList cannot be null.");
        }
        boolean outValue = false;
        for (Role role : inRoleList) {
            if (hasRole(role)) {
                outValue = true;
                break;
            }
        }
        return outValue;
    }

    /**
     * Removes all roles for the given application.
     * 
     * @param inApplicationName
     *            a String stating the application name to have the roles
     *            removed.
     * 
     * @return a boolean being <code>true</code> if the directory contained any
     *         roles for the given application, <code>false</code> otherwise.
     * 
     * @throws IllegalArgumentException
     *             in case inApplicationName is <code>null</code> or empty
     *             string.
     */
    public boolean remove(String inApplicationName) {
        if (inApplicationName == null || "".equals(inApplicationName)) {
            throw new IllegalArgumentException(
                    "inApplicationName cannot be null or empty string.");
        }
        return this.myRoleMap.remove(inApplicationName) != null ? true : false;
    }

    /**
     * Removes the given Role the directory.
     * 
     * @param inRole
     *            the Role to be removed.
     * 
     * @return a boolean indicating if the directory changed as a result of the
     *         operation, <code>true</code> means that the role was found and
     *         removed, <code>false</code> means the role was not found.
     * 
     * @throws IllegalArgumentException
     *             in case the specified list is <code>null</code> or if the
     *             list contains a <code>null</code> value.
     */
    public boolean remove(Role inRole) {
        if (inRole == null) {
            throw new IllegalArgumentException("inRole cannot be null.");
        }
        boolean outValue = false;
        List<Role> roleList = this.myRoleMap.get(inRole.getApplicationName());
        if (roleList != null) {
            outValue = roleList.remove(inRole);
        }
        return outValue;
    }

    /**
     * Removes the Roles in the given List from the directory. The roles my be
     * related to different applications.
     * 
     * @param inRoleList
     *            a List containing Role objects.
     * 
     * @return boolean indicating if the directory was changed as a result of
     *         this operation; <code>true</code> means that at least one role
     *         was removed, <code>false</code> means that none of the roles
     *         existed.
     * 
     * @throws IllegalArgumentException
     *             in case the specified list is <code>null</code> or if the
     *             list contains a <code>null</code> value.
     */
    public boolean removeAll(List<Role> inRoleList) {
        if (inRoleList == null) {
            throw new IllegalArgumentException("inRoleList cannot be null.");
        }
        boolean outValue = false;
        for (Role role : inRoleList) {
            if (remove(role)) {
                outValue = true;
            }
        }
        return outValue;
    }

    /**
     * Counts how many roles that are found in total in the directory.
     * 
     * @return an int being the total count.
     */
    public int roleCount() {
        int outValue = 0;
        for (String appName : this.myRoleMap.keySet()) {
            outValue += this.myRoleMap.get(appName).size();
        }
        return outValue;
    }

    @Override
    public String toString() {
        StringBuilder builder = new StringBuilder();
        for (String appName : getApplicationNames()) {
            builder.append(appName);
            builder.append(": ");
            builder.append(getRoles(appName));
            builder.append("\n");
        }
        return builder.toString();
    }
}
