/*
 *    ALMA - Atacama Large Millimiter Array
 *    (c) European Southern Observatory, 2002
 *    Copyright by ESO (in the framework of the ALMA collaboration),
 *    All rights reserved
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License as published by the Free Software Foundation; either
 *    version 2.1 of the License, or (at your option) any later version.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
 *    MA 02111-1307  USA
 *
 */
package alma.userrepository.projectdirectory.ldap;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.naming.InvalidNameException;
import javax.naming.NameAlreadyBoundException;
import javax.naming.NameNotFoundException;
import javax.naming.NamingEnumeration;
import javax.naming.NamingException;
import javax.naming.NoPermissionException;
import javax.naming.directory.Attribute;
import javax.naming.directory.AttributeInUseException;
import javax.naming.directory.Attributes;
import javax.naming.directory.DirContext;
import javax.naming.directory.InvalidAttributeValueException;
import javax.naming.directory.NoSuchAttributeException;
import javax.naming.directory.SearchControls;
import javax.naming.directory.SearchResult;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import alma.archive.projectcode.ProjectCode;
import alma.archive.projectcode.TypeCode;
import alma.userrepository.errors.DuplicateObjectException;
import alma.userrepository.errors.InvalidAttributeException;
import alma.userrepository.errors.InvalidOperationException;
import alma.userrepository.errors.ObjectNotFoundException;
import alma.userrepository.errors.PermissionException;
import alma.userrepository.errors.UserRepositoryException;
import alma.userrepository.projectdirectory.ProjectDirectorySession;
import alma.userrepository.projectdirectory.ProjectRole;
import alma.userrepository.shared.ldap.DnDirectory;

import com.sun.jndi.ldap.obj.GroupOfNames;

/**
 * @author Stewart
 * 
 */
public class LdapProjectDirectorySession implements ProjectDirectorySession {
    /**
     * Magic key that prevents groupOfNames objects from ever becoming empty
     */
    private static final String nullMember = "uid=nobody";

    protected Log log = LogFactory.getLog(this.getClass());

    /**
     * The DirContext for this session
     */
    DirContext ctx = null;

    @SuppressWarnings("unused")
    private LdapProjectDirectorySession() {
    }

    public LdapProjectDirectorySession(DirContext ctx) {
        this.ctx = ctx;
    }

    /*
     * (non-Javadoc)
     * 
     * @see ProjectDirectorySession#assignProjectRole(String, ProjectCode,
     * ProjectRole)
     */
    public void assignProjectRole(String member, ProjectCode projectCode,
            ProjectRole projectRole) throws UserRepositoryException {
        if (projectRole.equals(ProjectRole.ANY)) {
            throw new InvalidOperationException(
                    "Must add user to a specific role");
        }

        String roleDn = projectRole.getDn(projectCode);

        try {
            GroupOfNames group = (GroupOfNames) ctx.lookup(roleDn);
            group.addMember(member);
            group.close();
        } catch (AttributeInUseException e) {
            throw new DuplicateObjectException("The member '" + member
                    + "' already belongs to the project", e);
        } catch (NameNotFoundException e) {
            throw new ObjectNotFoundException("The role '" + projectRole
                    + "' was not found.", e);
        } catch (NoPermissionException e) {
            throw new PermissionException(
                    "Insufficient permission to add member to project '"
                            + projectCode + "'", e);
        } catch (InvalidAttributeValueException e) {
            throw new InvalidAttributeException("The member '" + member
                    + "' is of invalid format.", e);
        } catch (NamingException e) {
            throw new UserRepositoryException(
                    "Unexpected error occurred while adding member to project");
        }
    }

    /**
     * @throws UserRepositoryException
     */
    public void close() throws UserRepositoryException {
        try {
            ctx.close();
        } catch (NamingException e) {
            if (log.isErrorEnabled()) {
                log.error("Couldn't close session: ", e);
            }
            throw new UserRepositoryException("Couldn't close session: ", e);
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see ProjectDirectorySession#createNewProject(ProjectCode)
     */
    public void createNewProject(ProjectCode projectCode)
            throws UserRepositoryException {
        // create list of roles
        List<ProjectRole> roles = new ArrayList<ProjectRole>();
        Set<String> roleDns = new HashSet<String>();
        for (ProjectRole role : ProjectRole.values()) {
            if (!role.equals(ProjectRole.ANY)) {
                roles.add(role);
                roleDns.add(role.getDn(projectCode));
            }
        }

        // create top-level project groupOfNames, populate with project
        // roles
        GroupOfNames project = new GroupOfNames(roleDns);

        try {
            // must bind parent object before children, as it's top level
            ctx.bind(ProjectRole.ANY.getDn(projectCode), project);

            // then bind each role. note that binding the object sets the object
            // cn, so we need to create a new object for each role, we can't
            // just use the old one even though the data is the same.
            for (ProjectRole role : roles) {
                // groupOfNames objects must have at least one member defined on
                // the
                // LDAP server
                Set<String> nullSet = new HashSet<String>();
                nullSet.add(nullMember);
                GroupOfNames newRole = new GroupOfNames(nullSet);

                ctx.bind(role.getDn(projectCode), newRole);
            }
        } catch (NameAlreadyBoundException e) {
            throw new DuplicateObjectException("Project " + projectCode
                    + " already exists", e);
        } catch (NoPermissionException e) {
            throw new PermissionException(
                    "Insufficient permission to create project", e);
        } catch (InvalidNameException e) {
            throw new InvalidAttributeException(
                    "Cannot create valid LDAP dn from project id", e);
        } catch (NamingException e) {
            throw new UserRepositoryException("Error creating project", e);
        }
    }

    public void removeProject(ProjectCode projectCode)
            throws UserRepositoryException {
        SearchControls controls = new SearchControls();
        controls.setSearchScope(SearchControls.SUBTREE_SCOPE);
        controls.setReturningAttributes(new String[0]);
        String filter = "(cn=*)";

        NamingEnumeration<SearchResult> ne;
        try {
            ne = ctx.search("cn=" + projectCode + ","
                    + DnDirectory.PROJECT_BASE, filter, controls);
        } catch (NameNotFoundException e) {
            return;
        } catch (NamingException e) {
            throw new UserRepositoryException(
                    "Unexpected error occurred while searching for project " + projectCode,
                    e);
        }

        /*
         * Now that we have the project node and all nodes under it we start
         * sorting them based on the number of parent nodes (found in the
         * distinguished name). A node can only be deleted if it is a leaf. By
         * sorting the nodes like this we can delete them in the right order.
         */
        HashMap<Integer, List<String>> deleteMap = new HashMap<Integer, List<String>>();
        try {
            while (ne.hasMoreElements()) {
                SearchResult result = ne.next();
                String name = result.getNameInNamespace();
                int length = name.split(",").length;
                List<String> nameList = deleteMap.get(length);
                if (nameList == null) {
                    nameList = new ArrayList<String>();
                    deleteMap.put(length, nameList);
                }
                nameList.add(name);
            }
        } catch (NamingException e) {
            throw new UserRepositoryException(
                    "Unexpected error occurred while fetching roles.", e);
        }

        try {
            List<Integer> keys = new ArrayList<Integer>(deleteMap.keySet());
            Collections.sort(keys);
            Collections.reverse(keys);
            for (Integer key : keys) {
                for (String name : deleteMap.get(key)) {
                    ctx.destroySubcontext(name);
                }
            }
        } catch (NameNotFoundException e) {
        } catch (NoPermissionException e) {
            throw new PermissionException(
                    "Insufficient permission to remove project", e);
        } catch (NamingException e) {
            throw new UserRepositoryException(
                    "Unexpected error occurred while removing project", e);
        }

    }

    /*
     * (non-Javadoc)
     * 
     * @see ProjectDirectorySession#exists(ProjectCode)
     */
    public boolean exists(ProjectCode projectCode)
            throws UserRepositoryException {
        SearchControls controls = new SearchControls();
        controls.setSearchScope(SearchControls.ONELEVEL_SCOPE);

        NamingEnumeration<SearchResult> users;
        try {
            if (log.isDebugEnabled()) {
                log.debug("exists(): Searching for project with uid '"
                        + projectCode + "'.");
            }
            users = ctx.search(DnDirectory.PROJECT_BASE, "(cn=" + projectCode
                    + ")", controls);
        } catch (NamingException e) {
            if (log.isErrorEnabled()) {
                log.error("exists(): Unexpected error occured during search.");
            }
            throw new UserRepositoryException("Error searching for project", e);
        }

        if (users.hasMoreElements()) {
            return true;
        } else {
            return false;
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see ProjectDirectorySession#isMember(String, ProjectCode, ProjectRole)
     */
    public boolean isMember(String member, ProjectCode projectCode,
            ProjectRole projectRole) throws UserRepositoryException {
        String searchBaseDn = projectRole.getDn(projectCode);
        String searchFilter = "(member=" + member + ")";
        SearchControls controls = new SearchControls();
        controls.setSearchScope(SearchControls.SUBTREE_SCOPE);

        List<String> results;
        try {
            results = searchReturnResultList(searchBaseDn, searchFilter,
                    controls);
        } catch (NamingException e) {
            throw new UserRepositoryException(
                    "Error while searching for project member", e);
        }

        if (results.isEmpty()) {
            return false;
        } else {
            return true;
        }

    }

    /*
     * (non-Javadoc)
     * 
     * @see ProjectDirectorySession#listMembers(ProjectCode, ProjectRole)
     */
    public List<String> listMembers(ProjectCode projectCode,
            ProjectRole projectRole) throws UserRepositoryException {
        // search controls matching and returning all members
        String searchBaseDn = projectRole.getDn(projectCode);
        String searchFilter = "(member=*)";
        String[] searchAttributes = { "member" };
        SearchControls controls = new SearchControls();
        controls.setSearchScope(SearchControls.SUBTREE_SCOPE);
        controls.setReturningAttributes(searchAttributes);

        // search for objects satisfying the filter
        List<String> results;
        try {
            results = searchReturnResultList(searchBaseDn, searchFilter,
                    controls);
        } catch (NameNotFoundException e) {
            throw new ObjectNotFoundException("Project '" + projectCode
                    + "' not found", e);
        } catch (NamingException e) {
            e.printStackTrace();
            throw new UserRepositoryException("Error searching for members", e);
        }

        // if we're listing all members, don't want role placeholders
        List<String> removables = new ArrayList<String>();
        for (ProjectRole r : ProjectRole.values()) {
            removables.add(r.getDn(projectCode));
        }
        // LDAP groupOfNames must have at least one member, which we specify as
        // 'uid=nobody', so don't remove it!
        removables.add(nullMember);

        results.removeAll(removables);

        return results;
    }

    /*
     * (non-Javadoc)
     * 
     * @see ProjectDirectorySession#listProjects()
     */
    public List<ProjectCode> listProjects() throws UserRepositoryException {
        return listProjects("*");
    }

    /*
     * (non-Javadoc)
     * 
     * @see ProjectDirectorySession#listProjects(int, int)
     */
    public List<ProjectCode> listProjects(int inYear, int inSemester)
            throws UserRepositoryException {
        String cn = inYear + "." + inSemester + ".*";
        return listProjects(cn);
    }

    private List<ProjectCode> listProjects(String inCN)
            throws UserRepositoryException {
        SearchControls controls = new SearchControls();
        controls.setSearchScope(SearchControls.ONELEVEL_SCOPE);
        String[] searchAttributes = { "cn" };
        controls.setReturningAttributes(searchAttributes);
        String filter = "(cn=" + inCN + ")";

        // search for objects satisfying the filter
        List<String> resultList;
        try {
            if (log.isDebugEnabled()) {
                log.debug("listProjects(): asking for projects with filter '"
                        + filter + "'");
            }
            resultList = searchReturnResultList(DnDirectory.PROJECT_BASE,
                    filter, controls);
        } catch (NamingException e) {
            if (log.isErrorEnabled()) {
                log
                        .error("listProjects(): Unexpected error occured during search.");
            }
            throw new UserRepositoryException("Error searching for projects", e);
        }

        List<ProjectCode> outResultList = new ArrayList<ProjectCode>();
        for (String result : resultList) {
            try {
                outResultList.add(new ProjectCode(result));
            } catch (IllegalArgumentException e) {
                log.error("Error when creating ProjectCode object using ["
                        + result + "].", e);
            }
        }
        return outResultList;
    }

    public Map<TypeCode, Integer> getMaxSequenceNumbers(int inYear,
            int inSemester) throws UserRepositoryException {
        Map<TypeCode, Integer> outMap = new HashMap<TypeCode, Integer>();
        return outMap;
    }

    /*
     * (non-Javadoc)
     * 
     * @see ProjectDirectorySession#revokeProjectRole(String, ProjectCode,
     * ProjectRole)
     */
    public void revokeProjectRole(String member, ProjectCode projectCode,
            ProjectRole projectRole) throws UserRepositoryException {
        if (projectRole.equals(ProjectRole.ANY)) {
            throw new InvalidOperationException(
                    "Must remove from a specific role");
        }

        String roleDn = projectRole.getDn(projectCode);

        try {
            GroupOfNames group = (GroupOfNames) ctx.lookup(roleDn);
            group.removeMember(member);
            group.close();
        } catch (NoSuchAttributeException e) {
            throw new InvalidAttributeException("'" + member
                    + "' is not a member.", e);
        } catch (NameNotFoundException e) {
            throw new ObjectNotFoundException("The role was not found.", e);
        } catch (NoPermissionException e) {
            throw new PermissionException(
                    "Insufficient permission to remove member from role", e);
        } catch (NamingException e) {
            throw new UserRepositoryException(
                    "Unexpected error occurred while removing member from project",
                    e);
        }

    }

    private List<String> searchReturnResultList(String searchBase,
            String searchFilter, SearchControls controls)
            throws NamingException {
        // search for objects satisfying the filter
        List<String> results = new ArrayList<String>();
        for (NamingEnumeration<SearchResult> ne = ctx.search(searchBase,
                searchFilter, controls); ne.hasMoreElements();) {
            SearchResult result = ne.next();

            Attributes resultAttrs = result.getAttributes();
            for (NamingEnumeration<? extends Attribute> resultAttr = resultAttrs
                    .getAll(); resultAttr.hasMoreElements();) {
                Attribute attr = resultAttr.nextElement();
                for (NamingEnumeration<?> vals = attr.getAll(); vals.hasMore();) {
                    String s = (String) vals.next();
                    results.add(s);
                }
            }
        }
        return results;
    }
}
