/*
 *    ALMA - Atacama Large Millimiter Array
 *    (c) European Southern Observatory, 2002
 *    Copyright by ESO (in the framework of the ALMA collaboration),
 *    All rights reserved
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License as published by the Free Software Foundation; either
 *    version 2.1 of the License, or (at your option) any later version.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
 *    MA 02111-1307  USA
 *
 */
package alma.userrepository.projectdirectory;

import java.util.List;
import java.util.Map;

import alma.archive.projectcode.ProjectCode;
import alma.archive.projectcode.TypeCode;
import alma.userrepository.errors.UserRepositoryException;
import alma.userrepository.shared.DirectorySession;

/**
 * @author Stewart
 * 
 */
public interface ProjectDirectorySession extends DirectorySession {
    /**
     * Add a member to a project role. Members must be added to a specific role;
     * attempts to add a user to the ANY ProjectRole will throw an
     * IllegalOperationException.
     * 
     * @param member
     *            the member to add
     * 
     * @param projectCode
     *            the project code
     * 
     * @param projectRole
     *            the project role
     * 
     * @throws UserRepositoryException
     *             if an error occurs during the operation
     */
    public void assignProjectRole(String member, ProjectCode projectCode,
            ProjectRole projectRole) throws UserRepositoryException;

    /**
     * Create a new project. Throws DuplicateObjectException if a project with
     * the given ID already exists. Attempts to create a project without
     * sufficient privileges will result in a PermissionException.
     * 
     * @param projectCode
     *            the project code
     * 
     * @throws UserRepositoryException
     *             if an error occurs during the operation
     */
    public void createNewProject(ProjectCode projectCode)
            throws UserRepositoryException;

    public void removeProject(ProjectCode projectCode)
            throws UserRepositoryException;

    /**
     * Determine whether a project with the given code exists in the directory.
     * 
     * @param projectCode
     *            the project code to search for
     * 
     * @return true if a project with the given ID exists
     * 
     * @throws UserRepositoryException
     *             if an error occurs during the operation
     */
    public boolean exists(ProjectCode projectCode)
            throws UserRepositoryException;

    /**
     * Determine if an object is a member of the given project.
     * 
     * @param member
     *            the member to search for
     * 
     * @param projectCode
     *            the project code
     * 
     * @param projectRole
     *            the project role
     * 
     * @return true if the user is a member of the given project and role
     * 
     * @throws UserRepositoryException
     *             if an error occurs during the operation
     */
    public boolean isMember(String member, ProjectCode projectCode,
            ProjectRole projectRole) throws UserRepositoryException;

    /**
     * List project members with the given role.
     * 
     * @param projectCode
     *            the project code
     * 
     * @param projectRole
     *            the project role to query
     * 
     * @return the list of role members for the given project
     * 
     * @throws UserRepositoryException
     *             if an error occurs during the operation
     */
    public List<String> listMembers(ProjectCode projectCode,
            ProjectRole ProjectRole) throws UserRepositoryException;

    /**
     * List all projects. This function returns a list containing the IDs of all
     * projects in the directory.
     * 
     * @return a list of project codes
     * 
     * @throws UserRepositoryException
     *             if an error occurs during the operation
     */
    public List<ProjectCode> listProjects() throws UserRepositoryException;

    /**
     * List all projects registered for the given year and semester. This
     * function returns a list containing the IDs of all projects in the
     * directory.
     * 
     * @param inYear
     *            an int being the year of interest
     * @param inSemester
     *            an int being the semester of interest
     * 
     * @return a list of project codes
     * 
     * @throws UserRepositoryException
     *             if an error occurs during the operation
     */
    public List<ProjectCode> listProjects(int inYear, int inSemester)
            throws UserRepositoryException;

    /**
     * Remove a user from a project role. Members must be removed from a
     * specific role; attempts to remove a user from the ANY ProjectRole will
     * throw an IllegalOperation exception. Removing a member from a project
     * that does not exist will throw an ObjectNotFoundException. Removing a
     * non-existant member will throw an InvalidAttributeException. Performing
     * this operation without sufficient permission will result in a
     * PermissionException.
     * 
     * @param member
     *            the member to remove
     * 
     * @param projectCode
     *            the project code
     * 
     * @param projectRole
     *            the project role
     * 
     * @throws UserRepositoryException
     *             if an error occurs during the operation
     */
    public void revokeProjectRole(String member, ProjectCode projectCode,
            ProjectRole ProjectRole) throws UserRepositoryException;

    /**
     * Retrieves the maximum sequence number for each of the different type
     * codes for the given year and semester. In case no project codes, and
     * hence no sequence numbers, are found for a particular type code the type
     * code is excluded from the returned map.
     * 
     * @param inYear
     *            an int being the year of interest
     * @param inSemester
     *            an int being the semester of interest
     * 
     * @return a Map with various TypeCodes as keys and the value being the
     *         corresponding maximum sequence number as an Integer.
     * 
     * @throws UserRepositoryException
     *             if an error occurs during the operation
     */
    public Map<TypeCode, Integer> getMaxSequenceNumbers(int inYear,
            int inSemester) throws UserRepositoryException;
}
