/*
 *    ALMA - Atacama Large Millimiter Array
 *    (c) European Southern Observatory, 2002
 *    Copyright by ESO (in the framework of the ALMA collaboration),
 *    All rights reserved
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License as published by the Free Software Foundation; either
 *    version 2.1 of the License, or (at your option) any later version.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
 *    MA 02111-1307  USA
 *
 */
package alma.userrepository.addressbook.ldap;

import static alma.userrepository.addressbook.ldap.AttributeUtilities.MODIFY_ARRAY;
import static alma.userrepository.addressbook.ldap.AttributeUtilities.MODIFY_ATTR;

import java.util.ArrayList;
import java.util.Hashtable;
import java.util.List;

import javax.naming.Context;
import javax.naming.Name;
import javax.naming.NamingException;
import javax.naming.directory.Attribute;
import javax.naming.directory.Attributes;
import javax.naming.directory.DirContext;
import javax.naming.spi.DirObjectFactory;

import alma.userrepository.addressbook.ldap.beans.LdapAddress;
import alma.userrepository.addressbook.ldap.beans.LdapAddressBookEntry;
import alma.userrepository.addressbook.ldap.beans.LdapPreferences;
import alma.userrepository.addressbook.ldap.beans.LdapUser;
import alma.userrepository.domainmodel.Address;
import alma.userrepository.domainmodel.AddressBookEntry;
import alma.userrepository.domainmodel.Preferences;
import alma.userrepository.domainmodel.User;
import alma.userrepository.errors.CommunicationException;
import alma.userrepository.errors.UserRepositoryException;

public class UserRepositoryObjectFactory implements DirObjectFactory {

	public UserRepositoryObjectFactory() {
	}

	public Object getObjectInstance(Object obj, Name name, Context ctx,
			Hashtable<?, ?> env, Attributes attrs) throws Exception {
		Attribute oc = (attrs != null ? attrs.get("objectclass") : null);

		// secondary addresses (ie almaUser child nodes) are LDAP entries that
		// do not have the almaUser objectClass
		if (oc != null && oc.contains("almaAddress")
				&& !oc.contains("almaUser")) {
			return recreateAddress(attrs, name, ctx);
		}

		// default address and settings are attached to user entry
		if (oc != null && oc.contains("almaUser")) {
			return recreateEntry(attrs, name, ctx);
		}

		// otherwise we can't handle this entry, so return null to pass through
		// to other object factories.
		return null;
	}

	// ObjectFactory version
	public Object getObjectInstance(Object obj, Name name, Context ctx,
			Hashtable<?, ?> env) throws Exception {
		// Don't do anything if we can't see the attributes
		return null;
	}

	private AddressBookEntry recreateEntry(Attributes attrs, Name name,
			Context ctx) throws Exception {
		User user = recreateUser(attrs, name, ctx);
		Preferences preferences = recreatePreferences(attrs, name, ctx);
		Address address = recreateAddress(attrs, name, ctx);
		List<Address> addresses = new ArrayList<Address>();
		addresses.add(address);

		return new LdapAddressBookEntry(user, addresses, preferences);
	}

	private User recreateUser(Attributes attrs, Name name, Context ctx)
			throws NamingException, UserRepositoryException {
		// attrs only contains operational attributes, so we must retrieve the
		// last modified timestamp ourselves.
		Attribute modAttr = getModifiedTimestamp(name, ctx);
		attrs.put(modAttr);

		// Now we can fully recreate the user
		LdapUser user = new LdapUser(attrs);
		return user;
	}

	private Preferences recreatePreferences(Attributes attrs, Name name,
			Context ctx) throws NamingException {
		LdapPreferences preferences = new LdapPreferences(attrs);
		return preferences;
	}

	private Address recreateAddress(Attributes attrs, Name name, Context ctx)
			throws NamingException, UserRepositoryException {
		// attrs only contains operational attributes, so we must retrieve the
		// last modified timestamp ourselves.
		Attribute modAttr = getModifiedTimestamp(name, ctx);
		attrs.put(modAttr);

		LdapAddress address = new LdapAddress(attrs);
		return address;
	}

	// - private methods ------------------------------------------------------

	private static Attribute getModifiedTimestamp(Name name, Context ctx)
			throws UserRepositoryException {
		// ctx will be an instance of LdapContext, which subclasses
		// DirContext
		if (ctx instanceof DirContext) {
			Attributes a = null;
			try {
				a = ((DirContext) ctx).getAttributes(name, MODIFY_ARRAY);
			} catch (NamingException e) {
				throw new CommunicationException(
						"Could not access ModifyTimestamp.", e);
			}
			Attribute attr = a.get(MODIFY_ATTR);
			return attr;
		} else {
			throw new UserRepositoryException(
					"LDAP object factory is receiving non-LDAP objects!");
		}

	}
}
