/*
 *    ALMA - Atacama Large Millimiter Array
 *    (c) European Southern Observatory, 2002
 *    Copyright by ESO (in the framework of the ALMA collaboration),
 *    All rights reserved
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License as published by the Free Software Foundation; either
 *    version 2.1 of the License, or (at your option) any later version.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
 *    MA 02111-1307  USA
 *
 */
package alma.userrepository.addressbook;

import java.util.List;
import java.util.Map;

import alma.userrepository.domainmodel.AddressBookEntry;
import alma.userrepository.domainmodel.Preferences;
import alma.userrepository.domainmodel.User;
import alma.userrepository.errors.UserRepositoryException;
import alma.userrepository.shared.DirectorySession;

public interface AddressBookSession extends DirectorySession {

    /**
     * Commit the given entry to the address book. This function writes a new
     * entry if the user does not exist, and overwrites any attributes that have
     * changed should the AddressBookEntryJ object represent an existing entry.
     * 
     * For an LDAP backend, a PermissionException will be thrown if the
     * connection does not have the privileges required to commit the entry, and
     * an InvalidAttributeException will be thrown if the UID is in an invalid
     * format. If the entry has been updated since the entry was retrieved, a
     * DataExpiredException will be thrown and the entry will not be committed
     * to the address book.
     * 
     * @param entry
     *            the AddressBookEntryJ to commit
     * 
     * @throws UserRepositoryErrorEx
     *             if the server reports an error while performing the operation
     */
    public void commitEntry(AddressBookEntry entry)
            throws UserRepositoryException;

    /**
     * Delete the given user and address book entry from the user repository.
     * 
     * For an LDAP backend, a PermissionException will be thrown if the
     * connection does not have the privileges required to delete the entry, and
     * an ObjectNotFoundException will be thrown if the entry does not
     * correspond to a user in the database.
     * 
     * @param entry
     *            the entry to delete from the server.
     * 
     * @throws UserRepositoryException
     *             if the server reports an error while performing the operation
     */
    public void deleteEntry(AddressBookEntry entry)
            throws UserRepositoryException;

    /**
     * Delete the given user and address book entry from the user repository.
     * For an LDAP backend, a PermissionException will be thrown if the
     * connection does not have the privileges required to delete the entry, and
     * an ObjectNotFoundException will be thrown if the entry does not
     * correspond to a user in the database.
     * 
     * @param uid
     *            the uid of the user to delete
     * 
     * @throws UserRepositoryException
     *             if the server reports an error while performing the operation
     */
    public void deleteEntryByUid(String uid) throws UserRepositoryException;

    /**
     * Determine whether a user with the given uid exists in the user
     * repository.
     * 
     * @param uid
     *            the user uid to search for
     * 
     * @return true if a user with the given uid exists
     * 
     * @throws UserRepositoryException
     *             if the server raises an error while searching for users.
     */
    public boolean exists(String uid) throws UserRepositoryException;

    /**
     * Retrieve an AddressBookEntry from the address book. For an LDAP backend,
     * a PermissionException will be thrown if the connection does not have the
     * privileges required to retrieve the entry, and an ObjectNotFoundException
     * will be thrown if the entry does not correspond to a user in the
     * database.
     * 
     * @param uid
     *            the uid of the user entry to retrieve
     * 
     * @return the AddressBookEntry entry
     * 
     * @throws UserRepositoryErrorEx
     *             if the server reports an error while retrieving the user.
     */
    public AddressBookEntry getEntry(String uid) throws UserRepositoryException;

    /**
     * Retrieve a User from the address book. For an LDAP backend, a
     * PermissionException will be thrown if the connection does not have the
     * privileges required to retrieve the entry, and an ObjectNotFoundException
     * will be thrown if the entry does not correspond to a user in the
     * database.
     * 
     * @param uid
     *            the uid of the user entry to retrieve
     * 
     * @return the User with the given uid
     * 
     * @throws UserRepositoryErrorEx
     *             if the server reports an error while retrieving the user.
     */
    public User getUser(String uid) throws UserRepositoryException;

    /**
     * Retrieve preferences from address book. For an LDAP backend, a
     * PermissionException will be thrown if the connection does not have the
     * privileges required to retrieve the entry, and an ObjectNotFoundException
     * will be thrown if the entry does not correspond to a user in the
     * database.
     * 
     * @param uid
     *            the uid of the user entry to retrieve
     * 
     * @return the Preferences of the user with the given uid
     * 
     * @throws UserRepositoryErrorEx
     *             if the server reports an error while retrieving the user.
     */
    public Preferences getPreferences(String uid)
            throws UserRepositoryException;

    /**
     * List the UIDs of all users in the address book.
     * 
     * @returns the list of user UIDs
     * 
     * @throws UserRepositoryErrorEx
     *             if the server reports an error while retrieving the user.
     */
    public List<String> listUsers() throws UserRepositoryException;

    /**
     * Return selected attributes for user/address entries matching the given
     * filter. The result is a list of dictionaries, where each dictionary in
     * the list has keys for the requested attribute(s).
     * <p>
     * The format of the search filter argument depends on the back-end
     * technology used, i.e. the search query format should be LDAP for an LDAP
     * server, SQL for a database etc. An LDAP backend will throw an
     * InvalidSearchFilterException if the search filter is invalid.
     * 
     * @returns a List of Maps mapping attributes name to attribute value
     * 
     * @param searchFilter
     *            the search filter to use
     * 
     * @param attributes
     *            the attribute to return
     * 
     * @throws UserRepositoryErrorEx
     *             if the server reports an error while performing the search
     */
    public List<Map<String, String>> searchGetAttributes(String searchFilter,
            List<String> attributes) throws UserRepositoryException;

    /**
     * Return selected attributes for entries in the given location. The result
     * is a list of dictionaries, where each dictionary in the list has keys for
     * the requested attribute.
     * <p>
     * The format of the search filter argument depends on the back-end
     * technology used, i.e. the search query format should be LDAP for an LDAP
     * server, SQL for a database etc. The LDAP implementation will throw an
     * ObjectNotFoundException if the search base is invalid, and an
     * InvalidSearchFilterException if the search filter is invalid.
     * 
     * @returns the requested attributes of entries matching the search filter
     * 
     * @param searchBase
     *            the location from where to begin the search
     * 
     * @param searchFilter
     *            the search filter to use
     * 
     * @param attributes
     *            the attribute to return
     * 
     * @throws UserRepositoryException
     *             if the server reports an error while performing the search
     */
    public List<Map<String, String>> searchLocationGetAttributes(
            String searchBase, String searchFilter, List<String> attributes)
            throws UserRepositoryException;

    /**
     * Return the location of users matching the given search filter. The format
     * of the search filter argument depends on the back-end technology used,
     * i.e. the search query format should be LDAP for an LDAP server, SQL for a
     * database etc. For an LDAP backend, this function returns the canonical
     * identifier of all almaUsers matching the search filter, and will throw an
     * InvalidSearchFilterException if the search filter is invalid.
     * 
     * @returns the identifier of the users matching the search filter
     * 
     * @param searchFilter
     *            the search filter to use
     * 
     * @throws UserRepositoryErrorEx
     *             if the server reports an error while performing this
     *             operation.
     */
    public List<String> searchGetLocation(String searchFilter)
            throws UserRepositoryException;

    /**
     * Set the password for the user with the given uid. For an LDAP backend, a
     * PermissionException will be thrown if the connection does not have the
     * privileges required to retrieve the entry, and an ObjectNotFoundException
     * will be thrown if the entry does not correspond to a user in the
     * database.
     * 
     * @param uid
     *            the user uid
     * 
     * @param password
     *            the password to set
     * 
     * @throws UserRepositoryErrorEx
     *             if the server reports an error while changing the password
     */
    public void setPassword(String uid, String password)
            throws UserRepositoryException;

    /**
     * Sets preferences for the user with the given UID. For an LDAP backend, a
     * PermissionException will be thrown if the connection does not have the
     * privileges required to commit to the entry, and an
     * ObjectNotFoundException will be thrown if the entry does not correspond
     * to a user in the database.
     * 
     * @param uid
     *            the user UID
     * @param preferences
     *            the preferences to commit
     * @throws UserRepositoryException
     *             if the server reports an error while committing the
     *             preferences
     */
    public void setPreferences(String uid, Preferences preferences)
            throws UserRepositoryException;

}
