/*
 * ALMA - Atacama Large Millimeter Array
 * (c) Associated Universities Inc., 2007
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 */

/**
 * @author  srankin
 * @version $Id: TMCDBCommonQueries.java,v 1.1.4.2 2009/12/10 19:05:08 pburgos Exp $
 * @since
 */

package alma.TMCDB.Query;

import alma.TMCDB.types.ArrayTime;
import alma.TMCDB.TimeValue;
import alma.TMCDB.TMCDB;
import alma.TmcdbErrType.wrappers.AcsJTmcdbErrTypeEx;
import alma.TmcdbErrType.wrappers.AcsJTmcdbInvalidDataTypeEx;
import alma.TmcdbErrType.wrappers.AcsJTmcdbSqlEx;

import alma.archive.tmcdb.DAO.*;

import java.io.BufferedReader;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Timestamp;
import java.sql.Statement;
import java.util.ArrayList;
import java.util.List;
import java.util.Properties;

/*
 * This class is currently a mess!
 * 
 * Most code was designed following the current (2007-10) design of the TMCDB.
 * Parts have been quickly hacked to use data now at the ATF, which is collected
 * by software designed before the current TMCDB design was completed. All of
 * this code must be updated to reflect design changes driven by ACS 7.0 and the
 * ACACORR FBT.
 * 
 * TODO: Review the ACACORR FBT TMCDB work and correct this class as required.
 * 
 * TMCDBCommonQueries provides what we expect to be commonly used queries to
 * collect data about the current configuration of the telescope or the past or
 * current state of assemblies in the telescope.
 * 
 * To set up an instance of this class, do the following:
 * 
 *     TMCDBCommonQueries tmcdb = new TMCDBCommonQueries(); 
 *     tmcdb.connectToDB(); 
 *     try {
 *         tmcdb.initialize("TMCDB Configuration Name"); 
 *     } catch (AcsJTmcdbErrTypeEx ex) { 
 *         // TODO: Handle any exceptions that may occur 
 *     }
 * 
 * where "TMCDB Configuration Name" is the name of a configuration already in
 * the TMCDB. After this you may call query methods.
 * 
 * When done with the tmcdb instance created above:
 * 
 *     tmcdb.terminate();
 * 
 * This is necessary to release the database connection and clean up the super
 * classes.
 */
public class TMCDBCommonQueries extends TMCDB {

    //
    // Initialization state must be tracked because this class uses a database connection,
    // which must be initialized before use and released when no longer required.
    private enum InitializationState {
        NotReady, Ready, Terminated
    };

    protected PreparedStatement assemblyIdFromAssemblyNameAntennaNameQuery;
    protected PreparedStatement assemblyIdFromAssemblySNQuery;
    protected PreparedStatement assemblyIdFromCompNameQuery;
    protected PreparedStatement assemblyIdsWithNameQuery;
    protected PreparedStatement assemblyIdsQuery;
    protected PreparedStatement assemblySNsQuery;
    protected PreparedStatement assemblySnFromAssemblyIdQuery;
    protected PreparedStatement assemblyNameFromAssemblyIdQuery;
    protected PreparedStatement componentNamesQuery;
    protected PreparedStatement baseElementIdsFromAntennaNameQuery;
    protected PreparedStatement propertyDataTypeFromPropertyNameQuery;
    protected PreparedStatement propertyNameFromAssemblyNameQuery;
    protected PreparedStatement monitorPointNameFromAssemblyNameQuery;
    protected PreparedStatement monitorPointIdFromAssemblyIdQuery;
    protected PreparedStatement lastMonitorDataTime;
    protected PreparedStatement lastMonitorDataValue;

    private InitializationState tmcdbState;

    //
    // Instance life cycle methods

    public TMCDBCommonQueries() {
        super();
        tmcdbState = InitializationState.NotReady;
    }

    /**
     * Connect to the back end database.
     * 
     * This MUST be called after the constructor, but before initialize().
     */
    public void connectToDB() {
        try {
            connectDB();
//System.out.println("finished connectToDB()");
        } catch (AcsJTmcdbErrTypeEx ex) {
            logger.severe(
                "Failed to connect to DB in TMCDBCommonQueries.connectToDB() - not recoverable - please report a bug.");
            ex.printStackTrace();
        }
    }

    /**
     * Initialize instance for the named TMCDB configuration.
     * 
     * This MUST be called after connectDB(), but before any other methods.
     */
    public void initialize(String config) throws AcsJTmcdbErrTypeEx {
//System.out.println("entered initialize()");
        setConfigurationName(config);
        super.initialize();
        prepareStatements();
        tmcdbState = InitializationState.Ready;
//System.out.println("finished initialize()");
    }

    public String readConfigName() {
	Properties props = new Properties();

	String cwdConfLoc = "./dbConfig.properties";
	File cwdConf = new File(cwdConfLoc);

	String introotPath = System.getenv("ACSDATA");
	String introotConfLoc = introotPath + "/config/dbConfig.properties";
	File introotConf = new File(introotConfLoc);

	FileInputStream fis = null;
	if (cwdConf.exists()) {
	    logger.info("Reading configuration file from: " + cwdConfLoc);
	    try {
	        fis = new FileInputStream(cwdConfLoc);
	    } catch (FileNotFoundException ex) {
	        ex.printStackTrace();
	    }
	} else if (introotConf.exists()) {
	    logger.info("Reading configuration file from: " + introotConfLoc);
	    try {
	        fis = new FileInputStream(introotConfLoc);
	    } catch (FileNotFoundException ex) {
	        ex.printStackTrace();
	    }
	} else {
	// handle the case where there's no dbConfig.properties
	}

	try {
	    props.load(fis);
	} catch (IOException ex) {
	    ex.printStackTrace();
	}

	return props.getProperty("tmcdb.confname");
    }

    /**
     * Terminate this instance of the TMCDB, disconnect from the database, and
     * release superclass resources.
     * 
     * This MUST be called when done with this class.
     */
    public void terminate() {

        if (tmcdbState == InitializationState.Terminated)
            return;

        tmcdbState = InitializationState.Terminated;
        try {
            super.release();
        } catch (AcsJTmcdbErrTypeEx ex) {
            logger.severe(
                "super.release() threw a TMCDBException in TMCDBCommonQueries.terminate() - please report a bug.");
            ex.printStackTrace();
        }
    }

    /**
     * Call terminate() explicitly.  This is only a weak safety net.
     */
    protected void finalize() throws Throwable {
        try {
            terminate();
        } finally {
            super.finalize();
        }
    }

    /**
     * Before doing ANYTHING with the TMCDB, ensure it is ready for use.
     */
    private void ensureTmcdbReady() {
        if (tmcdbState != InitializationState.Ready)
            throw new IllegalStateException("TMCDBCommonQueries instance not ready for queries.");
    }

    /**
     * Initialize all prepared statements.
     * 
     * Where ConfigurationId matters, use the stored ConfigurationId.
     */
    private void prepareStatements() {

        // TODO: Audit all queries and verify ConfigurationId is used where
        // available.
        try {
            assemblyIdFromAssemblySNQuery = conn
                    .prepareStatement("SELECT AssemblyId FROM Assembly WHERE ConfigurationId = ? AND SerialNumber = ?");
            assemblyIdFromAssemblySNQuery.setInt(1, configurationId);
            if (useOracle)
            assemblyIdFromCompNameQuery = conn
                    .prepareStatement("SELECT AssemblyId from Monitorpoint, Baciproperty, Component WHERE Component.ComponentName = ? AND MonitorPoint.BaciPropertyId = BaciProperty.BaciPropertyId AND BaciProperty.ComponentId = Component.ComponentId AND ROWNUM = 1");
            else
            assemblyIdFromCompNameQuery = conn
                    .prepareStatement("SELECT TOP 1 AssemblyId from Monitorpoint, Baciproperty, Component WHERE Component.ComponentName = ? AND MonitorPoint.BaciPropertyId = BaciProperty.BaciPropertyId AND BaciProperty.ComponentId = Component.ComponentId");
            assemblyIdsQuery = conn
                    .prepareStatement("SELECT AssemblyId FROM Assembly WHERE ConfigurationId = ?");
            assemblyIdsQuery.setInt(1, configurationId);
            assemblyIdsWithNameQuery = conn
                    .prepareStatement("SELECT AssemblyId FROM Assembly WHERE ConfigurationId = ? AND AssemblyTypeName = ?");
            assemblyIdsWithNameQuery.setInt(1, configurationId);
            baseElementIdsFromAntennaNameQuery = conn
                    .prepareStatement("SELECT BaseElementId FROM BaseElement WHERE ConfigurationId = ? AND BaseElementName = ?");
            baseElementIdsFromAntennaNameQuery.setInt(1, configurationId);
            assemblySNsQuery = conn
                    .prepareStatement("SELECT SerialNumber FROM Assembly WHERE ConfigurationId = ?");
            assemblySNsQuery.setInt(1, configurationId);
            assemblySnFromAssemblyIdQuery = conn
                    .prepareStatement("SELECT SerialNumber FROM Assembly WHERE ConfigurationId = ? AND AssemblyId = ?");
            assemblySnFromAssemblyIdQuery.setInt(1, configurationId);
            assemblyNameFromAssemblyIdQuery = conn
                    .prepareStatement("SELECT AssemblyTypeName FROM Assembly WHERE AssemblyId = ?");

            monitorPointIdFromAssemblyIdQuery = conn
                    .prepareStatement("SELECT MonitorPointId FROM MonitorPoint WHERE MonitorPointName = ? AND AssemblyId = ?");

            componentNamesQuery = conn
                    .prepareStatement("SELECT ComponentName FROM Component WHERE ConfigurationId = ?");
            componentNamesQuery.setInt(1, configurationId);
            propertyNameFromAssemblyNameQuery = conn
                    .prepareStatement("SELECT MonitorPointName FROM MonitorPoint WHERE AssemblyId = ?");
            propertyDataTypeFromPropertyNameQuery = conn
                    .prepareStatement("SELECT DataType FROM MonitorPoint WHERE MonitorPointName = ?");

            lastMonitorDataTime = conn
                    .prepareStatement("SELECT MAX(MonitorTS) FROM MonitorData  WHERE MonitorPointId = ?");
            lastMonitorDataValue = conn
                    .prepareStatement("SELECT MeanStat FROM MonitorData WHERE MonitorPointId = ? AND MonitorTS = ?");

        } catch (SQLException err) {
            logger
                    .warning("SQLException in alma.TMCDB.Query.TMCDBCommonQueries.prepareStatements() "
                            + err.toString());
        }
    }

    //
    // Query methods

    /**
     * Get all antenna names from the TMCDB.
     * 
     * @return All antenna names found.
     */
    public List<String> getAntennaNames() {

        ensureTmcdbReady();
        List<String> names = new ArrayList<String>();

        // try {
        // ResultSet result = antennaNamesQuery.executeQuery();
        // while (result.next()) {
        // names.add(result.getString("AntennaName"));
        // }
        // } catch (SQLException err) {
        // logger.warning("SQLException in
        // alma.TMCDB.Query.TMCDBCommonQueries.getAntennaNames() - please report
        // a bug.");
        // }

        /*
         * The following code was quickly hacked together to deal with missing
         * configuration data in the current (2007-11-13) TMCDB at the ATF.
         * 
         * The only way to get a list of antennas at the ATF now is to depend on
         * the convention that assembly serial numbers have the form "CONTROL/<antenna
         * name>/<assembly>" where <antenna name> is the name of the antenna
         * containing the assembly, and <assembly> is the name of the assembly
         * (device).
         */

        List<String> assemblySNs = getComponentNames();
        String antennaName;
        for (String assemblySN : assemblySNs) {
//System.out.println("getAntennaNames() assemblySN: " + assemblySN);
            antennaName = getAntennaFromSN(assemblySN);
            if (!names.contains(antennaName))
                names.add(antennaName);
//System.out.println("getAntennaNames() antenna name: " + antennaName);
        }
        return names;
    }

    /**
     * Get the antenna name from an assembly serial number.
     * 
     * CAUTION: This is based on the current conventions for assembly serial
     * numbers. This will break when we start using serial numbers from AMBSIs.
     * When this happens, we must modify all code that uses this method.
     * 
     * @param serialNumber
     *            of the assembly
     * @return the antenna name embedded in the serialNumber
     */
    String getAntennaFromSN(String serialNumber) {
        int antennaNameIndex = 1;
        if (serialNumber.startsWith("CORR"))
        {
            antennaNameIndex = 0;
        }
        String[] snParts = serialNumber.split("/");
        try {
            return snParts[antennaNameIndex];
        } catch (ArrayIndexOutOfBoundsException err) {
            return "";
        }
    }

    /**
     * Get the assembly name from an assembly serial number.
     * 
     * CAUTION: This is based on the current conventions for assembly serial
     * numbers. This will break when we start using serial numbers from AMBSIs.
     * When this happens, we must modify all code that uses this method.
     * 
     * @param serialNumber
     *            of the assembly
     * @return the assembly name embedded in the serialNumber
     * 
     * Note: The returned assembly name may contain extra characters that
     * uniquely identify this assembly in a collection of similar assemblies.
     */
    String getAssemblyFromSN(String serialNumber) {
        int assemblyNameIndex = 2;
        String[] snParts;

	if (serialNumber.contains("FrontEnd")||serialNumber.contains("PhotonicReference"))
	    snParts = serialNumber.split("/",3);
        else if (serialNumber.contains("Mount")) {
            snParts = serialNumber.split("/");
            if (snParts[1].contains("DV")) {
                snParts[2] = "MountVertex";
            } else if (snParts[1].contains("PM")) {
                snParts[2] = "MountACA";
            } else {
                snParts[2] = "MountAEM";
            }

	} else
	    snParts = serialNumber.split("/");

        try {
            if (serialNumber.startsWith("CORR"))
            {
                return snParts[assemblyNameIndex - 1];
            }

            return snParts[assemblyNameIndex];
        } catch (ArrayIndexOutOfBoundsException err) {
            return "";
        }
    }

    /**
     * Get the ID for the given assembly serial number.
     * 
     * @param serialNumber
     *            the assembly to search for
     * @return the ID of the assembly found
     */
    public int getAssemblyIDforSN(String serialNumber) {
//System.out.println("getAssemblyIDforSN() => serialNumber = " + serialNumber);
        ensureTmcdbReady();
        List<Integer> ids = new ArrayList<Integer>();
        try {
            assemblyIdFromCompNameQuery.setString(1, serialNumber);
            ResultSet result = assemblyIdFromCompNameQuery.executeQuery();
            //assemblyIdFromAssemblySNQuery.setString(2, serialNumber);
            //ResultSet result = assemblyIdFromAssemblySNQuery.executeQuery();
            while (result.next())
                ids.add(result.getInt("AssemblyId"));
        } catch (SQLException err) {
            logger.warning(
                "SQLException in alma.TMCDB.Query.TMCDBCommonQueries.getAssembyIds() - please report a bug.");
        }
        if (ids.size() == 0)
            return 0;
        if (ids.size() == 1)
            return ids.get(0);
        logger.warning(
            "Found more than 1 assembly ID for assembly serial number in alma.TMCDB.Query.TMCDBCommonQueries.getAssemblyIDforSN(serialNumber) - possible TMCDB issue or bug in this method.");
        return ids.get(0);
    }

    /**
     * Get the ID for the given assembly name in the given antenna.
     * 
     * @param assemblyName
     *            assembly to search for
     * @param antennaName
     *            antenna to search in
     * @return the ID of the assembly found
     */
    public int getAssemblyId(String assemblyName, String antennaName) {
        ensureTmcdbReady();
//System.out.println("getAssemblyId() => entering: assemblyName = " + assemblyName + ", antennaName = " + antennaName);
        List<Integer> assemblyIdsInAntenna = getAssemblyIdsInAntenna(antennaName);
        List<Integer> assemblyIdsForAssemblyName = getAssemblyIdsWithName(assemblyName);
        List<Integer> intersection = new ArrayList<Integer>();
        for (int idInAntenna : assemblyIdsInAntenna)
            if (assemblyIdsForAssemblyName.contains(idInAntenna))
{
//System.out.println("getAssemblyId() => idInAntenna = " + idInAntenna);
                intersection.add(idInAntenna);
}
        if (intersection.size() == 0)
            return 0;
        if (intersection.size() == 1)
            return intersection.get(0);
        logger.warning(
            "Found more than 1 assembly ID for assembly name in antenna in alma.TMCDB.Query.TMCDBCommonQueries.getAssemblyId(assemblyName, antennaName) - possible TMCDB issue or bug in this method.");
        return intersection.get(0);
    }

    public String getAssemblySerialNumber(int assemblyId) {
        ensureTmcdbReady();
//System.out.println("getAssemblySerialNumber() => entering: assemblyId = " + assemblyId);
        String serialNumber = null;
        try {
            assemblySnFromAssemblyIdQuery.setInt(2, assemblyId);
            ResultSet result = assemblySnFromAssemblyIdQuery.executeQuery();
            while (result.next()) {
                serialNumber = result.getString("SerialNumber");
            }
        } catch (SQLException err) {
            logger.warning(
                "SQLException in alma.TMCDB.Query.TMCDBCommonQueries.getAssembySerialNumber() - please report a bug.");
        }
        return serialNumber;
    }

    public long getMonitorPointIdFromAssemblyId(String monitorPointName, int assemblyId)  {
        ensureTmcdbReady();
//System.out.println("getMonitorPointIdFromAssemblyId() => entering: monitorPointName = " + monitorPointName + ", assemblyId = " + assemblyId);
        long monitorPointId = 0;
        try {
            monitorPointIdFromAssemblyIdQuery.setString(1, monitorPointName);
            monitorPointIdFromAssemblyIdQuery.setInt(2, assemblyId);
            ResultSet result = monitorPointIdFromAssemblyIdQuery.executeQuery();
            while (result.next()) {
                monitorPointId = result.getInt("MonitorPointId");
            }
        } catch (SQLException err) {
            logger.warning(
                "SQLException in alma.TMCDB.Query.TMCDBCommonQueries.getMonitorPointIdFromAssemblyId() - please report a bug.");
        }
        return monitorPointId;
    }

    /**
     * Get all assembly IDs from the TMCDB.
     * 
     * @return All assembly IDs found.
     */
    public List<Integer> getAssemblyIds() {

        ensureTmcdbReady();
        List<Integer> ids = new ArrayList<Integer>();
        try {
            ResultSet result = assemblyIdsQuery.executeQuery();
            while (result.next()) {
                ids.add(result.getInt("AssemblyId"));
            }
        } catch (SQLException err) {
            logger.warning(
                "SQLException in alma.TMCDB.Query.TMCDBCommonQueries.getAssembyIds() - please report a bug.");
        }
        return ids;
    }

    /**
     * Get all assembly IDs contained in the given antenna.
     * 
     * @param antennaName
     *            the name of the antenna to check.
     * @return all assembly IDs found for the given antenna name.
     */
    List<Integer> getAssemblyIdsInAntenna(String antennaName) {
//System.out.println("getAssemblyIdsInAntenna() => entering, antenna Name = " + antennaName);
        /*
         * The following code was quickly hacked together to deal with missing
         * configuration data in the current (2007-11-13) TMCDB at the ATF.
         */

        ensureTmcdbReady();
        List<Integer> assemblyIds = new ArrayList<Integer>();
        // List<Integer> baseElementIds = getBaseElementIds(antennaName);
        // for (int baseElementId: baseElementIds) {
        // assemblyIds.addAll(getAssemblyIds(baseElementId));
        // }
        List<String> assemblySNs = getComponentNames();
        String antName;
        for (String assemblySN : assemblySNs) {
            antName = assemblySN.split("/")[1];
//System.out.println("getAssemblyIdsInAntenna() => antName = " + antName);
            if (antName.equals(antennaName)) {
                assemblyIds.add(getAssemblyIDforSN(assemblySN));
//System.out.println("getAssemblyIdsInAntenna() => adding Id = " + getAssemblyIDforSN(assemblySN) + ", SN = " + assemblySN);
            }
        }
        return assemblyIds;
    }

    /**
     * Get all assembly IDs with the given assembly name.
     * 
     * @param assemblyName
     *            the name of the assembly to find IDs for.
     * @return all assembly IDs found for the given assembly name.
     */
    List<Integer> getAssemblyIdsWithName(String assemblyName) {

        ensureTmcdbReady();
        List<Integer> ids = new ArrayList<Integer>();

        /*
         * The following code was quickly hacked together to deal with missing
         * configuration data in the current (2007-11-13) TMCDB at the ATF.
         */

        // try {
        // assemblyIdsWithNameQuery.setString(2, assemblyName);
        // ResultSet result = assemblyIdsWithNameQuery.executeQuery();
        // while (result.next()) {
        // ids.add(result.getInt("AssemblyId"));
        // }
        // } catch (SQLException err) {
        // logger.warning(
        // "Caught SQLException in
        // alma.TMCDB.Query.TMCDBCommonQueries.getPropertyName(assemblyName) " +
        // err.toString());
        // }
        List<String> allAssemblySNs = getComponentNames();
        List<String> candidateSNs = new ArrayList<String>();
        for (String candidateAssemblySN : allAssemblySNs)
            if (getAssemblyFromSN(candidateAssemblySN).equals(assemblyName))
                candidateSNs.add(candidateAssemblySN);
        for (String assemblySN : candidateSNs) {
            ids.add(getAssemblyIDforSN(assemblySN));
        }
        return ids;
    }

    /**
     * @param assemblyId
     *            the Id of the assembly to look for.
     * @return the AssemblyName found in the TMCDB for a given AssemblyId.
     */
    public String getAssemblyName(int assemblyId) {

//        ensureTmcdbReady();
        List<String> names = new ArrayList<String>();
        try {
            assemblyNameFromAssemblyIdQuery.setInt(1, assemblyId);
            ResultSet result = assemblyNameFromAssemblyIdQuery.executeQuery();
            while (result.next()) {
                names.add(result.getString("AssemblyName"));
            }
        } catch (SQLException err) {
            logger.warning(
                "Caught SQLException in alma.TMCDB.Query.TMCDBCommonQueries.getAssemblyName(assemblyId) "
                + err.toString());
        }
        if (names.size() == 0)
            return "";
        else if (names.size() == 1)
            return names.get(0);
        else {
            logger.warning(
                "alma.TMCDB.Query.TMCDBCommonQueries.getAssemblyName(assemblyId) found more than one name for assemblyId: "
                + assemblyId
                + ", returning the fist name found.");
            return names.get(0);
        }
    }

    /**
     * Find all assembly names for assemblies associated with the given antenna.
     * 
     * @param antennaName
     *            the name of the antenna containing assemblies we want names
     *            for.
     * @return a List of names of assemblies in named antenna.
     */
    public List<String> getAssemblyNames(String antennaName) {

        /*
         * The following code was quickly hacked together to deal with missing
         * configuration data in the current (2007-11-13) TMCDB at the ATF.
         * 
         * The only way to get assemblies contained in antennas at the ATF now
         * is to depend on the convention that assembly serial numbers have the
         * form "CONTROL/<antenna name>/<assembly>" where <antenna name> is
         * the name of the antenna containing the assembly, and <assembly> is
         * the name of the assembly (device).
         */

        ensureTmcdbReady();
        List<String> names = new ArrayList<String>();

        // List<Integer> IDs = getAssemblyIdsInAntenna(antennaName);
        // for (int ID: IDs) {
        // names.add(getAssemblyName(ID));
        // }

        List<String> serialNumbers = getComponentNames();

        String antenna = null;
        for (String SN : serialNumbers) {
//System.out.println("getAssemblyNames() Serial Number: " + SN);
            antenna = getAntennaFromSN(SN);
            if (antenna.equals(antennaName)) {
                names.add(getAssemblyFromSN(SN));
            }
        }
        return names;
    }

    // TODO: document
    List<String> getSerialNumbers() {

        ensureTmcdbReady();
        List<String> serialNumbers = new ArrayList<String>();
        try {
            ResultSet result = assemblySNsQuery.executeQuery();
            while (result.next()) {
                serialNumbers.add(result.getString("SerialNumber"));
//System.out.println("getSerialNumbers() SerialNumber: " + result.getString("SerialNumber"));
            }
        } catch (SQLException err) {
            logger.warning(
                "Caught SQLException in alma.TMCDB.Query.TMCDBCommonQueries.getSerialNumbers() "
                + err.toString());
        }
        return serialNumbers;
    }

    // TODO: document
    List<String> getComponentNames() {

        ensureTmcdbReady();
        List<String> componentNames = new ArrayList<String>();
/*
        List<String> serialNumbers = getSerialNumbers();
        MonitorDAO monitorDAO = new MonitorDAOImpl();

        for (String SN : serialNumbers) {
//System.out.println(SN);
          try {
            componentNames.add(monitorDAO.getComponentName(SN));
	  }catch(javax.persistence.NoResultException nre){

	  }	
        }
*/

        try {
            ResultSet result = componentNamesQuery.executeQuery();
            while (result.next()) {
                componentNames.add(result.getString("ComponentName"));

//System.out.println("getComponentNames() ComponentName: " + result.getString("ComponentName"));

            }
        } catch (SQLException err) {
            logger.warning(
                "Caught SQLException in alma.TMCDB.Query.TMCDBCommonQueries.getComponentNames() "
                + err.toString());
        }

        return componentNames;
    }

    /**
     * @param antennaName
     *            the name of the antenna to look for.
     * @return the BaseElementIds found in the TMCDB for the given antennaName.
     */
    List<Integer> getBaseElementIds(String antennaName) {

        ensureTmcdbReady();
        List<Integer> ids = new ArrayList<Integer>();
        try {
            baseElementIdsFromAntennaNameQuery.setString(2, antennaName);
            ResultSet result = baseElementIdsFromAntennaNameQuery
                    .executeQuery();
            while (result.next()) {
                ids.add(result.getInt("BaseElementId"));
            }
        } catch (SQLException err) {
            logger.warning(
                "Caught SQLException in alma.TMCDB.Query.TMCDBCommonQueries.getBaseElementId(antennaName) "
                + err.toString());
        }
        return ids;
    }

    /**
     * Get monitor data for the given antenna, assembly, and property within the
     * given time range.
     * 
     * @param antennaName
     *            the antenna containing the assembly to monitor
     * @param assemblyName
     *            the assembly to monitor
     * @param propertyName
     *            the property of the assembly to monitor
     * @param begin
     *            ArrayTime for the beginning of the time range of interest
     * @param end
     *            ArrayTime for the end of the time range of interest
     * @return an array of time-value pairs
     * 
     * Implementation details: The method converts from data types useful for
     * UIs to data expected by TMCDBMonitor.getMonitorData().
     */
    public TimeValuePager getMonitorData(String antennaName, String assemblyName,
            String propertyName, ArrayTime begin, ArrayTime end, long pageNum) {
//System.out.println("getMonitorData => entering: antennaName = " + antennaName + ", assemblyName = " + assemblyName + ", propertyName = " + propertyName);
        ensureTmcdbReady();
        TimeValue[] monitorData = new TimeValue[0];
//        List resultList;
        int assemblyId = getAssemblyId(assemblyName, antennaName);
        long monitorPointId = getMonitorPointIdFromAssemblyId(propertyName,assemblyId);

        TimeValuePager tvp = null;
        QueryDAO queryDAO = new QueryDAOImpl(logger);
        try {
            //resultList = queryDAO.getMonitorData(monitorPointId,java.sql.Timestamp.valueOf(begin.toFITS()),java.sql.Timestamp.valueOf(end.toFITS()));
            //monitorData = queryDAO.getMonitorData(monitorPointId,java.sql.Timestamp.valueOf(begin.toFITS()),java.sql.Timestamp.valueOf(end.toFITS()));
            tvp = queryDAO.getMonitorData(monitorPointId,java.sql.Timestamp.valueOf(begin.toFITS_alt()),java.sql.Timestamp.valueOf(end.toFITS_alt()));
        }catch(javax.persistence.NoResultException nre){
            logger.warning(
                "Caught NoResultException in alma.TMCDB.Query.TMCDBCommonQueries.getMonitorData) "
                + nre.toString());
        }

/*
        try {
            monitorData = getMonitorData(assemblyId, propertyId, dataType,
                    begin, end);
        } catch (AcsJTmcdbErrTypeEx err) {
            logger.warning(
                "Caught AcsJTmcdbErrTypeEx in alma.TMCDB.Query.TMCDBCommonQueries.getMonitorData) "
                + err.toString());
        }
*/
        //return monitorData;
        return tvp;
    }

    /**
     * Get the latest(one) monitor data point for the given antenna, assembly,
     * and property. TODO Implement this method
     * 
     * @param antennaName
     *            the antenna containing the assembly to monitor
     * @param assemblyName
     *            the assembly to monitor
     * @param propertyName
     *            the property of the assembly to monitor
     * @return a time-value pair
     * 
     */
    public TimeValue getLastMonitorData(String antennaName,
            String assemblyName, String propertyName) {
//System.out.println("getLastMonitorData => entering: antennaName = " + antennaName + ", assemblyName = " + assemblyName + ", propertyName = " + propertyName);
        ensureTmcdbReady();
        TimeValue monitorData = null;

        int assemblyId = getAssemblyId(assemblyName, antennaName);
        String serialNumber = getAssemblySerialNumber(assemblyId);
//System.out.println("getLastMonitorData => assemblyId = " + assemblyId + ", serialNumber = " + serialNumber);
        long monitorPointId = getMonitorPointIdFromAssemblyId(propertyName,assemblyId);
//System.out.println("getLastMonitorData => monitorPointId = " + monitorPointId + ", propertyName = " + propertyName + ", assemblyId = " + assemblyId);

/*
        QueryDAO queryDAO = new QueryDAOImpl();
        try {
            monitorData = queryDAO.getLastMonitorData(serialNumber,monitorPointId);
        }catch(javax.persistence.NoResultException nre){
            logger.warning(
                "Caught NoResultException in alma.TMCDB.Query.TMCDBCommonQueries.getLastMonitorData) "
                + nre.toString());
        }
*/

        try {
            monitorData = getLastMonitorDataPoint(monitorPointId);
        } catch (AcsJTmcdbErrTypeEx err) {
            logger.warning(
                "Caught AcsJTmcdbErrTypeEx in alma.TMCDB.Query.TMCDBCommonQueries.getMonitorData) "
                + err.toString());
        }

        return monitorData;
    }

    private TimeValue getLastMonitorDataPoint(long monitorPointId) throws AcsJTmcdbSqlEx {

        try {
//System.out.println("getLastMonitorDataPoint => entering");
            lastMonitorDataTime.setLong(1, monitorPointId);
            ResultSet timeStampResults = lastMonitorDataTime
                    .executeQuery();
//System.out.println("getLastMonitorDataPoint => executed lastMonitorDataTime ");
            if (timeStampResults.next() == false)
                return null;
            Timestamp lastTimeStamp = timeStampResults.getTimestamp(1);
//System.out.println("getLastMonitorDataPoint => lastTimeStamp = " + lastTimeStamp);
            lastMonitorDataValue.setLong(1, monitorPointId);
            lastMonitorDataValue.setTimestamp(2, lastTimeStamp);
            ResultSet valueResults = lastMonitorDataValue.executeQuery();
            if (valueResults.next() == false) {
                logger.warning(
                    "Did not find Value data matching time stamp in alma.TMCDB.Query.TMCDBCommonQueries.getLastMonitorDataPoint()");
                return null;
            }

            String lastValue = valueResults.getString(1);
            if (valueResults.next() != false) {
                logger.warning(
                    "Found more than one Value data matching time stamp in alma.TMCDB.Query.TMCDBCommonQueries.getLastMonitorDataPoint()");
            }

            //TimeValue result = new TimeValue(new ArrayTime(lastTimeStamp),
            TimeValue result = new TimeValue(new ArrayTime(lastTimeStamp.toString()), lastValue);
            return result;

        } catch (SQLException err) {
            err.printStackTrace();
            throw new AcsJTmcdbSqlEx(err);
        }
    }

    /**
     * Find the property data type for the given property name.
     * 
     * @param propertyName
     *            the name of the property to find the data type for.
     * @return a String containing the data type of the named property.
     */
    public String getPropertyDataType(String propertyName) {

        ensureTmcdbReady();
        String dataType = "";
        try {
            propertyDataTypeFromPropertyNameQuery.setString(1, propertyName);
            ResultSet result = propertyDataTypeFromPropertyNameQuery
                    .executeQuery();
            // This should give 0 or 1 results.
            if (result.next())
                dataType = result.getString("DataType");
        } catch (SQLException err) {
            logger.warning(
                "Caught SQLException in alma.TMCDB.Query.TMCDBCommonQueries.getPropertyDataType(" + propertyName + ") "
                + err.toString());
        }
        return dataType;
    }

    private String getGeneralAssemblyName(String specificAssemblyName) {
        // TODO: This conversion is brittle, but it is required until the full TMCDB table design is 
        // implemented.
        if (specificAssemblyName.startsWith("DRX"))
        	return "DRX";
        else if (specificAssemblyName.startsWith("DTX"))
           	return "DTX";
        else if (specificAssemblyName.startsWith("IFProc"))
            return "IFProc";
        else if (specificAssemblyName.startsWith("LO2"))
            return "LO2";
        else if (specificAssemblyName.startsWith("PSLLC"))
            return "PSLLC";
        else if (specificAssemblyName.startsWith("PSSAS"))
            return "PSSAS";
        else if (specificAssemblyName.startsWith("FrontEnd"))
            return specificAssemblyName.split("/")[1];
        else if (specificAssemblyName.startsWith("PhotonicReference"))
            return specificAssemblyName.split("/")[1];
        return specificAssemblyName;

    }

    /**
     * Find all property names for the given assembly.
     * 
     * @param assemblyName
     *            the name of the assembly to find properties for.
     * @return a List of names of properties for named assembly.
     */
    public List<String> getPropertyNames(String assemblyName) {

        ensureTmcdbReady();
        List<String> names = new ArrayList<String>();

        /*
         * The following code was quickly hacked together to deal with missing
         * configuration data in the current (2007-11-13) TMCDB at the ATF.
         */

//System.out.println("assemblyName: " + assemblyName);

        try {
            String generalAssemblyName = getGeneralAssemblyName(assemblyName);
            assemblyIdsWithNameQuery.setString(2,generalAssemblyName);
//System.out.println("generalassemblyName: " + generalAssemblyName);
            ResultSet result1 = assemblyIdsWithNameQuery.executeQuery();
            if(result1.next()){
                propertyNameFromAssemblyNameQuery.setString(1, result1.getString("AssemblyId"));
//System.out.println("assemblyId: " + result1.getString("AssemblyId"));
                ResultSet result = propertyNameFromAssemblyNameQuery.executeQuery();
                while (result.next()) {
                    names.add(result.getString("MonitorPointName"));
//System.out.println("MonitorPointName: " + result.getString("MonitorPointName"));
                }
            }
        } catch (SQLException err) {
            logger.warning(
                "Caught SQLException in alma.TMCDB.Query.TMCDBCommonQueries.getPropertyNames(" + assemblyName + ") "
                + err.toString());
        }
        return names;
    }

    /**
     * Call super class method using database connection already stored.
     * 
     * @param filename
     *            of file containing SQL code to pass to database.
     */
    public void loadAndExecuteScript(String filename) throws Exception {
        // super.loadAndExecuteScript(conn, filename);

        /*
         * The following is temporary code to allow progress while
         * TMCDBUtil.loadAndExecuteScript() is being redesigned. After that
         * redesign is complete, the following code will be removed.
         */
        // TODO: Delete the following after TMCDBUtil.loadAndExecuteScript() is complete.
        BufferedReader csr = null;
        Statement stmt = null;

        try {
            stmt = conn.createStatement();
            csr = new BufferedReader(new FileReader(filename));
        } catch (SQLException e) {
            // TODO Replace auto-generated catch block.
            e.printStackTrace();
        } catch (FileNotFoundException e) {
            // TODO Replace auto-generated catch block.
            e.printStackTrace();
        }
        String line;
        StringBuffer sb = new StringBuffer();
        if (csr != null && stmt != null)
            try {
                while ((line = csr.readLine()) != null) {
                    if (line.startsWith("--"))
                        continue; // SQL Comment
                    if (line.contains(";")) {
                        line = line.replace(";", "");
                        sb.append(line);
                        line = sb.toString();
                        try {
                            stmt.execute(line);
                        } catch (SQLException e) {
                            // TODO Replace Auto-generated catch block.
                            e.printStackTrace();
                        }
                        sb = new StringBuffer();
                    } else {
                        sb.append(line);
                    }
                }
            } catch (IOException e) {
                // TODO Replace auto-generated catch block.
                e.printStackTrace();
            }
    }


} // class

//O_o
