package alma.archive.projectcode;

import java.util.Formatter;

public class ProjectCode implements Comparable<ProjectCode> {

    private Integer myYear;

    private Integer mySemester;

    private Integer mySequenceNumber;

    private TypeCode myTypeCode;

    public ProjectCode(int inYear, int inSemester, int inSeqNum,
            TypeCode inTypeCode) {
        setYear(inYear);
        setSemester(inSemester);
        setSequenceNumber(inSeqNum);
        setTypeCode(inTypeCode);
    }

    public ProjectCode(String inProjectCode) {
        populate(inProjectCode);
    }

    @Override
    public int compareTo(ProjectCode inCode) {
        int outValue = this.myYear.compareTo(inCode.getYear());
        if (outValue == 0) {
            outValue = this.mySemester.compareTo(inCode.getSemester());
            if (outValue == 0) {
                outValue = this.mySequenceNumber.compareTo(inCode
                        .getSequenceNumber());
                if (outValue == 0) {
                    outValue = this.myTypeCode.compareTo(inCode.getTypeCode());
                }
            }
        }
        return outValue;
    }

    @Override
    public int hashCode() {
        return this.myYear + this.mySemester + this.mySequenceNumber
                + this.myTypeCode.ordinal();
    }

    @Override
    public boolean equals(Object inObject) {
        boolean outResult = false;
        if (inObject instanceof ProjectCode) {
            if (compareTo((ProjectCode) inObject) == 0) {
                outResult = true;
            }
        }
        return outResult;
    }

    public String getData() {
        StringBuilder builder = new StringBuilder();
        builder.append("Year: ");
        builder.append(this.myYear);
        builder.append(" Semester: ");
        builder.append(this.mySemester);
        builder.append(" SequenceNumber: ");
        builder.append(this.mySequenceNumber);
        builder.append(" TypeCode: ");
        builder.append(this.myTypeCode);
        builder.append(" ProjectCode: ");
        builder.append(toString());
        return builder.toString();
    }

    public int getYear() {
        return this.myYear;
    }

    public int getSemester() {
        return this.mySemester;
    }

    public int getSequenceNumber() {
        return this.mySequenceNumber;
    }

    public TypeCode getTypeCode() {
        return this.myTypeCode;
    }

    private void populate(String inProjectCode) {
        String[] parts = inProjectCode.split("\\.");
        if (parts.length != 4) {
            throw new IllegalArgumentException("Incorrect format; received ["
                    + inProjectCode
                    + "]. Expected is year.semester.sequence number.type code.");
        }

        try {
            setYear(Integer.parseInt(parts[0]));
        } catch (NumberFormatException e) {
            throw new IllegalArgumentException(
                    "Incorrect format of year; received [" + parts[0]
                            + "], integer expected.");
        }

        try {
            setSemester(Integer.parseInt(parts[1]));
        } catch (NumberFormatException e) {
            throw new IllegalArgumentException(
                    "Incorrect format of semester; received [" + parts[1]
                            + "], integer expected.");
        }

        try {
            setSequenceNumber(Integer.parseInt(parts[2]));
        } catch (NumberFormatException e) {
            throw new IllegalArgumentException(
                    "Incorrect format of sequence number; received ["
                            + parts[2] + "], integer expected.");
        }

        setTypeCode(parts[3]);
    }

    public void setSemester(int inSemester) {
        if (inSemester < 1 || inSemester > 9) {
            throw new IllegalArgumentException(
                    "Illegal value of semester; received [" + inSemester
                            + "], value should be between 1 and 9.");
        }
        this.mySemester = inSemester;
    }

    public void setSequenceNumber(int inSeqNum) {
        if (inSeqNum < 1 || inSeqNum > 99999) {
            throw new IllegalArgumentException(
                    "Illegal value of sequence number; received [" + inSeqNum
                            + "], value should be between 1 and 99999.");
        }
        this.mySequenceNumber = inSeqNum;
    }

    public void setTypeCode(String inTypeCode) {
        TypeCode code = TypeCode.typeCodeAbbreviation2enum(inTypeCode);
        if (code == null) {
            throw new IllegalArgumentException("Unknown type code ["
                    + inTypeCode + "] received.");
        }
        setTypeCode(code);
    }

    public void setTypeCode(TypeCode inTypeCode) {
        this.myTypeCode = inTypeCode;
    }

    public void setYear(int inYear) {
        if (inYear < 2009 || inYear > 2200) {
            throw new IllegalArgumentException(
                    "Unrealistic value of year; received [" + inYear
                            + "], value between 2009 and 2200 expected.");
        }
        this.myYear = inYear;
    }

    @Override
    public String toString() {
        StringBuilder builder = new StringBuilder();
        Formatter formatter = new Formatter();
        builder.append(this.myYear);
        builder.append(".");
        builder.append(this.mySemester);
        builder.append(".");
        builder.append(formatter.format("%05d", this.mySequenceNumber));
        builder.append(".");
        builder.append(this.myTypeCode.getTypeCodeAbbreviation());
        return builder.toString();
    }
}
