/*
 * 	  Created on 15-Oct-2003
 * 
 *    ALMA - Atacama Large Millimiter Array
 *    (c) European Southern Observatory, 2002
 *    Copyright by ESO (in the framework of the ALMA collaboration),
 *    All rights reserved
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License as published by the Free Software Foundation; either
 *    version 2.1 of the License, or (at your option) any later version.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
 *    MA 02111-1307  USA
 */
package alma.archive.commands;

import java.io.IOException;
import java.io.StringReader;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.List;
import java.util.ListIterator;
import java.util.StringTokenizer;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.jdom.Document;
import org.jdom.Element;
import org.jdom.JDOMException;
import org.jdom.input.SAXBuilder;
import org.jdom.output.Format;
import org.jdom.output.XMLOutputter;
import org.jdom.xpath.XPath;

import alma.archive.database.interfaces.InternalIF;
import alma.archive.database.interfaces.InternalIFFactory;
import alma.archive.exceptions.ArchiveException;
import alma.archive.exceptions.ModuleCriticalException;
import alma.archive.exceptions.general.DatabaseException;
import alma.archive.exceptions.general.EntityExistsException;
import alma.archive.wrappers.ArchiveTimeStamp;
import alma.archive.wrappers.Permissions;

/**
 * @author simon
 */
public class PathManager
{
	private static PathManager _instance = null;
	private static int instanceCounter = 0;
	
	public static PathManager instance(Logger logger) throws DatabaseException, ModuleCriticalException
	{
		instanceCounter++;
		if (_instance == null)
		{
			_instance = new PathManager(logger);
		} 
		return _instance;
	}
	
	public void close() throws DatabaseException, ModuleCriticalException
	{
		instanceCounter--;
		 if (instanceCounter == 0)
		 {
		 	_instance = null;
		 	internal.close();
		 }
	}
	
	private Logger logger = null;
	private InternalIF internal = null;
	private String _pathmanager = "archive://pathmanager";
	private String _nullns = "http://null";
	private URI nullns;
	private URI pathmanager;
	
	private Document doc;
	private Element root;
	
	private PathManager(Logger logger) throws DatabaseException, ModuleCriticalException
	{
		this.logger = logger;
		try 
		{
			pathmanager = new URI(_pathmanager);
			nullns = new URI(_nullns);
		}
		catch (URISyntaxException e) {e.printStackTrace();}
		
		root = new Element("path");
		doc = new Document(root);
		
		internal = InternalIFFactory.getInternalIF(logger);
		load();
	}
	
	public void addPath(URI path) throws ModuleCriticalException
	{
		if (path.getScheme().equalsIgnoreCase("curl"))
		{
			StringTokenizer st = new StringTokenizer(path.getAuthority()+ "/" + path.getPath(),"/");
			create(st,root);
			save();
		}
	}
	
	private void create(StringTokenizer st, Element current)
	{
		if (st.hasMoreTokens())
		{
			String token = st.nextToken();
			Element sub = current.getChild(token);
			if (sub == null)
			{
				sub = new Element(token);
				current.addContent(sub);
			}
			create(st,sub);
		}
	}
	
	
	public void removePath(URI path) throws ModuleCriticalException
	{
		if (path.getScheme().equalsIgnoreCase("curl"))
		{
			StringTokenizer st = new StringTokenizer(path.getAuthority()+ "/" + path.getPath(),"/");
			delete(st,root);
			save();
		}
	}
	
	private boolean delete(StringTokenizer st, Element current)
	{
		if (st.hasMoreTokens())
		{
			if (current.getChildren().size()>0)
			{
				String token = st.nextToken();
				Element sub = current.getChild(token);
				if (delete(st,sub))
				{
					current.removeContent(sub);
					if (current.getChildren().size()>0) return false;
					else return true;
				}
				return false;
			}
			else
			{
				return true;
			}
		}
		else
		{
			return true;
		}
	}
	
	public String listSubnodes(String subPath)
	{
		try
		{
			String _query = "/path" + subPath;
			XPath query = XPath.newInstance(_query);
			Element res = (Element)query.selectSingleNode(root);
			if (res != null)
			{
				List results = res.getChildren();
				String subnodes = "";
				boolean first = true;
				ListIterator iter = results.listIterator();
				while (iter.hasNext())
				{
					if (!first) subnodes += " ";
					Element sub = (Element)iter.next();
					subnodes += sub.getName();
					
					first = false;
				}
				return subnodes;
			}
			else return "";
		}
		catch (JDOMException e)
		{
			return "";
		}
	}
	
	public String toString()
	{
		XMLOutputter out = new XMLOutputter(Format.getPrettyFormat());
		// XMLOutputter out = new XMLOutputter("  ",true,"UTF-8");
		return out.outputString(doc);
	}
	
	private void save() throws ModuleCriticalException
	{
		try
		{
			//internal.store(pathmanager,toString());
			try
			{
				internal.store(pathmanager,toString(),nullns,"","pathmanager",
					new Permissions(),"pathmanager",true);
			}
			catch (EntityExistsException e)
			{
				internal.update(pathmanager,new ArchiveTimeStamp(),toString(),nullns,false,"pathmanager");
			}
		}
		catch (ArchiveException e){logger.log(Level.SEVERE,"ARCHIVE: Cannot save listenermanager");}	
	}
	
	private void load() throws ModuleCriticalException
	{
		SAXBuilder builder = new SAXBuilder();
		builder.setIgnoringElementContentWhitespace(true);
		
		try
		{
			doc = builder.build(new StringReader(internal.get(pathmanager,"pathmanager")));
			root = doc.getRootElement();
		}
		catch (JDOMException e) {logger.log(Level.SEVERE,"JDOM: Cannot parse Listeners string");}
		catch (IOException e) {logger.log(Level.SEVERE,"IO: Cannot parse Listeners string");}
		catch (ArchiveException e) {logger.log(Level.INFO,"ARCHIVE: Problem retriving the string");}
	}
}
