/*
* Copyright (C) The Community OpenORB Project. All rights reserved.
*
* This software is published under the terms of The OpenORB Community Software
* License version 1.0, a copy of which has been included with this distribution
* in the LICENSE.txt file.
*/
package org.openorb.util;

import java.io.UnsupportedEncodingException;

import java.util.ArrayList;
import java.util.Iterator;

import org.omg.CosNaming.NameComponent;
import org.omg.CosNaming.NamingContext;
import org.omg.CosNaming.NamingContextHelper;
import org.omg.CosNaming.NamingContextPackage.InvalidName;
import org.omg.CosNaming.NamingContextPackage.NotFound;
import org.omg.CosNaming.NamingContextPackage.CannotProceed;
import org.omg.CosNaming.NamingContextPackage.AlreadyBound;
import org.omg.CosNaming.NamingContextPackage.NotFoundReason;

/**
 * The functions in this class perform various translations on stringified 
 * CosNaming names.
 *
 * @author Chris Wood
 */
public abstract class NamingUtils
{
    /**
     * Encodes a string according to RFC2396.
     * All escaped chars use UTF-8 
     * encoding ( RFC2396 has been updated to RFC2732 ).
     *
     * @param str The string to encode.
     * @return The encoded string.
     * @throws UnsupportedEncodingException When the string can't be converted
     * into UTF-8 format.
     */
    public static String encodeRFC2396( String str )
        throws UnsupportedEncodingException
    {
        StringBuffer sb = new StringBuffer();
        int start = 0;
        byte [] tmp;

        for ( int i = 0; i < str.length(); ++i )
        {
            char c = str.charAt( i );

            if ( ( c >= 'a' && c <= 'z' )
                  || ( c >= 'A' && c <= 'Z' )
                  || ( c >= '0' && c <= '9' ) )
                continue;

            switch ( c )
            {
                // reserved

            case ';':

            case '/':

            case '?':

            case ':':

            case '@':

            case '&':

            case '=':

            case '+':

            case '$':

            case ',':
                // unreserved

            case '-':

            case '_':

            case '.':

            case '!':

            case '~':

            case '*':

            case '\'':

            case '(':

            case ')':
                break;

            default:
                sb.append( str.substring( start, i ) );
                tmp = str.substring( i, i + 1 ).getBytes( "UTF-8" );

                for ( int j = 0; j < tmp.length; ++j )
                {
                    sb.append( '%' );

                    if ( ( tmp[ j ] & 0xF0 ) < 0xA0 )
                        sb.append( ( tmp[ j ] & 0xF0 ) >> 4 );
                    else
                        sb.append( ( char ) ( 'A' + ( ( tmp[ j ]
                              & 0xF0 - 0xA0 ) >> 4 ) ) );

                    if ( ( tmp[ j ] & 0xF ) < 0xA )
                        sb.append( tmp[ j ] & 0xF );
                    else
                        sb.append( ( char ) ( 'A' + ( tmp[ j ]
                              & 0xF - 0xA ) ) );
                }

                start = i + 1;
            }
        }

        sb.append( str.substring( start ) );
        return sb.toString();
    }

    /**
     * Decodes a RFC2396 encoded string. 
     * 
     * @param enc The string to encode.
     * @return The decoded string.
     * @throws UnsupportedEncodingException
     * @throws NumberFormatException
     */
    public static String decodeRFC2396( String enc )
        throws UnsupportedEncodingException
    {
        StringBuffer sb = new StringBuffer();
        int start = 0;

        for ( int i = 0; i < enc.length(); ++i )
        {
            if ( sb.charAt( i ) == '%' )
            {
                sb.append( enc.substring( start, i ) );
                int count = 1;

                while ( sb.charAt( i + 2 * count ) == '%' )
                    ++count;

                byte [] buf = new byte[ count ];

                for ( int j = 0; j < count; ++j )
                    buf[ i ] = Byte.parseByte(
                        enc.substring( i + 2 * j + 1, i + 2 * j + 2 ), 16 );

                sb.append( new String( buf, "UTF-16" ) );

                i += count * 3 - 1;
            }
        }
        sb.append( enc.substring( start ) );
        return sb.toString();
    }

    /**
     * This function checks an address for the correct format.
     *
     * @param addr The address to check.
     * @return True if the format is correct, false otherwise.
     */
    private static boolean checkAddress( String addr )
    {
        int end = addr.length();
        int start;

        do
        {
            start = addr.lastIndexOf( ",", end );

            if ( start < 0 )
                start = 0;

            int proto = addr.indexOf( ":", start );

            // parse the protocol
            if ( proto < 0 || proto > end )
                return false;

            // TODO: check the protocol for correctness
            // addr.substring(start, proto)
            int vers = addr.indexOf( "@", proto );

            if ( vers > 0 && vers < end )
            {
                // parse the version.
                int major = addr.indexOf( ".", proto );

                if ( major < 0 || major > vers )
                    return false;

                try
                {
                    Integer.parseInt( addr.substring( proto + 1, major ) );
                    Integer.parseInt( addr.substring( major + 1, vers ) );
                }
                catch ( NumberFormatException ex )
                {
                    return false;
                }
            }
            else
            {
                vers = proto;
            }

            int host = addr.indexOf( ":", vers + 1 );

            if ( host > 0 && host < end )
            {
                // parse the port
                try
                {
                    Integer.parseInt( addr.substring( host + 1, end ) );
                }
                catch ( NumberFormatException ex )
                {
                    return false;
                }
            }
            else
            {
                host = end;
            }

            // TODO: check the address for correctness.
            // addr.substring(vers+1, host)
            end = start;
        }
        while ( start > 0 );
        
        return true;
    }

    //////////////////////////////////////////////////////////////
    // Deprecated methods and member fields.
    // Will be removed after the 1.3.0 release.
    //////////////////////////////////////////////////////////////

    /**
     * Binds a objet to a name, creating any required subcontexts.
     *
     * @param context The name context root.
     * @param name The name
     * @param obj The object to bind.
     * @exception InvalidName invalid name
     * @exception NotFound context not found
     * @exception CannotProceed canno bind
     * @exception AlreadyBound name already bound
     * @deprecated This method uses interfaces from the CosNaming module.
     * Which results in an implicit dependency to the naming service. This
     * dependency is going to be removed in the future.
     */
    public static void deepBind( NamingContext context, String name, org.omg.CORBA.Object obj )
        throws InvalidName, NotFound, CannotProceed, AlreadyBound
    {
        deepBind( context, to_name( name ), obj );
    }

    /**
     * Binds a objet to a name, creating any required subcontexts.
     *
     * @param context The name context root.
     * @param name The name
     * @param obj The object to bind.
     * @exception InvalidName if the name is invalid
     * @exception NotFound context not found
     * @exception CannotProceed if the binding cannot be completed
     * @exception AlreadyBound if already bound
     * @deprecated This method uses interfaces from the CosNaming module.
     * Which results in an implicit dependency to the naming service. This
     * dependency is going to be removed in the future.
     */
    public static void deepBind( NamingContext context, NameComponent [] name,
      org.omg.CORBA.Object obj )
      throws InvalidName, NotFound, CannotProceed, AlreadyBound
    {
        NameComponent [] ctxName = new NameComponent[ name.length - 1 ];
        System.arraycopy( name, 0, ctxName, 0, ctxName.length );
        NamingContext sub = resolveOrCreateContext( context, ctxName );
        sub.bind( new NameComponent [] { name[ name.length - 1 ] }, obj );
    }

    /**
     * Binds a objet to a name, creating any required subcontexts.
     *
     * @param context The name context root.
     * @param name The name
     * @param obj The object to bind.
     * @exception InvalidName if the name is invalid
     * @exception NotFound if the context cannot be found
     * @exception CannotProceed if the rebind conot proceed
     * @deprecated This method uses interfaces from the CosNaming module.
     * Which results in an implicit dependency to the naming service. This
     * dependency is going to be removed in the future.
     */
    public static void deepRebind( NamingContext context, String name, 
       org.omg.CORBA.Object obj )
       throws InvalidName, NotFound, CannotProceed
    {
        deepRebind( context, to_name( name ), obj );
    }

    /**
     * Binds a objet to a name, creating any required subcontexts.
     *
     * @param context The name context root.
     * @param name The name
     * @param obj The object to bind.
     * @exception InvalidName if the name is invalid
     * @exception NotFound if the context cannot be found
     * @exception CannotProceed if the bind conot proceed
     * @deprecated This method uses interfaces from the CosNaming module.
     * Which results in an implicit dependency to the naming service. This
     * dependency is going to be removed in the future.
     */
    public static void deepRebind( NamingContext context, NameComponent [] name, 
       org.omg.CORBA.Object obj )
        throws InvalidName, NotFound, CannotProceed
    {
        NameComponent [] ctxName = new NameComponent[ name.length - 1 ];
        System.arraycopy( name, 0, ctxName, 0, ctxName.length );
        NamingContext sub = resolveOrCreateContext( context, ctxName );
        sub.rebind( new NameComponent [] { name[ name.length - 1 ] }, obj );
    }

    /**
     * Resolve contexts as far as they exist in the name, and create the remainder
     *
     * @param context The name context root.
     * @param name The name
     * @return NamingContext the naming context
     * @exception InvalidName if the name is invalid
     * @exception NotFound if the context cannot be found
     * @exception CannotProceed if the resolve/context creation connot proceed
     * @deprecated This method uses interfaces from the CosNaming module.
     * Which results in an implicit dependency to the naming service. This
     * dependency is going to be removed in the future.
     */
    public static NamingContext resolveOrCreateContext( NamingContext context, 
      NameComponent [] name )
        throws InvalidName, NotFound, CannotProceed
    {
        org.omg.CORBA.Object obj;

        try
        {
            obj = context.resolve( name );
        }
        catch ( NotFound ex )
        {
            switch ( ex.why.value() )
            {

            case NotFoundReason._not_context:

            case NotFoundReason._not_object:
                throw ex;
            }

            NameComponent [] bname = new NameComponent [ 
              name.length - ex.rest_of_name.length ];
            System.arraycopy( name, 0, bname, 0, bname.length );

            NamingContext base = context;

            if ( bname.length > 0 )
                base = NamingContextHelper.narrow( context.resolve( bname ) );

            for ( int i = 0; i < ex.rest_of_name.length; ++i )
            {
                NameComponent [] next = new NameComponent[] { ex.rest_of_name[ i ] };

                try
                {
                    base = base.bind_new_context( next );
                }
                catch ( AlreadyBound ex1 )
                {
                    base = NamingContextHelper.narrow( base.resolve( next ) );
                }
            }

            return base;
        }

        try
        {
            return NamingContextHelper.narrow( obj );
        }
        catch ( final org.omg.CORBA.BAD_PARAM ex )
        {
            throw (NotFound) ExceptionTool.initCause( new NotFound( 
                    NotFoundReason.not_context, 
                    new NameComponent [] { name[ name.length - 1 ] } ), ex );
        }
    }

    /**
     * This operation accepts a stringified name and returns a Name.       
     *
     * @param sn the stringified name to transform to a name.
     * @return NameComponent[] the name
     * @throws InvalidName This exception is raised if the
     * stringified name is syntactically malformed or
     * violates an implementation limit.
     * @exception InvalidName if a name is invalid
     *
     * @deprecated This method uses interfaces from the CosNaming module.
     * Which results in an implicit dependency to the naming service. This
     * dependency is going to be removed in the future. Additionally this
     * method is part of the NamingContextExt package and there is no reason
     * for duplicating it here.
     */
    public static NameComponent[] to_name( String sn )
        throws InvalidName
    {
        if ( sn.length() == 0 )
            throw new InvalidName();

        ArrayList list = new ArrayList();

        int start = 0;

        boolean id = true;

        StringBuffer buf = null;

        for ( int i = 0; i < sn.length(); ++i )
            switch ( sn.charAt( i ) )
            {

            case '\\':

                if ( buf == null )
                    buf = new StringBuffer();

                buf.append( sn.substring( start, i ) );

                start = ++i;

                break;

            case '/':
                if ( buf != null )
                  list.add( 
                    buf.append( sn.substring( start, i ) ).toString() );
                else
                    list.add( sn.substring( start, i ) );

                if ( id )
                {
                    if ( i == start && buf == null )
                        throw new InvalidName();

                    list.add( "" );
                }
                else
                {
                    id = true;
                }

                buf = null;
                start = i + 1;
                break;

            case '.':

                if ( !id )
                    throw new InvalidName();

                if ( buf != null )
                {
                    list.add( buf.append( sn.substring( start, i ) ).toString() );
                    buf = null;
                }
                else
                    list.add( sn.substring( start, i ) );

                id = false;

                start = i + 1;

                break;
            }

        // the end state is identical to a '/'
        if ( buf != null )
            list.add( buf.append( sn.substring( start ) ).toString() );
        else
            list.add( sn.substring( start ) );

        if ( id )
        {
            if ( start == sn.length() && buf == null )
                throw new InvalidName();

            list.add( "" );
        }

        NameComponent [] name = new NameComponent[ list.size() / 2 ];
        Iterator itt = list.iterator();

        for ( int i = 0; i < list.size(); i += 2 )
        {
            name[ i / 2 ] = new NameComponent( 
             ( String ) itt.next(), ( String ) itt.next() );
        }
        return name;
    }

    /**
     * This operation accepts Name and returns a stringified name. 
     *
     * @param  n the name to stringified.
     * @return String the stringified name
     * @throws InvalidName This exception is raised if the name is
     * invalid.
     * @exception InvalidName if a name is invalid
     * @deprecated This method uses interfaces from the CosNaming module.
     * Which results in an implicit dependency to the naming service. This
     * dependency is going to be removed in the future. Additionally this
     * method is part of the NamingContextExt package and there is no reason
     * for duplicating it here.
     */
    public static String to_string( NameComponent[] n )
        throws InvalidName
    {
        if ( n.length == 0 )
            throw new InvalidName();

        int start, pos;

        StringBuffer buf = new StringBuffer();

        for ( int i = 0; i < n.length; ++i )
        {
            if ( n[ i ].id == null || n[ i ].id.length() == 0 )
            {
                buf.append( '.' );

                if ( n[ i ].kind == null || n[ i ].kind.length() == 0 )
                {
                    buf.append( '/' );
                    continue;
                }
            }
            else
            {
                start = 0;

                for ( int j = 0; j < n[ i ].id.length(); ++j )
                    switch ( n[ i ].id.charAt( j ) )
                    {

                    case '/':

                    case '\\':

                    case '.':
                        buf.append( n[ i ].id.substring( start, j ) );
                        buf.append( '\\' );
                        start = j;
                        break;
                    }

                buf.append( n[ i ].id.substring( start ) );

                if ( n[ i ].kind == null || n[ i ].kind.length() == 0 )
                {
                    buf.append( '/' );
                    continue;
                }

                buf.append( '.' );
            }

            start = 0;

            for ( int j = 0; j < n[ i ].kind.length(); ++j )
                switch ( n[ i ].kind.charAt( j ) )
                {

                case '/':

                case '\\':

                case '.':
                    buf.append( n[ i ].kind.substring( start, j ) );
                    buf.append( '\\' );
                    start = j;
                    break;
                }

            buf.append( n[ i ].kind.substring( start ) );
            buf.append( '/' );
        }

        return buf.substring( 0, buf.length() - 1 );
    }

    /**
     * This operation takes an URL address and performs any escapes
     * necessary on the stringified name and returns a fully formed
     * URL string.
     *
     * @param  addr the address ( for example myhost.xyz.com )
     * @param  sn  the stringified name to add to the URL
     * @return  the URL string format.
     *
     * @throws org.omg.CosNaming.NamingContextExtPackage.InvalidAddress 
     * This exception is raised if an address
     * is invalid ( it means that the address does not
     * respect the address format ).
     * @throws InvalidName This exception is raised if the
     * stringified name is syntactically malformed or
     * violates an implementation limit.
     *
     * @deprecated This method uses interfaces from the CosNaming module.
     * Which results in an implicit dependency to the naming service. This
     * dependency is going to be removed in the future. Additionally this
     * method is part of the NamingContextExt package and there is no reason
     * for duplicating it here.
     */
    public static String to_url( String addr, String sn )
        throws org.omg.CosNaming.NamingContextExtPackage.InvalidAddress, InvalidName
    {
        // check the address format.
        if ( checkAddress( addr ) == false )
            throw new org.omg.CosNaming.NamingContextExtPackage.InvalidAddress(); 

        // check the string name format.
        to_name( sn );
        
        String url = null;
        try
        {
            url = "corbaname:" + addr + "#" + encodeRFC2396( sn );
        }
        catch ( java.io.UnsupportedEncodingException ex )
        {
            return null;
        }
        
        return url;
    }

    /**
     * @deprecated This method uses interfaces from the CosNaming module.
     * Which results in an implicit dependency to the naming service. This
     * dependency is going to be removed in the future. 
     */
    private static final NameComponent [] EMPTY_NAME = new NameComponent[ 0 ];

    /**
     * Get the parent name of the given name. The parent of the empty name
     * will be also empty.
     * 
     * @param name the name.
     * @return the parent name.
     *
     * @deprecated This method uses interfaces from the CosNaming module.
     * Which results in an implicit dependency to the naming service. This
     * dependency is going to be removed in the future. 
     */
    public static NameComponent [] parent( final NameComponent [] name )
    {
        if ( name.length <= 1 )
            return EMPTY_NAME;

        NameComponent [] ret = new NameComponent[ name.length - 1 ];

        System.arraycopy( name, 0, ret, 0, ret.length );

        return ret;
    }

    /**
     * Get the parent name of the given name. The parent of the empty name
     * will be also empty. It is assumed that the name is in the correct format.
     * 
     * @param name the name.
     * @return the parent name.
     *
     * @deprecated This method uses interfaces from the CosNaming module.
     * Which results in an implicit dependency to the naming service. This
     * dependency is going to be removed in the future. 
     */
    public static String parent( String name )
    {
        try
        {
            return to_string( parent( to_name( name ) ) );
        }
        catch ( Exception ex )
        {
            return "";
        }
    }

}
